/*********************************************************************

  IPXLIB.H

  Header file for IPXLIB.OBJ, and any applications which call IPXLIB

*********************************************************************/

/* Beginning of IPXLIB.H */


#ifndef IPXLIB_H
#define IPXLIB_H


/**********************************************************************
  IPX_farp
    Defines the standard structure of a far pointer for other IPX
    structures.
**********************************************************************/

struct IPX_farp {
    unsigned int offset;  /* offset address of something              */
    unsigned int segment; /* segment (paragraph) address of something */
    };


/**********************************************************************
  IPX_node
    Defines the standard structure of a local network node address.  
    For Ethernet, this will be the full ethernet address.
**********************************************************************/

struct IPX_node {
    unsigned char ch[6];
    };


/**********************************************************************
  IPX_address
    This structure describes a full address, including network number,
      node address, and socket number.
**********************************************************************/

struct IPX_address {
    unsigned long int Network;  /* HI-LO */
    struct IPX_node   Node;     /* HI-LO */
    unsigned int      Socket;   /* HI-LO */
    };


/**********************************************************************
  IPX_work, IPX_dwork
    Basically, we don't have to do anything with these.  They exist.
**********************************************************************/

struct IPX_work {
    unsigned char ch[4];
    };

struct IPX_dwork {
    unsigned char ch[12];
    };


/**********************************************************************
  IPX_frag
    Describes a single fragment of an entire message buffer.
**********************************************************************/

struct IPX_frag {
    struct IPX_farp Address;     /* Address of buffer for fragment */
    unsigned int    Size;        /* Length of fragment in bytes    */
    };


/**********************************************************************
  IPX_ECB
    This is the event control block.  You must fill in some of the
    various fields depending on the particular call you invoke.
**********************************************************************/

struct IPX_ECB {
    struct IPX_farp   Link_Address;      /* used by IPX for chaining */
    struct IPX_farp   ESR_Address;       /* address of ESR routine   */
    unsigned char     In_Use;            /* used for polling ECB     */
    unsigned char     Completion_Code;   /* status of ECB request    */
    unsigned int      Socket_Number;     /* HI-LO socket number      */
    struct IPX_work   Workspace;         /* Don't worry about this   */
    struct IPX_dwork  Driver_Workspace;  /* Don't worry about this   */
    struct IPX_node   Immediate_Address; /* Node address send/recv   */
    unsigned int      Fragment_Count;    /* LO-HI # frag descriptors */
    struct IPX_frag   Fragment_Desc[16]; /* up to 16 frag descripts  */
    };


/***********************************************************************
  IPX_header
    Generally, ECB.Fragment_Desc.Address will point to one of these.
    You must initialize some of the various fields before posting a
    send of listen request.
***********************************************************************/

struct IPX_header {
    unsigned int       Checksum;          /* Not used anymore        */
    unsigned int       Length;            /* Set by IPX              */
    unsigned char      Transport_Control; /* Set by IPX              */
    unsigned char      Packet_Type;       /* 00 for unknown type     */
    struct IPX_address Destination;
    struct IPX_address Source;
    unsigned char      Data[546];         /* Packet data             */
    };


/***********************************************************************
  IPX_Is_Loaded
    Returns true (0x01) if IPX has been loaded, or false (0x00) if
    it has not been.  ALWAYS call this routine before calling any of
    the others.
***********************************************************************/

unsigned int IPX_Is_Loaded (void);


/***********************************************************************
  IPX_Open_Socket
    Opens an IPX socket.
    Return values include:
      0x00: Successful
      0xFE: Socket table is full
      0xFF: Socket is already open
***********************************************************************/

unsigned int IPX_Open_Socket (unsigned int Socket_Number);


/***********************************************************************
  IPX_Close_Socket
    Closes a socket previously opened by IPX_Open_Socket.
***********************************************************************/

void IPX_Close_Socket (unsigned int Socket_Number);


/***********************************************************************
  IPX_Get_Local_Target
    Returns the six-byte node address of the local target.
    The calling routine supplies a full address (network, node, and
      socket) and the file server returns the address of the closest
      router to Target.  The value in Target should be used as the
      Immediate_Address in a subsequent call to IPX_Send_Packet.
    Destination and Target must be within the same segment.
    Return value is the estimated amount of time to deliver the packet
      to the destination.
***********************************************************************/

unsigned int IPX_Get_Local_Target (struct IPX_address far *Destination,
                                   struct IPX_node    far *Target);


/***********************************************************************
  IPX_Send_Packet
    Initiates the sending of an IPX packet
    Calling routine must initialize the following fields:
      ECB.ESR_Address         IPX_header.Packet_Type
      ECB.Socket_Number       IPX_header.Destination
      ECB.Immediate_Address
      ECB.Fragment_Count
      ECB.Fragment_Desc
    ECB.ESR_Address should be set to zero.
    After calling this routine, the application should call
      IPX_Relinquish_Control repeatedly until ECB.In_Use == 0.
      At this point, ECB.Completion_Code will be set to:
        0x00: Successful (sent, not necessarily received)
        0xFC: Send request was canceled
        0xFD: The packet was malformed
        0xFE: The packet was undeliverable
        0xFF: IPX was unable to send the packet due to hardware or
                network failure
***********************************************************************/

void IPX_Send_Packet (struct IPX_ECB far *ECB);


/***********************************************************************
  IPX_Listen_For_Packet
    Prepares IPX to receive a packet, then returns while IPX attempts
      to listen.
    Calling routine must initialize the following fields:
      ECB.ESR_Address
      ECB.Socket_Number
      ECB.Fragment_Count
      ECB.Fragment_Desc
    ECB.ESR_Address points to the address of a routine which is
      called by IPX when a packet is received.  This routine is
      treated as an interrupt service routine, and must be terminated
      with an IRET instruction.  On entrance to the ESR, ES:SI -> the
      associated ECB.  Since the C language is not well suited for
      this type of processing, ECB.ESR_Address should be set to zero,
      which will disable ESR processing.
    After calling this routine, the application should occasionally
      call IPX_Relinquish_Control to ensure that the IPX driver gets
      enough execution time to actually listen for a packet.
    After calling this routine, the application should monitor
      ECB.In_Use.  This field is immediately set to FEh, and remains
      at that value until a packet is received, an error occurs, or
      IPX_Cancel_Event is invoked against the ECB.
***********************************************************************/

void IPX_Listen_For_Packet (struct IPX_ECB far *ECB);


/***********************************************************************
  IPX_Cancel_Event
    Cancel an ECB that is currently In_Use.
***********************************************************************/

void IPX_Cancel_Event (struct IPX_ECB far *ECB);


/***********************************************************************
  IPX_Get_Internetwork_Address
    Returns the network and node address of the workstation.
***********************************************************************/

void IPX_Get_Internetwork_Address (struct IPX_address far *Address);


/***********************************************************************
  IPX_Relinquish_Control
    Gives the IPX driver some CPU time to accomplish what it needs to
    do.
***********************************************************************/

void IPX_Relinquish_Control (void);


/***********************************************************************
  IPX_Flipword
    Reverses the order of bytes within a two-byte unsigned int.
***********************************************************************/

unsigned int IPX_Flipword (unsigned int Inword);


/***********************************************************************
  IPX_Fliplong
    Reverses the order of the bytes within a 4-byte sequence of type
    unsigned long.
***********************************************************************/

void IPX_Fliplong (unsigned long *Inlong);

#endif

/* End of IPXLIB.H */

