////////////////////////////////////////////////////////////
//
//	ImgDLL - a Win32 Image File Processing DLL
//
//	Copyright 1997, Smaller Animals Software
//
//	Author 
//		Chris Losinger, smallest@smalleranimals.com
//		http://www.smalleranimals.com/index.htm
//
//		This DLL provides JPG, PNG, TIFF and BMP read and write support.
//	Also included are routines to do various image processing
//	functions such as 24-bit to 8-bit color quantization and
//	gray-scaling, resizing, sharpening and blurring, 
//	DWORD-alignment utilities, image size querys, etc..
//
//	Note : GIF support has been disabled as of 9/23/97. If you
//  absolutely need GIF support, I may be able to help you. Ask.
//
//		This DLL is freely distributable. The full-access key is not.
//	If you have a full-access key without paying for it, you are
//	using this without permission, and that is illegal. I hope you 
//	feel bad.
//	
//
//	Tech Credit:
//
//		Color quantization code modified from DL1Quant, by Dennis Lee 
//			(denlee@ecf.utoronto.ca)
//
//		JPEG code from IJG v6a.
//
//		GIF code from "Programming for Graphics Files" by John Levine.
//
//		All other code written by Smaller Animals Software, Chris Losinger.
//
//
//	Usage notes :
//
//		0)	DLL must be initialized with ImgDLLInitDLL. the value passed to
//	this function may be the key to unlock the full version of the DLL.
//	Registered users will be given the real key. Everyone else is free to
//	try any value they like.
//
//		1)	all strings are to be zero-terminated - some languages don't do this
//	by default. 
//
//		2)	UINT is a 32-bit unsigned integer. this works as a "long" in VB. 
//	VB's "Integer" is too small!!
//
//		3)	BOOL is a 32-bit unsigned integer. zero for FALSE, non-zero for TRUE.
//
//		4)	RGBQUAD is a structure defined as follows :
//
//		typedef struct tagRGBQUAD { 
//			BYTE    rgbBlue; 
//			BYTE    rgbGreen; 
//			BYTE    rgbRed; 
//			BYTE    rgbReserved; 		  // rarely used
//		} RGBQUAD; 
//
//		5)	"int" is a 32-bit signed integer.
//
//		6)	HGLOBAL is a HANDLE to memory allocated by GlobalAlloc.
//	If a routine returns a non-NULL HGLOBAL value, it is up to
//	the caller to call GlobalFree to release this memory back
//	to the system.
//
//		7)  RGB means 24-bit (3 BYTEs per pixel)  Red,Green,Blue, one BYTE each.
//	If you have an RGB buffer of an image that is 100 x 100 pixels, you
//	MUST have a buffer of 100 x 3 x 100 BYTEs!! All calls that says RGB
//	assume this arrangement. RGB is the same as "packed RGB".
//	
//		8)	double designates a 64-bit floating-point number.
//
//		9)	"colormapped 8-bit" means an image which is made up of 8-bit pixels
//	each of which is an index into a palette of 24-bit RGB values. 
//	a "colormapped 8-bit" buffer is not the same as a 24-bit (RGB) buffer.
//
////////////////////////////////////////////////////////////

#ifndef IMGDLLH
#define IMGDLLH

#define IMGEXP FAR PASCAL

////////////////////////////////////////////////////////////
// error codes

#define TRIALVERSION	-1		// LIB was not initialized with a registered key

#define IMGOK			0		// no err
#define MEMERR			1		// out of mem
#define FILEOPENERR		2		// error on file open
#define FILEREADERR		3		// error on file read
#define FILEWRITEERR	4		// error on file write
#define BADPARAMERR		5		// bad user param
#define INVALIDBMPERR	6		// bad BMP file
#define BMPRLEERR		7		// we don't do compressed (RLE) BMP files
#define INVALIDGIFERR	8		// bad GIF file
#define INVALIDJPGERR	9		// bad JPG file
#define IMGDCERR		10		// error with device context
#define IMGDIBERR		11		// problem with a GetDIBits call
#define NOGIFERR		12		// GIF support disabled
#define IMGNORESOURCE	13		// resource not found
#define CALLBACKCANCEL	14		// callback returned FALSE - operation aborted
#define INVALIDPNGERR	15		// bad PNG file
#define PNGCREATEERR	16		// internal PNG lib behavior - contact smaller animals s.w.
#define IMGDLLINTERNAL	17		// misc unexpected behavior error - contact smaller animals s.w.
#define IMGFONTERR		18		// trouble creating a font object
#define INTTIFFERR		19		// misc internal TIFF error
#define	INVALIDTIFFERR	20		// invalid TIFF file
#define TIFFLZWNOTSUPPORTED	21	// this will not read TIFF-LZW iamges
#define INVALIDPCXERR	22		// invalid PCX image

#define IMGDLL_LOCK_KEY	0		// key to unlock DLL provided to registered
								// users only. any other key will prevent the DLL
								// from offering full functionality.

// channelMask values : e.g.
//	BYTE mask = CHRED | CHGREEN;// use only red and green values in this operation

#define CHRED		1			// include RED channel if this is set in channelMask
#define CHGREEN		2
#define CHBLUE		4

#ifdef __cplusplus
extern "C" {
#endif

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLStringInitDLL(const char * initString)
//
//	Purpose :
//		Perform DLL initialization / unlock. This is intended
//	for VB people who cannot use 32-bit unsigned ints. You will
//	need to generate a string representation of the numeric key.
//
//	e.g. if the key is 123456789, you will pass the 
//	string "123456789" as the argument.
//
//	This must be a string of the decimal representation (not hex, octal or
//	binary) of the key!!!!
//
//
//	Param				Type			Use
//	initString			const char *	can be used to unlock DLL 
//										functionality. see value
//										of LOCK_KEY in imgdll.h
//						
//										users who have paid for the DLL
//										will be given a value for LOCK_KEY
//										that will allow them to use all
//										of the DLLs functions without limits.
//
//	Note :
//	This call should only be called once per session. All calls
//	after the first will have no effect. That is, you can't put this
//	in a loop and try new initVal values until you find the right one.
//
//	Besides unlocking the DLL, this performs other useful initialization
//	functions.
//
//	Note :
//
//	Return
//		none
//
////////////////////////////////////////////////////////////

void IMGEXP ImgDLLStringInitDLL(const char * initString);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLInitDLL(UINT initVal)
//
//	Purpose :
//		Perform DLL initialization / unlock.
//
//	Param				Type			Use
//	initVal				UINT			can be used to unlock DLL 
//										functionality. see value
//										of LOCK_KEY in imgdll.h
//						
//										users who have paid for the DLL
//										will be given a value for LOCK_KEY
//										that will allow them to use all
//										of the DLLs functions without limits.
//
//	Note :
//	This call should only be called once per session. All calls
//	after the first will have no effect. That is, you can't put this
//	in a loop and try new initVal values until you find the right one.
//
//	Besides unlocking the DLL, this performs other useful initialization
//	functions.
//
//	Note :
//	A UINT is an "unsigned int". 32 bits. 4,294,967,296 possible values.
//
//	Return
//		none
//
////////////////////////////////////////////////////////////

void IMGEXP ImgDLLInitDLL(UINT initVal);

////////////////////////////////////////////////////////////
//
//	int IMGEXP ImgDLLGetLastError()
//
//	Purpose :
//		query the last error value
//
//	Param				Type			Use
//	none
//
//	Return
//		int value of error - see above for values and meaning
//
////////////////////////////////////////////////////////////

int IMGEXP ImgDLLGetLastError();

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLSetJPGErrorMsgBox(BOOL enable)
//
//	Enable JPG error message boxes
//
//	Param				Type			Use
//	enable				BOOL			TRUE, enable modal msg boxes, else 
//										disable
//
//
//	Return
//		none
//
////////////////////////////////////////////////////////////

void IMGEXP ImgDLLSetJPGErrorMsgBox(BOOL enable);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLSetJPGDCT(UINT dct)
//
//	Switch the compression method : dct is one of :
//
//	0 - default (usually JDCT_ISLOW)
//	1 - JDCT_ISLOW: slow but accurate integer algorithm
//	2 - JDCT_IFAST: faster, less accurate integer method
//	3 - JDCT_FLOAT: floating-point method
//
//	It is not necessary to use this function to be able save JPG files.
//	This is only for people who need more speed or higher quality
//	in their output.
//
//	All JPG files written will be written using this setting.
//	
//	Param		Type			Use
//	dct			UINT			which level to set
//
//	Return
//		none
//
////////////////////////////////////////////////////////////
void IMGEXP ImgDLLSetJPGDCT(UINT dct);

/////////////////////////////////////////////////////////////////////////////
// callback type definition
//
//	ex.  this prints the row #'s, as they are processed, to the debug window
//
//	BOOL CALLBACK functionName(UINT curRow, UINT totalRows)
//	{
//		TRACE2("cur row : %u    total rows : %u\n",curRow, totalRows);
//		return TRUE;
//	}
//
//	...
//		ImgDLLSetCallback(functionName);
//	...

typedef BOOL (CALLBACK* CALLBACKPTR)(UINT uCurRow,					// current row
									 UINT uHeight);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLSetCallback(CALLBACKPTR pCallback);
//
//	Purpose :
//	The DLL will call this function once for every scan line
//	read or written to a JPG, PNG or BMP file. if this function returns FALSE
//	the operation aborts and ImgDLLGetLastError will return CALLBACKCANCEL
//
//	you can use this to monitor the progress of read / write operations.
//
//	It is not neccessary for you to use this function. If you do not
//	set the callback function or pass a NULL to this call, no callback
//	will be performed.
//
//	Param				Type			Use
//	pCallback			CALLBACKPTR		ptr to a function defined to 
//										accept 2 UINTs and return a BOOL.
//	Return
//		none
//
////////////////////////////////////////////////////////////

void IMGEXP ImgDLLSetCallback(CALLBACKPTR pCallback);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveRGBToJPG(const char * fileName,
//						BYTE * pBuf,
//						UINT uWidthPix,
//						UINT uHeight,
//						UINT uQuality,
//						BOOL bColor)
//
//	Purpose :
//		Save an RGB buffer to a JPG file . 
//		Buffer must be packed RGB.
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//	uQuality			UINT			quality from 1-100
//	bColor				BOOL			FALSE = Grayscale
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveRGBToJPG(const char * fileName,
						BYTE * pBuf,
						UINT uWidthPix,
						UINT uHeight,
						UINT uQuality,
						BOOL bColor);

////////////////////////////////////////////////////////////
//
//BOOL IMGEXP ImgDLLGrayscaleToJPG(const char * fileName, 
//							BYTE *dataBuf,
//							UINT widthPix,
//							UINT height,
//							int quality);
//
//	Purpose :
//		Save an 8-bit grayscale buffer to a JPG file . 
//		Buffer must be 8 bit data. 0 for black 255 for white.
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//	uQuality			UINT			quality from 1-100
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLGrayscaleToJPG(const char * fileName, 
							BYTE *dataBuf,
							UINT widthPix,
							UINT height,
							int quality);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLRead8bitGrayscaleFromJPG(const char * fileName,
//							 UINT *width,
//							 UINT *height)
//
//
//	Purpose :
//		Read a JPG file to a buffer of width * height bytes.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLRead8bitGrayscaleFromJPG(const char * fileName,
							 UINT *width,
							 UINT *height);

////////////////////////////////////////////////////////////
//
//	UINT IMGEXP ImgDLLSaveRGBToJPGMem(BYTE * outBuf,
//										UINT outBufSize,
//										BYTE * inBuf,
//										UINT uWidthPix,
//										UINT uHeight,
//										UINT uQuality,
//										BOOL bColor)
//
//	Purpose :
//		Save an RGB buffer to a JPG image in memory. Output
//		buffer must have been previously allocated by the 
//		caller. For safety, make this at least (width * height * 3) BYTEs.
//
//		Input buffer must be packed RGB.
//
//
//	Param				Type			Use
//	outBuf				BYTE *			memory at which to write the JPG image
//	outBufSize			UINT			size of output buffer in BYTEs
//	pBuf				BYTE *			input image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//	uQuality			UINT			quality from 1-100
//	bColor				BOOL			FALSE = Grayscale
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		UINT representing number of bytes actually written to memory.
//
////////////////////////////////////////////////////////////


UINT IMGEXP ImgDLLSaveRGBToJPGMem(BYTE * outBuf,
									UINT outBufSize,
									BYTE * inBuf,
									UINT uWidthPix,
									UINT uHeight,
									UINT uQuality,
									BOOL bColor);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLReadRGBFromJPG(const char * fileName,
//							 UINT *width,
//							 UINT *height)
//
//
//	Purpose :
//		Read a JPG file to a buffer of RGB bytes.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////
HGLOBAL IMGEXP ImgDLLReadRGBFromJPG(const char * fileName,
							 UINT *width,
							 UINT *height);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLReadRGBFromJPGMem(BYTE * inBuf,   
//											UINT inBufSize,
//											UINT *width,
//											UINT *height)
//
//
//	Purpose :
//		Read a JPG memory stream to a buffer of RGB bytes.
//
//	Param				Type			Use
//	inBuf				BYTE *			buffer containing JPG file image
//	inBufSize			UINT			size of input buffer
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLReadRGBFromJPGMem(BYTE * inBuf,   
											UINT inBufSize,
											UINT *width,
											UINT *height);

////////////////////////////////////////////////////////////
//
//	HBITMAP IMGEXP ImgDLLReadHBITMAPFromJPG(const char * fileName,
//											HPALETTE hPal)
//
//
//	Purpose :
//		Read a JPG file to an HBITMAP - if hPal is non-NULL and the
//	display is at <= 256 colors, hPal is used to render the image. 
//	if NULL and display is <= 256 colors, the system palette will be used 
//	instead.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	hPal				HPALETTE		palette
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HBITMAP - the new bitmap!
//		
//		be sure to call DeleteObject(..) on this when you're done!
//
////////////////////////////////////////////////////////////

HBITMAP IMGEXP ImgDLLReadHBITMAPFromJPG(const char * fileName,
										HPALETTE hPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveRGBToBMP24(const char * fileName,
//						BYTE * pBuf,
//						UINT uWidthPix,
//						UINT uHeight)
//
//	Purpose :
//		Save an RGB buffer to a 24-bit BMP file . 
//		Buffer must be packed RGB (not DWORD aligned), 
//		vertically oriented as viewed.
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveRGBToBMP24(const char * fileName,
						BYTE * pBuf,
						UINT uWidthPix,
						UINT uHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveColormappedToBMP(const char * fileName,
//									BYTE * pBuf,
//									UINT uWidthPix,
//									UINT uHeight,
//									UINT uBPP,
//									UINT uColors,
//									RGBQUAD *rgbPalette)
//
//
//	Purpose :
//		Save a colormapped buffer to a colormapped BMP file . 
//		Buffer must be colormapped, 8-bit per pixel,
//		vertically oriented as viewed. 
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//	uBPP				UINT			bites per pixel to save as
//	uColors				UINT			colors in palette
//	rgbPalette			RGBQUAD *		palette array
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////


BOOL IMGEXP ImgDLLSaveColormappedToBMP(const char * fileName,
									BYTE * pBuf,
									UINT uWidthPix,
									UINT uHeight,
									UINT uBPP,
									UINT uColors,
									RGBQUAD *rgbPalette);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveDIBColormappedToBMP(const char * fileName,
//											BYTE * pBuf,
//											BITMAPINFO * pBMI,
//											RGBQUAD *colorMap)
//
//
//	Purpose :
//		Save a colormapped buffer to a colormapped BMP file . 
//		Buffer must be in DIB format - 1,4,8 or 24 bits.
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	pBMI				BITMAPINFO *	filled in BITMAPINFO
//	colorMap			RGBQUAD *		palette - ognored for 24-bit DIBs
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveDIBColormappedToBMP(const char * fileName,
											BYTE * pBuf,
											BITMAPINFO * pBMI,
											RGBQUAD *colorMap);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLReadRGBFromBMP(const char * fileName,
//							 UINT *width,
//							 UINT *height)
//
//
//	Purpose :
//		Read a BMP file to a buffer of RGB bytes. Does not read
//	RL-encoded BMP files.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////


HGLOBAL IMGEXP ImgDLLReadRGBFromBMP(const char * fileName,
									 UINT *width,
									 UINT *height);

////////////////////////////////////////////////////////////
//
//	HBITMAP IMGEXP ImgDLLReadHBITMAPFromBMP(const char * fileName)
//
//
//	Purpose :
//		Read a BMP file to a HBITMAP.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HBITMAP - the new bitmap!
//		
//		be sure to call DeleteObject(..) on this when you're done!
//
////////////////////////////////////////////////////////////

HBITMAP IMGEXP ImgDLLReadHBITMAPFromBMP(const char * fileName);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLGetJPGDimensions(const char * fileName,
//							UINT *width,
//							UINT *height)
//
//	Purpose :
//		Query a JPG file for image dimensions
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	width				UINT*			width, in pixels
//	height				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////


BOOL IMGEXP ImgDLLGetJPGDimensions(const char * fileName,
							UINT *width,
							UINT *height);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLGetBMPDimensions(const char * fileName,
//							UINT *width,
//							UINT *height)
//
//	Purpose :
//		Query a BMP file for image dimensions
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	width				UINT*			width, in pixels
//	height				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLGetBMPDimensions(const char * fileName,
							UINT *width,
							UINT *height);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLRGBFromDWORDAligned(BYTE *inBuf,
//									 UINT widthPix,
//								 	UINT widthBytes,
//									UINT height)
//
//	Purpose :
//		Create a packed RGB buffer from a DWORD aligned buffer
//
//	Param				Type			Use
//	inBuf				BYTE *			input buffer
//	uWidthPix			UINT			width, in pixels
//	uWidthBytes			UINT			width, in bytes	of input buffer
//	uHeight				UINT			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to the buffer
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////


HGLOBAL IMGEXP ImgDLLRGBFromDWORDAligned(BYTE *inBuf,
									UINT widthPix,
								 	UINT widthBytes,
									UINT height);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLDWORDAlignBuf(BYTE *dataBuf,
//										UINT widthPix,
//								 		UINT height,
//										UINT *uiOutWidthBytes)
//
//	Purpose :
//		Create a DWORD-aligned buffer from a packed RGB buffer
//
//	Param				Type			Use
//	dataBuf				BYTE *			input buffer
//	WidthPix			UINT			width, in pixels
//	height				UINT			height 
//	uiOutWidthBytes		UINT *			width, in bytes, of new buffer
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to the buffer
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////


HGLOBAL IMGEXP ImgDLLDWORDAlignBuf(BYTE *dataBuf,
									 UINT widthPix,		// pixels!!
									 UINT height,
									 UINT *uiOutWidthBytes);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLDWORDAlignBuf(BYTE *dataBuf,
//										UINT widthBytes,
//								 		UINT height,
//										UINT *uiOutWidthBytes)
//
//	Purpose :
//		Create a DWORD-aligned buffer from a buffer
//
//	Param				Type			Use
//	dataBuf				BYTE *			input buffer
//	WidthBytes			UINT			width, in bytes
//	height				UINT			height 
//	uiOutWidthBytes		UINT *			width, in bytes, of new buffer
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to the buffer
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLDWORDAlignBufBytes(BYTE *dataBuf,
									 UINT widthBytes,		// bytes!!
									 UINT height,
									 UINT *uiOutWidthBytes);// bytes!!!

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLVerticalFlipBuf(BYTE  * inbuf, 
//										UINT widthBytes,
//								 		UINT height)
//
//	Purpose :
//		Vertically flip buffer
//
//	Param				Type			Use
//	inbuf				BYTE *			input buffer
//	widthBytes			UINT			width, in bytes
//	height				UINT			height 
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLVerticalFlipBuf(BYTE  * inbuf, 
					   UINT widthBytes, 
					   UINT height);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLBGRFromRGB(BYTE *buf, UINT widthPix, UINT height)
//
//	Purpose :
//		Swap red and blue values in a buffer
//
//	Param				Type			Use
//	dataBuf				BYTE *			buffer
//	WidthPix			UINT			width, in pixels
//	height				UINT			height 
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////


BOOL IMGEXP ImgDLLBGRFromRGB(BYTE *buf, UINT widthPix, UINT height);
////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSetAlphaChannelToImage(BYTE *destRGBA,
//										UINT w,
//										UINT h,
//										BYTE *alphaImage);
//
//	Purpose :
//		Fill in the bytes of the alpha channel in an RGBA image.
//
//	Param			Type			Use
//	destRGBA		BYTE *			32-bit RGBA image to receive new alpha data
//	w				UINT			width in pixels
//	h				UINT			height
//	alphaImage		BYTE *			8-bit image that will become the alpha channel
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSetAlphaChannelToImage(BYTE *destRGBA,
										UINT w,
										UINT h,
										BYTE *alphaImage);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP  ImgDLLRGBAFromRGB24(BYTE *inBuf,
//							UINT widthPix,
//							UINT height,
//							BYTE defAlpha)
//
//	Purpose :
//		Given a 24-bit RGB image, create a 32-bit RGBA image.
//	The alpha channel will be set to defAlpha.
//
//	Param			Type			Use
//	inBuf			BYTE *			24-bit RGB image
//	widthPix		UINT			width in pixels
//	height			UINT			height
//	defAlpha		BYTE 			default value for all aplha BYTEs
//
//	Return
//		HGLOBAL to new buffer
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLRGBAFromRGB24(BYTE *inBuf,
							UINT widthPix,
							UINT height,
							BYTE defAlpha);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLMake8BitGrayScale(BYTE *inBuf, 
//									UINT widthPix, 
//									UINT height,
//									BYTE *outBuf)
//
//	Purpose :
//		convert RGB to 8-bir grayscale uses luminance formula.
//
//	Param				Type			Use
//	inBuf				BYTE *			input RGB buffer
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	outBuf				BYTE *			output 8-bit buffer
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////
BOOL IMGEXP ImgDLLMake8BitGrayScale(BYTE *inBuf, 
									UINT widthPix, 
									UINT height, 
									BYTE *outBuf);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLMakeGrayScale(BYTE *buf, UINT widthPix, UINT height)
//
//	Purpose :
//		convert RGB to GGG (all three bytes the same). 
//	uses luminance formula.
//
//	Param				Type			Use
//	dataBuf				BYTE *			buffer
//	WidthPix			UINT			width, in pixels
//	height				UINT			height 
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLMakeGrayScale(BYTE *buf, UINT widthPix, UINT height);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLHorizontalFlipRGB(BYTE  * inbuf, 
//										UINT widthPix,
//								 		UINT height)
//
//	Purpose :
//		Horizontally flip a buffer of RGB pixels
//
//	Param				Type			Use
//	dataBuf				BYTE *			input buffer
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to the buffer
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////


BOOL IMGEXP ImgDLLHorizontalFlipRGB(BYTE  * inbuf, 
									   UINT widthPix, 
									   UINT height);

////////////////////////////////////////////////////////////
//
//	BOOL IMGDLLEXP ImgDLLResizeRGB2(BYTE * rgbIn,
//							UINT inWidth,
//							UINT inHeight,
//							BYTE *rgbOut,
//							UINT outWidth,
//							UINT outHeight)
//
//	Purpose :
//		given an RGB image, resize it to the dimensions specified.
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	inWidth				UINT			input width
//	inHeight			UINT			input height
//	rgbOut				BYTE *			buffer to receive image
//	outWidth			UINT			output height
//	outHeight			UINT			output width
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLResizeRGB2(BYTE * rgbIn,
							UINT inWidth,
							UINT inHeight,
							BYTE *rgbOut,
							UINT outWidth,
							UINT outHeight);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLResizeRGB(BYTE * rgbIn,
//							UINT inWidth,
//							UINT inHeight,
//							UINT outWidth,
//							UINT outHeight)
//
//	Purpose :
//		given an RGB image, resize it to the dimensions specified.
//		return a handle to the new image.
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	inWidth				UINT			input width
//	inHeight			UINT			input height
//	outWidth			UINT			output height
//	outHeight			UINT			output width
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLResizeRGB(BYTE * rgbIn,
							UINT inWidth,
							UINT inHeight,
							UINT outWidth,
							UINT outHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSharpenRGB(BYTE * rgbIn,
//							BYTE * rgbOut,
//							UINT widthPix,
//							UINT height,
//							UINT sharpLevel)
//
//
//	Purpose :
//		sharpen an RGB buffer into a different RGB buffer. 3 x 3 matrix
//
//		buffers must be the same size!!!
//
//		rgbIn must not be the same as rgbOut !!!
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	rgbOut				BYTE *			24-bit RGB image data
//	widhtPix			UINT			width in pixels
//	height				UINT			height
//	sharpLevel			UINT			0..100 sharpening intensity
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSharpenRGB(BYTE * rgbIn,
							BYTE * rgbOut,
							UINT widthPix,
							UINT height,
							UINT sharpLevel);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLBlurRGB(BYTE * rgbIn,
//							BYTE * rgbOut,
//							UINT widthPix,
//							UINT height,
//							UINT blurLevel)
//
//
//	Purpose :
//		blur an RGB buffer into a different RGB buffer. 3 x 3 matrix
//
//		buffers must be the same size!!!
//
//		rgbIn must not be the same as rgbOut !!!
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	rgbOut				BYTE *			24-bit RGB image data
//	widhtPix			UINT			width in pixels
//	height				UINT			height
//	blurLevel			UINT			0..100 blur intensity
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLBlurRGB(BYTE * rgbIn,
							BYTE * rgbOut,
							UINT widthPix,
							UINT height,
							UINT blurLevel);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLQuantizeRGBTo8Bit(BYTE *inBuf,
//										BYTE *outBuf,
//										UINT widthPix,
//										UINT height,
//										UINT colors,
//										RGBQUAD *pPal)
//
//	Purpose :
//		generate a palette and a dithered image with the specified
//	number of colors.
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	outBuf				BYTE *			output image
//	widthPix			UINT 			width
//	height				UINT 			height
//	colors				UINT			colors to reduce to	(1..256)
//	pPal				RGBQUAD *		ptr to an array of RGBQUADs. 
//										there must be at least "colors" 
//										entries. this receives the
//										new palette.
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLQuantizeRGBTo8Bit(BYTE *inBuf,
										BYTE *outBuf,
										UINT widthPix,
										UINT height,
										UINT colors,
										RGBQUAD *pPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLGet8BitPalette(BYTE *inbuf, 
//									UINT nWidth, 
//									UINT nHeight, 
//									UINT nColors,
//									RGBQUAD *pPal);
//
//	Purpose :
//		generate a palette with the specified number of colors. does not
//	generate an output image
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	widthPix			UINT 			width
//	height				UINT 			height
//	colors				UINT			colors to reduce to	(1..256)
//	pPal				RGBQUAD *		ptr to an array of RGBQUADs. there must be at
//										least "colors" entries. this receives the
//										new palette.
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLGet8BitPalette(BYTE *inbuf, 
									UINT nWidth, 
									UINT nHeight, 
									UINT nColors,
									RGBQUAD *pPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLRGBFrom8Bit(BYTE *inBuf,
//								BYTE *outBuf,
//								UINT widthPix,
//								UINT height,
//								UINT colors,
//								RGBQUAD *pPal)
//
//	Purpose :
//		apply a palette to an 8-bit image to generate an equivalent
//	24-bit image
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	outBuf				BYTE *			output image
//	widthPix			UINT 			width
//	height				UINT 			height
//	colors				UINT			colors in palette
//	pPal				RGBQUAD *		ptr to an array of RGBQUADs. there must be at
//										least "colors" entries.
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLRGBFrom8Bit(BYTE *inBuf,
								BYTE *outBuf,
								UINT widthPix,
								UINT height,
								UINT colors,
								RGBQUAD *pPal);


////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLDIBToRGB( BYTE * inBuf,
//									BITMAPINFOHEADER *pBMI,
//									RGBQUAD * colorMap)
//	Purpose :
//		given an image in DIB format, create a 24-bit RGB buffer
//
//	Param			Type				Use
//	inBuf			BYTE *				DI-Bits
//	pBMI			BITMAPINFOHEADER *	a filled-in BMIH
//	colorMap		RGBQUAD *			space large enough to hold a palette
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLDIBToRGB( BYTE * inBuf,
								BITMAPINFOHEADER *pBMI,
								RGBQUAD * colorMap);


////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLDIBToRGB2(BYTE * pDIB,
//								UINT *width, 
//								UINT *height)
//
//	Purpose :
//		given a DIB, create an RGB buffer, using the 
//	screen device driver to generate 1,4,8 bits per pixel images.
//	all returned in 24-bit buffers!!
//
//	Param				Type			Use
//	inhBmp				BYTE *			BYTE ptr to DIB 
//	width				UINT *			width
//	height				UINT *			height
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLDIBToRGB2(BYTE * pDIB,
								UINT *width, 
								UINT *height);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLRGBToDIB( BYTE * inBuf,
//								UINT widthPix,
//								UINT height)
//	Purpose :
//		given a 24-bit RGB image, create a 24-bit DIB. 
//
//	Param				Type			Use
//	inBuf				BYTE *			RGB buffer
//	widthPix			UINT			buffer width in pixels
//	height				UINT			height
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////
HGLOBAL IMGEXP ImgDLLRGBToDIB( BYTE * inBuf,
								UINT widthPix,
								UINT height);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLHBITMAPToRGB(HBITMAP inhBmp, 
//								  UINT *width, 
//								  UINT *height)
//
//	Purpose :
//		given an HBITMAP create an RGB buffer, using the 
//	screen device driver to generate 1,4,8 bits per pixel images.
//	all returned in 24-bit buffers!!
//
//	Param				Type			Use
//	inhBmp				HBITMAP			handle to bitmap
//	width				UINT *			width
//	height				UINT *			height
//	hPal				HPALETTE		palette to use in rendering the image.
//										if NULL, the system palette is used
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLHBITMAPToRGB(HBITMAP inhBmp, 
								  UINT *width, 
								  UINT *height,
								  HPALETTE hPal);

////////////////////////////////////////////////////////////
//
//	HBITMAP IMGEXP ImgDLLRGBToHBITMAP( BYTE * inBuf,
//								UINT widthPix,
//								UINT height,
//								HPALETTE hPal)
//	Purpose :
//		given a 24-bit RGB image, create a DIB. 
//
//	if hPal is non-NULL and the display is at <= 256 colors, 
//	hPal is used to render the image. if NULL and display is <= 
//	256 colors, the system palette will be used instead.
//
//	Param				Type			Use
//	inBuf				BYTE *			RGB buffer
//	widthPix			UINT			buffer width in pixels
//	height				UINT			height
//	hPal				HPALETTE		palette
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////
HBITMAP IMGEXP ImgDLLRGBToHBITMAP( BYTE * inBuf,
								UINT widthPix,
								UINT height,
								HPALETTE hPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawHBITMAP(HDC outHDC, 
//								HBITMAP hBmp,
//								int xPos,
//								int yPos,
//								UINT outWidth,
//								UINT outHeight,
//								HPALETTE hPal)
//
//	Purpose :
//		given an HBITMAP and an HDC, draw the image to the DC. this
//	is just a quick way to do the usual CreateCompatibleDC, SelectObject,
//	BitBlt stuff.
//
//	Param				Type			Use
//	outHDC				HDC				device context to draw into
//	hBmp				HBITMAP			handle to bitmap
//	xPos				int				X position in DC to draw at
//	yPos				int				Y position in DC to draw at
//	outWidth			UINT 			width of dest rect
//	outHeight			UINT 			height of dest rect
//	hPal				HPALETTE		palette to use in rendering the image.
//										if NULL, the system palette is used
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawHBITMAP(HDC outHDC, 
								HBITMAP hBmp,
								int xPos,
								int yPos,
								UINT outWidth,
								UINT outHeight,
								HPALETTE hPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawRGB(HDC outDC,
//							BYTE *pRGB,
//							UINT inWidth,
//							UINT inHeight,
//							int xPos,
//							int yPos,
//							UINT outWidth,
//							UINT outHeight,
//							HPALETTE hPal)
//	Purpose :
//		given a 24-bit RGB image and an HDC, draw the image to the DC.
//	this call wraps a call to ImgDLLRGBToHBITMAP as well as all the
//	usual CreateCompatibleDC, SelectObject, BitBlt stuff
//
//	Param				Type			Use
//	outHDC				HDC				device context to draw into
//	pRGB				BYTE *			RGB (24-bit) image
//	inWidth				UINT			width of RGB image
//	inHeight			UINT			height of RGB image
//	xPos				int				X position in DC to draw at
//	yPos				int				Y position in DC to draw at
//	outWidth			UINT 			width of dest rect
//	outHeight			UINT 			height of dest rect
//	hPal				HPALETTE		palette to use in rendering the image.
//										if NULL, the system palette is used
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawRGB(HDC outDC,
							BYTE *pRGB,
							UINT inWidth,
							UINT inHeight,
							int xPos,
							int yPos,
							UINT outWidth,
							UINT outHeight,
							HPALETTE hPal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawRGB2(HDC outDC,
//							BYTE *pRGB,
//							UINT inWidth,
//							UINT inHeight,
//							int xPos,
//							int yPos,
//							UINT outWidth,
//							UINT outHeight,
//							HPALETTE hPal)
//	Purpose :
//		given an HBITMAP and an HDC, draw the image to the DC.
//	this uses StretchDIBBits. 
//	this is 20-30 % faster than ImgDLLDrawRGB(), but it can't
//	use a palette.
//
//	NOTE: using this function, an output size different than the
//	input size results in a scaled image. the scaling operation
//	produces images which are acceptable for most purposes, but
//	are not as good as you would get from using ResizeRGB.
//
//	this is different behavior than ImgDLLDrawRGB, which clips 
//	the image, if necessary.
//
//	Param				Type			Use
//	outHDC				HDC				device context to draw into
//	pRGB				BYTE *			RGB (24-bit) image
//	inWidth				UINT			width of RGB image
//	inHeight			UINT			height of RGB image
//	xPos				int				X position in DC to draw at
//	yPos				int				Y position in DC to draw at
//	outWidth			UINT 			width of dest rect
//	outHeight			UINT 			height of dest rect
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawRGB2(HDC outDC,
							BYTE *pRGB,
							UINT inWidth,
							UINT inHeight,
							int xPos,
							int yPos,
							UINT outWidth,
							UINT outHeight);

/////////////////////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawTransparentHBITMAP(HDC hdc, 
//										 HBITMAP hBitmap, 
//										 int xStart,
//										 int yStart, 
//										 COLORREF cTransparentColor)
//
//
//	Description:
//	Draws a bitmap with one color "transparent"
//	Code taken almost exactly from Microsoft Developer's Network
//	October 1995.
//	
//	parameter			type			use
//	---------			----			-----
//	hdc					HDC				handle to a DC to draw into
//	hBitmap				HBITMAP			handle to a BMP
//	xPos				int				xpos
//	ypos				int				ypos
//	cTransparentColor	COLORRREF		color to use as trasparent
//
//	return - TRUE on success
//
/////////////////////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawTransparentHBITMAP(HDC outDC, 
								HBITMAP hBmp,
								int xPos,
								int yPos,
								COLORREF cTransparentColor);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawTranparentRGB(HDC outDC,
//							BYTE *pRGB,
//							UINT inWidth,
//							UINT inHeight,
//							int xPos,
//							int yPos,
//							COLORREF cTransparentColor)
//	Purpose :
//		given an RGB image and an HDC, draw the image to the DC with
//	one color transparent. this call wraps a call to ImgDLLRGBToHBITMAP 
//	and a call to ImgDLLDrawTransparentHBITMAP
//
//	Param				Type			Use
//	outHDC				HDC				device context to draw into
//	pRGB				BYTE *			RGB (24-bit) image
//	inWidth				UINT			width of RGB image
//	inHeight			UINT			height of RGB image
//	xPos				int				X position in DC to draw at
//	yPos				int				Y position in DC to draw at
//	cTransparentColor	COLORREF		color to use as trasparent
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		NULL on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawTranparentRGB(HDC outDC,
										BYTE *pRGB,
										UINT inWidth,
										UINT inHeight,
										int xPos,
										int yPos,
										COLORREF cTransparentColor);

////////////////////////////////////////////////////////////
//
//	HBITMAP IMGEXP ImgDLLLoadResourceBMP(HINSTANCE hInstance,
//									LPCSTR lpString, 
//									HPALETTE hInPalette);
//
//	use ImgDLLLoadResourceBMP, instead of CBitmap::LoadBitmap!
//
//	on 8-bit displays, LoadBitmap maps the bitmap to the 16-colors
//	of the standard Windows palette - this is no good. you don't even
//	get a chance to use a palette on the image because LoadBitmap mangles
//	the color info!! yuck!!
//
//	if you use this function, with a palette that represents a
//  spread of colors from the image, or even just a nice spread of
//	colors all across the spectrum, you'll get much better results.
//	trust me. 
//	
//	if you use a NULL palette, this function will use a 
//	set of colors from the system palette, which will give slightly
//	better results than LoadBitmap.
//
//	Param			type			use
//	hInstance		HINSTANCE		app instance to find a resource in
//	lpString		LPCSTR			resource name, or ID
//	hInPalette		HPALETTE		palette to map the bitmap through, may be NULL
//
//	returns an HBITMAP, else NULL
//
//	be sure to call DeleteObject(..) on this when you're done!
//
////////////////////////////////////////////////////////////

HBITMAP IMGEXP ImgDLLLoadResourceBMP(HINSTANCE hInstance,
									LPCSTR lpString, 
									HPALETTE hInPalette);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLApplyMatrixToRGB(BYTE * rgbIn,
//							BYTE * rgbOut,
//							UINT widthPix,
//							UINT height,
//							BYTE channelMask,
//							BOOL zero,
//							BOOL bDoFeedback,
//							double dLevel,
//							double *pMatrix)
//
//	Purpose :
//		apply a convolution matrix to an RGB image.
//
//		buffers must be the same size!!!
//		rgbIn must not be the same as rgbOut !!!
//
//		the matrix is an array of _9_ doubles, arranged as :
//
//			0  1  2
//			3  4  5
//			6  7  8
///
//			where "4" is the source pixel.
//
//		it is applied for pixel(x,y) as :
//
//			double tempSum = (
//			pMatrix[0] * pixel(x-1, y-1) +
//			pMatrix[1] * pixel(x,   y-1) +
//			pMatrix[2] * pixel(x+1, y-1) +
//			pMatrix[3] * pixel(x-1, y )  +
//			pMatrix[4] * pixel(x,   y )  +
//			pMatrix[5] * pixel(x+1, y )  +
//			pMatrix[6] * pixel(x-1, y+1) +
//			pMatrix[7] * pixel(x,   y+1) +
//			pMatrix[8] * pixel(x+1, y+1));		// step 1
//
//			tempSum = tempSum * dLevel;			// step 2
//			tempSum = min(tempSum, 255);		// step 3
//			tempSum = max(tempSum, 0);			// step 4
//
//			outval = (BYTE) tempSum;			// step 5
//
//	bDoFeedback
//		if the bDoFeedback options is TRUE, step 2 becomes :
//
//			tempSum = pixel(x,y) * (1.0 - dLevel) + (dLevel * tempSum);
//
//
//	channelMask
//		if the CHRED bit is set in channelMask, the matrix is
//		applied to all red pixels.
//	
//		if the CHGREEN bit is set in channelMask, the matrix is
//		applied to all green pixels.
//
//		if the CHBLUE bit is set in channelMask, the matrix is
//		applied to all blue pixels.
//
//	zero
//		if zero is TRUE, the input channels excluded by the channelMask
//		will have all values set to zero.
//		if zero is FALSE, the input channels excluded by the channelMask
//		will retain the values from the source image.
//
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	rgbOut				BYTE *			24-bit RGB image data allocated by caller
//	widhtPix			UINT			width in pixels
//	height				UINT			height
//	channelMask			BYTE			controls which channels are to be modified
//										CHRED | CHGREEN | CHBLUE
//	zero				BOOL			controls channels excluded by channelMask
//	bDoFeedback			BOOL			controls arithmetic
//	dLevel				double			multiplier
//	pMatrix				double *		3 x 3 matrix to apply.
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////


BOOL IMGEXP ImgDLLApplyMatrixToRGB(BYTE * rgbIn,
							BYTE * rgbOut,
							UINT widthPix,
							UINT height,
							BYTE channelMask,
							BOOL zero,
							BOOL bDoFeedback,
							double dLevel,
							double *pMatrix);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLApplyConvolutionFilter(BYTE * rgbIn,
//												BYTE * rgbOut,
//												UINT widthPix,
//												UINT height,
//												UINT matrixCols,
//												UINT matrixRows,
//												double dLevel,
//												double *pMatrix)
//
//	Purpose :
//		apply a convolution filter/matrix to an RGB image.
//
//		buffers must be the same size!!!
//		rgbIn must not be the same as rgbOut !!!
//
//		the number of columns and the number of rows in the matrix
//		must both be odd!
//
//		it is applied as follows :
//
//			M = matrix ( 3x3, in this example)
//		
//				1  1  1
//			M = 1 -8  1			this is a sharpening matrix
//				1  1  1
//		
//			the matrix is centered on the current pixel. in this example, 
//			M[4] is the center of the matrix.
//
//			dLevel is a level 
//
//			P = pixels . since the matrix is centered on the current pixel,
//			P(x,y), the pixels used in this calculation will be :
//				P(x-1,y-1),	P(x, y-1),	P(x+1, y-1)
//				P(x-1,y),	P(x, y),	P(x+1, y)
//				P(x-1,y+1),	P(x, y+1),	P(x+1, y+1)
//
//			t = sum ( M[i] * P[i] )  [for all elements of the matrix]
//
//			outPixel = curPixel - (dLevel * t)
//
//		this example demonstrates the sharpening filter used by this library.
//
//		besides the ability to use matricies of arbitrary size, this function 
//		differs from ImgDLLApplyMatrixToRGB in the way the final caluclation 
//		is performed. 
//
//	Param				Type			Use
//	rgbIn				BYTE *			24-bit RGB image data
//	rgbOut				BYTE *			24-bit RGB image data allocated by caller
//	widhtPix			UINT			width of image in pixels
//	height				UINT			height of image
//	matrixCols			UINT			width of matrix, must be an odd number
//	matrixRows			UINT			height of matrix, must be an odd number
//	dLevel				double			multiplier
//	pMatrix				double *		matrix to apply . be sure you have 
//										matrixCols x matrixRows elements!!
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLApplyConvolutionFilter(BYTE * rgbIn,
											BYTE * rgbOut,
											UINT widthPix,
											UINT height,
											UINT matrixCols,
											UINT matrixRows,
											double dLevel,
											double *pMatrix);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLApplyLUTToRGB(BYTE *rgbBuf,
//								   UINT widthPix,
//								   UINT height,
//								   BYTE channelMask,
//								   BYTE *pLUT)
//
//	Purpose :
//		Apply a Look Up Table (LUT) to an RGB image.
//
//	Param				Type			Use
//	rgbBuf				BYTE *			buffer to apply LUT to
//	widthPix			UINT			width
//	height				UINT			height
//	channelMask			BYTE			control which channels are modified
//										CHRED | CHGREEN | CHBLUE
//	pLUT				BYTE			array of 256 BYTE values (0...255)
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLApplyLUTToRGB(BYTE *rgbBuf,
								   UINT widthPix,
								   UINT height,
								   BYTE channelMask,
								   BYTE *pLUT);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLHistogramEqualizeRGB(BYTE *inBuf, 
//								UINT widthPix, 
//								UINT height,
//								UINT loThresh,
//								UINT hiThresh,
//								BYTE channelMask);
//
//
//	Purpose :
//		Perform histogram equalization on an image.
//
//	Param				Type			Use
//	inBuf				BYTE *			buffer
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	loThresh			UINT			0..255 low limit on candidate pixel values
//	hiThresh			UINT			0..255 hi limit on candidate pixel values. 
//										must be < loThresh
//	channelMask			BYTE			control which channels are modified
//										CHRED | CHGREEN | CHBLUE
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLHistogramEqualizeRGB(BYTE *inBuf, 
								UINT widthPix, 
								UINT height,
								UINT loThresh,
								UINT hiThresh,
								BYTE channelMask);


////////////////////////////////////////////////////////////
//
//	BOOL ImgDLLGetBrightnessHistogram(BYTE *inBuf,
//									UINT widthPix,
//									UINT height,
//									int *pHisto);
//
//	Purpose :
//		Calculate the brightness/luminance histogram for an image.
//	For 0..255, every pixel with a calculated luminance of "b" will
//	increase the value of pHisto[b].
//
//	Param				Type			Use
//	inBuf				BYTE *			buffer
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	pHisto				int *			array of 256 ints
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLGetBrightnessHistogram(BYTE *inBuf,
								UINT widthPix,
								UINT height,
								int *pHisto);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLGetChannelHistogram(BYTE *inBuf,
//											UINT widthPix,
//											UINT height,
//											BYTE chMask,
//											int *pHisto);
//
//	Purpose :
//		Calculate the histogram for a single channel (R,G or B) in 
//	an RGB image.
//	Every pixel in the image with a value of "b" in the specified
//	channel will increase the value of pHisto[b] by one.
//
//	Param				Type			Use
//	inBuf				BYTE *			buffer
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	chMask				BYTE			_one of_ CHRED, CHGREEN, CHBLUE
//	pHisto				int *			array of 256 ints
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLGetChannelHistogram(BYTE *inBuf,
										UINT widthPix,
										UINT height,
										BYTE chMask,
										int *pHisto);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLRotateRGB(BYTE *inBuf, 
//								UINT widthPix, 
//								UINT height,
//								double radians,
//								COLORREF backColor,
//								UINT * outWidth,
//								UINT *outHeight);
//
//
//	Purpose :
//		Rotate an RGB image by a number of radians.
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	radians				double			0..2pi radians to rotate
//	backColor			COLORREF		color to draw where there is no image
//	outWidth			UINT *			output width
//	outHeight			UINT *			output height
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to a new image, else NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLRotateRGB(BYTE *inBuf, 
								UINT widthPix, 
								UINT height,
								double radians,
								COLORREF backColor,
								UINT * outWidth,
								UINT *outHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLQuickRotateRGB(BYTE *inBuf, 
//											UINT widthPix, 
//											UINT height,
//											int rotationType)
//
//
//	Purpose :
//		Rotate an RGB image 90, 180 or 270 degrees.
//	This rotates the image in-place.
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	widthPix			UINT			width, in pixels
//	height				UINT			height 
//	rotationType		int				rotation type : 
//										0 - 90 degrees cw
//										1 - 180 deg
//										2 - 270 deg cw
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		BOOL - TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLQuickRotateRGB(BYTE *inBuf, 
											UINT widthPix, 
											UINT height,
											int rotationType);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLCropRGB(BYTE *inBuf,
//						   UINT uInWidthPix,
//						   UINT uInHeight,
//						   UINT uLeft,
//						   UINT uTop,
//						   UINT uRight,
//						   UINT uBottom)
//	Purpose :
//		Crop an image to the rectangle specified.
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	uInWidthPix			UINT			width, in pixels
//	uInHeight			UINT			height 
//	uLeft				UINT			left edge of crop rect
//	uTop				UINT			top edge of crop rect
//	uRight				UINT			right edge...
//	uBottom				UINT			bottom edge
//
//	uTop <= uBottom
//	uLeft <= uRight
//
//	uRight - uLeft < uInWidthPix
//	uBottom - uTop < uInHeight
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to a new image, else NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLCropRGB(BYTE *inBuf,
						   UINT uInWidthPix,
						   UINT uInHeight,
						   UINT uLeft,
						   UINT uTop,
						   UINT uRight,
						   UINT uBottom);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLZoomRGB(BYTE *inBuf,
//						   UINT uInWidthPix,
//						   UINT uInHeight,
//						   UINT uLeft,
//						   UINT uTop,
//						   UINT uRight,
//						   UINT uBottom,
//						   UINT uOutWidthPix,
//						   UINT uOutHeight)
//	Purpose :
//		Zoom into an image. Performed as a crop to the rectangle
//	specified, then a resize to the output dimensions. This actually
//	uses ImgDLLCropRGB and then ImgDLLResizeRGB.
//
//	Param				Type			Use
//	inBuf				BYTE *			input image
//	uInWidthPix			UINT			width, in pixels
//	uInHeight			UINT			height 
//	uLeft				UINT			left edge of crop rect
//	uTop				UINT			top edge of crop rect
//	uRight				UINT			right edge...
//	uBottom				UINT			bottom edge
//	uOutWidthPix		UINT			width of output image (may be > uInWidthPix)
//	uOutHeight			UINT			height of output image (may be > uInHeight)
//
//	uTop <= uBottom
//	uLeft <= uRight
//
//	uRight - uLeft < uInWidthPix
//	uBottom - uTop < uInHeight
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to a new image, else NULL on failure
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLZoomRGB(BYTE *inBuf,
						   UINT uInWidthPix,
						   UINT uInHeight,
						   UINT uLeft,
						   UINT uTop,
						   UINT uRight,
						   UINT uBottom,
						   UINT uOutWidthPix,
						   UINT uOutHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLOverlayRGB(BYTE *pImage1,
//						   UINT uImage1Width,
//						   UINT uImage1Height,
//						   BYTE *pImage2,
//						   UINT uImage2Width,
//						   UINT uImage2Height,
//						   int iXPos,
//						   int iYPos,
//						   double opacity)
//	Purpose :
//		Overlay one image on another, using variable opacity.
//
//	pImage1 must point to an RGB image that is larger or
//	equal in size to the image in pImage2.
//
//	Param				Type		Use
//	pImage1				BYTE *		larger image to receive the overlay
//  uImage1Width		UINT		width of pImage1	
//  uImage1Height		UINT		height of pImage1
//	pImage2				BYTE *		smaller image to be overlayed
//  uImage2Width		UINT		width of pImage2
//  uImage2Height		UINT		height of pImage2
//  iXPos				int		x position on pImage1 where overlay will start
//  iYPos				int		y position on pImage1 where overlay will start
//	opacity				double		overlay image opacity (0...1 . 1 is opaque)
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLOverlayRGB(BYTE *pImage1,
						   UINT uImage1Width,
						   UINT uImage1Height,
						   BYTE *pImage2,
						   UINT uImage2Width,
						   UINT uImage2Height,
						   int iXPos,
						   int iYPos,
						   double opacity);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLOverlayRGBTransparentBYTE *pImage1,
//						   UINT uImage1Width,
//						   UINT uImage1Height,
//						   BYTE *pImage2,
//						   UINT uImage2Width,
//						   UINT uImage2Height,
//						   int  iXPos,
//						   int  iYPos,
//						   double opacity,
//							COLORREF transColor)
//
//	Purpose :
//		Overlay one image on another, using variable opacity and
//	one transparent color.
//
//	transColor specifies a color in image2 that is to be considered
//	transparent - image1 will show through image2 where the color of
//	image 2 is transColor.
//
//	pImage1 must point to an RGB image that is larger or
//	equal in size to the image in pImage2.
//
//	Param				Type		Use
//	pImage1				BYTE *		larger image to receive the overlay
//  uImage1Width		UINT		width of pImage1	
//  uImage1Height		UINT		height of pImage1
//	pImage2				BYTE *		smaller image to be overlayed
//  uImage2Width		UINT		width of pImage2
//  uImage2Height		UINT		height of pImage2
//  iXPos				int 		x position on pImage1 where overlay will start
//  iYPos				int 		y position on pImage1 where overlay will start
//	opacity				double		overlay image opacity (0.0 - 1.0 1 is opaque)
//	transColor			COLORREF	color in image2 to be considered trasnparent
//
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLOverlayRGBTransparent(BYTE *pImage1,
						   UINT uImage1Width,
						   UINT uImage1Height,
						   BYTE *pImage2,
						   UINT uImage2Width,
						   UINT uImage2Height,
						   int  iXPos,
						   int  iYPos,
						   double opacity,
						   COLORREF clr);

////////////////////////////////////////////////////////////
//
//	UINT IMGEXP ImgDLLCountRGBColors(BYTE *inBuf,
//						UINT widthPix,
//						UINT height,
//						BOOL fast)
//
//	Purpose :
//		Count colors used in RGB image
//
//	Param				Type		Use
//	InBuf				BYTE *		input image
//  widthPix			UINT		width of image
//  height				UINT		height of image
//	fast				BOOL		if TRUE, this uses 1 MB of memory and runs
//									about 18% faster than if FALSE. if FALSE,
//									this uses 512 K of memory.
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		FALSE on failure
//
////////////////////////////////////////////////////////////

UINT IMGEXP ImgDLLCountRGBColors(BYTE *inBuf,
						UINT widthPix,
						UINT height,
						BOOL fast);


////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLDrawTextOnRGB(BYTE *pRGB,
//							UINT width,
//							UINT height,
//							const char *drawText,
//							const char *fontName,
//							UINT fontPixels,
//							int xpos,
//							int ypos,
//							COLORREF textColor)
//
//	Purpose :
//		draw text on an RGB image.
//
//
//	Param			Type			Use
//	pRGB			BYTE *			ptr to your image data
//	width			UINT			image width, in pixels
//	height			UINT			image height
//	drawText		const char *	text to draw
//	fontName		const char *	name of font to use
//	fontPixels		UINT			max height of text, in pixels
//	xpos			int				x position
//	ypos			int				y position
//	textColor		COLORREF		text color
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success, else FALSE
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLDrawTextOnRGB(BYTE *pRGB,
							UINT width,
							UINT height,
							const char *drawText,
							const char *fontName,
							UINT fontPixels,
							int xpos,
							int ypos,
							COLORREF textColor);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLColorSubRGB(BYTE *image,
//					   UINT w,
//					   UINT h,
//					   COLORREF oldColor,
//					   COLORREF newColor)
//
//	Purpose :
//		replace all pixels of a given color with a new color
//
//	Param			Type			Use
//	image			BYTE *			RGB image
//	w				UINT			width
//	h				UINT			height
//	oldColor		COLORREF		color to look for
//	newColor		COLORREF		color to replace it with
//	
//
//	Return
//		none
//
////////////////////////////////////////////////////////////
void IMGEXP ImgDLLColorSubRGB(BYTE *image,
					   UINT w,
					   UINT h,
					   COLORREF oldColor,
					   COLORREF newColor);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLReadRGBFromPNG(const char * fileName,
//							 UINT *width,
//							 UINT *height)
//
//
//	Purpose :
//		Read a PNG file to a buffer of RGB bytes. all files types
//	are converted to 24-bit RGB images!!
//
//		screen gamma defaults to 2.9 . this can (and should) be changed
//	to your own monitor's gamma value by using ImgDLLPNGSetScreenGamma.
//
//		background color for alpha blending channel is set by using
//	ImgDLLPNGSetDefBackground.
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLReadRGBFromPNG(const char *fileName,
										UINT *width, 
										UINT *height);

////////////////////////////////////////////////////////////
//
//	double IMGEXP ImgDLLPNGSetScreenGamma(double screenGamma)
//
//	Purpose :
//		Set the screen gamma value for PNG reads and writes
//
//	Param			Type			Use
//	screenGamma		double			value to set as gamma. 
//
//	Return
//		last PNG gamma value
//
////////////////////////////////////////////////////////////

double IMGEXP ImgDLLPNGSetScreenGamma(double screenGamma);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLPNGSetDefBackground(BYTE palIndex,
//										 UINT red,
//										 UINT green,
//										 UINT blue,
//										 UINT gray)
//
//	Purpose :
//		Set the background colors to be used in blending the
//	alpha channel from PNG images that have one. This is used
//	for both reading and writing of PNG images - if the input image
//	uses an alpha channel but has no background specified, this
//	value is used. When writing a file, this value is written
//	as the default background.
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	palIndex		BYTE			palIndex to use for background
//									(if you know the incoming image has a 
//									certain palette)
//	red				UINT			red component of RGB background value
//	green			UINT			green component of RGB background value
//	blue			UINT			blue component of RGB background value
//	gray			UINT			gray value to be used for grayscale images
//
//	Return
//		none
//
////////////////////////////////////////////////////////////

void IMGEXP ImgDLLPNGSetDefBackground(BYTE palIndex,
										 UINT red,
										 UINT green,
										 UINT blue,
										 UINT gray);


////////////////////////////////////////////////////////////
//
//	BOOL IMGDLLEXP ImgDLLSaveToPNG(const char *imageFileName, 
//							  BYTE *image, 
//							  UINT inWidthPix, 
//							  UINT inHeight, 
//							  UINT inWidthBytes,
//							  UINT bitDepth,
//							  UINT colors,
//							  RGBQUAD * inPalette, 
//							  UINT colorType,
//							  double gammaVal,
//							  UINT filterMask,
//							  UINT interlaceType)
//
//
//	Purpose :
//		Save a buffer to a PNG file. 
//
//		This routine can take a large variety of input image
//	formats, not just packed RGB!. There is not much error checking 
//	done on the input data. Be sure that what you are passing is in the
//	format you say it is!! 
//
//	Writes any text fields added with ImgDLLAddPNGText
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	imageFileName	const char *	file to write
//	image			BYTE *			ptr to your image data
//	inWidthPix		UINT			image width, in pixels
//	inHeight		UINT			image height
//	inWidthBytes	UINT			width of an image row, in BYTEs
//	bitDepth		UINT			1, 2, 4, 8, or 16 bits/channel
//	colors			UINT			number of colors in palette
//
//	inPalette		RGBQUAD *		palette entries. there must be at least
//									as many RGBQUADs as specified by "colors".
//									if your image does not use a palette, pass
//									NULL for this parameter.
//
//	colorType		UINT			this tells the PNG writer which format the
//									input image is in, see PNG_COLOR_* definitions
//
//	gammaVal		double			if you know the display gamma of the device this 
//									image was created on, pass a value equal 
//									to 1 / display_gamma. according to the PNG docs,
//									a value of .45 is acceptable for most PCs
//
//	filterMask		UINT			filter to apply to the image. as of LibPng-0.96,
//									the only valid value is 0.
//
//	interlaceType	UINT			see PNG_INTERLACE_* definitions
//
//	title			char *			optional text to write under key "Title"
//	description		char *			optional text to write under key "Description"
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success, else FALSE
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveToPNG(const char *imageFileName, 
							  BYTE *image, 
							  UINT inWidthPix, 
							  UINT inHeight, 
							  UINT inWidthBytes,
							  UINT bitDepth,
							  UINT colors,
							  RGBQUAD * inPalette, 
							  UINT colorType,
							  double gammaVal,
							  UINT filterMask,
							  UINT interlaceType);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveRGB24ToPNGRGB(const char *imageFileName, 
//							  BYTE *image, 
//							  UINT inWidthPix, 
//							  UINT inHeight, 
//							  double gammaVal)
//
//	Purpose :
//		Save an RGB24 buffer to a PNG-RGB24 file. 
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Writes any text fields added with ImgDLLAddPNGText
//
//	Param			Type			Use
//	imageFileName	const char *	file to write
//	image			BYTE *			ptr to your image data
//	inWidthPix		UINT			image width, in pixels
//	inHeight		UINT			image height
//	gammaVal		double			if you know the display gamma of the device this 
//									image was created on, pass a value equal 
//									to 1 / display_gamma. according to the PNG docs,
//									a value of .45 is acceptable for most PCs
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success, else FALSE
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveRGB24ToPNGRGB(const char *imageFileName, 
							  BYTE *image, 
							  UINT inWidthPix, 
							  UINT inHeight, 
							  double gammaVal);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSave8BitToPNG8Bit(const char *imageFileName, 
//							  BYTE *image, 
//							  UINT inWidthPix, 
//							  UINT inHeight, 
//							  UINT colors,
//							  RGBQUAD * inPalette,
//							  double gammaVal)
//
//	Purpose :
//		Save an RGB24 buffer to a PNG-RGB24 file. 
//
//	Writes any text fields added with ImgDLLAddPNGText
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	imageFileName	const char *	file to write
//	image			BYTE *			ptr to your image data
//	inWidthPix		UINT			image width, in pixels
//	inHeight		UINT			image height
//	colors			UINT			number of colors in palette
//	inPalette		RGBQUAD			palette
//	gammaVal		double			if you know the display gamma of the device this 
//									image was created on, pass a value equal 
//									to 1 / display_gamma. according to the PNG docs,
//									a value of .45 is acceptable for most PCs
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success, else FALSE
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSave8BitToPNG8Bit(const char *imageFileName, 
									  BYTE *image, 
									  UINT inWidthPix, 
									  UINT inHeight, 
									  UINT colors,
									  RGBQUAD * inPalette,
									  double gammaVal);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLGetPNGKey(UINT index);
//
//	Purpose :
//		return a buffer containing the key of one of the PNG text
//	fields.
//
//	a PNG file may contain any number text fields. Each of these fields
//	consists of a key and the text. These strings can be any length.
//	When ImgDLL reads a PNG file, it stores these strings in the 
//	ImgDLL PNG text buffer. These strings remain valid until the next 
//	PNG file is read or ImgDLLClearPNGText is called. 
//
//	To find the number of strings read from the last PNG file, use
//	ImgDLLGetPNGTextCount.
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	index			UINT			index of string you'd like to fetch
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to a zero-terminated string, else NULL on error
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLGetPNGKey(UINT index);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLGetPNGText(UINT index);
//
//	Purpose :
//		return a buffer containing the text of one of the PNG text
//	fields.
//
//	a PNG file may contain any number text fields. Each of these fields
//	consists of a key and the text. These strings can be any length.
//	When ImgDLL reads a PNG file, it stores these strings in the 
//	ImgDLL PNG text buffer. These strings remain valid until the next 
//	PNG file is read or ImgDLLClearPNGText is called. 
//
//	To find the number of strings read from the last PNG file, use
//	ImgDLLGetPNGTextCount.
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	index			UINT			index of string you'd like to fetch
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL to a zero-terminated string, else NULL on error
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLGetPNGText(UINT index);

////////////////////////////////////////////////////////////
//
//	void IMGEXP ImgDLLClearPNGText();
//
//	Purpose :
//		clear the stored PNG text fields, both input and output. 
//	when you read a PNG file, the stored text remains in memory 
//	until you read a new PNG file, or you explicitly clear it with 
//	this call.
//
//	see ImgDLLAddPNGText for a brief example
//
//	Param			Type			Use
//	none
//
//	Return
//		none
//
////////////////////////////////////////////////////////////
void IMGEXP ImgDLLClearPNGText();

////////////////////////////////////////////////////////////
//
//	UINT IMGEXP ImgDLLGetPNGTextCount();
//
//	Purpose :
//		Ask ImgDLL how many text fields were read when the last
//	PNG file was read. 
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	none
//
//	Return
//		UINT - number of fields read
//
////////////////////////////////////////////////////////////
UINT IMGEXP ImgDLLGetPNGTextCount();

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLAddPNGText(const char *key,
//								const char *text,
//								BOOL compress);
//
//	Purpose :
//		Add a text field to the ImgDLL global PNG text buffer.
//	PNG text fields consist of a mandatory 1-79 character key and 
//	an optional text buffer of any size. The text can be written 
//	compressed or uncompressed. It is not recommended that you compress text 
//	fields of length < 1000.
//
//	Keys should be plain ASCII, no control or non-printable chars.
//
//	You may add as many fields as you wish. 
//
//  The keywords that are given in the PNG Specification are:
//
//            Title            Short (one line) title or caption for image
//            Author           Name of image's creator
//            Description      Description of image (possibly long)
//            Copyright        Copyright notice
//            Creation Time    Time of original image creation
//            Software         Software used to create the image
//            Disclaimer       Legal disclaimer
//            Warning          Warning of nature of content
//            Source           Device used to create the image
//            Comment          Miscellaneous comment; conversion from other
//                             image format
//
//	NOTE : All fields will be cleared after the next ImgDLL PNG file write
//	action.
//
//	Please read the PNG Notes in the ImgDLLv3.RTF file.
//
//	Param			Type			Use
//	key				char *			key string 1-79 ASCII chars
//	title			char *			text string
//	compress		BOOL			compress text if TRUE
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////
BOOL IMGEXP ImgDLLAddPNGText(const char *key,
								const char *text,
								BOOL compress);


////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLTIFFToRGB(const char * path,
//								  UINT *pWidth,
//								  UINT *pHeight);
//
//	Purpose :
//		read a 24-bit image from a non-LZW-compressed TIFF image
//
//
//	Param			Type			Use
//	path			const char *	path to open for reading
//	pWidth			UINT*			width, in pixels
//	pHeight			UINT*			height, in pixels
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalUnlock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLTIFFToRGB(const char * path,
								  UINT *pWidth,
								  UINT *pHeight);

////////////////////////////////////////////////////////////
//	BOOL IMGEXP ImgDLLSaveRGB24ToTIFFRGB(LPCSTR lpFileName,
//							BYTE *pRGB,
//							UINT nWidth,
//							UINT nHeight);
//
//	Purpose :
//		Save an RGB buffer to a non-LZW-compressed TIFF file . 
//		Buffer must be packed RGB.
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveRGB24ToTIFFRGB(LPCSTR lpFileName,
							BYTE *pRGB,
							UINT nWidth,
							UINT nHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGEXP ImgDLLSaveColormappedToTIFF(LPCSTR lpFileName,
//							BYTE *pRGB,
//							UINT nWidth,
//							UINT nHeight,
//							RGBQUAD *pPal);
//
//	Purpose :
//		Save an 8-bit colormapped image to a non-LZW-compressed TIFF file . 
//
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pBuf				BYTE *			image buffer
//	uWidthPix			UINT			width, in pixels
//	uHeight				UINT			height, in pixels
//	pPal				RGBQUAD *		ptr to array of 256 RGBQUAD values -the image palette
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////
BOOL IMGEXP ImgDLLSaveColormappedToTIFF(LPCSTR lpFileName,
							BYTE *pRGB,
							UINT nWidth,
							UINT nHeight,
							RGBQUAD *pPal);

////////////////////////////////////////////////////////////
//
//	HGLOBAL IMGEXP ImgDLLReadRGBFromPCX(const char * fileName,
//							 UINT *width,
//							 UINT *height)
//
//
//	Purpose :
//		Read a PCX file to a buffer of RGB bytes.
//
//	Param				Type			Use
//	fileName			const char *	path to open for reading
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		HGLOBAL . 
//		Caller must use GlobalLock to get a ptr to this buffer!
//		Caller is responsible for calling GlobalFree on this buffer!
//
////////////////////////////////////////////////////////////

HGLOBAL IMGEXP ImgDLLReadRGBFromPCX(const char* filename,
                                          UINT *width,
                                          UINT *height);


////////////////////////////////////////////////////////////
//
//	BOOL IMGDLLEXP ImgLibSaveRGB24ToPCXRGB(LPCSTR lpFileName,
//							BYTE *pRGB,
//							UINT nWidth,
//							UINT nHeight)
//
//
//	Purpose :
//		Write a 24-bit PCX file from a 24-bit RGB buffer
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pRGB				BYTE *			ptr to the image
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////
						 
BOOL IMGEXP ImgDLLSaveRGB24ToPCXRGB(LPCSTR lpFileName,
							BYTE *pRGB,
							UINT nWidth,
							UINT nHeight);

////////////////////////////////////////////////////////////
//
//	BOOL IMGLIBEXP ImgDLLSaveColormappedToPCX(LPCSTR lpFileName,
//							BYTE *pRGB,
//							UINT nWidth,
//							UINT nHeight,
//							RGBQUAD *pPal)
//
//	Purpose :
//		Write an 8-bit PCX file from a colormapped buffer
//
//	Param				Type			Use
//	fileName			const char *	path to open for writing
//	pRGB				BYTE *			ptr to the 8-bit image
//	uWidth				UINT*			width, in pixels
//	uHeight				UINT*			height, in pixels
//	pPal				RGBQUAD *		ptr to 256 RGBQUADs
//
//	modifies global error value, check with	ImgDLLGetLastError()
//
//	Return
//		TRUE on success
//
////////////////////////////////////////////////////////////

BOOL IMGEXP ImgDLLSaveColormappedToPCX(LPCSTR lpFileName,
							BYTE *pRGB,
							UINT nWidth,
							UINT nHeight,
							RGBQUAD *pPal);

#ifdef __cplusplus
}
#endif


#endif