/*
/--------------------------------------------------------------------
|
|      TGADEC.CPP            Targa Decoder Class
|
|      Targa file decoder. Decodes 8, 15, 16, 24 and 32 bpp
|      targa files (compressed and uncompressed) and returns a 32
|      bpp bitmap. Preserves the alpha channel.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "tgadec.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
#endif

IMPLEMENT_DYNAMIC (CTGADecoder, CPicDecoder);


CTGADecoder::CTGADecoder
    ()
    // Creates a decoder
{
}


CTGADecoder::~CTGADecoder
    ()
{
}


void CTGADecoder::DoDecode
    ()
{
  TGAHEADER TgaHead;

  trace (2, "Decoding TGA.\n");

  readTgaHeader (&TgaHead);

  CalcDestBPP (TgaHead.PixelDepth);

  if (TgaHead.PixelDepth == 16 || TgaHead.PixelDepth == 32)
    // Image has alpha channel
    m_pBmp->Create (TgaHead.ImageWidth, TgaHead.ImageHeight,
                    m_DestBPP, TRUE);
  else
    // Image doesn't have alpha channel
    m_pBmp->Create (TgaHead.ImageWidth, TgaHead.ImageHeight,
                    m_DestBPP, FALSE);

  // Read the color map data (Version 1.0 and 2.0).
  if (TgaHead.CmapType != 0)
    readPalette (TgaHead.CmapIndex,
                 TgaHead.CmapLength,
                 TgaHead.CmapEntrySize);

  readImage (&TgaHead);
}


void CTGADecoder::readTgaHeader
    ( TGAHEADER * pTgaHead       // Pointer to TGA header structure
    )
{
  // Read the TGA header (Version 1.0 and 2.0).
  pTgaHead->IdLength      = ReadByte ();
  pTgaHead->CmapType      = ReadByte ();
  pTgaHead->ImageType     = ReadByte ();
  pTgaHead->CmapIndex     = ReadIWord ();
  pTgaHead->CmapLength    = ReadIWord ();
  pTgaHead->CmapEntrySize = ReadByte ();
  pTgaHead->X_Origin      = ReadIWord ();
  pTgaHead->Y_Origin      = ReadIWord ();
  pTgaHead->ImageWidth    = ReadIWord ();
  pTgaHead->ImageHeight   = ReadIWord ();
  pTgaHead->PixelDepth    = ReadByte ();
  pTgaHead->ImagDesc      = ReadByte ();

  // Skip image ID
  m_pDataSrc->Skip (pTgaHead->IdLength);
}


void CTGADecoder::readPalette
    ( int StartIndex,           // Index of first palette entry.
      int Length,               // Number of palette entries stored.
      int EntrySize             // Size of palette entries in bits.
    )
    // Reads the TGA palette and creates a windows palette.
{
  int   i;

  m_pPal = new RGBAPIXEL [256];
  if (!m_pPal)
    raiseError (ERR_NO_MEMORY, "Out of memory for palette.");

  for (i=StartIndex; i<StartIndex+Length; i++)
  {
    m_pPal[i] = readPixel32 (EntrySize);
  }

  if (m_DestBPP == 8)
    m_pBmp->SetPalette (m_pPal);
}


void CTGADecoder::setGrayPalette
    ()
    // Creates a grayscale palette.
{
  int i;
  m_pPal = new RGBAPIXEL [256];
  for (i=0; i<256; i++)
  {
    SetRGBAPixel (m_pPal+i, i, i, i, 0xFF);
  }
}


void CTGADecoder::readImage
    ( TGAHEADER * pTgaHead       // Pointer to TGA header structure
    )
{
  BOOL bCompressed;

  if (pTgaHead->ImageType == TGA_Mono ||
      pTgaHead->ImageType == TGA_RLEMono)
    setGrayPalette ();

  switch (pTgaHead->ImageType)
  {
    case TGA_Map:
    case TGA_RGB:
    case TGA_Mono:
      bCompressed = FALSE;
      break;
    case TGA_RLEMap:
    case TGA_RLERGB:
    case TGA_RLEMono:
      bCompressed = TRUE;
      break;
    default:
      raiseError (ERR_FORMAT_UNKNOWN, "Unknown TGA image type.");
  }
  readData (pTgaHead, bCompressed);
}


void CTGADecoder::readData
    ( TGAHEADER * pTgaHead,       // Pointer to TGA header structure
      BOOL bCompressed
    )
{
  int Width = pTgaHead->ImageWidth;
  int Height = pTgaHead->ImageHeight;
  int bpp = pTgaHead->PixelDepth;

  // Bits 4 & 5 of the Image Descriptor byte control the ordering of
  // the pixels.
  BOOL bXReversed = ((pTgaHead->ImagDesc & 16) == 16);
  BOOL bYReversed = ((pTgaHead->ImagDesc & 32) == 32);

  BYTE * pDest;
  BYTE ** pLineArray = m_pBmp->GetLineArray();

  int y;

  for (y=0; y < Height; y++)
  {
    if (bYReversed)
      pDest = pLineArray[y];
     else
      pDest = pLineArray[Height-y-1];

    if (!bCompressed)
      expandUncompressedLine (pDest, Width, bXReversed, bpp);
     else
      expandCompressedLine (pDest, Width, bXReversed, bpp);
  }
}


void CTGADecoder::expandUncompressedLine
    ( BYTE * pDest,
      int Width,
      BOOL bReversed,
      int bpp
    )
{
  int x;

  for (x=0; x<Width; x++)
  {
    if (m_DestBPP == 32)
    {
      *((RGBAPIXEL *)pDest) = readPixel32 (bpp);
      pDest += 4;
    }
    else
    {
      *pDest = readPixel8 (bpp);
      pDest ++;
    }
  }
}


void CTGADecoder::expandCompressedLine
    ( BYTE * pDest,
      int Width,
      BOOL bReversed,
      int bpp
    )
{
  int  x;
  int  i;
  BYTE Count;

  for (x=0; x<Width; )
  {
    Count = ReadByte ();
    if (Count & 128)
    { // RLE-Encoded packet
      Count -= 127; // Calculate real repeat count.
      if (m_DestBPP == 32)
      {
        *((RGBAPIXEL *)pDest) = readPixel32 (bpp);
        for (i=1; i<Count; i++)
          *((RGBAPIXEL *)(pDest+i*4)) = *(RGBAPIXEL *)pDest;
      }
      else
      {
        *pDest = readPixel8 (bpp);
        for (i=1; i<Count; i++)
          *(pDest+i) = *pDest;
      }
    }
    else
    { // Raw packet
      Count += 1; // Calculate real repeat count.
      for (i=0; i<Count; i++)
      {
        if (m_DestBPP == 32)
          *((RGBAPIXEL *)(pDest+i*4)) = readPixel32 (bpp);
        else
          *(pDest+i) = readPixel8 (bpp);
      }
    }
    if (m_DestBPP == 32)
      pDest += Count*4;
    else
      pDest += Count;
    x += Count;
  }
}


RGBAPIXEL CTGADecoder::readPixel32
    ( int bpp
    )
    // The decoder could be made much faster if the big switch (bpp)
    // statement was taken out of the inner loop. On the other hand,
    // doing that would cause a lot of code to be repeated half a
    // dozen times...
{
  RGBAPIXEL Dest;
  WORD Src;
  BYTE * pCurEntry;

  switch (bpp)
  {
    case 8:
      Dest = m_pPal[ReadByte()];
      break;
    case 15:
    case 16:
      Src = ReadIWord();
      if (bpp == 16)
        SetRGBAPixel (&Dest,
                      (BYTE)(((Src >> 10) & 31)*8),
                      (BYTE)(((Src >> 5) & 31)*8),
                      (BYTE)((Src & 31)*8),
                      (BYTE)(Src & 32786 >> 8));
       else
        SetRGBAPixel (&Dest,
                      (BYTE)(((Src >> 10) & 31)*8),
                      (BYTE)(((Src >> 5) & 31)*8),
                      (BYTE)((Src & 31)*8),
                      0xFF);
      break;
    case 24:
      pCurEntry = m_pDataSrc->ReadNBytes (3);
      SetRGBAPixel (&Dest, *(pCurEntry+2),
                    *(pCurEntry+1), *(pCurEntry), 0xFF);
      break;
    case 32:
      pCurEntry = m_pDataSrc->ReadNBytes (4);
      SetRGBAPixel (&Dest, *(pCurEntry+2), *(pCurEntry+1),
                    *(pCurEntry), *(pCurEntry+3));
      break;
  }
  return Dest;
}

BYTE CTGADecoder::readPixel8
    ( int bpp
    )
{
  BYTE Dest;

  ASSERT (bpp == 8);

  Dest = ReadByte();

  return Dest;
}

