/*
/--------------------------------------------------------------------
|
|      PICDEC.CPP            Generic Picture Decoder Class
|
|      Abstract base class to construct CBmps from picture data in
|      memory or in a file. Classes derived from this class implement
|      concrete decoders for specific file formats. The data is
|      returned in 32-bit DIBs with a valid alpha channel.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "picdec.h"
#include "filesrc.h"
#ifdef _WINDOWS // Klaube
#include "ressrc.h" 
#endif
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CPicDecoder, CObject);
#endif

// Set default trace configuration here. The defined levels are
// explained in picdec.h.
int CPicDecoder::m_TraceLevel = 0;
char * CPicDecoder::m_pszTraceFName = NULL;


CPicDecoder::CPicDecoder
    ()
    // Creates a decoder
{
  m_bDecoding = FALSE;
}


CPicDecoder::~CPicDecoder
    ()
{
  // Can't destroy a decoder while decoding!
  ASSERT (!m_bDecoding);

  if (m_pszTraceFName)
    delete m_pszTraceFName;
  m_pszTraceFName = NULL;
}


void CPicDecoder::MakeBmpFromFile
    ( const char * pszFName,
      CBmp * pBmp,
      int BPPWanted
    )
    // Decodes a picture in a file by creating a file data source and
    // calling MakeBmp with this data source.
{
  CFileSource * pFileSrc = NULL;
  int err;

  char sz[256];
  sprintf (sz, "--- Decoding file %s. ---\n", pszFName);
  trace (1, sz);

  try
  {
    pFileSrc = new CFileSource ();
    err = pFileSrc->Open (pszFName);
    if (err)
    {
      sprintf (sz, "Opening %s failed", pszFName);
      raiseError (err, sz);
    }

    MakeBmp (pFileSrc, pBmp, BPPWanted);
    pFileSrc->Close ();
  }
  catch (CTextException)
  {
    // Clean up on error
    if (pFileSrc) delete pFileSrc;
    throw;
  }

  // Clean up
  delete pFileSrc;
}

#ifdef _WINDOWS
void CPicDecoder::MakeBmpFromResource
    ( int ResourceID,
      CBmp * pBmp,
      int BPPWanted
    )
    // Decodes a picture in a resource by creating a resource data
    // source and calling MakeBmp with this data source.
{
  CResourceSource * pResSrc = NULL;
  int err;

  char sz[256];
  sprintf (sz, "--- Decoding resource ID %i. ---\n", ResourceID);
  trace (1, sz);

  try
  {
    pResSrc = new CResourceSource ();
    err = pResSrc->Open (ResourceID);
    if (err)
    {
      sprintf (sz, "Opening resource %i failed", ResourceID);
      raiseError (err, sz);
    }

    MakeBmp (pResSrc, pBmp, BPPWanted);
    pResSrc->Close ();
  }
  catch (CTextException)
  {
    // Clean up on error
    if (pResSrc) delete pResSrc;
    throw;
  }

  // Clean up
  delete pResSrc;
}
#endif


void CPicDecoder::MakeBmp
    ( CDataSource * pDataSrc,
      CBmp * pBmp,
      int BPPWanted
    )
    // Decodes a picture by getting the encoded data from pDataSrc.
    // Stores the results in pBmp.
{
  ASSERT (!m_bDecoding);
    // Make sure only one decoding process happens at one time. This
    // is needed because we keep the decoding status in member variables.
    // If you need several decoding threads, you will need several
    // decoder objects.

  ASSERT (BPPWanted == 0 || BPPWanted == 8 || BPPWanted == 32);

  m_pBmp = pBmp;
  m_pPal = NULL;
  m_pDataSrc = pDataSrc;
  m_DestBPP = BPPWanted;

  m_bDecoding = TRUE;

  try
  {
    DoDecode ();
    if (m_pPal) delete m_pPal;
    m_bDecoding = FALSE;
  }
  catch (CTextException)
  {
    m_bDecoding = FALSE;
    if (m_pPal) delete m_pPal;
    throw;  // Pass the error on.
  }
}

void CPicDecoder::SetTraceConfig
    ( int Level,
      char * pszFName
    )
{
  // Level has to be between 0 and 3.
  ASSERT (Level < 4);

  m_TraceLevel = Level;

  if (m_pszTraceFName)
    delete m_pszTraceFName;

  if (pszFName)
  {
    m_pszTraceFName = new char[strlen (pszFName)+1];
    strcpy (m_pszTraceFName, pszFName);

    // Delete any old trace file with the same name.
    remove (m_pszTraceFName);
  }
  else
    m_pszTraceFName = NULL;
}

void CPicDecoder::raiseError
    ( int Code,
      char * pszErr
    )
    // This function is needed by callbacks outside of any object,
    // so it's public and static. It should not be called from
    // outside of the library.
{
  char sz[256];
  sprintf (sz, "Decoder error: %s\n", pszErr);
  trace (0, sz);
  throw (CTextException (Code, sz));
}

void CPicDecoder::DoDecode
    ()
    // Implements the actual decoding process. Uses variables local to
    // the object to retrieve and store the data. Implemented in
    // derived classes.
{
  // This routine should never be called. It's here so derived classes
  // can override MakeDIB directly if they want to. (CAnyDecoder does
  // this).
  ASSERT (FALSE);
}

/////////////////////////////////////////////////////////////////////
// Protected routines



// These routines expand pixel data of various bit depths to 32 bpp.
// The original intent was for several derived classes to use them.
// As it is, they are too slow & therefore almost unused.

void CPicDecoder::Expand1bpp
    ( BYTE * pDest,
      BYTE * pSrc,
      int Width      // Width in pixels
    )
{
  int i;

  for (i=0; i<Width/8; i++)
  {
    *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> 7) & 1));
    *((RGBAPIXEL *)(pDest+4)) = *(m_pPal+((*pSrc >> 6) & 1));
    *((RGBAPIXEL *)(pDest+8)) = *(m_pPal+((*pSrc >> 5) & 1));
    *((RGBAPIXEL *)(pDest+12)) = *(m_pPal+((*pSrc >> 4) & 1));
    *((RGBAPIXEL *)(pDest+16)) = *(m_pPal+((*pSrc >> 3) & 1));
    *((RGBAPIXEL *)(pDest+20)) = *(m_pPal+((*pSrc >> 2) & 1));
    *((RGBAPIXEL *)(pDest+24)) = *(m_pPal+((*pSrc >> 1) & 1));
    *((RGBAPIXEL *)(pDest+28)) = *(m_pPal+(*pSrc & 1));
    pSrc++;
    pDest += 32;
  }
  if (Width & 7)  // Check for leftover pixels
    for (i=7; i>(8-Width & 7); i--)
    {
      *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> i) & 1));
      pDest += 4;
    }
}

void CPicDecoder::Expand2bpp
    ( BYTE * pDest,
      BYTE * pSrc,
      int Width      // Width in pixels
    )
{
  int i;
  for (i=0; i<Width/4; i++)
  {
    *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> 6) & 3));
    *((RGBAPIXEL *)(pDest+4)) = *(m_pPal+((*pSrc >> 4) & 3));
    *((RGBAPIXEL *)(pDest+8)) = *(m_pPal+((*pSrc >> 2) & 3));
    *((RGBAPIXEL *)(pDest+12)) = *(m_pPal+(*pSrc & 3));
    pSrc++;
    pDest += 16;
  }
  if (Width & 3)  // Check for leftover pixels
    for (i=6; i>8-(Width & 3)*2; i-=2)
    {
      *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> i) & 3));
      pDest += 4;
    }
}

void CPicDecoder::Expand4bpp
    ( BYTE * pDest,
      BYTE * pSrc,
      int Width      // Width in pixels
    )
{
  int i;

  for (i=0; i<Width/2; i++)
  {
    *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> 4) & 15));
    *((RGBAPIXEL *)(pDest+4)) = *(m_pPal+(*pSrc & 15));
    pSrc++;
    pDest += 8;
  }
  if (Width & 1) // Odd Width?
  {
    *((RGBAPIXEL *)pDest) = *(m_pPal+((*pSrc >> 4) & 15));
    pDest += 4;
  }
}

void CPicDecoder::Expand8bpp
    ( BYTE * pDest,
      BYTE * pSrc,
      int Width      // Width in pixels
    )
{
  int i;
  for (i=0; i<Width; i++)
  {
    *((RGBAPIXEL *)pDest) = *(m_pPal+*pSrc);
    pSrc++;
    pDest += 4;
  }
}

void CPicDecoder::CalcDestBPP
    ( int SrcBPP
    )
{
  if (m_DestBPP == 0)
  {
    if (SrcBPP <= 8)
      m_DestBPP = 8;
    else
      m_DestBPP = 32;
  }

  ASSERT (!(m_DestBPP == 8 && SrcBPP > 8));
    // Can't do color quantization yet.
}

void CPicDecoder::trace
    ( int TraceLevel,
      char * pszMessage
    )
    // Outputs debugging data to a file or to the MSVC debug console.
{
  if (TraceLevel <= m_TraceLevel)
  {
    if (m_pszTraceFName)
    {
      // The file is closed after every call so no data is lost
      // if the program crashes.
      FILE * pFile = fopen (m_pszTraceFName, "a+t");
      if (pFile != (FILE *)0)
      {
        fprintf (pFile, pszMessage);
        fclose (pFile);
      }
      else
      { // No permission? File locked? Filename nonsense?
        TRACE ("Error opening trace file!\n");
      }
    }
    else
      TRACE (pszMessage);
  }
}
