/*
/--------------------------------------------------------------------
|
|      bitmap.cpp         Device independent bitmap class
|
|        Manipulates uncompressed device independent bitmaps
|        of all color depths.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "bitmap.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CBmp, CObject);
#endif


CBmp::CBmp
    ()
    // Creates an empty bitmap.
    // Derived classes have to create a small bitmap here so the
    // class can assume that a valid bitmap is available all the
    // time.
{
}


CBmp::~CBmp
    ()
{
}


void CBmp::CreateCopy
    ( CBmp & rSrcBmp,
      int BPPWanted
    )
    // Creates a copy of rSrcBmp, converting color depth if nessesary.
    // Supports 8 and 32 BPP. Alpha channel information is preserved.
{
  ASSERT_VALID (this);
  ASSERT_VALID (&rSrcBmp);
  ASSERT (BPPWanted == 8 || BPPWanted == 32 || BPPWanted == 0);
  ASSERT (rSrcBmp.GetBitsPerPixel() == 8 ||
          rSrcBmp.GetBitsPerPixel() == 32);

  if (BPPWanted == rSrcBmp.GetBitsPerPixel() ||
      BPPWanted == 0)
  {
    if (&rSrcBmp != this)
    {
      // Create empty bitmap.
      Create (rSrcBmp.GetWidth(), rSrcBmp.GetHeight(),
              rSrcBmp.GetBitsPerPixel(), rSrcBmp.HasAlpha());
      BYTE ** pSrcLines = rSrcBmp.GetLineArray();
      BYTE ** pDstLines = GetLineArray();
      int LineLen = (GetWidth()*GetBitsPerPixel())/8;
         // Minimum possible line length.

      for (int y = 0; y<GetHeight(); y++)
      {
        memcpy (pDstLines[y], pSrcLines[y], LineLen);
      }

      if (GetBitsPerPixel() == 8)
        SetPalette (rSrcBmp.GetPalette());

      ASSERT_VALID (this);
    }
  }
  else
  {
    // Can't copy to self while changing bit depth.
    ASSERT (&rSrcBmp != this);

    Create (rSrcBmp.GetWidth(), rSrcBmp.GetHeight(),
            BPPWanted, rSrcBmp.HasAlpha());
    if (BPPWanted == 32)
    { // Conversion 8->32 BPP
      BYTE ** pSrcLines = rSrcBmp.GetLineArray();
      BYTE ** pDstLines = GetLineArray();
      int SrcLineLen = GetWidth();
      int DestLineLen = GetWidth()*4;
      RGBAPIXEL * pPal = rSrcBmp.GetPalette();

      for (int y = 0; y<GetHeight(); y++)
      { // For each line
        BYTE * pSrcPixel = pSrcLines[y];
        BYTE * pDstPixel = pDstLines[y];

        for (int x = 0; x < SrcLineLen; x++)
        { // For each pixel
          *((RGBAPIXEL *)pDstPixel) = pPal[*pSrcPixel];
          pSrcPixel++;
          pDstPixel += 4;
        }
      }
    }
    else
      quantize (rSrcBmp);
    ASSERT_VALID (this);
  }
}

#ifdef _DEBUG
void CBmp::AssertValid
    () const
{
  ASSERT (m_pBits);

  ASSERT (m_Height >= 0);
  ASSERT (m_Width >= 0);

  // Color table only if 8 bpp.
  ASSERT ((m_bpp > 8) == (m_pClrTab == NULL));

  // Alpha channel only if 32 bpp
  ASSERT ((m_bpp == 32) || !m_bAlphaChannel);
}
#endif


/////////////////////////////////////////////////////////////////////
// CBmp creation

void CBmp::Create
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
    // Create a new empty bitmap. Bits are uninitialized.
{
  ASSERT_VALID (this);

  freeMembers ();
  internalCreate (Width, Height, BitsPerPixel, bAlphaChannel);

  ASSERT_VALID (this);
}


/////////////////////////////////////////////////////////////////////
// CBmp manipulation

void CBmp::SetGrayPalette
    ()
    // Fills the color table with a grayscale palette.
{
  ASSERT (m_pClrTab); // Bitmap must contain a palette!

  int i;
  int NumColors = GetNumColors();
  int ColFactor = 256/NumColors;

  for (i=0; i<NumColors; i++)
    SetPaletteEntry (i, i*ColFactor, i*ColFactor, i*ColFactor, 0xFF);
}

void CBmp::SetPalette
    ( RGBAPIXEL * pPal
    )
{
  ASSERT (m_pClrTab); // Bitmap must contain a palette!

  memcpy (m_pClrTab, pPal, 256*sizeof(RGBAPIXEL));
}


void CBmp::SetAlphaChannel
    ( CBmp * pAlphaBmp
    )
    // Replaces the alpha channel with a new one.
{
  BYTE * pLine;
  BYTE * pAlphaLine;
  int x,y;
  BYTE ** pLineArray;
  BYTE ** pAlphaLineArray;

  ASSERT_VALID (this);
  // Current bitmap must be 32 bpp.
  ASSERT (GetBitsPerPixel() == 32);

  ASSERT_VALID (pAlphaBmp);
  // Alpha channel must be 8 bpp.
  ASSERT (pAlphaBmp->GetBitsPerPixel() == 8);

  // The two bitmaps must have the same dimensions
  ASSERT (pAlphaBmp->GetWidth() == GetWidth());
  ASSERT (pAlphaBmp->GetHeight() == GetHeight());

  pLineArray = GetLineArray();
  pAlphaLineArray = pAlphaBmp->GetLineArray();

  for (y=0; y < GetHeight(); y++)
  {
    pLine = pLineArray[y];
    pAlphaLine = pAlphaLineArray[y];
    for (x=0; x < GetWidth(); x++)
    {
      pLine[x*4+RGBA_ALPHA] = pAlphaLine[x];
    }
  }

  m_bAlphaChannel = TRUE;

  ASSERT_VALID (this);
}


/////////////////////////////////////////////////////////////////////
// Local functions


void CBmp::initLocals
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
{
  m_Width = Width;
  m_Height = Height;
  m_bpp = BitsPerPixel;
  m_bAlphaChannel = bAlphaChannel;

  // Initialize pointers to lines.
  initLineArray ();

  if (BitsPerPixel < 16)
  { // Color table exists
    SetGrayPalette ();
  }

  ASSERT_VALID (this);
}


void CBmp::quantize
    ( CBmp & rSrcBmp
    )
    // Performs color quantization on rSrcBmp. Assumes that memory
    // for the destination has been allocated already.
{
  ASSERT (FALSE);
}