/*
/--------------------------------------------------------------------
|
|      DIBGRIT.CPP          Bitmap Graphic item class
|
|      A bitmap on a canvas.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdafx.h"
#include "dibgrit.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CDIBGrItem, CGrItem);


CDIBGrItem::CDIBGrItem
    ( int x,            // Position on the canvas
      int y,
      int w,            // Width
      int h,            // Height
      int z,            // Position in z-Order
      BYTE Opacity,     // Opacity of the object. 255 is completely
                        // opaque, 0 is completely transparent.
      CBmp * pBmp
    )
  : CGrItem (x, y, w, h, z, Opacity)
{
  ASSERT_VALID (pBmp);
  ASSERT (pBmp->GetBitsPerPixel() == 32);

  m_pBmp = pBmp;
  if (m_pBmp)
    m_pLineArray = m_pBmp->GetLineArray();
  else
    m_pLineArray = NULL;
}


CDIBGrItem::~CDIBGrItem
    ()
{
}

void CDIBGrItem::Draw
    ( CBmp * pCanvas,
      CRect * pUpdateRect
    )
    // Responsible for drawing the object on the canvas.
{
  ASSERT_VALID (pCanvas);
  ASSERT (pCanvas->GetBitsPerPixel() == 32);

  ASSERT_VALID (m_pBmp);
  ASSERT (m_pBmp->GetBitsPerPixel() == 32);

  // Perform clipping.
  CRect ClipRect (pUpdateRect);
  clip (pCanvas, &ClipRect);

  // Draw
  if (m_w == m_pBmp->GetWidth() &&
      m_h == m_pBmp->GetHeight())
    drawClippedNoScale (pCanvas, &ClipRect);
  else
    drawClipped (pCanvas, &ClipRect);

  // If this fails, the procedure destroyed something.
  ASSERT_VALID (pCanvas);
  ASSERT_VALID (m_pBmp);
}


void CDIBGrItem::drawClippedNoScale
    ( CBmp * pCanvas,
      CRect * pRect
    )
    // Draws the object. pRect must have been clipped already.
    // Assumes that no scaling is nessesary.
{
  BYTE * pDest;
  BYTE * pSrc;
  BYTE ** pDestLineArray = pCanvas->GetLineArray();

  int SrcXOfs = (pRect->left-m_x); // Starting X-Coordinate in the
                                   // source bitmap. 0 unless clipped.

  for (int sy = pRect->top; sy<pRect->bottom; sy++)
  { // For each line

    // Set up pointers to beginnings of lines.
    pDest = pDestLineArray[sy] + pRect->left*4;
    pSrc = m_pLineArray[sy-m_y] + SrcXOfs*4;

    if (m_pBmp->HasAlpha())
      drawAlphaLine (pDest, pSrc, pRect);
    else
      // No alpha channel.
      if (m_Opacity == 255)
        memcpy (pDest, pSrc, 4*pRect->Width());
      else
        drawFadeLine (pDest, pSrc, pRect);
  }
}


inline void CDIBGrItem::drawAlphaLine
    ( BYTE * pDest,
      BYTE * pSrc,
      CRect * pRect
    )
    // Draws one line. No scaling. Assumes alpha channel exists.
{
  int alpha;
  int negalpha;
  for (int sx = pRect->left; sx<pRect->right; sx++)
  { // For each pixel
    alpha = ((pSrc[RGBA_ALPHA]) * m_Opacity)>>8;
    negalpha = 255-alpha;
    pDest[RGBA_RED] = (pDest[RGBA_RED]*negalpha +
                       pSrc[RGBA_RED]*alpha)>>8;
    pDest[RGBA_GREEN] = (pDest[RGBA_GREEN]*negalpha +
                         pSrc[RGBA_GREEN]*alpha)>>8;
    pDest[RGBA_BLUE] = (pDest[RGBA_BLUE]*negalpha +
                        pSrc[RGBA_BLUE]*alpha)>>8;
    pDest += 4;
    pSrc += 4;
  }
}


inline void CDIBGrItem::drawFadeLine
    ( BYTE * pDest,
      BYTE * pSrc,
      CRect * pRect
    )
    // Draws one line. No scaling. Assumes alpha channel doesn't
    // exist.
{
  int negalpha;
  negalpha = 255-m_Opacity;

  for (int sx = pRect->left; sx<pRect->right; sx++)
  { // For each pixel
    pDest[RGBA_RED] = (pDest[RGBA_RED]*negalpha +
                       pSrc[RGBA_RED]*m_Opacity)>>8;
    pDest[RGBA_GREEN] = (pDest[RGBA_GREEN]*negalpha +
                         pSrc[RGBA_GREEN]*m_Opacity)>>8;
    pDest[RGBA_BLUE] = (pDest[RGBA_BLUE]*negalpha +
                        pSrc[RGBA_BLUE]*m_Opacity)>>8;
    pDest += 4;
    pSrc += 4;
  }
}


void CDIBGrItem::drawClipped
    ( CBmp * pCanvas,
      CRect * pRect
    )
    // Draws the object. pRect must have been clipped already.
{
  BYTE * pDest;
  BYTE * pSrc;
  BYTE * pAlpha = NULL;
  BYTE ** pDestLineArray = pCanvas->GetLineArray();

  double YScale = (double)(m_pBmp->GetHeight())/m_h;
  int SrcLineNum;

  for (int sy = pRect->top; sy<pRect->bottom; sy++)
  { // For each line

    pDest = pDestLineArray[sy] + pRect->left*4;
    SrcLineNum = (int)((sy-m_y)*YScale);
    pSrc = m_pLineArray[SrcLineNum];
    drawScaleLine (pDest, pSrc, m_pBmp->HasAlpha(), pRect);
  }
}


inline void CDIBGrItem::drawScaleLine
    ( BYTE * pDest,
      BYTE * pSrc,
      BOOL bAlpha,
      CRect * pRect
    )
    // Draws one line. Takes into account scaling and alpha channel
    // information.
    // Fixed-point-version.
{
  int alpha;
  int negalpha;
  int SrcOfs;
  double XScale = (double)(m_pBmp->GetWidth())/m_w;
  int xs = (int)(XScale*65536);

  for (int sx = pRect->left-m_x; sx<pRect->right-m_x; sx++)
  { // For each pixel
    SrcOfs = (sx*xs)>>16;
    if (bAlpha)
      alpha = ((pSrc[SrcOfs*4+RGBA_ALPHA]) * m_Opacity)>>8;
     else
      alpha = m_Opacity;
    negalpha = 255-alpha;
    pDest[RGBA_RED] = (pDest[RGBA_RED]*negalpha +
                       pSrc[SrcOfs*4+RGBA_RED]*alpha)>>8;
    pDest[RGBA_GREEN] = (pDest[RGBA_GREEN]*negalpha +
                         pSrc[SrcOfs*4+RGBA_GREEN]*alpha)>>8;
    pDest[RGBA_BLUE] = (pDest[RGBA_BLUE]*negalpha +
                        pSrc[SrcOfs*4+RGBA_BLUE]*alpha)>>8;

    pDest += 4;
  }
}

