/*
/--------------------------------------------------------------------
|
|      TIFFDEC.CPP            TIFF Decoder Class
|
|      TIFF file decoder. Uses LIBTIFF to do the actual conversion.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "tiffdec.h"
#include "except.h"

extern "C"
{
#include "tiffio.h"
#include "tif_msrc.h"
}

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CTIFFDecoder, CPicDecoder);
#endif

char CTIFFDecoder::m_szLastErr[256];

/////////////////////////////////////////////////////////////////////
// Class functions

CTIFFDecoder::CTIFFDecoder
    ()
    // Creates a decoder
{
}


CTIFFDecoder::~CTIFFDecoder
    ()
{
}

/* I used this function to test the LIBTIFF rouines without a custom
   data source. If something goes wrong, it may still be useful to
   isolate bugs.

CBmp * CTIFFDecoder::MakeBmpFromFile
    ( char * pszFName
    )
{
  char sz[256];
  sprintf (sz, "Decoding file %s.\n", pszFName);
  trace (1, sz);

  TIFFSetErrorHandler (Win32ErrorHandler);
  TIFFSetWarningHandler (Win32WarningHandler);

  TIFF* tif = TIFFOpen(pszFName, "r");
  if (tif)
  {
    uint32 w, h;
    size_t npixels;
    uint32* raster;

    TIFFGetField (tif, TIFFTAG_IMAGEWIDTH, &w);
    TIFFGetField (tif, TIFFTAG_IMAGELENGTH, &h);
    npixels = w * h;

    m_pBmp->Create (w, h, 32, (img.alpha != 0));

    //raster = (uint32*) _TIFFmalloc(npixels * sizeof (uint32));
    raster = (uint32*) m_pDIB->GetBits();
    if (raster != NULL)
    {
      TIFFReadRGBAImage(tif, w, h, raster, 0)
      _TIFFfree(raster);
    }
    TIFFClose(tif);
  }

  return m_pDIB;
}
*/

void CTIFFDecoder::DoDecode
    ()
{
  int ok;
  ULONG x, y;

  TIFFRGBAImage img;
  char emsg[1024];
  BYTE * pBits;

  CalcDestBPP (32); // libtiff always decodes to 32 bpp.

  // Set up function pointers to error handling routines.
  TIFFSetErrorHandler (Win32ErrorHandler);
  TIFFSetWarningHandler (Win32WarningHandler);

  TIFF* tif = TIFFOpenMem (m_pDataSrc->ReadEverything(),
                           m_pDataSrc->GetFileSize());
  if (!tif)
    raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);

  ok = TIFFRGBAImageBegin(&img, tif, 0, emsg);

  if (ok == 0)
  {
    TIFFClose (tif);
    raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);
  }

  try
  {
    m_pBmp->Create (img.width, img.height, 32, (img.alpha != 0));
    pBits = new BYTE [img.width*img.height*4];
    if (pBits == NULL)
      raiseError (ERR_NO_MEMORY, "Out of memory allocating TIFF buffer.");
  }
  catch (CTextException)
  {
    TIFFClose (tif);
    throw;
  }

  ok = TIFFRGBAImageGet(&img, (uint32 *) pBits, img.width, img.height);
  if (!ok)
  {
    TIFFRGBAImageEnd(&img);
    TIFFClose(tif);
    raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);
  }

  BYTE ** pLineArray = m_pBmp->GetLineArray();

  // Correct the byte ordering
  for (y=0; y<img.height; y++)
  {
    BYTE * pSrc = pBits+(img.height-y-1)*img.width*4;
    RGBAPIXEL * pPixel = (RGBAPIXEL *)(pLineArray[y]);
    for  (x=0; x<img.width; x++)
    {
      SetRGBAPixel (pPixel, *pSrc, *(pSrc+1),
                    *(pSrc+2), *(pSrc+3));
      pPixel++;
      pSrc += 4;
    }
  }

  // Clean up.
  delete pBits;
  TIFFRGBAImageEnd(&img);
  TIFFClose(tif);
}

/////////////////////////////////////////////////////////////////////
// Static functions used as Callbacks from the TIFF library

void CTIFFDecoder::Win32ErrorHandler
    ( const char* module,
      const char* fmt,
      va_list ap
    )
{
  char szMessage [256];
  sprintf(szMessage, fmt, ap);

  strcpy (m_szLastErr, szMessage);

  return;
}

void CTIFFDecoder::Win32WarningHandler
    ( const char* module,
      const char* fmt,
      va_list ap
    )
{
  char szTemp[256];
  char szMessage [256];
  sprintf(szMessage, fmt, ap);

  if (module != NULL)
    sprintf (szTemp, "Warning in LIBTIFF: %s\n", szMessage);
   else
    sprintf (szTemp, "Warning in LIBTIFF: %s\n", szMessage);

  trace (2, szTemp);

  return;
}
