/*
/--------------------------------------------------------------------
|
|      PICTDEC.CPP            Macintosh pict Decoder Class
|
|      This class decodes macintosh PICT files with 1,2,4,8,16 and 32
|      bits per pixel as well as PICT/JPEG. If an alpha channel is
|      present in a 32-bit-PICT, it is decoded as well.
|      The PICT format is a general picture file format and can
|      contain a lot of other elements besides bitmaps. These elements
|      are ignored.
|      There are several opcodes for which I did not find examples.
|      I have marked the appropriate code as "untested". It'll
|      probably work anyway.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "pictdec.h"
#ifdef SUPPORT_JPEG
#include "jpegdec.h"
#endif
#include "optable.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
#endif

IMPLEMENT_DYNAMIC (CPictDecoder, CPicDecoder);

CPictDecoder::CPictDecoder
    ( CJPEGDecoder * pJPEGDecoder
    )
    // Creates a decoder
{
  m_pJPEGDecoder = pJPEGDecoder;
}


CPictDecoder::~CPictDecoder
    ()
{
}


void CPictDecoder::DoDecode
    ()
    // Called by MakeDIB to do the actual decoding.
{
  trace (2, "Decoding mac pict.\n");

  // Skip empty 512 byte header.
  m_pDataSrc->Skip (512);

  // Read PICT header
  readHeader ();

  interpretOpcodes ();
}

void CPictDecoder::readHeader
    ()
    // Decodes header and version information.
    // Sets m_Version.
    // Performs checks to make sure the data is really a pict file.
{
  BYTE ch;
  WORD PicSize;  // Version 1 picture size. Ignored in version 2.
  char sz[256];

  PicSize = ReadMWord ();

  readRect (&m_Frame);

  while ((ch = ReadByte()) == 0);
  if (ch != 0x11)
                raiseError (ERR_WRONG_SIGNATURE,
                "Error decoding pict: Version number missing.");

  switch (ReadByte())
  {
    case 1:
      m_Version = 1;
      break;
    case 2:
      if (ReadByte() != 0xff)
        raiseError (ERR_WRONG_SIGNATURE,
                    "Illegal version number.");
      m_Version = 2;
      break;
    default:
      raiseError (ERR_WRONG_SIGNATURE,
                  "Illegal version number.");
  }

  sprintf (sz, "PICT version %d found.\n", m_Version);
  trace (2, sz);
}


void CPictDecoder::interpretOpcodes
    ()
    // This is the main decoder loop. The functions reads opcodes,
    // skips some, and hands the rest to opcode-specific functions.
    // It stops decoding after the first opcode containing bitmapped
    // data.
{
  WORD   Opcode;
  char   sz[256];

  BOOL   bDone = FALSE;

  while (!bDone)
  {
    Opcode = readOpcode();

    if (Opcode == 0xFF || Opcode == 0xFFFF)
    {
      bDone = TRUE;
      trace (2, "Opcode: End of pict.\n");
      raiseError (ERR_FORMAT_NOT_SUPPORTED,
                  "PICT contained only vector data!\n");
    }
    else if (Opcode < 0xa2)
    {
      if (!strcmp(optable[Opcode].name, "reserved"))
        sprintf (sz, "Opcode: reserved=0x%x\n", Opcode);
      else
        sprintf (sz, "Opcode: %s\n", optable[Opcode].name);
      trace (2, sz);

      switch (Opcode)
      {
        case 0x01: // Clip
          clip ();
          break;
        case 0x12:
        case 0x13:
        case 0x14:
          pixPat ();
          break;
        case 0x70:
        case 0x71:
        case 0x72:
        case 0x73:
        case 0x74:
        case 0x75:
        case 0x76:
        case 0x77:
          skipPolyOrRegion ();
          break;
        case 0x90:
        case 0x98:
          bitsRect ();
          bDone = TRUE;
          break;
        case 0x91:
        case 0x99:
          bitsRegion ();
          bDone = TRUE;
          break;
        case 0x9a:
          opcode9a ();
          bDone = TRUE;
          break;
        case 0xa1:
          longComment ();
          break;
        default:
          // No function => skip to next Opcode
          if (optable[Opcode].len == WORD_LEN)
            m_pDataSrc->Skip(ReadMWord());
           else
            m_pDataSrc->Skip(optable[Opcode].len);
      }
    }
    else if (Opcode == 0xc00)
    {
      trace (2, "Opcode: Header.\n");
      m_pDataSrc->Skip(24);
    }
    else if (Opcode == 0x8200)
    {
      trace (2, "Opcode: JPEG.\n");
      jpegOp ();
      bDone = TRUE;
    }
    else if (Opcode >= 0xa2 && Opcode <= 0xaf)
    {
      sprintf (sz, "Opcode: reserved 0x%x.\n", Opcode);
      trace (2, sz);
      m_pDataSrc->Skip(ReadMWord());
    }
    else if ((Opcode >= 0xb0 && Opcode <= 0xcf) ||
             (Opcode >= 0x8000 && Opcode <= 0x80ff))
    {
      // just a reserved Opcode, no data
      sprintf (sz, "Opcode: reserved 0x%x.\n", Opcode);
      trace (2, sz);
    }
    else if ((Opcode >= 0xd0 && Opcode <= 0xfe) ||
             (Opcode >= 8100 && Opcode <= 0xffff))
    {
      sprintf (sz, "Opcode: reserved 0x%x.\n", Opcode);
      trace (2, sz);
      m_pDataSrc->Skip(ReadMLong());
    }
    else if (Opcode >= 0x100 && Opcode <= 0x7fff)
    {
      sprintf (sz, "Opcode: reserved 0x%x.\n", Opcode);
      trace (2, sz);
      m_pDataSrc->Skip((Opcode >> 7) & 255);
    }
    else
    {
      char sz[256];
      sprintf (sz, "Can't handle Opcode %x.\n", Opcode);
      raiseError (ERR_FORMAT_UNKNOWN, sz);
    }
  }
}


WORD CPictDecoder::readOpcode
    ()
    // moves to an even byte position in the file and returns the
    // opcode found.
{
  if (m_Version == 2)
    m_pDataSrc->AlignToWord();

  if (m_Version == 1)
    return ReadByte ();
   else
    return ReadMWord ();
}


/////////////////////////////////////////////////////////////////////
// Opcode functions

void CPictDecoder::clip
    ()
    // skips clipping rectangle or region.
{
  MacRect ClipRect;

  WORD len = ReadMWord();

  if (len == 0x000a)
    { /* null rgn */
      readRect(&ClipRect);
    }
   else
    m_pDataSrc->Skip(len - 2);
}

void CPictDecoder::pixPat
    ()
    // skips pattern definition.
{
  WORD        PatType;
  WORD        rowBytes;
  MacpixMap   p;
  RGBAPIXEL * pCT;
  WORD        NumColors;

  PatType = ReadMWord();

  switch (PatType)
    {
      case 2:
        m_pDataSrc->Skip(8);
        m_pDataSrc->Skip(5);
        break;
      case 1:
        m_pDataSrc->Skip(8);
        rowBytes = ReadMWord();
        readRect(&p.Bounds);
        readPixmap(&p);

        pCT = readColourTable(&NumColors);
        skipBits(&p.Bounds, rowBytes, p.pixelSize);
        delete pCT;
        break;
      default:
        raiseError (ERR_FORMAT_UNKNOWN,
                    "Unknown pattern type in pixPat.");
    }
}

void CPictDecoder::skipPolyOrRegion
    ()
{
  trace (3, "Skipping polygon or region.\n");
  m_pDataSrc->Skip (ReadMWord () - 2);
}

void CPictDecoder::bitsRect
    ()
    // Bitmap/pixmap data clipped by a rectangle.
{
  WORD rowBytes;

  rowBytes = ReadMWord();    // Bytes per row in source when
                             // uncompressed.
  if (rowBytes & 0x8000)
    doPixmap(rowBytes, FALSE);
   else
    doBitmap(rowBytes, FALSE);
}

void CPictDecoder::bitsRegion
    ()
    // Bitmap/pixmap data clipped by a region.
    // Untested...
{
  WORD rowBytes;

  rowBytes = ReadMWord();    // Bytes per row in source when
                             // uncompressed.
  if (rowBytes & 0x8000)
    doPixmap(rowBytes, TRUE);
   else
    doBitmap(rowBytes, TRUE);
}

void CPictDecoder::opcode9a
    ()
    // DirectBitsRect.
{
  MacpixMap PixMap;

  m_pDataSrc->Skip(4); // Skip fake len and fake EOF.
  ReadMWord();         // bogus row bytes.

  // Read in the PixMap fields.
  readRect(&PixMap.Bounds);
  readPixmap (&PixMap);

  // Ignore source & destination rectangle as well as transfer mode.
  MacRect TempRect;
  readRect (&TempRect);
  readRect (&TempRect);
  WORD mode = ReadMWord();

  // Create empty DIB
  CalcDestBPP (PixMap.pixelSize);
  createOutputBmp (PixMap);

  // Do the actual unpacking.
  switch (PixMap.pixelSize)
  {
    case 32:
      unpack32bits (&PixMap.Bounds, 0, PixMap.cmpCount);
      break;
    case 8:
      unpack8bits (&PixMap.Bounds, 0);
      break;
    default:
      unpackbits (&PixMap.Bounds, 0, PixMap.pixelSize);
  }
}

void CPictDecoder::longComment
    ()
    // Untested!!
{
  WORD type;
  WORD len;

  type = ReadMWord();
  len = ReadMWord();
  if (len > 0)
    m_pDataSrc->Skip (len);
}

void CPictDecoder::jpegOp
    ()
    // Invoke the JPEG decoder for this PICT.
{
  long OpLen = ReadMLong();
  BOOL bFound = FALSE;
  int i = 0;

  // Make sure the client allowed 32-bit output.
  CalcDestBPP (32);

  // skip to JPEG header.
  while (!bFound && i < OpLen)
  {
    BYTE * pData = m_pDataSrc->GetBufferPtr (3);
    if (pData[0] == 0xFF && pData[1] == 0xD8 && pData[2] == 0xFF)
      bFound = TRUE;
    else
    {
      ReadByte();
      i++;
    }
  }
  if (bFound)
    // Pass the data to the JPEG decoder.
    if (m_pJPEGDecoder)
#ifdef SUPPORT_JPEG
      m_pJPEGDecoder->MakeBmp (m_pDataSrc, m_pBmp);
#else
      raiseError (ERR_FORMAT_NOT_SUPPORTED,
                  "Library not compiled for PICT/JPEG.");
#endif
     else
      raiseError (ERR_FORMAT_NOT_SUPPORTED,
                  "Library not compiled for PICT/JPEG.");
   else
    raiseError (ERR_FORMAT_NOT_SUPPORTED,
                "PICT file contains unrecognized quicktime data.\n");
}

/////////////////////////////////////////////////////////////////////
// Bitmap & Pixmap functions

void CPictDecoder::createOutputBmp
    ( MacpixMap PixMap
    )
{
  // Create empty DIB
  if (m_DestBPP == 32)
  { // 32-bit output.
    if (PixMap.cmpCount == 4)
    { // Alpha channel in picture.
      m_pBmp->Create (PixMap.Bounds.right - PixMap.Bounds.left,
                      PixMap.Bounds.bottom - PixMap.Bounds.top,
                      32,
                      TRUE);
    }
    else
    { // No alpha channel in picture.
      m_pBmp->Create (PixMap.Bounds.right - PixMap.Bounds.left,
                      PixMap.Bounds.bottom - PixMap.Bounds.top,
                      32,
                      FALSE);
    }
  }
  else
  { // 8-bit-output.
    m_pBmp->Create (PixMap.Bounds.right - PixMap.Bounds.left,
                    PixMap.Bounds.bottom - PixMap.Bounds.top,
                    8,
                    FALSE);
  }
}


void CPictDecoder::doBitmap
    ( int rowBytes,
      BOOL bIsRegion
    )
    // Decode version 1 bitmap: 1 bpp.
{
  MacRect Bounds;
  MacRect SrcRect;
  MacRect DstRect;
  WORD    mode;
  RGBAPIXEL pPal[] = { 0xFFFFFFFF, 0x00000000 };
  ((BYTE *)pPal)[4+RGBA_ALPHA] = 0xFF;

  WORD    width;        // Width in pixels
  WORD    height;       // Height in pixels

  trace (2, "Reading version 1 bitmap.\n");

  readRect(&Bounds);
  dumpRect ("  Bounds", &Bounds);
  readRect(&SrcRect);
  readRect(&DstRect);

  width = Bounds.right - Bounds.left;
  height = Bounds.bottom - Bounds.top;
  m_pPal = pPal;

  CalcDestBPP (1);
  // Create empty DIB without resolution info.
  m_pBmp->Create (width, height, m_DestBPP, FALSE);

  mode = ReadMWord();

  if (bIsRegion)
    skipPolyOrRegion ();

  unpackbits (&Bounds, rowBytes, 1);
  m_pPal = NULL;
}

void CPictDecoder::doPixmap
    ( int rowBytes,
      BOOL bIsRegion
    )
    // Decode version 2 pixmap
{
  MacpixMap   PixMap;
  WORD        NumColors;    // Palette size.

  readRect(&PixMap.Bounds);
  readPixmap(&PixMap);

  CalcDestBPP (PixMap.pixelSize);
  createOutputBmp (PixMap);

  // Read mac colour table into windows palette.
  m_pPal = readColourTable (&NumColors);
  if (m_DestBPP == 8)
  { // Copy palette
    m_pBmp->SetPalette (m_pPal);
  }

  // Ignore source & destination rectangle as well as transfer mode.
  MacRect TempRect;
  readRect (&TempRect);
  readRect (&TempRect);
  WORD mode = ReadMWord();

  if (bIsRegion)
    skipPolyOrRegion ();

  switch (PixMap.pixelSize)
  {
    case 32:
      unpack32bits (&PixMap.Bounds, rowBytes, PixMap.cmpCount);
      break;
    case 8:
      unpack8bits (&PixMap.Bounds, rowBytes);
      break;
    default:
      unpackbits (&PixMap.Bounds, rowBytes, PixMap.pixelSize);
  }
}

void CPictDecoder::unpack32bits
    ( MacRect* pBounds,
      WORD rowBytes,
      int NumBitPlanes    // 3 if RGB, 4 if RGBA
    )
    // This routine decompresses BitsRects with a packType of 4 (and
    // 32 bits per pixel). In this format, each line is separated
    // into 8-bit-bitplanes and then compressed via RLE. To decode,
    // the routine decompresses each line & then juggles the bytes
    // around to get pixel-oriented data.
{
  BYTE * pSrcLine;           // Pointer to source line in file.
  int    i,j;
  WORD   BytesPerRow;        // bytes per row when uncompressed.
  int    linelen;            // length of source line in bytes.
  BYTE * pDestLine;
  BYTE   FlagCounter;
  int    len;

  BYTE * pLinebuf;           // Temporary buffer for line data. In
                             // this buffer, pixels are uncompressed
                             // but still plane-oriented.
  BYTE * pBuf;               // Current location in pLinebuf.
  BYTE ** pLineArray = m_pBmp->GetLineArray();


  int Height = pBounds->bottom - pBounds->top;
  int Width = pBounds->right - pBounds->left;

  BytesPerRow = Width*NumBitPlanes;

  if (rowBytes == 0)
    rowBytes = BytesPerRow;

  // Allocate temporary line buffer.
  pLinebuf = new BYTE [BytesPerRow];

  try
  {
    for (i = 0; i < Height; i++)
    { // for each line do...
      if (rowBytes > 250)
        linelen = ReadMWord();
       else
        linelen = ReadByte();

      pSrcLine = m_pDataSrc->ReadNBytes(linelen);
      pBuf = pLinebuf;

      // Unpack bytewise RLE.
      for (j = 0; j < linelen; )
      {
        FlagCounter = pSrcLine[j];
        if (FlagCounter & 0x80)
        {
          if (FlagCounter == 0x80)
            // Special case: repeat 0 times.
            // Apple sais this happens with third-party encoders.
            // Ignore, since it's nonsense anyway.
            j++;
          else
          { // Real packed data.
            len = ((FlagCounter ^ 255) & 255) + 2;
            memset (pBuf, *(pSrcLine+j+1), len);
            pBuf += len;
            j += 2;
          }
        }
        else
        { // Unpacked data
          len = (FlagCounter & 255) + 1;
          memcpy (pBuf, pSrcLine+j+1, len);
          pBuf += len;
          j += len + 1;
        }
      }

      // Convert plane-oriented data into pixel-oriented data &
      // copy into destination bitmap.
      pDestLine = pLineArray[i];
      pBuf = pLinebuf;

      if (NumBitPlanes == 3)
      { // No alpha channel.
        for (j = 0; j < Width; j++)
        { // For each pixel in line...
          *(pDestLine+RGBA_BLUE) = *(pBuf+Width*2); // Blue
          *(pDestLine+RGBA_GREEN) = *(pBuf+Width);  // Green
          *(pDestLine+RGBA_RED) = *pBuf;            // Red
          *(pDestLine+RGBA_ALPHA) = 0xFF;           // Alpha
          pDestLine +=4;
          pBuf++;
        }
      }
      else
      { // Image with alpha channel.
        for (j = 0; j < Width; j++)
        { // For each pixel in line...
          *(pDestLine+RGBA_BLUE) = *(pBuf+Width*3);  // Blue
          *(pDestLine+RGBA_GREEN) = *(pBuf+Width*2); // Green
          *(pDestLine+RGBA_RED) = *(pBuf+Width);     // Red
          *(pDestLine+RGBA_ALPHA) = *pBuf;           // Alpha
          pDestLine+=4;
          pBuf++;
        }
      }
    }
  }
  catch (CTextException)
  {
    delete pLinebuf;
    throw;
  }
  delete pLinebuf;
}


void CPictDecoder::unpack8bits
    ( MacRect* pBounds,
      WORD rowBytes
    )
    // Decompression routine for 8 bpp. rowBytes is the number of
    // bytes each source row would take if it were uncompressed.
    // This _isn't_ equal to the number of pixels in the row - it
    // seems apple pads the data to a word boundary and then
    // compresses it. Of course, we have to decompress the excess
    // data and then throw it away.
{
  BYTE  * pSrcLine;           // Pointer to source line in file.
  int     i,j,k;
  int     linelen;            // length of source line in bytes.
  BYTE  * pDestPixel;
  BYTE    FlagCounter;
  int     len;
  BYTE  * pLineBuf;
  BYTE ** pLineArray = m_pBmp->GetLineArray();

  int Height = pBounds->bottom - pBounds->top;
  int Width = pBounds->right - pBounds->left;

  // High bit of rowBytes is flag.
  rowBytes &= 0x7fff;

  if (rowBytes == 0)
    rowBytes = Width;

  pLineBuf = new BYTE [rowBytes * 4];

  try
  {
    if (rowBytes < 8)
    { // Ah-ha!  The bits aren't actually packed.  This will be easy.
      for (i = 0; i < Height; i++)
      {
        pSrcLine = m_pDataSrc->ReadNBytes (rowBytes);
        if (m_DestBPP == 32)
        {
          pDestPixel = pLineArray[i];
          expandBuf (pDestPixel, pSrcLine, Width, 8);
        }
        else
        {
          pDestPixel = pLineArray[i];
          memcpy (pDestPixel, pSrcLine, Width);
        }
      }
    }
    else
    {
      for (i = 0; i < Height; i++)
      { // For each line do...
        if (rowBytes > 250)
          linelen = ReadMWord();
         else
          linelen = ReadByte();

        pSrcLine = m_pDataSrc->ReadNBytes(linelen);
        pDestPixel = pLineBuf;

        // Unpack RLE. The data is packed bytewise.
        for (j = 0; j < linelen; )
        {
          FlagCounter = pSrcLine[j];
          if (FlagCounter & 0x80)
          {
            if (FlagCounter == 0x80)
              // Special case: repeat value of 0.
              // Apple sais ignore.
              j++;
            else
            { // Packed data.
              len = ((FlagCounter ^ 255) & 255) + 2;

              if (m_DestBPP == 32)
              {
                RGBAPIXEL PixValue = m_pPal[*(pSrcLine+j+1)];
                for (k = 0; k < len; k++)
                { // Repeat the pixel len times.
                  *(RGBAPIXEL *)(pDestPixel+k*4) = PixValue;
                }
                pDestPixel += len*4;
              }
              else
              {
                BYTE PixValue = *(pSrcLine+j+1);
                for (k = 0; k < len; k++)
                { // Repeat the pixel len times.
                  *(pDestPixel+k) = PixValue;
                }
                pDestPixel += len;
              }
              j += 2;
            }
          }
          else
          { // Unpacked data
            len = (FlagCounter & 255) + 1;
            if (m_DestBPP == 32)
            {
              for (k=0; k<len; k++)
              {
                *((RGBAPIXEL *)pDestPixel) =
                           m_pPal[*(pSrcLine+j+k+1)];
                pDestPixel += 4;
              }
            }
            else
            {
              for (k=0; k<len; k++)
              {
                *(pDestPixel) = *(pSrcLine+j+k+1);
                pDestPixel ++;
              }
            }
            j += len + 1;
          }
        }
        if (m_DestBPP == 32)
        {
          pDestPixel = pLineArray[i];
          memcpy (pDestPixel, pLineBuf, 4*Width);
        }
        else
        {
          pDestPixel = pLineArray[i];
          memcpy (pDestPixel, pLineBuf, Width);
        }
      }
    }
  }
  catch (CTextException)
  {
    delete pLineBuf;
    throw;
  }
  delete pLineBuf;
}

void CPictDecoder::unpackbits
    ( MacRect* pBounds,
      WORD rowBytes,
      int pixelSize          // Source bits per pixel.
    )
    // Decompression routine for everything but 8 & 32 bpp. This
    // routine is slower than the two routines above since it has to
    // deal with a lot of special cases :-(.
    // It's also a bit chaotic because of these special cases...
    // unpack8bits is basically a dumber version of unpackbits.
{
  BYTE * pSrcLine;           // Pointer to source line in file.
  int    i,j,k;
  WORD   pixwidth;           // bytes per row when uncompressed.
  int    linelen;            // length of source line in bytes.
  int    pkpixsize;
  BYTE * pDestLine;
  BYTE   FlagCounter;
  int    len;
  int    PixelPerRLEUnit;
  BYTE * pLineBuf;
  BYTE * pBuf;
  BYTE ** pLineArray = m_pBmp->GetLineArray();

  int Height = pBounds->bottom - pBounds->top;
  int Width = pBounds->right - pBounds->left;

  // High bit of rowBytes is flag.
  if (pixelSize <= 8)
    rowBytes &= 0x7fff;

  pixwidth = Width;
  pkpixsize = 1;          // RLE unit: one byte for everything...
  if (pixelSize == 16)    // ...except 16 bpp.
  {
    pkpixsize = 2;
    pixwidth *= 2;
  }

  if (rowBytes == 0)
    rowBytes = pixwidth;

  try
  {
    // I allocate the temporary line buffer here. I allocate too
    // much memory to compensate for sloppy (& hence fast)
    // decompression.
    switch (pixelSize)
    {
      case 1:
        PixelPerRLEUnit = 8;
        pLineBuf = new BYTE [(rowBytes+1) * 32];
        break;
      case 2:
        PixelPerRLEUnit = 4;
        pLineBuf = new BYTE [(rowBytes+1) * 16];
        break;
      case 4:
        PixelPerRLEUnit = 2;
        pLineBuf = new BYTE [(rowBytes+1) * 8];
        break;
      case 8:
        PixelPerRLEUnit = 1;
        pLineBuf = new BYTE [rowBytes * 4];
        break;
      case 16:
        PixelPerRLEUnit = 1;
        pLineBuf = new BYTE [rowBytes * 2 + 4];
        break;
      default:
        char sz[256];
        sprintf (sz,
                 "Illegal bpp value in unpackbits: %d\n",
                 pixelSize);
        raiseError (ERR_FORMAT_UNKNOWN, sz);
    }

    if (rowBytes < 8)
    { // ah-ha!  The bits aren't actually packed.  This will be easy.
      for (i = 0; i < Height; i++)
      {
        pDestLine = pLineArray[i];
        pSrcLine = m_pDataSrc->ReadNBytes (rowBytes);
        if (m_DestBPP == 32)
          expandBuf(pDestLine, pSrcLine, Width, pixelSize);
        else
          expandBuf8(pDestLine, pSrcLine, Width, pixelSize);
      }
    }
    else
    {
      for (i = 0; i < Height; i++)
      { // For each line do...
        if (rowBytes > 250)
          linelen = ReadMWord();
         else
          linelen = ReadByte();

        pSrcLine = m_pDataSrc->ReadNBytes(linelen);
        pBuf = pLineBuf;

        // Unpack RLE. The data is packed bytewise - except for
        // 16 bpp data, which is packed per pixel :-(.
        for (j = 0; j < linelen; )
        {
          FlagCounter = pSrcLine[j];
          if (FlagCounter & 0x80)
          {
            if (FlagCounter == 0x80)
              // Special case: repeat value of 0.
              // Apple sais ignore.
              j++;
            else
            { // Packed data.
              len = ((FlagCounter ^ 255) & 255) + 2;

              // This is slow for some formats...
              if (m_DestBPP == 32)
              {
                expandBuf (pBuf, pSrcLine+j+1, 1, pixelSize);
                for (k = 1; k < len; k++)
                { // Repeat the pixel (for 16 bpp) or
                  // byte (for everything else) len times.
                  memcpy (pBuf+(k*4*PixelPerRLEUnit), pBuf,
                          4*PixelPerRLEUnit);
                }
                pBuf += len*4*PixelPerRLEUnit;
              }
              else
              {
                expandBuf8 (pBuf, pSrcLine+j+1, 1, pixelSize);
                for (k = 1; k < len; k++)
                { // Repeat the expanded byte len times.
                  memcpy (pBuf+(k*PixelPerRLEUnit), pBuf,
                          PixelPerRLEUnit);
                }
                pBuf += len*PixelPerRLEUnit;
              }
              j += 1 + pkpixsize;
            }
          }
          else
          { // Unpacked data
            len = (FlagCounter & 255) + 1;
            if (m_DestBPP == 32)
            {
              expandBuf (pBuf, pSrcLine+j+1, len, pixelSize);
              pBuf += len*4*PixelPerRLEUnit;
            }
            else
            {
              expandBuf8 (pBuf, pSrcLine+j+1, len, pixelSize);
              pBuf += len*PixelPerRLEUnit;
            }
            j += len * pkpixsize + 1;
          }
        }
        pDestLine = pLineArray[i];
        if (m_DestBPP == 32)
          memcpy (pDestLine, pLineBuf, 4*Width);
        else
          memcpy (pDestLine, pLineBuf, Width);
      }
    }
  }
  catch (CTextException)
  {
    delete pLineBuf;
    throw;
  }

  delete pLineBuf;
}

void CPictDecoder::skipBits
    ( MacRect* pBounds,
      WORD rowBytes,
      int pixelSize         // Source bits per pixel.
    )
    // skips unneeded packbits.
{
  int    i;
  WORD   pixwidth;           // bytes per row when uncompressed.
  int    linelen;            // length of source line in bytes.

  int Height = pBounds->bottom - pBounds->top;
  int Width = pBounds->right - pBounds->left;

  // High bit of rowBytes is flag.
  if (pixelSize <= 8)
    rowBytes &= 0x7fff;

  pixwidth = Width;

  if (pixelSize == 16)
    pixwidth *= 2;

  if (rowBytes == 0)
    rowBytes = pixwidth;

  if (rowBytes < 8)
  {
    m_pDataSrc->Skip (rowBytes*Height);
  }
  else
  {
    for (i = 0; i < Height; i++)
    {
      if (rowBytes > 250)
        linelen = ReadMWord();
       else
        linelen = ReadByte();
      m_pDataSrc->Skip (linelen);
    }
  }
}


void CPictDecoder::expandBuf
    ( BYTE * pDestBuf,
      BYTE * pSrcBuf,
      int Width,       // Width in bytes for 8 bpp or less.
                       // Width in pixels for 16 bpp.
      int bpp          // bits per pixel
    )
    // Expands Width units to 32-bit pixel data.
{
  BYTE * pSrc;
  BYTE * pDest;
  int i;

  pSrc = pSrcBuf;
  pDest = pDestBuf;

  switch (bpp)
  {
    case 16:
      for (i=0; i<Width; i++)
      {
        WORD Src = pSrcBuf[1]+(pSrcBuf[0]<<8);
        *(pDestBuf+RGBA_BLUE) = ((Src) & 31)*8;          // Blue
        *(pDestBuf+RGBA_GREEN) = ((Src >> 5) & 31)*8;    // Green
        *(pDestBuf+RGBA_RED) = ((Src  >> 10) & 31)*8;    // Red
        *(pDestBuf+RGBA_ALPHA) = 0xFF;                   // Alpha
        pSrcBuf += 2;
        pDestBuf += 4;
      }
      break;
    case 8:
      Expand8bpp (pDestBuf, pSrcBuf, Width);
      break;
    case 4:
      Expand4bpp (pDestBuf, pSrcBuf, Width*2);
      break;
    case 2:
      Expand2bpp (pDestBuf, pSrcBuf, Width*4);
      break;
    case 1:
      Expand1bpp (pDestBuf, pSrcBuf, Width*8);
      break;
    default:
      raiseError (ERR_FORMAT_UNKNOWN,
                  "Bad bits per pixel in expandBuf.");
  }
  return;
}


void CPictDecoder::expandBuf8
    ( BYTE * pDestBuf,
      BYTE * pSrcBuf,
      int Width,       // Width in bytes.
      int bpp          // bits per pixel.
    )
    // Expands Width units to 8-bit pixel data.
    // Max. 8 bpp source format.
{
  BYTE * pSrc;
  BYTE * pDest;
  int i;

  pSrc = pSrcBuf;
  pDest = pDestBuf;

  switch (bpp)
  {
    case 8:
      memcpy (pDestBuf, pSrcBuf, Width);
      break;
    case 4:
      for (i=0; i<Width; i++)
      {
        *pDest = (*pSrc >> 4) & 15;
        *(pDest+1) = (*pSrc & 15);
        pSrc++;
        pDest += 2;
      }
      if (Width & 1) // Odd Width?
      {
        *pDest = (*pSrc >> 4) & 15;
        pDest++;
      }
      break;
    case 2:
      for (i=0; i<Width; i++)
      {
        *pDest = (*pSrc >> 6) & 3;
        *(pDest+1) = (*pSrc >> 4) & 3;
        *(pDest+2) = (*pSrc >> 2) & 3;
        *(pDest+3) = (*pSrc & 3);
        pSrc++;
        pDest += 4;
      }
      if (Width & 3)  // Check for leftover pixels
        for (i=6; i>8-(Width & 3)*2; i-=2)
        {
          *pDest = (*pSrc >> i) & 3;
          pDest++;
        }
      break;
    case 1:
      for (i=0; i<Width; i++)
      {
        *pDest = (*pSrc >> 7) & 1;
        *(pDest+1) = (*pSrc >> 6) & 1;
        *(pDest+2) = (*pSrc >> 5) & 1;
        *(pDest+3) = (*pSrc >> 4) & 1;
        *(pDest+4) = (*pSrc >> 3) & 1;
        *(pDest+5) = (*pSrc >> 2) & 1;
        *(pDest+6) = (*pSrc >> 1) & 1;
        *(pDest+7) = *pSrc  & 1;
        pSrc++;
        pDest += 8;
      }
      if (Width & 7)  // Check for leftover pixels
        for (i=7; i>(8-Width & 7); i--)
        {
          *pDest = (*pSrc >> i) & 1;
          pDest++;
        }
      break;
    default:
      raiseError (ERR_FORMAT_UNKNOWN,
                  "Bad bits per pixel in expandBuf8.");
  }
  return;
}


/////////////////////////////////////////////////////////////////////
// Auxillary functions

void CPictDecoder::readPixmap
    ( MacpixMap * pPixMap
    )
{
  pPixMap->version = ReadMWord();
  pPixMap->packType = ReadMWord();
  pPixMap->packSize = ReadMLong();
  pPixMap->hRes = ReadMLong();
  pPixMap->vRes = ReadMLong();
  pPixMap->pixelType = ReadMWord();
  pPixMap->pixelSize = ReadMWord();
  pPixMap->cmpCount = ReadMWord();
  pPixMap->cmpSize = ReadMWord();
  pPixMap->planeBytes = ReadMLong();
  pPixMap->pmTable = ReadMLong();
  pPixMap->pmReserved = ReadMLong();

  tracePixMapHeader (2, pPixMap);
}

RGBAPIXEL * CPictDecoder::readColourTable
    ( WORD * pNumColors
    )
    // Reads a mac colour table into a bitmap palette.
{
  LONG        ctSeed;
  WORD        ctFlags;
  WORD        val;
  int         i;
  RGBAPIXEL * pColTable;

  trace (3, "Getting color table info.\n");

  ctSeed = ReadMLong();
  ctFlags = ReadMWord();
  *pNumColors = ReadMWord()+1;

  char sz[256];
  sprintf (sz, "Palette Size:  %d\n", *pNumColors);
  trace (2, sz);
  trace (3, "Reading Palette.\n");

  pColTable = new RGBAPIXEL[256];

  if (!pColTable)
    raiseError (ERR_NO_MEMORY, "Out of memory allocationg color table.");

  for (i = 0; i < *pNumColors; i++)
  {
    val = ReadMWord();
    if (ctFlags & 0x8000)
      // The indicies in a device colour table are bogus and
      // usually == 0, so I assume we allocate up the list of
      // colours in order.
      val = i;
    if (val >= *pNumColors)
    {
      delete pColTable;
      raiseError (ERR_FORMAT_UNKNOWN,
                  "pixel value greater than colour table size.");
    }
    // Mac colour tables contain 16-bit values for R, G, and B...
    *(((BYTE*)(pColTable+val))+RGBA_RED) = HIBYTE (ReadMWord());
    *(((BYTE*)(pColTable+val))+RGBA_GREEN) = HIBYTE (ReadMWord());
    *(((BYTE*)(pColTable+val))+RGBA_BLUE) = HIBYTE (ReadMWord());
    *(((BYTE*)(pColTable+val))+RGBA_ALPHA) = 0xFF;
  }

  return pColTable;
}

void CPictDecoder::readRect
    ( MacRect * pr
    )
{
  pr->top = ReadMWord();
  pr->left = ReadMWord();
  pr->bottom = ReadMWord();
  pr->right = ReadMWord();
}


void CPictDecoder::dumpRect
    ( char * psz,
      MacRect * pr
    )
{
  char sz[256];
  sprintf (sz, "%s (%d,%d) (%d,%d).\n",
           psz, pr->left, pr->top, pr->right, pr->bottom);
  trace (2, sz);
}


void CPictDecoder::tracePixMapHeader
    ( int Level,
      MacpixMap * pPixMap
    )
{
  char sz[256];
  trace (Level, "PixMap header info:\n");
  dumpRect ("  Bounds:", &(pPixMap->Bounds));

  sprintf (sz, "  version: 0x%x\n", pPixMap->version);
  trace (Level, sz);
  sprintf (sz, "  packType: %d\n", pPixMap->packType);
  trace (Level, sz);
  sprintf (sz, "  packSize: %ld\n", pPixMap->packSize);
  trace (Level, sz);
  sprintf (sz, "  pixelSize: %d\n", pPixMap->pixelSize);
  trace (Level, sz);
  sprintf (sz, "  cmpCount: %d\n", pPixMap->cmpCount);
  trace (Level, sz);
  sprintf (sz, "  cmpSize: %d.\n", pPixMap->cmpSize);
  trace (Level, sz);
  sprintf (sz, "  planeBytes: %ld.\n", pPixMap->planeBytes);
  trace (Level, sz);
}

