/*
/--------------------------------------------------------------------
|
|      ANYBMP.CPP         Device independent bitmap class
|
|        Plattform-independent version
|
|        Manipulates uncompressed device independent bitmaps.
|
|        Supported data formats are 8 and 32 bpp. The data is stored
|        sequentially without padding in the bitmap.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "anybmp.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
IMPLEMENT_DYNAMIC (CAnyBmp, CBitmap);
#endif


CAnyBmp::CAnyBmp
    ()
    // Creates an empty bitmap.
{
  internalCreate(16, 16, 8, FALSE);

  ASSERT_VALID(this);
}


CAnyBmp::~CAnyBmp
    ()
{
  // Free the memory.
  freeMembers();
}


long CAnyBmp::GetMemUsed
    ()
    // Returns the memory used by the object.
{
  ASSERT_VALID (this);

  return GetMemNeeded (GetWidth(), GetHeight(), GetBitsPerPixel())+
         sizeof (*this);
}


long CAnyBmp::GetBytesPerLine
    ()
    // Returns number of bytes used per line.
{
  return GetWidth()*(GetBitsPerPixel()/8);
}


/////////////////////////////////////////////////////////////////////
// Static functions

long CAnyBmp::GetBitsMemNeeded
    ( LONG width,
      LONG height,
      WORD BitsPerPixel
    )
    // Returns memory needed by bitmap bits.
{
  // Calculate memory per line.
  int LineMem = width*(BitsPerPixel/8);

  // Multiply by number of lines
  return LineMem*height;
}


long CAnyBmp::GetMemNeeded
    ( LONG width,
      LONG height,
      WORD BitsPerPixel
    )
    // Returns memory needed by a bitmap with the specified attributes.
{
  int HeaderMem = sizeof (CAnyBmp);
  if (BitsPerPixel < 16)
  { // Palette memory
    HeaderMem += (1 << BitsPerPixel)*sizeof (RGBAPIXEL);
  }

  return HeaderMem+GetBitsMemNeeded (width, height, BitsPerPixel);
}


/////////////////////////////////////////////////////////////////////
// Local functions


void CAnyBmp::internalCreate
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
    // Create a new empty bitmap. Bits are uninitialized.
    // Assumes that no memory is allocated before the call.
{
  // Allocate memory
  int MemNeeded = GetMemNeeded (Width, Height, BitsPerPixel);

#ifdef MAX_BITMAP_SIZE
  if (MemNeeded > MAX_BITMAP_SIZE)
    throw CTextException(ERR_DIB_TOO_LARGE, "Bitmap size too large.\n");
#endif

//  m_pBits = (BYTE *) malloc (GetBitsMemNeeded (Width, Height,
//                                               BitsPerPixel));
  m_pBits = new BYTE [GetBitsMemNeeded (Width, Height, BitsPerPixel)];

  if (BitsPerPixel == 8)
    m_pClrTab = new RGBAPIXEL [256];
   else
    m_pClrTab = NULL;
  initLocals (Width, Height, BitsPerPixel, bAlphaChannel);

  ASSERT_VALID (this);
}


void CAnyBmp::initLineArray
    ()
{
//  m_pLineArray = (BYTE **) malloc (m_Height * sizeof (BYTE *));
  m_pLineArray = new BYTE * [m_Height];
  int LineLen = GetBytesPerLine();

  for (int y=0; y<m_Height; y++)
    m_pLineArray[y] = m_pBits + y*LineLen;
}

void CAnyBmp::freeMembers
    ()
{
  delete m_pBits;
  m_pBits = NULL;

  if (m_pClrTab)
  {
    delete m_pClrTab;
    m_pClrTab = NULL;
  }

  delete m_pLineArray;
  m_pLineArray = NULL;
}

