///////////////////////////////////////////////////////////////////
//
//  CodeMax sample app : CodeMax Editor
//
//  (c) WinMain Software 1998
//
//

#include "precomp.h"
#include "resource.h"
#include "editapp.h"
#include "codefile.h"

CCodeFile::CCodeFile( CEditorApp *pApp )
{
	m_pApp = pApp;
	m_hWnd = m_hWndEdit = NULL;
	m_bDoIdleProcessing = FALSE;
	*m_szPath = _T('\0');
	UpdateFileTime();
}

CCodeFile::~CCodeFile()
{
}

BOOL CCodeFile::RegisterClass( HINSTANCE hInstance )
{
	// register the CodeMax edit control
	CMRegisterControl();

	WNDCLASS    wndclass;

	wndclass.style         = 0;
	wndclass.lpfnWndProc   = WndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor( NULL, IDC_ARROW );
	wndclass.hbrBackground = NULL;
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = CODEFILEWNDCLASS;

	return ::RegisterClass( &wndclass );
}

BOOL CCodeFile::UnregisterClass( HINSTANCE hInstance )
{
	// unregister the CodeMax edit control
	CMUnregisterControl();

	return ::UnregisterClass( CODEFILEWNDCLASS, hInstance );
}

void CCodeFile::Initialize( HWND hWnd )
{
	m_hWnd = hWnd;
	m_hWndEdit = CreateWindow( CODEMAXWNDCLASS,
	                           _T(""),
							   WS_CHILD | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL,
							   0,
							   0,
							   100,
							   100,
							   m_hWnd,
							   NULL,
							   m_pApp->GetHINSTANCE(),
							   NULL );
	ASSERT( IsWindow( m_hWndEdit ) );
	// default language to C++
	VERIFY( SendMessage( m_hWndEdit, CMM_SETLANGUAGE, 0, ( LPARAM ) CMLANG_CPP ) == CME_SUCCESS );

	// Save the settings in the registry
	m_pApp->LoadProfile( m_hWndEdit );
}

// Open() will load a file into a code window
void CCodeFile::Open( LPCTSTR pszPath )
{
	SetPath( pszPath );
	if ( SendMessage( m_hWndEdit, CMM_OPENFILE, 0, ( LPARAM ) m_szPath ) != CME_SUCCESS )
	{
		DWORD dwLastError = GetLastError();
		if ( dwLastError != ERROR_FILE_NOT_FOUND )
		{
			TCHAR szMsg[ 1024 ];
			_tcscpy( szMsg, _T("Failed to open document:\n\n") );
			LPTSTR lpMessageBuffer;

			FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
	  					  NULL,
						  dwLastError,
	  					  MAKELANGID( LANG_ENGLISH, SUBLANG_ENGLISH_US ),
	  					  (LPTSTR) &lpMessageBuffer, 0, NULL );
			_tcscat( szMsg, lpMessageBuffer );
			MessageBox( m_pApp->GetHWND(), szMsg, APPNAME, MB_OK | MB_ICONEXCLAMATION );
		}
	}
	UpdateWindowTitle();
	// Tell CodeMax what the language is so it can do color syntax stuff
	// and auto indentation
	SetLanguageBasedOnFileType();
	UpdateFileTime();
}

// Save() will save the contents of the edit window.  This function should only be
// called after calling SetPath()
BOOL CCodeFile::Save()
{
	ASSERT( *m_szPath );
	BOOL bSuccess = SendMessage( m_hWndEdit, CMM_SAVEFILE, 1, ( LPARAM ) m_szPath ) == CME_SUCCESS;
	if ( !bSuccess )
	{
		// CodeMax could not save the file -- out of diskspace or readonly or something.
		TCHAR szMsg[ _MAX_PATH + 500 ];
		TCHAR szTitle[ _MAX_PATH + 1 ];
		GetTitle( szTitle );
		wsprintf( szMsg, _T("Failed to save %s:\n\n"), szTitle );
		DWORD dwLastError = GetLastError();
		LPTSTR lpMessageBuffer;

		FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
	  				  NULL,
					  dwLastError,
	  				  MAKELANGID( LANG_ENGLISH, SUBLANG_ENGLISH_US ),
	  				  (LPTSTR) &lpMessageBuffer, 0, NULL );
		_tcscat( szMsg, lpMessageBuffer );
		MessageBox( m_pApp->GetHWND(), szMsg, APPNAME, MB_OK | MB_ICONEXCLAMATION );
	}
	UpdateFileTime();
	return ( bSuccess );
}

// SetLanguageBasedOnFileType() will configure the CodeMax control to the language specific to
// this file
void CCodeFile::SetLanguageBasedOnFileType()
{
	// based on the file name extension, set the language
	LPCTSTR pszDot = _tcsrchr( m_szPath, '.' );
	if ( pszDot )
	{
		TCHAR szExt[ _MAX_PATH ];
		_tcscpy( szExt, pszDot + 1 );
		CharUpper( szExt );

		LPCTSTR pszLang = NULL;

		if ( !_tcscmp( szExt, _T("C") ) || !_tcscmp( szExt, _T("CPP") ) || 
			 !_tcscmp( szExt, _T("H") ) || !_tcscmp( szExt, _T("HPP") ) )
		{
			// C/C++
			pszLang = CMLANG_CPP;
		}
		else if ( !_tcscmp( szExt, _T("JAVA") ) )
		{
			// Java
			pszLang = CMLANG_JAVA;
		}
		else if ( !_tcscmp( szExt, _T("BAS") ) || !_tcscmp( szExt, _T("INC") ) )
		{
			// Basic
			pszLang = CMLANG_BASIC;
		}
		else if ( !_tcscmp( szExt, _T("PAS") ) )
		{
			// Pascal
			pszLang = CMLANG_PASCAL;
		}
		else if ( !_tcscmp( szExt, _T("SQL") ) || !_tcscmp( szExt, _T("DDL") ) )
		{
			// SQL
			pszLang = CMLANG_SQL;
		}

		VERIFY( SendMessage( m_hWndEdit, CMM_SETLANGUAGE, 0, ( LPARAM ) pszLang ) == CME_SUCCESS );
	}
}

// GetTitle() calculates a window title suitable for display
LPCTSTR CCodeFile::GetTitle( LPTSTR pszBuff ) const
{
	ASSERT( pszBuff );
	if ( *m_szPath )
	{
		_tcscpy( pszBuff, m_szPath );
	}
	else
	{
		_tcscpy( pszBuff, _T("Untitled") );
	}
	return pszBuff;
}

// UpdateWindowTitle() updates the file caption based on the current state of the window contents.
// An asterisk (*) is appended if the window contents are modified
void CCodeFile::UpdateWindowTitle()
{
	TCHAR szTitle[ _MAX_PATH + 1 ];

	GetTitle( szTitle );
	if ( SendMessage( m_hWndEdit, CMM_ISMODIFIED, 0, 0 ) )
	{
		_tcscat( szTitle, _T("*") );
	}

	SetWindowText( m_hWnd, szTitle );
}

// SetPath() updates the file path of this code file
void CCodeFile::SetPath( LPCTSTR pszPath )
{
	_tcsncpy( m_szPath, pszPath, _MAX_PATH );
	UpdateWindowTitle();
	SetLanguageBasedOnFileType();
	UpdateFileTime();
	UpdateReadOnlyState();
	m_bDoIdleProcessing = FALSE;
	
	// get the root dir
	TCHAR szRootDir[ _MAX_PATH ];
	_tcscpy( szRootDir, m_szPath );
	if ( *szRootDir == _T('\\') && szRootDir[ 1 ] == _T('\\') )
	{
		// file is on a network resource
		// e.g. \\fileserver1\disk1\file.cpp
		// don't allow background processing -- it may be too slow
	}
	else
	{
		// file is on a disk with a drive letter (e.g. a:\file.cpp)
		LPTSTR psz = szRootDir;
		while ( *psz && *psz != _T('\\') )
		{
			psz++;
		}
		*psz = _T('\0');
		switch ( GetDriveType( szRootDir ) )
		{
			case DRIVE_UNKNOWN:
			case DRIVE_NO_ROOT_DIR:
			case DRIVE_REMOVABLE:
			case DRIVE_CDROM:
				// pointless, or slow
				m_bDoIdleProcessing = FALSE;
				break;
			case DRIVE_FIXED:
			case DRIVE_REMOTE:
			case DRIVE_RAMDISK:
				m_bDoIdleProcessing = TRUE;
				break;
		}
	}
}

// IsCodeUpToDate() will check to see the file on disk is newer than the window contents.
// If bStale is TRUE, then the code is not up to date if the file on disk is newer.
// If bStale is FALSE, then the code is not up to date if the file on disk is newer or older
BOOL CCodeFile::IsCodeUpToDate( BOOL bStale ) const
{
	BOOL bUpToDate = TRUE;
	if ( *m_szPath )
	{
		FILETIME time;
		GetFileTimeFromDisk( time );
		int nCompare = CompareFileTime( &time, &m_timeFileSaved );
		bUpToDate = bStale ? ( nCompare <= 0 ) : ( nCompare == 0 );
	}

	return bUpToDate;
}

// updates the internal file time
void CCodeFile::UpdateFileTime()
{
	FILETIME time;
	GetFileTimeFromDisk( time );
	m_timeFileSaved = time;
}

// simple function to get the FILETIME for the underlying file on disk.  If there is no
// file, then this function will set the time to 'zero'.
void CCodeFile::GetFileTimeFromDisk( FILETIME &time ) const
{
	time.dwLowDateTime = time.dwHighDateTime = 0;
	if ( *m_szPath )
	{
		HANDLE hFile = CreateFile( m_szPath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL );
		if ( hFile != INVALID_HANDLE_VALUE )
		{
			GetFileTime( hFile, NULL, NULL, &time );
			CloseHandle( hFile );
		}
	}
}

// Abort dialog proc for the 'Cancel Printing' dialog
BOOL CALLBACK AbortDlgProc( HWND /* hWndDlg */, UINT uMsg, WPARAM wParam, LPARAM lParam )
{
	static LPBOOL pbAbort = NULL;
	switch ( uMsg )
	{
		case WM_INITDIALOG:
		{
			ASSERT( lParam );
			pbAbort = ( LPBOOL ) lParam;
			return TRUE;
		}

		case WM_COMMAND:
		{
			WORD wID = LOWORD( wParam );  

			if ( LOWORD( wParam ) == IDCANCEL && HIWORD( wParam ) == BN_CLICKED )
			{
				*pbAbort = TRUE;
			}
			break;
		}
	}

	return 0;
}

// Prints the contents of a code file.  This function will display the print dialog and then the
// print status dialog box.
void CCodeFile::Print() const
{
	PRINTDLG pd = {
		sizeof( PRINTDLG ),
		m_hWnd,
		m_pApp->GetDevMode(),
		m_pApp->GetDevNames(),
		NULL,
		PD_RETURNDC,
		0,
		0,
		0,
		0,
		0,
		m_pApp->GetHINSTANCE(),
		0,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL
	};
	if ( !PrintDlg( &pd ) )
	{
		return;
	}

	HDC hDC = pd.hDC;
	
	// re-calc the printer settings -- the PrintDlg() function may realloc them
	m_pApp->SetDevMode( pd.hDevMode );
	m_pApp->SetDevNames( pd.hDevNames );

	TCHAR szTitle[ _MAX_PATH + 1 ];
	GetTitle( szTitle );
	DOCINFO di = {
		sizeof( DOCINFO ),
		szTitle,
		NULL,
		NULL,
		0
	};

	// use an 11 point font
	int cyLine = MulDiv( 11, GetDeviceCaps( hDC, LOGPIXELSY ), 72 );
	HFONT hFont = CreateFont( cyLine, 0, 0, 0, FW_NORMAL, 0, 0, 0, 0, 0, 0, 0, 0, _T("Courier New") );
	SelectObject( hDC, hFont );

	// calc the paper size
	int cyPaper = GetDeviceCaps( hDC, VERTRES );
	cyPaper = ( cyPaper / cyLine ) * cyLine;

	BOOL bAbort = FALSE;

	// display the cancel dialog
	HWND hWndAbort = CreateDialogParam( m_pApp->GetHINSTANCE(), MAKEINTRESOURCE( IDD_ABORTPRINT ), m_pApp->GetHWND(), ( DLGPROC ) AbortDlgProc, ( LPARAM ) &bAbort );
	ShowWindow( hWndAbort, SW_SHOW );

	VERIFY( StartDoc( hDC, &di ) > 0 );

	TEXTMETRIC tm;
	VERIFY( GetTextMetrics( hDC, &tm ) );
	int cxTab = tm.tmAveCharWidth * CM_GetTabSize( m_hWndEdit );
	int nLineCount = CM_GetLineCount( m_hWndEdit );
	int nLine = 0;

	BOOL bContinue = TRUE;
	while ( bContinue && !bAbort )
	{
		VERIFY( StartPage( hDC ) > 0 );
		int cy = 0;
		while ( cy < cyPaper )
		{
			if ( nLine >= nLineCount )
			{
				// end of doc
				bContinue = FALSE;
				break;
			}
			else
			{
				// print a line
				int cbLine = CM_GetLineLength( m_hWndEdit, nLine );
				LPTSTR pszLine = new TCHAR[ cbLine + 1 ];
				CM_GetLine( m_hWndEdit, nLine, pszLine );
				TabbedTextOut( hDC, 0, cy, pszLine, cbLine, 1, &cxTab, 0 );
				cy += cyLine;
				delete pszLine;
				nLine++;
			}
		}
		VERIFY( EndPage( hDC ) > 0 );
		// flush the message queue for the abort dlg so it can process the Cancel button
		MSG msg;
		while ( PeekMessage( &msg, hWndAbort, NULL, NULL, PM_REMOVE ) )
		{
			TranslateMessage( &msg );
			DispatchMessage( &msg );
		}
	}
	

	if ( bAbort )
	{
		VERIFY( AbortDoc( hDC ) > 0 );
	}
	else
	{
		VERIFY( EndDoc( hDC ) > 0 );
	}

	DestroyWindow( hWndAbort );
	DeleteDC( hDC );
	DeleteObject( hFont );
}

LRESULT CCodeFile::OnCreate( HWND hWnd, WPARAM wParam, LPARAM lParam )
{
	ASSERT( lParam );
	CREATESTRUCT *pcs = ( CREATESTRUCT * )lParam;
	ASSERT( pcs->lpCreateParams );
	MDICREATESTRUCT *pmcs = ( MDICREATESTRUCT * ) pcs->lpCreateParams;

	// associate the CCodeFile object with the window
	CCodeFile *pCode = ( CCodeFile * ) pmcs->lParam;
	SetWindowLong( hWnd, GWL_USERDATA, ( DWORD ) pCode );
	pCode->Initialize( hWnd );
	return DefMDIChildProc( hWnd, WM_CREATE, wParam, lParam );
}

LRESULT CCodeFile::OnNCDestroy( WPARAM wParam, LPARAM lParam )
{
	// clean up this memory
	LRESULT lRes = DefMDIChildProc( m_hWnd, WM_NCDESTROY, wParam, lParam );
	delete this;
	return lRes;
}


LRESULT CCodeFile::OnSize( WPARAM wParam, LPARAM lParam )
{
	int cx = LOWORD( lParam );
	int cy = HIWORD( lParam );

	// resize the CodeMax control
	ASSERT( IsWindow( m_hWndEdit ) );
	SetWindowPos( m_hWndEdit, NULL, 0, 0, cx, cy, SWP_NOZORDER );

	if ( ( wParam == SIZE_RESTORED || wParam == SIZE_MAXIMIZED ) && IsWindowVisible( m_hWnd ) )
	{
		m_pApp->WriteProfileInt( CODE_SECTION, KEY_CODEMAXIMIZED, wParam == SIZE_MAXIMIZED );
	}

	return DefMDIChildProc( m_hWnd, WM_SIZE, wParam, lParam );
}

LRESULT CCodeFile::OnSetFocus( WPARAM wParam, LPARAM lParam )
{
	ASSERT( IsWindow( m_hWndEdit ) );
	SetFocus( m_hWndEdit );

	return DefMDIChildProc( m_hWnd, WM_SETFOCUS, wParam, lParam );
}

LRESULT CCodeFile::OnMDIActivate( WPARAM wParam, LPARAM lParam )
{
	// let the app know that this file is now the active file
	m_pApp->SetActiveCode( m_hWnd == ( HWND ) lParam ? this : NULL );
	return DefMDIChildProc( m_hWnd, WM_MDIACTIVATE, wParam, lParam );
}

LRESULT CCodeFile::OnModifiedChanged()
{
	// Update the dirty flag (*) on the caption
	UpdateWindowTitle();
	return 0;
}

LRESULT CCodeFile::OnPropsChanged()
{
	// Save the settings in the registry
	m_pApp->SaveProfile( m_hWndEdit );
	return 0;
}

LRESULT CCodeFile::OnSelChanged()
{
	// CodeMax edit selection changed -- update the status bar
	CM_RANGE sel;
	CM_GetSel( m_hWndEdit, &sel, FALSE );
	m_pApp->OnSelChanged( sel.posEnd.nLine, sel.posEnd.nCol );
	return 0;
}

LRESULT CCodeFile::OnClose( WPARAM wParam, LPARAM lParam )
{
	BOOL bClose = TRUE;
	// if window contents are modified, prompt user to save
	if ( SendMessage( m_hWndEdit, CMM_ISMODIFIED, 0, 0 ) )
	{
		TCHAR szMsg[ _MAX_PATH + 100 ];
		TCHAR szTitle[ _MAX_PATH + 1 ];
		GetTitle( szTitle );
		wsprintf( szMsg, _T("Save changes to %s?"), szTitle );
		int nResult = MessageBox( m_pApp->GetHWND(), szMsg, APPNAME, MB_YESNOCANCEL | MB_ICONEXCLAMATION );
		if ( nResult == IDYES )
		{
			// yes -- save
			bClose = m_pApp->OnFileSave( this );
		}
		else if ( nResult == IDCANCEL )
		{
			// cancel the close operation
			bClose = FALSE;
		}
	}

	if ( lParam )
	{
		// tell the caller whether the window was closed.
		*( BOOL * ) lParam = bClose;
	}
	return bClose ? DefMDIChildProc( m_hWnd, WM_CLOSE, wParam, lParam ) : 0;
}

LRESULT CCodeFile::OnEraseBkgnd( WPARAM /* wParam */, LPARAM /* lParam */ )
{
	// don't bother erasing the bkgnd.  The edit control will do it.
	return 0;
}

// UpdateReadOnlyState() syncs the buffer's readonly state with the underlying file's 
// readonly state
void CCodeFile::UpdateReadOnlyState()
{
	if ( *m_szPath )
	{
		DWORD dwFlags = GetFileAttributes( m_szPath );
		if ( dwFlags != 0xffffffff )
		{
			BOOL bReadOnly = ( ( dwFlags & FILE_ATTRIBUTE_READONLY ) == FILE_ATTRIBUTE_READONLY );
			if ( bReadOnly != IsReadOnly() )
			{
				CM_SetReadOnly( m_hWndEdit, bReadOnly );
			}
		}
	}
}
