/*
 * Some kind of Ultima comment block goes here
 */

/*+
 *
 * Module = List
 *
 * :Overview
 *	Generic linked list data type.  
 *
 * :Prefix
 *	LS
 *
 * :Description
 *	This module implements a simple generic linked list data type.  
 *	It uses a doubly linked list structure and provides some standard 
 *	operations for storing and retrieving data from the list.
 *
 *	The (client) data objects stored in a list are ^lvoid*^n pointers.
 *
 * :Data Types
 *	@ ^xLs_Pool^n - list memory pool
 *	@ ^xLs_List^n - list
 *	@ ^xLs_Elem^n - element handle
 *	@ ^xLs_Gen^n - generator
 *	@ ^xLs_Status^n - return codes
 *
 * :Functions
 *
 *	List creation, deletion and copying:
 *	@ ^xLs_MakePool^n - create list memory pool
 *	@ ^xLs_Create^n - create empty list
 *	@ ^xLs_CreateLP^n - create empty list (using list pool)
 *	@ ^xLs_Destroy^n - destroy a list
 *	@ ^xLs_Copy^n - copy a list 
 *	@ ^xLs_CopyLP^n - copy a list (using list pool)
 *
 *	Accessing or manipulating elements at list ends:
 *	@ ^xLs_FirstElem^n - get head element of list
 *	@ ^xLs_LastElem^n - get tail element of list
 *	@ ^xLs_NewBegin^n - add element to head of list
 *
 *	Generating and iterating:
 *	@ ^xLs_Start^n - create generator before list head
 *	@ ^xLs_End^n - create generator after list tail
 *	@ ^xLs_ElemGen^n - create generator before/after given element
 *	@ ^xLs_Next^n - generate next element
 *	@ ^xLs_Prev^n - generate previous element

 *	@ ^xLs_Foreach^n - apply a function to each element (first to last)
 *	@ ^xLs_Backeach^n - apply a function to each element (last to first)
 *
 *
 *	Element membership, counting and locating:
 *	@ ^xLs_Length^n - get number of elements in list
 *
 *
 *	Miscellaneous:
 *	@ ^bLs_PoolStats^n - print some statistics about a list pool
 *
 * :Remarks
 *	This module is based on the ^e List Management Package^n 
 *	(a.k.a. ^llsList^n module) written by David Harrison of UC Berkeley.
 *
 * :See Also
 *	<memory pool module>
 *
+*/


#ifndef LS_DEFINED
#define LS_DEFINED

#include "vmSys.h"	/* memory pool manager stuff */

/* 
 * ==========================================================================
 *
 * 			    TYPE DEFINITIONS
 *
 * ==========================================================================
 */


/*+
 *
 * Data Type = Ls_Pool
 *
 * :Overview
 *	Handle to a list memory pool.
 *
 * :Syntax
 *	| typedef struct Ls_PoolDesc *Ls_Pool;
 *
 * :Defined In
 *	| list.h
 *
 * :Description
 *	In order to work properly, the list module needs to be able
 *	to allocate memory for internal objects and maintain certain
 *	internal state (e.g. free lists, statistics, etc.).
 *	The list module can allocate memory internally or use a memory 
 *	pool supplied by the client for this purpose.
 *
 *	For the latter case, the function ^xLs_MakePool^n can be used 
 *	to create appropriate internal state and associate it with a
 *	given memory pool, and then return a handle to this state as
 *	an object of type ^xLs_Pool^n.
 *	This handle can then be passed to various list creation functions.
 *
 *	All list state and storage (both in use and free) belonging to
 *	a given ^xLs_Pool^n will be released (and hence become invalid)
 *	when its associated memory pool is freed or refurbished.
 *
 *      It is possible to define more than one list pool within
 *	a given memory pool.  In some cases this may be desirable from
 *	a software organization point of view.  However, sharing list
 *	objects within the same pool leads to a more efficient use of
 *	resources.
 *
 * :Remarks
 *	The ^xLs_Pool^n handle type is implemented as a pointer to
 *	a undefined structure type.  This permits proper type checking
 *	without exposing its implementation details.
 *
 * :Example
 *	| Vm_MemId memoryPool;
 *	| Ls_Pool listPool;
 *	| Ls_List list1, list2;
 *	| 
 *	| /# 
 *	|  * make a memory pool and an associated list pool
 *	|  #/
 *	| memoryPool = Vm_GetMemPoolId( 4000, 24 ); 
 *	| listPool = Ls_MakePool( memoryPool );
 *	| 
 *	| list1 = Ls_CreateLP( listPool );  /# create an empty list #/
 *	| 
 *	| ... do some list operations ...
 *	| 
 *	| list2 = Ls_Copy( list1, NULL );   /# copy it #/
 *	| 
 *	| ... more operations ...
 *	| 
 *	| Vm_FreeMemPool( memoryPool );
 *
 *
 * :See Also
 *	^xList^n
 *	<memory manager overview>
 *	^xVm_CreateMemPoolID^n
 *      ^xVm_FreeMemPool^n
 *      ^xVm_ReburbishMemPool^n
 *	^xLs_MakePool^n
 *	<functions that take a Ls_Pool>
 *
+*/
typedef struct Ls_PoolDesc	*Ls_Pool;


/*+
 *
 * Data Type = Ls_List
 *
 * :Overview
 *	Handle to a list.
 *
 * :Syntax
 *	| typedef struct Ls_ListDesc *Ls_List;	
 *
 * :Defined In
 *	| list.h
 *
 * :Description
 *	The ^xLs_List^n type is represents handle to a list instance.
 *	Empty lists can be created with ^xLs_Create^n and many other
 *	list functions take or return this type of object.
 *
 * :Remarks
 *	The ^xLs_List^n handle type is implemented as a pointer to
 *	a undefined structure type.  This permits proper type checking
 *	without exposing its implementation details.
 *
 * :See Also
 *	^xList^n,
 *	^xLs_Create^n,
 *	^xLs_Destroy^n,
 *	functions that return lists.
 *
+*/
typedef struct Ls_ListDesc	*Ls_List;	/* list handle */


/*+
 *
 * Data Type = Ls_Gen
 *
 * :Overview
 *	Generator
 *
 * :Syntax
 *	| typedef struct Ls_GenDesc *Ls_Gen;	
 *
 * :Defined In
 *	| list.h
 *
 * :Description
 *	A generator object represents a specific postion in a given list.
 *	They are used for iterating, accessing, and/or manipulating items
 * 	in a list.
 *	A generator represents a position ^ebetween^n two list elements.
 *	In the extreme cases, a generator may be positioned before the 
 *	first element in the list, or after the last element.
 *
 *	Generators can be created and destroyed with: 
 *	^xLs_Start^n, ^xLs_End^n, ^xLs_ElemGen^n and ^xLs_Finish^n.
 *
 *	The functions ^xLs_Next^n and ^xLs_Prev^n 
 *	move a generator forwards or backwards over one element.
 *
 *	To insert or delete elements near a generator, use:
 *	^xLs_InBefore^n, ^xLs_InAfter^n, ^xLs_DelBefore^n, or
 *	^xLs_DelAfter^n. 
 *	
 * :Remarks
 *	The ^xLs_Gen^n handle type is implemented as a pointer to
 *	a undefined structure type.  This permits proper type checking
 *	without exposing its implementation details.
 *
 * :See Also
 *	^xList^n
 *
+*/
typedef struct Ls_GenDesc	*Ls_Gen;	/* generator handle */

/*+
 *
 * Data Type = Ls_Elem
 *
 * :Overview
 *	Handle to a list element.
 *
 * :Syntax
 *	| typedef struct Ls_ElemDesc *Ls_Elem;	
 *
 * :Defined In
 *	| list.h
 *
 * :Description
 *	Routines which return items of a list optionally return a 
 *	^elist element handle^n of type ^xLs_Elem^n.
 *	This handle can be stored in user data structures and later used to
 *	quickly access the item without searching through the list.  
 *	If you do not wish to use handles, you can pass the zero handle,
 *	^l(Ls_Elem *)0^n, to the routine.  
 *	For brevity, the ^xLS_NH^n macro may be used to specify
 *	no handle to routines which return a handle.
 *
 * :Remarks
 *	The ^xLs_Elem^n handle type is implemented as a pointer to
 *	a undefined structure type.  This permits proper type checking
 *	without exposing its implementation details.
 *
 * :See Also
 *	^xList^n,
 *	functions that return element handles
 *
+*/
typedef struct Ls_ElemDesc	*Ls_Elem;	/* item handle */


/*+
 *
 * Data Type = Ls_Status
 * :Overview
 *	Return codes
 *
 * :Syntax
 *	| typedef int Ls_Status;
 *
 * :Defined In
 *	| list.h
 *
 * :Description
 *	Many of the list functions provide an (integer) return code.
 *	The symbolic values and their meanings are summarized below.
 *	@ ^xLS_OK^n - routine completed successfully
 *	@ ^xLS_NOMORE^n - no item to return/delete/generate
 *	@ ^xLS_BADPARAM^n - client function in
 *	    ^xLs_Foreach^n or ^xLs_Backeach^n provided illegal
 *	    return code.
 *	@ ^xLS_STOP^n - ^xLs_Foreach^n or ^xLs_Backeach^n
 *	    stopped prematurely by client function.
 *
 * :See Also
 *	^xList^n
 *
+*/
typedef int	Ls_Status;	/* Return codes          */

/* typedef void	*Ls_Generic;	/* Generic pointer       */
/* typedef void	*Ls_GenericLoc;	/* Address of a generic pointer *
/* 				 * (this is used to get around  *
				 * the fact that void ** is not *
				 * a generic pointer)	 */

#define	LS_NIL		0	/* Nil for Ls_List       */

#define LS_BADPARAM	-2	/* Bad parameter value   */
#define LS_NOMORE	-1	/* No more items         */

#define	LS_OK		0

#define LS_BEFORE	1	/* Set spot before object */
#define LS_AFTER	2	/* Set spot after object  */
#define LS_STOP		3	/* Stop generating items  */
#define LS_DELETE	4	/* Delete generated item  */

/*
 * For all those routines that take a handle, this macro can be
 * used when no handle is required.
 */
#define LS_NH		(Ls_Elem *)0

#define LS_NULLPOOL	(Ls_Pool)0




/* 
 * ==========================================================================
 *
 * 			    FUNCTION PROTOTYPES
 *
 * ==========================================================================
 */


/*+
 *
 * Function = Ls_MakePool
 *
 * :Overview
 *	Create a list pool within a given memory pool.
 *
 * :Prototype
 *	| Ls_Pool Ls_MakePool( Vm_MemID memPool );
 *
 * :Prototype In
 *	| list.h
 *
 * :Description
 *	Most list creation functions come in two flavors: one that uses
 * 	memory internal to the list module, and one that uses memory 
 *	within a user defined memory pool.  If a memory pool is used,
 *	the list with allocate and release its memory within that pool.
 *	When the memory pool is freed or refurbished, all list storage
 *	associated with the pool (both in use and free) will become
 *	invalid.
 *
 *	In order to use a memory pool, the list module needs to keep some
 *	internal state (free lists, statistics, etc.).  The ^xLs_MakePool^n
 *	function initializes this state and returns a handle to it.
 *
 * :Remarks
 *      It is possible to define more than one list pool within
 *	a given memory pool.  In some cases this may be desirable from
 *	a software organization point of view.  However, sharing list
 *	objects within the same pool leads to a more efficient use of
 *	resources.
 *
 * :Return Value
 *	An ^xLs_Pool^n.
 *
 * :Example
 *	| Vm_MemId memoryPool;
 *	| Ls_Pool listPool;
 *	| Ls_List list1, list2;
 *	| 
 *	| /# 
 *	|  * make a memory pool and an associated list pool
 *	|  #/
 *	| memoryPool = Vm_GetMemPoolId( 4000, 24 ); 
 *	| listPool = Ls_MakePool( memoryPool );
 *	| 
 *	| list1 = Ls_CreateLP( listPool );  /# create an empty list #/
 *	| 
 *	| ... do some list operations ...
 *	| 
 *	| list2 = Ls_Copy( list1, NULL );   /# copy it #/
 *	| 
 *	| ... more operations ...
 *	| 
 *	| Vm_FreeMemPool( memoryPool );
 *
 *
 *
 * :See Also
 *	^xList^n
 *	<memory manager overview>
 *	^xVm_CreateMemPoolID^n
 *      ^xVm_FreeMemPool^n
 *      ^xVm_ReburbishMemPool^n
 *	<functions that take a Ls_Pool>
 *
+*/
extern Ls_Pool Ls_MakePool( Vm_MemID memPool );


/*+
 *
 * Function = Ls_Create, Ls_CreateLP
 *
 * :Overview
 *	Create a new list
 *
 * :Prototype
 *	| Ls_List Ls_Create( void );
 *	| Ls_List Ls_CreateLP( Ls_Pool pool );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	Each of these functions creates a new list and returns it to
 * 	the client.  ^xLs_Create^n allocates memory internally while
 * 	^xLs_CreateLP^n allocates memory from a client supplied list pool.
 *
 * :Return Value
 *	A newly created empty list.
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_List Ls_Create( void );
extern Ls_List Ls_CreateLP( Ls_Pool pool );


/*+
 * 
 * Function = Ls_Destroy
 *
 * :Overview
 *	Destroy a list.
 *
 * :Prototype
 *	| Ls_Destroy( Ls_List list, void (* delFunc)() );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 * 	^xLs_Destroy^n releases the resources associated with ^llist^n.
 * 	User data is released by calling ^ldelFunc^n on each of the elements 
 *	with the user data as an argument.
 *	Accessing a list after its destruction is a no-no.
 *
 * :Return Value
 *	none
 *
 * :See Also
 *	^xLs_Create^n
 *	^xList^n
 *
+*/
extern void Ls_Destroy( Ls_List list, void (* delFunc)() );




/*+
 * 
 * Function = Ls_Copy, Ls_CopyLP
 *
 * :Overview
 *	copy a list
 *
 * :Prototype
 *	| Ls_List Ls_Copy( Ls_List list, void *(*copyFunc)() );
 *	| Ls_List Ls_CopyLP( Ls_Pool listPool, Ls_List list, void *(*copyFunc)());
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	These functions return a copy of the specified list.
 *	^xLs_Copy^n allocates the list using internal list module storage,
 *	while ^xLs_CopyLP^n uses a user-supplied memory pool.
 *
 *	If `copyFunc' is defined (non-null), it will be called for each
 *	item in ^llist^n and the pointer it returns will be used in
 *	place of the original user data for the item in the newly
 *	created list.  The form of ^lcopyFunc^n should be:
 *
 *	| void *copyFunc( lsGeneric data );
 *
 *	This is normally used to make copies of the user data in the new list.
 *	If no ^lcopyFunc^n is provided, an identity function is used.
 *
 *
 * :Return Value
 *	A copy of ^llist^n.
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_List Ls_Copy( Ls_List list, void *(*copyFunc)() );
extern Ls_List Ls_CopyLP( Ls_Pool pool, Ls_List list, void *(*copyFunc)());



/*+
 * 
 * Function = Ls_FirstElem, Ls_LastElem
 *
 * :Overview
 *	get first/last item from a list
 *
 * :Prototype
 *	| Ls_Status Ls_FirstElem( Ls_List list, void *dataLoc, Ls_Elem *handleLoc );
 *	| Ls_Status Ls_LastElem( Ls_List list, void *dataLoc, Ls_Elem *handleLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_FirstElem^n and ^xLs_LastElem^n return the first and
 *	last items, respectively, from a list.  The item is stored in the
 *	location whose address is given by ^ldataLoc^n.  If ^lhandleLoc^n
 *	is non-zero, it will be filled with an item handle.
 *
 * :Return Value
 *	@ ^xLS_OK^n - success
 *	@ ^xLS_NOMORE^n - list is empty
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_Status 
    Ls_FirstElem( Ls_List list, void *dataLoc, Ls_Elem *handleLoc );
extern Ls_Status 
    Ls_LastElem( Ls_List list, void *dataLoc, Ls_Elem *handleLoc );



/*+
 * 
 * Function = Ls_NewBegin, Ls_NewEnd
 *
 * :Overview
 *	add a new item to the beginning/end of a list
 *
 * :Prototype
 *	| void Ls_NewBegin( Ls_List list, void *data, Ls_Elem *handleLoc );
 *	| void Ls_NewEnd( Ls_List list, void *data, Ls_Elem *handleLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_NewBegin^n and ^xLs_NewEnd^n add 
 *	a new element containing
 * 	^ldata^n to the beginning or end of a list.
 *	If ^lhandleLoc^n is non-zero, it will be filled with an element 
 * 	handle.
 *
 * :Return Value
 *	none
 *
 * :See Also
 *	^xList^n
 *
+*/
extern void Ls_NewBegin( Ls_List list, void *data, Ls_Elem * );
extern void Ls_NewEnd( Ls_List list, void *data, Ls_Elem * );



/*+
 * 
 * Function = Ls_DelBegin, Ls_DelEnd
 *
 * :Overview
 *	Delete the first/last item of a list
 *
 * :Prototype
 *	| Ls_Status Ls_DelBegin( Ls_List list, void *dataLoc );
 *	| Ls_Status Ls_DelEnd( Ls_List list, void *dataLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_DelBegin^n and ^xLs_DelEnd^n delete the first and
 *	last items, respectively, from a list.
 * 	The deleted item is stored in the location whose address is 
 *	given by ^ldataLoc^n.
 *
 * :Return Value
 *	@ ^xLS_OK^n - success
 *	@ ^xLS_NOMORE^n - list is empty
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_Status Ls_DelBegin( Ls_List list, void *dataLoc );
extern Ls_Status Ls_DelEnd( Ls_List list, void *dataLoc );



/*+
 * 
 * Function = Ls_Length
 *
 * :Overview
 *	get length of a list
 *
 * :Prototype
 *	| int Ls_Length( Ls_List list );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	Return the number of items in the given list.
 *
 * :Return Value
 *	length
 *
 * :See Also
 *	^xList^n
 *
+*/
extern int Ls_Length( Ls_List list );




/*+
 * 
 * Function = Ls_Start, Ls_End
 *
 * :Overview
 *	Create a list element generator
 *
 * :Prototype
 *	| Ls_Gen Ls_Start( Ls_list list );
 *	| Ls_Gen Ls_End( Ls_list list );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	These functions create a generator that can be used to iterate
 *	through the elements of a list.  ^xLs_Start^n generates from 
 *	the beginning of the list, while ^xLs_End^n begins generation at the
 * 	end of the list.
 *
 * :Return Value
 *	A generator.
 *
 * :Example
 *	|     /# 
 *	|      * process each element of a Foo list from front to back 
 *	|      #/
 *	|     
 *	|     Foo *fooPtr;
 *	|     Ls_Gen gen;	
 *	|    
 *	|     gen = Ls_Start( list );
 *	|     while( Ls_Next( gen, &fooPtr, LS_NH ) == LS_OK ) {
 *	|         ...
 *	|         /# do something exciting with fooPtr #/
 *	|         ...
 *	|     }
 *	|     Ls_Finish( gen );
 *
 * :See Also
 *	^xLs_Gen^n
 *      ^xLs_ElemGen^n
 *      ^xLs_Next^n
 *      ^xLs_Prev^n
 *      ^xLs_Finish^n
 *	^xList^n
 *
+*/
extern Ls_Gen Ls_Start( Ls_List list );
extern Ls_Gen Ls_End( Ls_List list );



/*+
 * 
 * Function = Ls_ElemGen
 *
 * :Overview
 *	create a list element generator near an element
 *
 * :Prototype
 *	| Ls_Gen Ls_ElemGen( Ls_Elem elem, void *dataLoc, int pos );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	This function creates a generator positioned either before of
 *	after a given element in a list.  The position is controlled
 *	by the ^lpos^n paramter:
 *	@ ^xLS_BEFORE^n - before the element
 *	@ ^xLS_AFTER^n - after the element
 *	The elements user data will be placed at the address given
 *	by ^ldataLoc^n.
 *
 * :Return Value
 *	A generator.
 *
 * :Example
 *	| Ls_Elem  e;
 *	| Ls_Gen   gen;
 *	| Item    *item;
 *	| 
 *	| e = ...   /# some element handle #/
 *	| 
 *	| /# process all items after element e #/
 *	| gen = Ls_ElemGen( e, NULL, LS_AFTER );
 *	| while( lsNext( gen, &item, LS_NH ) == LS_OK ) {
 *	|     ProcessItem( item );
 *	| }
 *	| Ls_Finish( gen );
 *
 * :See Also
 *	Ls_Gen Ls_Next Ls_Prev Ls_Finish 
 *	^xList^n
 *
+*/
extern Ls_Gen Ls_ElemGen(Ls_Elem handle, void *dataLoc, int pos);



/*+
 * 
 * Function = Ls_Next, Ls_Prev
 *
 * :Overview
 *	generate the next element in a sequence
 *
 * :Prototype
 *	| Ls_Status Ls_Next( Ls_Gen gen, void *dataLoc, Ls_Elem *handleLoc );
 *	| Ls_Status Ls_Prev( Ls_Gen gen, void *dataLoc, Ls_Elem *handleLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_Next^n and ^xLs_Prev^n generate the 
 *      list element after or before the current generator location.  
 *	The data from the generated element will be placed at ^ldataLoc^n.
 *	If ^lhandleLoc^n is non-zero, it will be used to return a handle
 *	to the generated element.
 *
 * :Return Value
 *	@ ^xLS_OK^n - successful
 *	@ ^xLS_NOMORE^n - no item generated (at end of list)
 *
 * :Example
 *	| /# 
 *	|  * process each element of a Foo list from front to back
 *	|  #/
 *	| 
 *	| Foo *fooPtr;
 *	| Ls_Gen gen;	
 *	|
 *	| gen = Ls_Start( list );
 *	| while( Ls_Next( gen, &fooPtr, LS_NH ) == LS_OK ) {
 *	|     ...
 *	|     /# do something exciting with fooPtr #/
 *	|     ...
 *	| }
 *	| Ls_Finish( gen );
 *
 * :See Also
 *	Ls_Gen Ls_Start Ls_Finish Ls_InBefore Ls_InAfter
 *	^xList^n
 *
+*/
extern Ls_Status 
    Ls_Next( Ls_Gen gen, void *dataLoc, Ls_Elem *handleLoc );
extern Ls_Status 
    Ls_Prev( Ls_Gen gen, void *dataLoc, Ls_Elem *handleLoc );



/*+
 * 
 * Function = Ls_InBefore, Ls_InAfter
 *
 * :Overview
 * 	insert an item before/after the most recently generated 
 *
 * :Prototype
 * 	| void Ls_InBefore( Ls_Gen gen, void *data, Ls_Elem *handleLoc );
 * 	| void Ls_InAfter( Ls_Gen gen, void *data, Ls_Elem *handleLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_InBefore^n and ^xLs_InAfter^n insert a 
 *	new element into a list before or after a specified generator.
 *
 * :Return Value
 *	none
 *
 * :See Also
 *	^xList^n
 *
+*/
extern void Ls_InBefore( Ls_Gen gen, void *data, Ls_Elem *handleLoc );
extern void Ls_InAfter( Ls_Gen gen, void *data, Ls_Elem *handleLoc );


/*+
 * 
 * Function = Ls_DelBefore, Ls_DelAfter
 *
 * :Overview
 * 	Delete an item relative to a generator.
 *
 * :Prototype
 *	| Ls_Status Ls_DelBefore( Ls_Gen gen, void *dataLoc );
 *	| Ls_Status Ls_DelAfter( Ls_Gen gen, void *dataLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	The functions ^xLs_DelBefore^n and ^xLs_DelAfter^n delete the
 *	list element before or after a specified generator.
 *	If ^ldataLoc^n is non-zero, it is used to return the data
 *	associated with the deleted item.
 *
 * :Return Value
 *	@ ^xLS_OK^n - successful
 *	@ ^xLS_NOMORE^n - no item to delete
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_Status Ls_DelBefore( Ls_Gen gen, void *dataLoc );
extern Ls_Status Ls_DelAfter( Ls_Gen gen, void *dataLoc );


/*+
 * 
 * Function = Ls_Finish
 *
 * :Overview
 * 	Release a generator.
 *
 * :Prototype
 *	| void Ls_Finish( Ls_Gen gen );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	Once a generator is no longer of use, pass it to 
 *	^xLs_Finish^n to release the resources associated with it.
 *
 * :See Also
 *	^xList^n,
 *	^xLs_Start^n,
 *	^xLs_End^n,
 *	^xLs_ElemGen^n
 *
+*/
extern void Ls_Finish( Ls_Gen gen );


/*+
 * 
 * Function = Ls_Foreach, Ls_Backeach
 *
 * :Overview
 * 	Iterate through a list.
 *
 * :Prototype
 *	| Ls_Status Ls_Foreach( Ls_List list, Ls_Status (*userFunc)(), void *itrData );
 *	| Ls_Status Ls_Backeach( Ls_List list, Ls_Status (*userFunc)(), void *itrData );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	These two functions iterate over the elements of a list, calling
 *	a user-specified function on each element.  
 *	^xLs_Foreach^n iterates from first to last, while ^xLs_Backeach^n 
 *	iterates last to first.
 *
 *	The function ^luserFunc^n should have the following form:
 *	| Ls_Status userFunc( void *data, void *itrData )
 *	where ^ldata^n is a data object stored at an element, and
 *	^litrData^n is the data passed into the iterator.
 *	The user function is expected to return one of three values:
 *	@ ^lLS_OK^n - keep iterating
 *	@ ^lLS_STOP^n - stop iterating
 *	@ ^lLS_DELETE^n - delete current element and keep iterating
 *
 * :Return Value
 *	@ ^xLS_OK^n - successfully iterated entire list
 *	@ ^xLS_STOP^n - iteration stopped prematurely by client function
 *	@ ^xLS_BADPARAM^n - unexpected return code from client function
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_Status 
    Ls_Foreach( Ls_List list, Ls_Status (*userFunc)(), void *itrData );
extern Ls_Status 
    Ls_Backeach( Ls_List list, Ls_Status (*userFunc)(), void *itrData );

/*+
 * 
 * Function = Ls_ElemList
 *
 * :Overview
 * 	Get list associated with element
 *
 * :Prototype
 *	| Ls_List Ls_ElemList( Ls_Elem elem );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	Given an element handle, this function returns the list that
 *	contains the element.
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_List Ls_ElemList( Ls_Elem elem );

/*+
 * 
 * Function = Ls_ElemData
 *
 * :Overview
 * 	Get user data of element.
 *
 * :Prototype
 *	| void *Ls_ElemData( Ls_Elem elem );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	^xLs_ElemData^n returns the user data stored at list element ^lelem^n.
 *
 * :See Also
 *	^xLs_SetElemData^n
 *	^xList^n
 *
+*/
extern void *Ls_ElemData( Ls_Elem elem );

/*+
 * 
 * Function = Ls_SetElemData
 *
 * :Overview
 * 	Set user data of element.
 *
 * :Prototype
 *	| void *Ls_SetElemData( Ls_Elem elem, void *newData );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	^xLs_SetElemData^n sets the user data of ^lelem^n to ^lnewData^n
 *	and returns the previously stored data.
 *
 * :See Also
 *	^xLs_ElemData^n,
 *	^xList^n
 *
+*/
extern void *Ls_SetElemData( Ls_Elem elem, void *newData );

/*+
 * 
 * Function = Ls_RemoveElem
 *
 * :Overview
 * 	Remove an element from its list
 *
 * :Prototype
 *	| Ls_Status Ls_RemoveElem( Ls_Elem elem, void *dataLoc );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	^xLs_RemoveElem^n removes the element ^lelem^n from its list.
 *	If ^ldataLoc^n is non-NULL, it is used to return the user data
 *	stored at ^lelem^n.
 *
 * :See Also
 *	^xList^n
 *
+*/
extern Ls_Status Ls_RemoveElem( Ls_Elem elem, void *dataLoc );

/*+
 * 
 * Function = Ls_Sort, Ls_SortCD
 *
 * :Overview
 * 	Sort a list
 *
 * :Prototype
 *	| void Ls_Sort( Ls_List list, int (*compare)() );
 *	| void Ls_SortCD( Ls_List list, int (*compare)(), void *compData );
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	These routines sort an input list, in place, according to a 
 *	user-specified comparison function.
 *	The comparison function for ^xLs_Sort^n should be of the following form:
 *	| int compare( void *item1, void *item2 );
 *	The routine should return ^l-1^n, if ^litem1^n is less than ^litem2^n,
 *	^l0^n if they are equal, and ^l1^n if ^litem1^n is greater than 
 *	^litem2^n.
 *
 *	In addition to two items, the comparision function for ^xLs_SortCD^n
 *	gets an additional piece of user provided data.  Its prototype
 *	should be:
 *	| int compare( void *item1, void *item2, void *compData );
 *	
 * :Remarks
 *	These sorting routines use a generic merge sort written
 *	by Rick Rudell that runs in ^eO(n log n)^n time.
 *
 * :See Also
 *	^xList^n,
 *	^xLs_Uniq^n
 *
+*/
extern void Ls_Sort( Ls_List list, int (*compare)() );
extern void Ls_SortCD( Ls_List list, int (*compare)(), void *compData );

/*+
 * 
 * Function = Ls_Uniq, Ls_UniqCD
 *
 * :Overview
 * 	Remove duplicates from a sorted list
 *
 * :Prototype
 *	| void Ls_Uniq( Ls_List list, int (*compare)(), void (*delFunc)());
 *	| void Ls_UniqCD( Ls_List list, int (*compare)(), void (*delFunc)(), void *compData);
 *
 * :Prototype In
 * 	| list.h
 *
 * :Description
 *	These functions each take a sorted list and remove all duplicate
 *	elements.  The comparison functions for ^xLs_Uniq^n and ^xLs_UniqCD^n
 *	have the following prototypes respectively.
 *	| int compare( void *item1, void *item2 );
 *	| int compare( void *item1, void *item2, void *compData );
 *	The routine should return ^l-1^n, if ^litem1^n is less than ^litem2^n,
 *	^l0^n if they are equal, and ^l1^n if ^litem1^n is greater than 
 *	^litem2^n.
 *
 *	The function ^ldelFunc^n will be called with the user data of each
 *	duplicate element to be destroyed.  If no clean up is required,
 *	^ldelFunc^n can be NULL.
 *	
 * :See Also
 *	^xList^n,
 *	^xLs_Sort^n,
 *	^luniq(1)^n
 *
+*/
extern void Ls_Uniq( Ls_List list, int (*compare)(), void (*delFunc)());
extern void Ls_UniqCD( Ls_List list, int (*compare)(), void (*delFunc)(), void *compData);


/*+
 * Note = To Go
 *
 * :Overview
 * 	functions that still need individual comments
 *
 * :Description

| /#
|  * return allocation stats
|  #/
| void Ls_ReportPoolStats( Ls_Pool pool )
| 
| 
| /# append source to target, copy items #/
| void Ls_AppendCopy( Ls_List target, Ls_List source, void *(*copyFunc)())
| void Ls_PrependCopy( Ls_List target, Ls_List source, void *(*copyFunc)())
| 
| /# append source to target, move items #/
| void Ls_AppendMove( Ls_List target, Ls_List source )
| void Ls_PrependMove( Ls_List target, Ls_List source )
| 
| 
| /# concatenate two lists, copy items #/
| Ls_List Ls_ConcatCopy( Ls_List list1, Ls_List list2, void *(*copyFunc)() )
| Ls_List Ls_ConcatCopyLP( Ls_Pool pool, Ls_List list1, Ls_List list2, void *(*copyFunc)() )
| 
| /# concatenate two lists, move items #/
| Ls_List Ls_ConcatMove( Ls_List list1, Ls_List list2 )
| 
| 
| /# filter specific items from list, copy items #/
| Ls_List Ls_FilterCopy( Ls_List list, int (*compFunc)(), void *data, 
| 	void *(*copyFunc)() )
| Ls_List Ls_FilterCopyLP( Ls_List list, int (*compFunc)(), void *data, 
| 	void *(*copyFunc)() )
| 
| /# filter specific items from list, move items #/
| Ls_List Ls_FilterMove( Ls_List list, int (*compFunc)(), void *data )
| 
| 
| /# count specific members of a list #/
| int Ls_CountMember( Ls_List list, void *object, int (*compFunc)() )
| int Ls_CountMemberCD( Ls_List list, void *object, int (*compFunc)(),
| 		 void *clientData)
| 
| 
| /# test list for specific item membership #/
| int Ls_IsMember( Ls_List list, void *object, int (*compFunc)() )
| int Ls_IsMemberCD( Ls_List list, void *object, int (*compFunc)(),
| 	      void *clientData )
| 
| /# locate specific item in a list #/
| Ls_Status Ls_LocateElem( Ls_List list, void *object, int (*compFunc)(),
| 	      void *dataLoc, Ls_Elem *itemHandle)
| Ls_Status Ls_LocateElemCD( Ls_List list, void *object, int (*compFunc)(),
| 		void *dataLoc, Ls_Elem *itemHandle,
| 		void *clientData )
| 
| 
| /# delete specific items from a list #/
| Ls_Status Ls_DeleteElem( Ls_List list, void *object, int (*compFunc)() )
| Ls_Status Ls_DeleteElemCD( Ls_List list, void *object, int (*compFunc)(),
| 		void *clientData )
| 
| 
| /# get next/previous of specific item #/
| Ls_Status Ls_NextElem( Ls_Elem item, void *dataLoc, Ls_Elem *nextLoc )
| Ls_Status Ls_PrevElem( Ls_Elem item, void *dataLoc, Ls_Elem *prevLoc )
| 
| 
| /# reverse a list in place #/
| Ls_Status Ls_Reverse( Ls_List list )
| 
| /# randomize list order in place #/
| Ls_Status Ls_Scramble( Ls_List list )  /# uses default seed #/
| Ls_Status Ls_ScrambleSeed( Ls_List list, long seed )
+*/


/*
 * return allocation stats
 */
extern void Ls_ReportPoolStats( Ls_Pool pool );


/* append source to target, copy items */
extern void 
    Ls_AppendCopy( Ls_List target, Ls_List source, void *(*copyFunc)());
extern void 
    Ls_PrependCopy( Ls_List target, Ls_List source, void *(*copyFunc)());

/* append source to target, move items */
extern void Ls_AppendMove( Ls_List target, Ls_List source );
extern void Ls_PrependMove( Ls_List target, Ls_List source );


/* concatenate two lists, copy items */
extern Ls_List 
    Ls_ConcatCopy( Ls_List list1, Ls_List list2, void *(*copyFunc)() );
extern Ls_List 
    Ls_ConcatCopyLP( Ls_Pool pool, Ls_List list1, Ls_List list2, void *(*copyFunc)() );

/* concatenate two lists, move items */
extern Ls_List Ls_ConcatMove( Ls_List list1, Ls_List list2 );


/* filter specific items from list, copy items */
extern Ls_List
    Ls_FilterCopy( Ls_List list, int (*compFunc)(), void *data, 
	void *(*copyFunc)() );
extern Ls_List
    Ls_FilterCopyLP( Ls_List list, int (*compFunc)(), void *data, 
	void *(*copyFunc)() );

/* filter specific items from list, move items */
extern Ls_List
    Ls_FilterMove( Ls_List list, int (*compFunc)(), void *data );


/* count specific members of a list */
extern int
    Ls_CountMember( Ls_List list, void *object, int (*compFunc)() );
extern int
    Ls_CountMemberCD( Ls_List list, void *object, int (*compFunc)(),
		 void *clientData);


/* test list for specific item membership */
extern int 
    Ls_IsMember( Ls_List list, void *object, int (*compFunc)() );
extern int
    Ls_IsMemberCD( Ls_List list, void *object, int (*compFunc)(),
	      void *clientData );

/* locate specific item in a list */
extern Ls_Status
    Ls_LocateElem( Ls_List list, void *object, int (*compFunc)(),
	      void *dataLoc, Ls_Elem *itemHandle);
extern Ls_Status
    Ls_LocateElemCD( Ls_List list, void *object, int (*compFunc)(),
		void *dataLoc, Ls_Elem *itemHandle,
		void *clientData );


/* delete specific items from a list */
extern Ls_Status
    Ls_DeleteElem( Ls_List list, void *object, int (*compFunc)() );
extern Ls_Status
    Ls_DeleteElemCD( Ls_List list, void *object, int (*compFunc)(),
		void *clientData );


/* get next/previous of specific item */
extern Ls_Status
    Ls_NextElem( Ls_Elem item, void *dataLoc, Ls_Elem *nextLoc );
extern Ls_Status
    Ls_PrevElem( Ls_Elem item, void *dataLoc, Ls_Elem *prevLoc );


/* reverse a list in place */
extern Ls_Status Ls_Reverse( Ls_List list );

/* randomize list order in place */
extern Ls_Status Ls_Scramble( Ls_List list ); /* uses default seed */
extern Ls_Status Ls_ScrambleSeed( Ls_List list, long seed );


#endif /* LS_DEFINED */
