//#############################################################################
// AGENT.PAS
//
// RUNTIME MONITORING AGENT
// Version 2.009
//
// SCA Software International S.A.
// Zoran M.Todorovic (codor@eunet.yu) & Miodrag Stojadinovic (micko@eunet.yu)
//
// Do not modify this file !!!
//
// This unit contains a class TAgentDrv which automatically loads an Agent DLL
// (if it exists) and offers DLL exported functions as a public class methods.
// If DLL is not found, class will still function normally (just returns).
// This unit also creates an object AgentDrv of class TAgentDrv so that it can
// be used anywhere in a Delphi project (just use this unit).
// For help, see Agent online help.
//#############################################################################

unit Agent;

interface

Uses
  Windows, SysUtils;

// Constants from AgentDll.h
const
  // Message types
  MT_TEXT     = $1000;
  MT_EXTTEXT  = $1001;
  MT_SYSTEM   = $2000;
  MT_PROCESS  = $2001;
  MT_FIBER    = $2002;
  // Maximum length of message fields
  MAX_NAME_LENGTH = 255;
  MAX_TEXT_LENGTH = 500;

// Types equal as in AgentDll.h
Type
  // Plain text message (MT_TEXT)
  TMtText = record
    Text: array [0..MAX_TEXT_LENGTH] of char;
  end;
  // Extended text message (MT_EXTTEXT)
  TMtExtText = record
    Filename: array [0.. MAX_NAME_LENGTH] of char;
    Linenum: Integer;
    ErrorCode: Integer;
    Text: array [0..MAX_TEXT_LENGTH] of char;
  end;
  // System alias definition message (MT_SYSTEM)
  TMtSystem = record
    Identifier: Integer;
    Name: array [0..MAX_NAME_LENGTH] of char;
  end;
  // Process alias definition message (MT_PROCESS)
  TMtProcess = record
    System: Integer;
    Identifier: Integer;
    Name: array [0..MAX_NAME_LENGTH] of char;
  end;
  // Fiber alias definition message (MT_FIBER)
  TMtFiber = record
    Process: Integer;
    Identifier: Integer;
    Name: array [0..MAX_NAME_LENGTH] of char;
  end;

Type
  // Function prototype to send a message (AgentDLL.DLL)
  TSendMsg = function(
            fiber: Integer;
            process: Integer;
            system: Integer;
            typ: Integer;
            len: Integer;
            data: Pointer
            ): Integer; stdcall;
  // Function prototype to convert binary buffer to user friendly format
  // (either in decimal or hexdecimal format) (AgentDLL.DLL)
  TConvertToReadable = function(
            buffer: PChar;
            buflen: Integer;
            maxlen: Integer;
            radix: Integer
            ): Boolean; stdcall;

  // This unit class. Unit creates an object of this class (see
  // Initialization/Finalization part of the unit)
  TAgentDrv = class
  private
    Ok: Boolean;
    Loaded: Boolean;
    FHandle: THandle;
    FSendMsg: TSendMsg;
    FConvertToReadable: TConvertToReadable;
  public
    constructor Create(name: String);
    destructor Destroy; override;
    function IsOk: Boolean;
    function SendMsg(
              fiber: Integer;
              process: Integer;
              system: Integer;
              typ: Integer;
              len: Integer;
              data: Pointer
              ): Integer;
    function ConvertToReadable(
              buffer: PChar;
              maxlen: Integer;
              radix: Integer
              ): Boolean;
    procedure DefineSystem(systemId: Integer; name: PChar);
    procedure DefineProcess(systemId: Integer; processId: Integer;
                            name: PChar);
    procedure DefineFiber(processId: Integer; fiberId: Integer;
                            name: PChar);
    procedure SendText1A(fiberId: Integer; processId: Integer;
                            systemId: Integer; text: PChar);
    procedure SendText1B(fiberId: Integer; text: PChar);
    procedure SendText2A(fiberId: Integer; processId: Integer;
                            systemId: Integer; filename: PChar;
                            linenum: Integer; text: PChar;
                            errcode: Longint);
    procedure SendText2B(fiberId: Integer; filename: PChar;
                            linenum: Integer; text: PChar;
                            errcode: Longint);
  end;

var
  AgentDrv: TAgentDrv;

implementation

constructor TAgentDrv.Create(name: String);
var
  n: array[0..255] of char;
begin
  Ok := False;
  Loaded := False;
  strpcopy(n,name);
  // Try to dynamically load a library (AgentDLL.DLL)
  FHandle := LoadLibrary(n);
  if (FHandle > 32) then begin
    // Loading OK
    Ok := True;
    Loaded := True;
  end;
  try
    // Get pointers to DLL exported functions
    FSendMsg := GetProcAddress(FHandle,'SendMsg');
    FConvertToReadable := GetProcAddress(FHandle,'ConvertToReadable');
  except
    Ok := False;
  end;
end;

destructor TAgentDrv.Destroy;
begin
  // If loaded then free DLL
  if Loaded and (FHandle > 32) then
    FreeLibrary(FHandle);
end;

function TAgentDrv.IsOk: Boolean;
begin
  Result := (Ok and Loaded);
end;

// Local function. Should not be used directly. It is used by other functions
// in this class.
function TAgentDrv.SendMsg(
              fiber: Integer;
              process: Integer;
              system: Integer;
              typ: Integer;
              len: Integer;
              data: Pointer
          ): Integer;
begin
  Result := -1;
  if assigned(FSendMsg) then
    Result := FSendMsg(fiber,process,system,typ,len,data);
end;

// Local function. Should not be used directly. It is used by other functions
// in this class.
function TAgentDrv.ConvertToReadable(
              buffer: PChar;
              buflen: Integer;
              maxlen: Integer;
              radix: Integer
          ): Boolean;
begin
  Result := False;
  if assigned(FConvertToReadable) then
    Result := FConvertToReadable(buffer,buflen,maxlen,radix);
end;

// Use this function to define a system alias.
// SystemId - Unique identifier of the system alias
// Name     - C like string specifying alias name
procedure TAgentDrv.DefineSystem(systemId: Integer; name: PChar);
var
  data: TMtSystem;
begin
  ZeroMemory(@data,sizeof(TMtSystem));
  data.Identifier := systemId;
  strlcopy(data.Name,name,MAX_NAME_LENGTH);
  SendMsg(0,0,0,MT_SYSTEM,sizeof(TMtSystem),@data);
end;

// Use this function to define a process alias.
// SystemId - Unique identifier of the system alias this process alias
//            belongs to.
// ProcessId - Unique identifier of the process alias
// Name     - C like string specifying alias name
procedure TAgentDrv.DefineProcess(systemId: Integer; processId: Integer;
                        name: PChar);
var
  data: TMtProcess;
begin
  ZeroMemory(@data,sizeof(TMtProcess));
  data.System := systemId;
  data.Identifier := processId;
  strlcopy(data.Name,name,MAX_NAME_LENGTH);
  SendMsg(0,0,0,MT_PROCESS,sizeof(TMtProcess),@data);
end;

// Use this function to define a fiber alias.
// ProcessId - Unique identifier of the process alias this fiber alias
//             belongs to.
// FiberId  - Unique identifier of the fiber alias
// Name     - C like string specifying alias name
procedure TAgentDrv.DefineFiber(processId: Integer; fiberId: Integer;
                        name: PChar);
var
  data: TMtFiber;
begin
  ZeroMemory(@data,sizeof(TMtFiber));
  data.Process := processId;
  data.Identifier := fiberId;
  strlcopy(data.Name,name,MAX_NAME_LENGTH);
  SendMsg(0,0,0,MT_FIBER,sizeof(TMtFiber),@data);
end;

// Send a simple text to the Agent.
// FiberId  - Fiber alias this message belongs to.
// ProcessId - Process alias fiber alias belongs to.
// SystemId - System alias process alias belongs to.
// Text     - C like string containing the message text
procedure TAgentDrv.SendText1A(fiberId: Integer; processId: Integer;
                        systemId: Integer; text: PChar);
var
  data: TMtText;
begin
  ZeroMemory(@data,sizeof(TMtText));
  strlcopy(data.Text,text,MAX_TEXT_LENGTH);
  SendMsg(fiberId,processId,systemId,MT_TEXT,sizeof(TMtText),@data);
end;

// Simpler version of the previous function (SendText1A).
// FiberId  - Fiber alias this message belongs to. Agent will search its
//            list of process aliases and try to find the alias this fiber
//            belongs to.
// Text     - C like string containing the message text
procedure TAgentDrv.SendText1B(fiberId: Integer; text: PChar);
var
  data: TMtText;
begin
  ZeroMemory(@data,sizeof(TMtText));
  strlcopy(data.Text,text,MAX_TEXT_LENGTH);
  SendMsg(fiberId,0,0,MT_TEXT,sizeof(TMtText),@data);
end;

// Send an extended text to the Agent.
// FiberId  - Fiber alias this message belongs to.
// ProcessId - Process alias fiber alias belongs to.
// SystemId - System alias process alias belongs to.
// Filename - Source code filename from which a message is sent.
// Linenum  - Line number from which a message is sent.
// Text     - C like string containing the message text
// Errcode  - Optional Win32 or application specific error code.
procedure TAgentDrv.SendText2A(fiberId: Integer; processId: Integer;
                        systemId: Integer; filename: PChar;
                        linenum: Integer; text: PChar;
                        errcode: Longint);
var
  data: TMtExtText;
begin
  ZeroMemory(@data,sizeof(TMtExtText));
  if (filename <> nil) then begin
    strlcopy(data.Filename,filename,MAX_NAME_LENGTH);
  end;
  data.Linenum := linenum;
  strlcopy(data.Text,text,MAX_TEXT_LENGTH);
  data.ErrorCode := errcode;
  SendMsg(fiberId,processId,systemId,MT_EXTTEXT,sizeof(TMtExtText),@data);
end;

// Simpler version of the previous function (SendText2A).
// FiberId  - Fiber alias this message belongs to. Agent will search its list
//            of process aliases and try to find a process alias this fiber
//            belongs to.
// Filename - Source code filename from which a message is sent.
// Linenum  - Line number from which a message is sent.
// Text     - C like string containing the message text
// Errcode  - Optional Win32 or application specific error code.
procedure TAgentDrv.SendText2B(fiberId: Integer; filename: PChar;
                        linenum: Integer; text: PChar;
                        errcode: Longint);
var
  data: TMtExtText;
begin
  ZeroMemory(@data,sizeof(TMtExtText));
  if (filename <> nil) then begin
    strlcopy(data.Filename,filename,MAX_NAME_LENGTH);
  end;
  data.Linenum := linenum;
  strlcopy(data.Text,text,MAX_TEXT_LENGTH);
  data.ErrorCode := errcode;
  SendMsg(fiberId,0,0,MT_EXTTEXT,sizeof(TMtExtText),@data);
end;

//
// On unit initialization, create an object of class TAgentDrv with a name
// of the DLL (AgentDLL.DLL must be somewhere in path or in a current dir.)
// On unit finalization, free this object and release resources.
//

initialization
  AgentDrv := TAgentDrv.Create('AgentDll.Dll');
finalization
  AgentDrv.Free;
end.

//#############################################################################
// End of AGENT.PAS
//#############################################################################