unit DBTreeView;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, db, dbtables;

type
  TDistanceEvent = procedure(Sender: TObject; Distance: integer) of object;
  TTreeDatalink = class(TDatalink)
  protected
    procedure DatasetChanged; override;
    procedure DatasetScrolled(Distance: integer); override;
    procedure ActiveChanged; override;
    procedure UpdateData; override;
  public
    OnDatasetChanged: TNotifyEvent;
    OnDatasetScrolled: TDistanceEvent;
    OnActiveChanged: TNotifyEvent;
    OnUpdateData: TNotifyEvent;
  end;

  TDBTreeView = class(TTreeView)
  private
    bChangingTree: boolean;
    bChangingData: boolean;
    bSiUpdateData: boolean;

    strFieldidRubroName: string;
    strFieldidParentName: string;
    strFieldNombreName: string;

    fieldIdRubro: TField;
    fieldIdParent: TField;
    fieldNombre: TField;

    OnEditedProc: TTVEditedEvent;
    OnChangeProc: TTVChangedEvent;

    {Operaciones con nodos}
    function NodeAgregar(treenodeParent: TTreeNode;
                         iIdRubro: integer;
                         strNombre: string): TTreeNode;
    procedure NodeAgregarHijos(treenodeParent: TTreeNode;
                               iHijosVisibles: integer);
    procedure NodeBorrarHijos(treenodeParent: TTreeNode);
    procedure NodeActualizarHijos(treenodeParent: TTreeNode;
                                  iHijosVisibles: integer);

    {Get/Set properties}
    procedure SetDatasource(Value: TDatasource);
    function GetDatasource: TDatasource;
  protected
    iIdValue: integer;
    strNuevoNombre: string;
    TreeDatalink: TTreeDatalink;

    procedure ReenlazarCampos;
    function ReOrdenar: boolean; {Nodos}

    procedure RecordModified;
    function MoverItem(iIdParentNuevo: integer): TTreeNode;

    procedure DatalinkDatasetChanged(Sender: TObject);
    procedure DatalinkDatasetScrolled(Sender: TObject; Distance: integer);
    procedure DatalinkActiveChanged(Sender: TObject);
    procedure DatalinkUpdateData(Sender: TObject);

    procedure Change(Sender: TObject; Node: TTreeNode);
    procedure Edited(Sender: TObject; Node: TTreeNode;
              var S: String);
    procedure CMExit(var Message: TWMNoParams); message CM_EXIT;
  public
    procedure Actualizar;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property KeyField: string read strFieldidRubroName write strFieldIdRubroName;
    property ParentKeyField: string read strFieldidParentName write strFieldIdParentName;
    property ListField: string read strFieldNombreName write strFieldNombreName;
    property DataSource: TDatasource read GetDatasource write SetDatasource;

    property OnEdited: TTVEditedEvent read OnEditedProc write OnEditedProc;
    property OnChange: TTVChangedEvent read OnChangeProc write OnChangeProc;
  end;

const
  I_IMAGEINDEXUNSELECTED = 0;
  I_IMAGEINDEXSELECTED = 1;
  STR_RUBROBORRARCONFIRMAR = 'Est seguro de eliminar el rubro ';
  STR_ITEMBORRARCONFIRMAR = 'Est seguro de eliminar ';
  STR_ITEMNUEVOCAPTION = 'Nuevo Item';
  STR_DATASETINACTIVE = 'No conectado';

  {Popup Rubros}
  STR_RUBROSAGREGARRUBROCAPTION = 'Agregar rubro junto a ''';
  STR_RUBROSAGREGARSUBRUBROCAPTION = 'Agregar subrubro de ''';
  STR_RUBROSELIMINARCAPTION = 'Eliminar ''';
  STR_RUBROSNOHIGHLIGHTED = '(rbol de rubros)';
  STR_COMILLAS = '''';
  STR_DOSPUNTOS = ':';
  STR_INTERROGACION = '?';
  STR_ITEMSNOHIGHLIGHTED = '(lista de Items)';

procedure Register;

implementation

procedure TDBTreeView.NodeAgregarHijos(treenodeParent: TTreeNode;
                                     iHijosVisibles: integer);
var strFilterAnterior: string;
    strBookmarkAnterior: string;
    strIdRubroParent: string;
    bFilteredAnterior: boolean;
    bEncontrado: boolean;
    iBucle: integer;
begin
     if Assigned(treenodeParent) then
        strIdRubroParent := IntToStr(Integer(treenodeParent.Data))
     else
        strIdRubroParent := '-1';

     with TreeDatalink.Dataset do
     begin
          {Guarda las condiciones anteriores para dejarlas
          como estaban (slo si viene con parent=nil)}
          strFilterAnterior := Filter;
          bFilteredAnterior := Filtered;
          strBookmarkAnterior := Bookmark;

          {Busca slo los hijos}
          Filter := strFieldIdParentName + '=' +
                    strIdRubroParent;
          Filtered := True;

          {Los recorre y los agrega}
          bEncontrado := FindFirst;
          while bEncontrado do
          begin
               NodeAgregar(treenodeParent,
                           fieldIdRubro.AsInteger,
                           fieldNombre.AsString);

               bEncontrado := FindNext;
          end;
     end;

     {Repetir esta funcin para los hijos}
     if Assigned(treenodeParent) then
        for iBucle := 0 to treenodeParent.Count-1 do
            NodeAgregarHijos(treenodeParent.Item[iBucle],
                             iHijosVisibles -1)
     else
        for iBucle := 0 to Items.Count-1 do
            NodeAgregarHijos(Items.Item[iBucle],
                             iHijosVisibles -1);

     {Devolver posicin y filtros originales
     si parentnode=nil}
     if not Assigned(treenodeParent) then
     begin
          ReOrdenar;
          with TreeDatalink.Dataset do
          begin
               Filter := strFilterAnterior;
               Filtered := bFilteredAnterior;
               Bookmark := strBookmarkAnterior;
          end;
     end;
end;

procedure TDBTreeView.NodeActualizarHijos(treenodeParent: TTreeNode;
                                        iHijosVisibles: integer);
begin
     NodeBorrarHijos(treenodeParent);
     NodeAgregarHijos(treenodeParent, iHijosVisibles);
end;

procedure TDBTreeView.NodeBorrarHijos(treenodeParent: TTreeNode);
var iBucleHijos: integer;
begin
     if Assigned(treenodeParent) then
        for iBucleHijos := treenodeParent.Count - 1 downto 0 do
            treenodeParent.Item[iBucleHijos].Delete
     else
        Items.Clear;
end;

function TDBTreeView.NodeAgregar(treenodeParent: TTreeNode;
                               iIdRubro: integer;
                               strNombre: string): TTreeNode;
var treenodeAgregado: TTreeNode;
begin
     treenodeAgregado := Items.
     AddChild(treenodeParent, strNombre);
     treenodeAgregado.Data := Pointer(iIdRubro);
     treenodeAgregado.ImageIndex := I_IMAGEINDEXUNSELECTED;
     treenodeAgregado.SelectedIndex := I_IMAGEINDEXSELECTED;

     if not bChangingTree then
        {Si no est actualizando es porque slo
        agreg uno, en ese caso:}
        ReOrdenar;

     Result := treenodeAgregado;
end;

procedure TDBTreeView.Actualizar;
var bRellenado: boolean;
begin
     bChangingTree := True;
     bChangingData := True;

     bRellenado := False;

     if Assigned(TreeDatalink) then
     begin
        if TreeDatalink.Active then
        begin
             NodeActualizarHijos(nil, 1);
             if Assigned(Selected) then
                iIdValue := Integer(Selected.Data)
             else
                iIdValue := 0;
             bRellenado := True;
        end;
     end;

     if not bRellenado and
        not (csDestroying in ComponentState) then
     begin
          Items.Clear;
          Items.AddChildFirst(nil, STR_DATASETINACTIVE);
     end;

     bChangingData := False;
     bChangingTree := False;
end;

function TDBTreeView.ReOrdenar: boolean;
begin
     Result := ((SortType <> stText) or AlphaSort);
     {Si SortType no es stText directamente devuelve True
     y si no hace el AlphaSort y de eso depender}
end;

procedure Register;
begin
  RegisterComponents('Data Controls', [TDBTreeView]);
end;

constructor TDBTreeView.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);

     {Inicializar conexin a datos}
     TreeDatalink := TTreeDatalink.Create;
     TreeDatalink.OnDatasetChanged := DatalinkDatasetChanged;
     TreeDatalink.OnDatasetScrolled := DatalinkDatasetScrolled;
     TreeDatalink.OnActiveChanged := DatalinkActiveChanged;
     TreeDatalink.OnUpdateData := DatalinkUpdateData;

     {Derivar estos eventos a procedimientos internos}
     {Se redefinen los eventhandlers para ser llamados
     desde estos nuevos procedimientos}
     {Evento -> OnChange viejo (va TTreeView) ->
      proc. Change -> OnChangeNuevo (segn usuario)}
     inherited OnChange := Change;
     inherited OnEdited := Edited;
end;

destructor TDBTreeView.Destroy;
begin
     TreeDatalink.Free;
     inherited Destroy;
end;

procedure TDBTreeView.Change(Sender: TObject; Node: TTreeNode);
var variantData: Variant;
begin
     if not bChangingTree then {Si no lo estoy cambiando}
     begin                     {a propsito}

          bChangingData := True; {Activa llamadas de la tabla}

          if Assigned(Node) then
          begin
               variantData := Integer(Node.Data);

               with TreeDatalink.Dataset do

               if not Locate(strFieldIdRubroName,
                      variantData,[]) then
               begin
                  iIdValue := 0;
                  Node.Delete {Si no lo encuentra borrar tem}
               end
               else
               begin
                  iIdValue := fieldIdRubro.AsInteger;
                  if Assigned(OnChangeProc) then
                       OnChangeProc(Sender, Node);
               end;
          end;

          bChangingData := False;
     end;
end;

procedure TDBTreeView.Edited(Sender: TObject; Node: TTreeNode;
  var S: String);
begin
     if not bChangingTree then {Si no lo estoy cambiando}
     begin                     {a propsito}
          if TreeDatalink.Edit then
          begin
               strNuevoNombre := S;
               bSiUpdateData := True;
               TreeDatalink.UpdateRecord;
          end;

          if Assigned(OnEditedProc) then OnEditedProc(Sender, Node, S);
     end;
end;

procedure TDBTreeView.DatalinkDatasetChanged(Sender: TObject);
begin
     if not bChangingData then {Si no se cambia a propsito}
     begin
          bChangingTree := True;
          if not fieldIdRubro.IsNull then
             if fieldIdRubro.value = iIdValue then
                RecordModified
             else
                DatalinkDatasetScrolled(Sender, 0);
          bChangingTree := False;
     end;
end;

procedure TDBTreeView.DatalinkDatasetScrolled(Sender: TObject; Distance: integer);
var iBucle: integer;
    bItemEncontrado: boolean;
    treenodeParent: TTreeNode;
    bookmarkNuevo: TBookmarkStr;
    iIdRubroNuevo, iIdParentNuevo: integer;
begin
     if not bChangingData then
     begin
          bChangingData := True;
          bChangingTree := True;


          with TreeDatalink.Dataset do
          begin
               bookmarkNuevo := Bookmark;
               if not Locate(strFieldIdRubroName,
                       Variant(Integer(Selected.Data)),[]) then
                  Selected.Delete;  {Si no est ms borrar el tem}
               Bookmark := bookmarkNuevo;
          end;

          {Busca a ver si el nuevo est en el tree}

          bItemEncontrado := False;
          treenodeParent := nil;
          iBucle := 0;
          iIdRubroNuevo := fieldIdRubro.AsInteger;
          iIdParentNuevo := fieldIdParent.AsInteger;

          while not bItemEncontrado
                and (iBucle < Items.Count) do
          begin
               bItemEncontrado :=
               Integer(Items[iBucle].Data) = iIdRubroNuevo;

               if Integer(Items[iBucle].Data) = iIdParentNuevo then
                  treenodeParent := Items[iBucle];

               Inc(iBucle);
          end;

          if  bItemEncontrado then
              Selected := Items[iBucle -1] {Hay un Inc(iBucle) de ms}
          else
              Selected := NodeAgregar(treenodeParent,
                                      fieldIdRubro.AsInteger,
                                      fieldNombre.AsString);
          iIdValue := fieldIdRubro.AsInteger;

          bChangingData := False;
          bChangingTree := False;
     end;
end;

procedure TDBTreeView.DatalinkActiveChanged(Sender: TObject);
begin
     if not (csDestroying in ComponentState) then
     begin
          if TreeDatalink.Active then ReenlazarCampos;
          Actualizar;
     end;
end;

procedure TDBTreeView.RecordModified;
var iIdParentData: integer;
    iIdParentTree: integer;
begin
     Selected.Text := fieldNombre.AsString;
     iIdParentData := fieldIdParent.AsInteger;
     if Assigned(Selected.Parent) then
        iIdParentTree := Integer(Selected.Parent.Data)
     else
        iIdParentTree := -1;
     if not (iIdParentTree = iIdParentData) then
  {      Selected := }MoverItem(iIdParentData);
end;

function TDBTreeView.MoverItem(iIdParentNuevo: integer): TTreeNode;
var iBucle: integer;
    pointerData: Pointer;
    strCaption: string;
    treenodeAgregado: TTreeNode;
    pointerIdParentNuevo: Pointer;
    bItemEncontrado: boolean;
begin
     bChangingTree := True;

     {Buscar item con idParent}
     pointerIdParentNuevo := Pointer(iIdParentNuevo);
     bItemEncontrado := False;
     iBucle := 0;

     while not bItemEncontrado and (iBucle < Items.Count) do
     begin
           bItemEncontrado :=
           (Items[iBucle].Data = pointerIdParentNuevo);
           Inc(iBucle);
     end;

     if bItemEncontrado then
        Selected.MoveTo(Items[iBucle-1], naAddChild);

     bChangingTree := False;

     Result := Selected;
end;

procedure TDBTreeView.DatalinkUpdateData(Sender: TObject);
begin
     if bSiUpdateData then
     begin
          bChangingData := True;
          fieldNombre.value := strNuevoNombre;
          bChangingData := False;
     end;
end;

procedure TDBTreeView.CMExit(var Message: TWMNoParams);
begin
     try
        if TreeDatalink.Dataset.State in [dsEdit, dsInsert] then
           TreeDataLink.UpdateRecord; {actualizame la tabla (si hubo Modified)}
     except
        on Exception do SetFocus; {si falla, quedarse en el control}
     end;
     inherited;
end;

procedure TDBTreeView.ReenlazarCampos;
begin
     with TreeDatalink.Dataset do
     begin
          fieldIdRubro := FieldByName(strFieldIdRubroName);
          fieldIdParent:= FieldByName(strFieldIdParentName);
          fieldNombre  := FieldByName(strFieldNombreName);
     end;
end;

function TDBTreeView.GetDatasource: TDatasource;
begin
     Result := TreeDatalink.Datasource;
end;

procedure TDBTreeView.SetDatasource(Value: TDatasource);
begin
     TreeDatalink.Datasource := Value;
{     if Assigned(Value) then Actualizar;}
end;

{******************************************************}
{TTreeDatalink}
{******************************************************}

procedure TTreeDatalink.DatasetChanged;
begin
     if Assigned(OnDatasetChanged) then OnDatasetChanged(Self);
end;

procedure TTreeDatalink.DatasetScrolled(Distance: integer);
begin
     if Assigned(OnDatasetScrolled) then OnDatasetScrolled(Self, Distance);
end;

procedure TTreeDatalink.ActiveChanged;
begin
     if Assigned(OnActiveChanged) then OnActiveChanged(Self);
end;

procedure TTreeDatalink.UpdateData;
begin
     if Assigned(OnUpdateData) then OnUpdateData(Self);
end;

end.
