//-----------------------------------------------------------------------------
// File: RMBegin1.cpp
//
// Desc: This minimal file creates a default device, loads a mesh from a 
//       DirectX file, builds a scene, and rotates the mesh object. It also
//       contains standard code which allows the sample to run in the
//       Windows environment. 
//
// Copyright (C) 1998 Microsoft Corporation. All Rights Reserved.
//
// Modified by Weston Software to demonstrate DirectXCPP
//-----------------------------------------------------------------------------

#define INITGUID       // Required for Direct3D applications 
#include <windows.h>   // Standard Windows header file 
#include <direct.h>    // DirectDraw definitions 
#include "resource.h"

#include "dxcpp.h"     // DirectXCPP classes by Weston Software
#include "spider.h"    // Animated spider class


// Macro to display a message box containing the given string. 
#define DISPLAYMSG(x) MessageBox(NULL, x, "D3DRM Sample", MB_OK);


// Global Variables 
CADirect3DRM *lpD3DRM = NULL;            // Direct3DRM object 
CADirectDrawClipper *lpDDClipper = NULL; // DirectDrawClipper object 

// Global Structure
struct _myglobs
{
    // Direct3DRM device. 
    CADirect3DRMDevice * dev;           

    // Direct3DRM viewport through which to view the scene. 
    CADirect3DRMViewport * view;  

    // Master frame in which other frames are placed. 
    CADirect3DRMFrame * scene;    

    // Frame describing the users point of view. 
    CADirect3DRMFrame * camera;   

    // mesh spider class
    CASpider  *spider;

    // Application is minimized. 
    BOOL bMinimized;            

    // All D3DRM objects have been initialized. 
    BOOL bInitialized;          

} myglobs;


// Prototypes 
BOOL             BuildScene( CADirect3DRM *, CADirect3DRMDevice *, 
                             CADirect3DRMFrame *, CADirect3DRMFrame * );
static BOOL      RenderScene();
BOOL             CreateObjects( HWND win );

static HWND      InitApp( HINSTANCE, int );
int APIENTRY     WinMain( HINSTANCE, HINSTANCE, LPSTR, int );
LRESULT CALLBACK WindowProc( HWND, UINT, WPARAM, LPARAM );




//-----------------------------------------------------------------------------
// Name: BuildScene()
// Desc: Create the scene to be rendered. 
//-----------------------------------------------------------------------------
BOOL BuildScene( CADirect3DRM * lpD3DRM,  CADirect3DRMDevice * dev, 
                 CADirect3DRMFrame * scene, CADirect3DRMFrame * camera )
{
    CADirect3DRMFrame *lights = NULL;
    CADirect3DRMLight  *light1 = NULL;
    CADirect3DRMLight  *light2 = NULL;

    CADirect3DRMFrame rotateframe( lpD3DRM, scene->I() );
    rotateframe.SetRotation(scene->I(), D3DVAL(0), D3DVAL(1), D3DVAL(0), D3DVAL(-0.01)); // angle 

    myglobs.spider = new CASpider( lpD3DRM, rotateframe.I() );
    myglobs.spider->SetPosition(rotateframe.I(), D3DVAL(4), D3DVAL(0), D3DVAL(0));

    // Set up the camera frame's position. Objects with the same x-value and
    // y-value as the camera will appear straight ahead.
    // Negative z-values are farther away, making the object look
    // smaller as the negative numbers increase.
    camera->SetPosition(scene->I(), D3DVAL(0), D3DVAL(8), -D3DVAL(16));
    camera->LookAt( rotateframe.I(), scene->I(), D3DRMCONSTRAIN_Z );

    // Initialize the lights in the scene, creating a light frame that
    // is a child of the scene.
    lights = new CADirect3DRMFrame( lpD3DRM, scene->I() );

    // Position the light frame within the scene.
    lights->SetPosition(scene->I(), D3DVAL(5), D3DVAL(0), -D3DVAL(7));
    lights->SetOrientation( scene->I(),  1, -1, 1,  1, 1, 1);

    light1 = new CADirect3DRMLight( lpD3DRM, D3DRMLIGHT_DIRECTIONAL,
                                    D3DRGB( D3DVAL(1.0), D3DVAL(0.8), D3DVAL(0.9)));
    lights->AddLight( light1->I() );

    // Create a dim, ambient light and add it to the scene frame,
    // applying it to the whole scene. Ambient light comes from all 
    // directions, so a bright ambient light would wash out the object.
    light2 = new CADirect3DRMLight( lpD3DRM, D3DRMLIGHT_AMBIENT, D3DRGB( 
                                    D3DVAL(.1), D3DVAL(0.1), D3DVAL(0.1)));
    lights->AddLight( light2->I() );
    
    // setup the shadows on the spider
    LPVOID pVoid = NULL;
    lpD3DRM->I()->QueryInterface( IID_IDirect3DRM2, (void **) &pVoid );
    if (pVoid)  //  DX5 or greater
    {
      LPDIRECT3DRMVISUAL pShadow;
      lpD3DRM->CreateShadow( myglobs.spider->I(), light1->I(),
        D3DVAL(0), D3DVAL(-1), D3DVAL(0),
        D3DVAL(0), D3DVAL(1), D3DVAL(0),
        &pShadow);

      rotateframe.AddVisual( pShadow );
      pShadow->Release();
    }
    else  // DX3 has limited shadow support
    {
      LPDIRECT3DRMVISUAL pShadow;
      lpD3DRM->CreateShadow( myglobs.spider->m_pMeshBuilder->I(), light1->I(),
        D3DVAL(0), D3DVAL(-1), D3DVAL(0),
        D3DVAL(0), D3DVAL(1), D3DVAL(0),
        &pShadow);

      myglobs.spider->AddVisual( pShadow );
      pShadow->Release();
    }

    // Clean up.
    delete lights;
    delete light1;
    delete light2;
    return TRUE;
}




//-----------------------------------------------------------------------------
// Name: RenderScene()
// Desc: Clear the viewport, render the next frame, and update the window.
//-----------------------------------------------------------------------------
static BOOL RenderScene()
{
    // Move the scene.
    myglobs.scene->Move(D3DVAL(1.0));
    myglobs.spider->Tick();

    // Clear the viewport.
    myglobs.view->Clear();

    // Render the scene to the viewport.
    myglobs.view->Render(myglobs.scene->I());

    // Update the window.
    myglobs.dev->Update();

    return TRUE;
}




//-----------------------------------------------------------------------------
// Name: AddMediaPath()
// Desc: Looks in the system registry to determine the media path for the
//       sample. Then, it adds that path to the string passed in, checks if the
//       file exists, and returns a path to the file.
//-----------------------------------------------------------------------------
VOID AddMediaPath( CADirect3DRM * pD3DRM )
{
	HKEY   key;
	LONG   result;
	TCHAR  strPath[512];
	DWORD  type, size = 512;

	// Open the registry
	result = RegOpenKeyEx( HKEY_LOCAL_MACHINE, "Software\\Microsoft\\DirectX",
						   0, KEY_READ, &key );
	if( ERROR_SUCCESS != result )
		return;

	// Search for the desired registry value, and close the registry
	result = RegQueryValueEx( key, "DX6SDK Samples Path", NULL, &type, 
		                      (BYTE*)strPath, &size );
	RegCloseKey( key );

	if( ERROR_SUCCESS != result )
		return;

	strcat( strPath, "\\D3DRM\\Media" );

	pD3DRM->AddSearchPath( strPath );

	return;
}




//-----------------------------------------------------------------------------
// Name: CreateObjects()
// Desc: Initialize globals, create the device and objects. 
//-----------------------------------------------------------------------------
BOOL CreateObjects( HWND win )
{
    RECT rc;      // Bounding rectangle for main window 
    int width;    // Device's width 
    int height;   // Device's height 


    // Initialize the entire global variable structure to zero. 
    memset(&myglobs, 0, sizeof(myglobs));

    // Create a DirectDrawClipper object and associate the window with it.
    lpDDClipper = new CADirectDrawClipper();
    lpDDClipper->SetHWnd(0, win);

    // Create the Direct3DRM object.
    lpD3DRM = new CADirect3DRM();
    
    // Create a default Direct3DRM device.
    GetClientRect(win, &rc);

    myglobs.dev = new CADirect3DRMDevice( lpD3DRM, lpDDClipper->I(),  
                           rc.right, rc.bottom);

    // Create the master scene frame and the camera frame.
    myglobs.scene = new CADirect3DRMFrame( lpD3DRM, NULL );
    myglobs.scene->SetSceneBackgroundRGB( D3DVAL(0), D3DVAL(0), D3DVAL(.4) );

    myglobs.camera = new CADirect3DRMFrame( lpD3DRM, myglobs.scene->I() );

    // Create the Direct3DRM viewport using the device, camera frame,
    // and the device's width and height.
    width = myglobs.dev->GetWidth();
    height = myglobs.dev->GetHeight();

    myglobs.view = new CADirect3DRMViewport( lpD3DRM, myglobs.dev->I(), myglobs.camera->I(), 0, 0, 
                                   width, height);

  	// Add the media path so our textures and .x files can be found
	  AddMediaPath( lpD3DRM );

    // Create the scene to be rendered.
    if (!BuildScene(lpD3DRM, myglobs.dev, myglobs.scene, myglobs.camera))
        return FALSE;

    // Globals are initialized.
    myglobs.bInitialized = TRUE;

    return TRUE;
}




//-----------------------------------------------------------------------------
// Name: InitApp()
// Desc: Create the main window and initialize objects. 
//-----------------------------------------------------------------------------
static HWND InitApp( HINSTANCE this_inst, int cmdshow )
{
    HWND win;     // Main window handle 
    WNDCLASS wc;  // Window class

    // Set up and register the window class.
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WindowProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = sizeof(DWORD);
    wc.hInstance = this_inst;
    wc.hIcon = LoadIcon(this_inst, (LPCTSTR) IDI_ICON1);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName = MAKEINTRESOURCE( IDR_MENU );
    wc.lpszClassName = "DirectXCPP Demo";
    if (!RegisterClass(&wc))
        return FALSE;

    // Create the window.
    win = CreateWindow(   
            "DirectXCPP Demo",                 // class 
            "DirectXCPP:  Spider Demo",        // caption 
            WS_OVERLAPPEDWINDOW,               // style 
            CW_USEDEFAULT,                     // init. x pos 
            CW_USEDEFAULT,                     // init. y pos 
            640,                               // init. x size 
            480,                               // init. y size 
            NULL,                              // parent window 
            NULL,                              // menu handle 
            this_inst,                         // program handle 
            NULL);                             // create parms 
        
    if (!win)
        return FALSE;

    // Initialize global variables and create the Direct3D Retained
    // Mode objects.
    if (!CreateObjects(win))
        return FALSE;

    // Display the window.
    ShowWindow(win, cmdshow);
    UpdateWindow(win);

    return win;
}




//-----------------------------------------------------------------------------
// Name: WinMain()
// Desc: Initialize the application, process messages, and render the scene.
//-----------------------------------------------------------------------------
int APIENTRY WinMain( HINSTANCE this_inst, HINSTANCE, LPSTR, int cmdshow )
{
    HWND    hwnd;
    MSG     msg;

    // Create the window and initialize objects. 
    if (!(hwnd = InitApp(this_inst, cmdshow)))
        return 1;

    while (TRUE) 
    {  
        // Monitor the message queue and process messages. PeekMessage 
        // returns control to the application immediately so the 
        // application can both process messages and continue rendering. 
        // If the message is WM_QUIT, break out of this function
        // because the application is terminating.
        if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) 
        {
            if (msg.message == WM_QUIT) 
                break;
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    
        // If the application is not minimized and the Direct3D Retained 
        // Mode objects are initialized, render.
        if (!myglobs.bMinimized && myglobs.bInitialized) 
        {
            // Render one frame of the scene. If rendering fails, post
            // the WM_DESTROY message, enabling the application to 
            // free resources before terminating the application.
            if (!RenderScene()) 
            {
                DISPLAYMSG("Rendering failed. Aborting execution.");
                PostMessage(NULL, WM_DESTROY, 0, 0);
                break;
            }

        // Yield to other applications if this application is
        // not currently rendering.
        } else 
              WaitMessage();

    }
    return msg.wParam;
}




//-----------------------------------------------------------------------------
// Name: WindowProc()
// Desc: Handle messages for the main window.
//-----------------------------------------------------------------------------
LRESULT CALLBACK WindowProc( HWND win, UINT msg, WPARAM wparam, LPARAM lparam )
{
    switch (msg) 
    {
        case WM_COMMAND:
	          switch( LOWORD( wparam ) )
	          {
	              case ID_ABOUT:
                    MessageBox( win,
                      "DirectXCPP 6.0 Spider Demo\n"
                      "Copyright (C) 1998, Weston Software\n\n"
                      "http:\\\\www.westonsoftware.com\n",
                      "About", MB_OK );
                    break;
	              case ID_FILE_EXIT:
	                  DestroyWindow( win );
	                  break;
	          }
	          break;

        case WM_SIZE:
            // Handle minimizing of the window. If the application was 
            // minimized, set a flag so rendering will stop while 
            // minimized. Otherwise, continue rendering.
            if (SIZE_MINIMIZED == wparam) 
                myglobs.bMinimized = TRUE;
            else
                myglobs.bMinimized = FALSE;
            break;

        case WM_DESTROY:
            // Clear the bInitialized flag, free objects, and post the 
            // WM_QUIT message to terminate the application. Be sure to
            // free the viewport before freeing the device or a general 
            // protection fault might occur.
            myglobs.bInitialized = FALSE;
            delete myglobs.spider;
            delete myglobs.view;
            delete myglobs.dev;
            delete myglobs.camera;
            delete myglobs.scene;
            delete lpD3DRM;
            delete lpDDClipper;
            PostQuitMessage(0);
            break;

        default:
            return DefWindowProc(win, msg, wparam, lparam);
    }
    return 0L;
}





