/*
Module : RASMONITOR.H
Purpose: Defines the interface for a number of wrapper 
         classes for monitoring Ras connections 
Created: PJN / 18-06-1997
History: PJN / 28-09-1998 1.  Updated code to include a virtual OnDial method
                          2.  Optimized the code by using CArray::ElementAt
                          3.  Added the following member variables to CRasConnection:
                              Dial Time, whether or not the connection is currently
                              connected, Dial Duration and the RAS handle
                          4.  Renamed the connection variable of CRasConnection
                          5.  Updated the sample program to v1.1
                          6.  Updated the version info in the sample app to include 
                              the build date
                          7.  Fixed a problem when enumerating more than one ras
                              connection at a time.
                          8.  Sample app now uses LoadImage instead of LoadIcon to avoid
                              the shell first scaling up to 32*32 and then back to 16*16.
                              What this means is that the icons RasMon puts in the tray
                              notification area will not look blurred.
                          9.  Fixed a memory leak in CRasMonitor::DoCheck() when multiple
                              connections were being enumerated.
                          10. Included a new selection of tray icons in the sample app.
                          11. Added a popup menu item to bring up the help file.
         PJN / 28-10-1998 1.  Major optimisation added to reduce memory usage (c. 3 MB) 
                              on NT when Ras service is not running. Thanks to Daniel 
                              Harth for this very neat optimisation.
                          2.  Internally the code now uses a hidden window to perform 
                              the monitoring instead of a worker thread. This helps avoid 
                              potential deadlock problems which can occur as previously 
                              the code used a worker thread.
                          3.  Now uses window messages instead of virtual functions to 
                              implement class customisation
         PJN / 02-02-1999 1.  Fixed a compile problem which was occurring when code was 
                              compiled for UNICODE.
                          2.  Fixed a crash which was occuring when the code was run on NT 
                              and you do not have Ras installed.
                          3.  When running on NT, the code now only uses 
                              SC_MANAGER_ENUMERATE_SERVICE priviledges when connecting to 
                              the Service Configuration Manager.
                          4.  CRasConnection connection now includes a standard MFC 
                              Serialize method.



Copyright (c) 1998-1999 by PJ Naughter.  
All rights reserved.

*/

#ifndef __RASMONITOR_H__
#define __RASMONITOR_H__
                      

#ifndef __AFXTEMPL_H__
#pragma message("RAS classes require afxtempl.h in your PCH")                                                                                
#endif

#ifndef _WINSVC_
#pragma message("RAS classes require winsvc.h in your PCH")                                                                                
#endif


//Function pointer typedefs used internally by CRasMonitor class
typedef DWORD (__stdcall *PFNRASENUMCONNECTIONS)(LPRASCONN, LPDWORD, LPDWORD);
typedef DWORD (__stdcall *PFNRASGETCONNECTSTATUS)(HRASCONN, LPRASCONNSTATUS);




//A class represeing a Dial-Up Networking connection
class CRasConnection
{
public:
//Constructors / Destructors
  CRasConnection();
	CRasConnection(const RASCONN& Connection);

//Methods
  void Serialize(CArchive& ar);
  static void SerializeTime(CArchive& ar, SYSTEMTIME& st);

//Data
  CString    m_sName;           //Name of the connection
	CString    m_sDeviceName;     //Name of the Device this connection is using
	CString    m_sDeviceType;     //Name of the Device Type this connection is using
  SYSTEMTIME m_ConnectionTime;  //The local time when the connection was started
  SYSTEMTIME m_DialTime;        //The local time when the connection was dialled
  BOOL       m_bConnected;      //True if currently connected, FALSE during the dialing
                                //stage of a connection
  DWORD      m_ConnectDuration; //how long the connection was made (in Seconds)
  DWORD      m_DialDuration;    //how long the connection took to be made (in Seconds)
  SYSTEMTIME m_HangupTime;      //The local time when the connection was closed
};


//Wparam which is used in notification message
const WPARAM RAS_DIAL_EVENT = 1;
const WPARAM RAS_CONNECT_EVENT = 2;
const WPARAM RAS_DISCONNECT_EVENT = 3;
const WPARAM RAS_CHECK_EVENT = 4;


//The Ras Monitor class which client applications should 
//derive from and override the virtual functions to perform
//whatever application specific behaviour they want
class CRasMonitor : public CWnd
{
public:
//Constructors / Destructors
  CRasMonitor();
  virtual ~CRasMonitor();

//Methods
	//Start monitoring ras connections
  BOOL Start(CWnd* pNotifyWnd, UINT nNotifyMessage);

	//Stop monitoring ras connections.
  BOOL Stop();

  //Determines if a ras connection handle is currently connected
  static BOOL IsConnected(HRASCONN hConnection);

  //Serializes the current state of the connections to and from disk
  void Serialize(CArchive& ar);

  //Accessors
  int NumberOfConnections() const { return m_Connections.GetSize(); };

protected:
//Data
  CArray<CRasConnection, CRasConnection&> m_Connections;
  UINT m_nTimerID;
  CWnd* m_pNotificationWnd;
  UINT m_nNotifyMessage;
  BOOL m_bStarted;
  BOOL m_bArchivedDataLoaded;
  SYSTEMTIME m_archiveTime;

//Methods
	BOOL ConnectionExists(const CString& sName);
  int GetConnection(const CString& sName);
	void DoCheck(BOOL bClosing);
  void OnDial(const CRasConnection& connection);
  void OnConnect(const CRasConnection& connection);
  void OnDisconnect(const CRasConnection& connection);
  void OnCheck(BOOL bClosing, RASCONN* pConnections, DWORD dwConnections);

//Static Methods
  static DWORD RasEnumConnections(LPRASCONN lprasconn, LPDWORD lpcb, LPDWORD lpcConnections); 
  static DWORD RasGetConnectStatus(HRASCONN hrasconn, LPRASCONNSTATUS lprasconnstatus);

//Static Data
  static SC_HANDLE m_hSCM;
  static SC_HANDLE m_hRasMan;
  static PFNRASENUMCONNECTIONS m_pfnRasEnumConnections;
  static PFNRASGETCONNECTSTATUS m_pfnRasGetConnectStatus;
  static HINSTANCE m_hRasApi32;
  static BOOL m_bRunningNT;

	//{{AFX_MSG(CRasMonitor)
	afx_msg int OnCreate(LPCREATESTRUCT lpCreateStruct);
	afx_msg void OnTimer(UINT nIDEvent);
	//}}AFX_MSG
  DECLARE_MESSAGE_MAP()

  DECLARE_DYNAMIC(CRasMonitor)
};




#endif //__RASMONITOR_H__