/*
/--------------------------------------------------------------------
|
|      FILESRC.CPP        File Data Source Class
|
|      This is a class which takes a file as a source of picture data.
|
|      Restriction: If memory-mapped files are not available, this
|      class reads all data into memory _before_ decoding can start.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"

#include "filesrc.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
#endif

IMPLEMENT_DYNAMIC (CFileSource, CDataSource);

CFileSource::CFileSource
    ()
{
#ifdef _WINDOWS
  m_hf = NULL;
  m_hm = NULL;
#else
  m_pFile = NULL;
#endif
}

CFileSource::~CFileSource
    ()
{
#ifdef _WINDOWS
  if (m_hf)
    Close();
#else
  if (m_pFile)
    Close();
#endif
}

int CFileSource::Open
    ( const char * pszFName
    )
{
  int FileSize;

#ifdef _WINDOWS
  BYTE * pBuffer = NULL;
  try
  {
    m_hf = CreateFile (pszFName, GENERIC_READ, FILE_SHARE_READ, NULL,
                       OPEN_EXISTING,
                       FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
                       NULL);

    if (m_hf == INVALID_HANDLE_VALUE)
    {
      switch (GetLastError())
      {
        case ERROR_PATH_NOT_FOUND:
          // sprintf (sz, "Path not found.", pszFName);
          return ERR_PATH_NOT_FOUND;
        case ERROR_FILE_NOT_FOUND:
          // sprintf (sz, "File not found.", pszFName);
          return ERR_FILE_NOT_FOUND;
        case ERROR_ACCESS_DENIED:
          // sprintf (sz, "Access denied.", pszFName);
          return ERR_ACCESS_DENIED;
        case ERROR_SHARING_VIOLATION:
          // sprintf (sz, "Sharing violation.", pszFName);
          return ERR_ACCESS_DENIED;
        default:
          // sprintf (sz, "CreateFile returned %d.",
          //          pszFName, GetLastError());
          return ERR_FILE_NOT_FOUND;
      }
    }

    FileSize = ::GetFileSize (m_hf, NULL);

    m_hm = CreateFileMapping (m_hf, NULL, PAGE_READONLY, 0, 0, NULL);

    // This happens if the file is empty.
    if (m_hm == NULL)
      // raiseError (ERR_ACCESS_DENIED, "CreateFileMapping failed.");
      return ERR_ACCESS_DENIED;

    pBuffer = (BYTE *) MapViewOfFile (m_hm, FILE_MAP_READ, 0, 0, 0);

    if (pBuffer == NULL)
      // raiseError (ERR_ACCESS_DENIED, "MapViewOfFile failed.");
      return ERR_ACCESS_DENIED;

    // We've got the file mapped to memory.
    CDataSource::Open (pszFName, pBuffer, FileSize);
  }
  catch (CTextException)
  {
    // Clean up on error
    if (pBuffer) UnmapViewOfFile (pBuffer);
    if (m_hm) CloseHandle (m_hm);
    if (m_hf) CloseHandle (m_hf);
    throw;
  }
  return 0;

#else

  // Generic code assuming memory mapped files are not available.
  m_pFile = fopen (pszFName, "rb");
  if ((int)m_pFile == -1)
  { // Crude...
    m_pFile = 0;
    return -1;
  }

  // Determine file size. Can this be done in an easier way using ANSI C?
  fseek (m_pFile, 0, SEEK_END);
  FileSize = ftell (m_pFile);
  fseek (m_pFile, 0, SEEK_SET);

  // Read the whole file into the buffer. I shouldn't be doing this.
  m_pBuffer = (BYTE *) malloc (FileSize);
  fread (m_pBuffer, 1, FileSize, m_pFile);
  fclose (m_pFile);

  CDataSource::Open (pszFName, m_pBuffer, FileSize);
  return 0;
#endif
}

void CFileSource::Close
    ()
{
#ifdef _WINDOWS
  UnmapViewOfFile (m_pStartData);
  CDataSource::Close ();
  CloseHandle (m_hm);
  CloseHandle (m_hf);
  m_hm = NULL;
  m_hf = NULL;
#else
  free (m_pBuffer);
  m_pBuffer = NULL;
  CDataSource::Close ();
  m_pFile = NULL;
#endif
}

