/*
/--------------------------------------------------------------------
|
|	VIEW.CPP    view class implementation
|
|	In the document/view model, this is where document rendering
|	(drawing in this case) occurs. Also handles palette issues.
|
|	Copyright (c) 1998 Bernard Delme
|
\--------------------------------------------------------------------
*/

#include "stdafx.h"
#include "piclook.h"
#include "doc.h"
#include "view.h"
#include "mainfrm.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CPLView

IMPLEMENT_DYNCREATE(CPLView, CScrollView)

BEGIN_MESSAGE_MAP(CPLView, CScrollView)
	//{{AFX_MSG_MAP(CPLView)
	ON_WM_ERASEBKGND()
	ON_WM_SIZE()
	ON_MESSAGE(WM_DOREALIZE, OnDoRealize)
	ON_UPDATE_COMMAND_UI(ID_VIEW_FITIMAGE, OnUpdateFitImage)
	ON_COMMAND(ID_VIEW_FITIMAGE, OnFitImage)
	ON_UPDATE_COMMAND_UI(ID_VIEW_DITHER, OnUpdateDither)
	ON_COMMAND(ID_VIEW_DITHER, OnDither)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	//}}AFX_MSG_MAP

	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPLView construction/destruction

CPLView::CPLView()
{
	m_bFit = FALSE;		// use real graphics dimensions
	m_bDither = TRUE;	// use vfw's DrawDib engine
}

CPLView::~CPLView()
{
}

/////////////////////////////////////////////////////////////////////////////
// CPLView drawing

void CPLView::OnDraw(CDC* pDC)
{
	CPLViewerDoc* pDoc = GetDocument();
	if ( ! pDoc ) return;

	CBmpClass *pDib = pDoc->GetDib();
	if ( ! pDib ) return;

	// else, we have a bitmap to show!

	CSize docsize = pDoc->GetDocSize();
	CRect rcImg(0,0,docsize.cx,docsize.cy);
	CRect rcDest;
	if (! pDC->IsPrinting())   // display DC
	{
		GetClientRect( &rcDest );
		if( ! m_bFit )	// display unaltered, centered
		{
			CPoint ptOffset;
			//rcDest = rcImg;
			ptOffset.x = max( 0, rcDest.Width() - rcImg.Width() ) / 2 ;
			ptOffset.y = max( 0, rcDest.Height() - rcImg.Height() ) / 2 ;
			rcDest = rcImg + ptOffset;
		}
	}
	else	// printer device context
	{
		// get size of printer page (in pixels)
		int cxPage = pDC->GetDeviceCaps(HORZRES);
		int cyPage = pDC->GetDeviceCaps(VERTRES);
		// get printer pixels per inch
		int cxInch = pDC->GetDeviceCaps(LOGPIXELSX);
		int cyInch = pDC->GetDeviceCaps(LOGPIXELSY);

		// create a rectangle which preserves the document's aspect ratio, 
		// and fills the page taking into account the relative size of a
		// printed pixel (cyInch / cxInch).

		rcDest.top = rcDest.left = 0;
		rcDest.right = cxPage;
		rcDest.bottom = MulDiv(docsize.cy * cyInch, cxPage, docsize.cx * cxInch);
		if (rcDest.bottom > cyPage)
		{
			rcDest.bottom = cyPage;
			rcDest.right  = MulDiv(docsize.cx * cxInch, cyPage, docsize.cy * cyInch);
		}
	}

#ifdef _NO_DRAWDIB
	double ratio;
	if ( rcDest != rcImg )
	{
		double rx = (double) rcDest.Width() / rcImg.Width();
		double ry = (double) rcDest.Height() / rcImg.Height();
		ratio  = min(rx,ry);
	}
	else ratio = 1.0;

	pDib->StretchDraw(pDC, 0, 0, ratio);
#else
	pDib->DrawEx(*pDC, &rcDest, &rcImg, m_bDither);
#endif
}

/////////////////////////////////////////////////////////////////////////////
// CPLView printing

BOOL CPLView::OnPreparePrinting(CPrintInfo* pInfo)
{
	pInfo->SetMaxPage(1);
	// default preparation
	return DoPreparePrinting(pInfo);
}

/////////////////////////////////////////////////////////////////////////////
// CPLView commands

/* 
 *	Woosh, we got the WM_DOREALIZE message from MDI supervisor,
 *	time to shine our brightest colors
 */
LRESULT CPLView::OnDoRealize(WPARAM wParam, LPARAM)
{
	BOOL bRet = FALSE;
#ifndef _IGNORES_PALETTE
	ASSERT(wParam != NULL);
	CPLViewerDoc* pDoc = GetDocument();
	if (pDoc->GetDib() == NULL)
		return bRet;  // must be a new document

	CPalette* pPal = pDoc->GetDocPalette();
	if (pPal != NULL)
	{
		CMainFrame* pAppFrame = (CMainFrame*) AfxGetApp()->m_pMainWnd;
		ASSERT_KINDOF(CMainFrame, pAppFrame);

		CClientDC appDC(pAppFrame);
		// All views but one should be a background palette.
		// wParam contains a handle to the active view, so the SelectPalette
		// bForceBackground flag is FALSE only if wParam == m_hWnd (this view)
		CPalette* oldPalette = appDC.SelectPalette(pPal, ((HWND)wParam) != m_hWnd);

		if (oldPalette != NULL)
		{
			UINT nColorsChanged = appDC.RealizePalette();
			if (nColorsChanged > 0)
			{
				//pDoc->UpdateAllViews(NULL);
				Invalidate(FALSE);
				//UpdateWindow();
				bRet = TRUE;
			}
			appDC.SelectPalette(oldPalette, TRUE);
		}
		else
		{
			TRACE0("\tSelectPalette failed in CPLView::OnDoRealize()\n");
			//MessageBeep(0);
		}
	}
#endif	// _IGNORES_PALETTE
	return (LRESULT) bRet;
}

void CPLView::OnInitialUpdate()
{
	CScrollView::OnInitialUpdate();
	ASSERT(GetDocument() != NULL);

	// set scrolling ranges
	SetScrollSizes(MM_TEXT, GetDocument()->GetDocSize());

	// adjust default view size to smaller documents
	CSize docdim = GetDocument()->GetDocSize();
	CRect vwrect; GetClientRect( vwrect );
	CSize vwdim = vwrect.Size();
	if ( docdim.cx < vwdim.cx || docdim.cy < vwdim.cy )
	{
		int cx = min( docdim.cx, vwdim.cx );
		int cy = min( docdim.cy, vwdim.cy );
		MoveWindow(0,0,cx,cy,FALSE);
		GetParentFrame()->RecalcLayout();	// rtfm!
		// adapt enclosing frame and get rid of scrollbar(s)
		ResizeParentToFit(TRUE);
	}
}

void CPLView::OnActivateView(BOOL bActivate, CView* pActivateView,
					CView* pDeactiveView)
{
	CScrollView::OnActivateView(bActivate, pActivateView, pDeactiveView);

	if (bActivate)
	//if (pDeactiveView != pActivateView)
	{
		// probably we switched from view/doc within MDI app, 
		// so this view deserves foreground palette 
		ASSERT(pActivateView == this);
		OnDoRealize((WPARAM)m_hWnd, 0);   // same as SendMessage(WM_DOREALIZE);
	}
}

BOOL CPLView::OnEraseBkgnd(CDC* pDC) 
{
#ifndef _NO_DRAWDIB
	if (m_bFit)
	// in this case, we cover the whole client area with our
	// drawing, so there's no ned to erase
		return TRUE;
#endif
	CRect crect; GetClientRect( crect );
	CSize delta = GetDocument()->GetDocSize() - crect.Size();
	if (delta.cx >= 0 && delta.cy >= 0)
		return TRUE;

	//pDC->FillSolidRect(crect,::GetSysColor(COLOR_3DFACE));
	pDC->FillSolidRect(crect,RGB(224,224,224));
	return TRUE;
	//return CScrollView::OnEraseBkgnd(pDC);
}

void CPLView::OnSize(UINT nType, int cx, int cy) 
{
    CScrollView::OnSize(nType, cx, cy);
    
	if (m_bFit)	// track new dimension, avoid scrollbars
		SetScaleToFitSize(CSize(cx,cy));
}

// user toggles the 'fit image to viewable area' button
void CPLView::OnFitImage() 
{
	if (m_bFit = ! m_bFit)
	{
	    CRect cr;
	    GetClientRect( cr );
	    SetScaleToFitSize(cr.Size());
	}
	else
	    SetScrollSizes(MM_TEXT, GetDocument()->GetDocSize());

	//pDoc->UpdateAllViews(NULL);
	Invalidate(FALSE);
}

void CPLView::OnUpdateFitImage(CCmdUI* pCmdUI) 
{
//#ifdef _NO_DRAWDIB
//	pCmdUI->Enable(FALSE);
//#else
	pCmdUI->SetCheck(m_bFit);
//#endif
}

// experiment GDI vs VFW DIB drawing
void CPLView::OnDither() 
{
	m_bDither = ! m_bDither;
	Invalidate(FALSE);
}

void CPLView::OnUpdateDither(CCmdUI* pCmdUI) 
{
#ifdef _NO_DRAWDIB
	pCmdUI->Enable(FALSE);
#else	// only meaningful when using the DrawDib API
    pCmdUI->SetCheck(m_bDither);
#endif
}

void CPLView::OnEditCopy() 
{
	CBmpClass *pDib = GetDocument()->GetDib();
	pDib->ToClipBoard();
}
