/*
/--------------------------------------------------------------------
|
|      TIFFDEC.CPP            TIFF Decoder Class
|
|      TIFF file decoder. Uses LIBTIFF to do the actual conversion.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "tiffdec.h"
#include "except.h"

extern "C"
{
#include "tiffio.h"
#include "tif_msrc.h"
}

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CTIFFDecoder, CPicDecoder);
#endif

char CTIFFDecoder::m_szLastErr[256];

/////////////////////////////////////////////////////////////////////
// Class functions

CTIFFDecoder::CTIFFDecoder
    ()
    // Creates a decoder
{
}


CTIFFDecoder::~CTIFFDecoder
    ()
{
}

/* I used this function to test the LIBTIFF routines without a custom
   data source. If something goes wrong, it may still be useful to
   isolate bugs.

CBmp * CTIFFDecoder::MakeBmpFromFile
    ( char * pszFName
    )
{
  char sz[256];
  sprintf (sz, "Decoding file %s.\n", pszFName);
  trace (1, sz);

  TIFFSetErrorHandler (Win32ErrorHandler);
  TIFFSetWarningHandler (Win32WarningHandler);

  TIFF* tif = TIFFOpen(pszFName, "r");
  if (tif)
  {
    uint32 w, h;
    size_t npixels;
    uint32* raster;

    TIFFGetField (tif, TIFFTAG_IMAGEWIDTH, &w);
    TIFFGetField (tif, TIFFTAG_IMAGELENGTH, &h);
    npixels = w * h;

    m_pBmp->Create (w, h, 32, (img.alpha != 0));

    //raster = (uint32*) _TIFFmalloc(npixels * sizeof (uint32));
    raster = (uint32*) m_pDIB->GetBits();
    if (raster != NULL)
    {
      TIFFReadRGBAImage(tif, w, h, raster, 0)
      _TIFFfree(raster);
    }
    TIFFClose(tif);
  }

  return m_pDIB;
}
*/

void CTIFFDecoder::DoDecode
    ()
{
	// Set up function pointers to error handling routines.
	TIFFSetErrorHandler (Win32ErrorHandler);
	TIFFSetWarningHandler (Win32WarningHandler);

	TIFF* tif = TIFFOpenMem (m_pDataSrc->ReadEverything(),
						   m_pDataSrc->GetFileSize());
	if (!tif)
		raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);

	uint16	BitsPerSample;
	uint16	SamplePerPixel;

	/* get tagged image parameters */
	TIFFGetFieldDefaulted(tif, TIFFTAG_BITSPERSAMPLE, &BitsPerSample);
	TIFFGetFieldDefaulted(tif, TIFFTAG_SAMPLESPERPIXEL, &SamplePerPixel);

	/* For the time being, paintlib bitmaps only actually support 8
	   or 32bpp; so the following mapping should cover all cases ... */
	if (SamplePerPixel == 1 && BitsPerSample <= 8)
		doLoColor(tif);
	else	
		// complicated decoding; use higher-level API
		// it will promote all images to 32bpp, though
		doHiColor(tif);

	TIFFClose(tif);
}

void CTIFFDecoder::doHiColor(TIFF* tif)
{
  int ok;
  ULONG x, y;

  TIFFRGBAImage img;
  char emsg[1024];
  BYTE * pBits;

  CalcDestBPP (32); // non-palette formats handled here

  ok = TIFFRGBAImageBegin(&img, tif, 0, emsg);

  if (ok == 0)
  {
    TIFFClose (tif);
    raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);
  }

  try
  {
    m_pBmp->Create (img.width, img.height, 32, (img.alpha != 0));
    pBits = new BYTE [img.width*img.height*4];
    if (pBits == NULL)
      raiseError (ERR_NO_MEMORY, "Out of memory allocating TIFF buffer.");
  }
  catch (CTextException)
  {
    TIFFClose (tif);
    throw;
  }

  ok = TIFFRGBAImageGet(&img, (uint32 *) pBits, img.width, img.height);
  if (!ok)
  {
    TIFFRGBAImageEnd(&img);
    TIFFClose(tif);
    raiseError (ERR_WRONG_SIGNATURE, m_szLastErr);
  }

  BYTE ** pLineArray = m_pBmp->GetLineArray();

  // Correct the byte ordering
  for (y=0; y<img.height; y++)
  {
    BYTE * pSrc = pBits+(img.height-y-1)*img.width*4;
    RGBAPIXEL * pPixel = (RGBAPIXEL *)(pLineArray[y]);
    for  (x=0; x<img.width; x++)
    {
      SetRGBAPixel (pPixel, *pSrc, *(pSrc+1),
                    *(pSrc+2), *(pSrc+3));
      pPixel++;
      pSrc += 4;
    }
  }

  // Clean up.
  delete pBits;
  TIFFRGBAImageEnd(&img);
}

/*
 * TIFF decoding for 1, 4 and 8 bit(s) per pixel
 * bdelmee; 10/98
 */

/* check if color map holds old-style 8-bit values */
static int checkcmap(int n, uint16* r, uint16* g, uint16* b)
{
	while (n-- > 0)
		if (*r++ >= 256 || *g++ >= 256 || *b++ >= 256)
			return (16);

	return (8);
}

#define CVT(x)      (((x) * 255L) / ((1L<<16)-1))

void CTIFFDecoder::doLoColor(TIFF* tif)
{
	uint32	imageLength;
	uint32	imageWidth;
	uint16	BitsPerSample;
	int32	LineSize;
	uint16	SamplePerPixel;
	uint32	RowsPerStrip;
	int16	PhotometricInterpretation;
	uint32	row;
	BYTE	*pBits;

	TIFFGetFieldDefaulted(tif, TIFFTAG_IMAGEWIDTH, &imageWidth);
	TIFFGetFieldDefaulted(tif, TIFFTAG_IMAGELENGTH, &imageLength);
	TIFFGetFieldDefaulted(tif, TIFFTAG_BITSPERSAMPLE, &BitsPerSample);
	TIFFGetFieldDefaulted(tif, TIFFTAG_ROWSPERSTRIP, &RowsPerStrip);
	TIFFGetFieldDefaulted(tif, TIFFTAG_PHOTOMETRIC, &PhotometricInterpretation);
	TIFFGetFieldDefaulted(tif, TIFFTAG_SAMPLESPERPIXEL, &SamplePerPixel);

	LineSize = TIFFScanlineSize(tif); //Number of bytes in one line

	CalcDestBPP (8); // palette formats handled here

	try
	{
		m_pBmp->Create (imageWidth, imageLength, 8, 0);
		m_pPal = new RGBAPIXEL[256];
		// memset( m_pPal, 0, 256 * sizeof(RGBAPIXEL));
		if (m_pPal == NULL)
			raiseError (ERR_NO_MEMORY, "Out of memory allocating TIFF palette.");
		pBits = new BYTE [LineSize];
		if (pBits == NULL)
			raiseError (ERR_NO_MEMORY, "Out of memory allocating TIFF buffer.");
	}
	catch (CTextException)
	{
		TIFFClose (tif);
		throw;
	}


	// phase one: build color map

	if /* monochrome (=bitonal) or grayscale */
	(PhotometricInterpretation == PHOTOMETRIC_MINISWHITE ||
	PhotometricInterpretation == PHOTOMETRIC_MINISBLACK)
	{
		int numColors = 1 << BitsPerSample;
		BYTE step = 255 / (numColors-1);
		BYTE *pb = (BYTE *) m_pPal;
		if (PhotometricInterpretation == PHOTOMETRIC_MINISWHITE)
			pb += (numColors-1) * sizeof(RGBAPIXEL);
		// warning: the following ignores possible halftone hints
		for (int i = 0; i < numColors ; ++i)
		{
			pb[RGBA_RED] = pb[RGBA_GREEN] = pb[RGBA_BLUE] = i * step;
			pb[RGBA_ALPHA] = 255;
			if (PhotometricInterpretation == PHOTOMETRIC_MINISWHITE)
				pb -= sizeof(RGBAPIXEL);
			else
				pb += sizeof(RGBAPIXEL);
		}
	}
	//PhotometricInterpretation = 2 image is RGB
	//PhotometricInterpretation = 3 image has a color palette
	else if (PhotometricInterpretation == PHOTOMETRIC_PALETTE)
	{
		uint16* red;
		uint16* green;
		uint16* blue;
		int16 i, Palette16Bits;

		// we get pointers to libtiff-owned colormaps
		i = TIFFGetField(tif, TIFFTAG_COLORMAP, &red, &green, &blue);

		//Is the palette 16 or 8 bits ?
		Palette16Bits = checkcmap(1<<BitsPerSample, red, green, blue) == 16;

		//load the palette in the DIB
		for (i = 0; i < 1<<BitsPerSample; ++i)
		{
			BYTE *pb = (BYTE *) (m_pPal+i);
			pb[RGBA_RED  ] = (BYTE) (Palette16Bits ? CVT(  red[i]) :   red[i]);
			pb[RGBA_GREEN] = (BYTE) (Palette16Bits ? CVT(green[i]) : green[i]);
			pb[RGBA_BLUE ] = (BYTE) (Palette16Bits ? CVT( blue[i]) :  blue[i]);
			pb[RGBA_ALPHA] = 255;
		}
	}
	else
		Trace( 2, "unexpected PhotometricInterpretation in CTIFFDecoder::DoLoColor()" );

	// phase two: read image itself

	//generally, TIFF images are ordered from upper-left to bottom-right
	// we implicitly assume PLANARCONFIG_CONTIG
	BYTE **pLineArray = m_pBmp->GetLineArray();

	if (BitsPerSample > 8)
		Trace( 2, "unexpected bit-depth in CTIFFDecoder::DoLoColor()" );
	else for ( row = 0; row < imageLength; ++row )
	{
		uint16 x;
		TIFFReadScanline( tif, pBits, row, 0 );
		if (BitsPerSample == 1)	// go ahead, waste space ;-)
			for (x=0; x < imageWidth; ++x)
				pLineArray[row][x] = pBits[x / 8] & (128 >> (x & 7)) ? 1 : 0;
		else if (BitsPerSample == 4)
		{
			for (x=0; x < imageWidth / 2; ++x)
			{
				pLineArray[row][2*x  ] = (pBits[x] & 0xf0) >> 4;
				pLineArray[row][2*x+1] = (pBits[x] & 0x0f);
			}
			// odd number of pixels
			if (imageWidth & 1)
				pLineArray[row][imageWidth-1] = (pBits[x] & 0xf0) >> 4;
		}
		else //if (BitsPerSample == 8)
			memcpy( pLineArray[row], pBits, LineSize );
	}

	// propagate colormap
	m_pBmp->SetPalette( m_pPal );
	// clean up
	delete pBits;
}

/////////////////////////////////////////////////////////////////////
// Static functions used as Callbacks from the TIFF library

void CTIFFDecoder::Win32ErrorHandler
    ( const char* module,
      const char* fmt,
      va_list ap
    )
{
  char szMessage [256];
  sprintf(szMessage, fmt, ap);

  strcpy (m_szLastErr, szMessage);

  return;
}

void CTIFFDecoder::Win32WarningHandler
    ( const char* module,
      const char* fmt,
      va_list ap
    )
{
  char szTemp[256];
  char szMessage [256];
  sprintf(szMessage, fmt, ap);

  if (module != NULL)
    sprintf (szTemp, "Warning in LIBTIFF: %s\n", szMessage);
   else
    sprintf (szTemp, "Warning in LIBTIFF: %s\n", szMessage);

  Trace (2, szTemp);

  return;
}
