/*
/--------------------------------------------------------------------
|
|      JPEGDEC.CPP            JPEG Decoder Class
|
|      JPEG file decoder. Uses the independent JPEG group's library
|      to do the actual conversion.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#define JPEG_INTERNALS
#include "jpegdec.h"

#include "except.h"

extern "C"
{
#include "jmemsrc.h"
}

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
#endif

// Check to see if the pixel format defined in jmorecfg.h matches the
// one defined in bitmap.h.
// If the program reports an error here, you're probably using a wrong
// version of jmorecfg.h. Several files in libjpeg and libtiff need to
// replaced for the libraries to be used with paintlib. See the
// documentation for details.
#if (RGBA_RED != RGB_RED) || (RGBA_GREEN != RGB_GREEN)   \
    || (RGBA_BLUE != RGB_BLUE) || (RGB_PIXELSIZE != PIXEL_SIZE)
#error Illegal JPEG pixel format
#endif

IMPLEMENT_DYNAMIC (CJPEGDecoder, CPicDecoder);

/////////////////////////////////////////////////////////////////////
// Error handling.

METHODDEF(void)
error_exit (j_common_ptr pcinfo)
// This procedure is called by the IJPEG library when an error
// occurs.
{
  /* Create the message string */
  char sz[256];
  (pcinfo->err->format_message) (pcinfo, sz);
  strcat (sz, "\n");

  CPicDecoder::raiseError (ERR_FORMAT_NOT_SUPPORTED, sz);
}

/////////////////////////////////////////////////////////////////////
// Class functions

CJPEGDecoder::CJPEGDecoder
    ()
    // Creates a decoder
{
  cinfo.err = jpeg_std_error (&jerr);
  jerr.error_exit = error_exit;  // Register custom error manager.

  jpeg_create_decompress (&cinfo);

  m_bFast = TRUE;
}


CJPEGDecoder::~CJPEGDecoder
    ()
{
  jpeg_destroy_decompress (&cinfo);
}

void CJPEGDecoder::SetFast
    ( BOOL bFast
    )
    // TRUE (the default) selects fast but sloppy decoding.
{
  m_bFast = bFast;
}

/* I used this function to test the IJPEG rouines without a custom
   data source. If something goes wrong, it may still be useful to
   isolate bugs.

CBmp * CJPEGDecoder::MakeBmpFromFile
    ( char * pszFName
    )
{
  char sz[256];
  sprintf (sz, "Decoding file %s.\n", pszFName);
  trace (1, sz);

  FILE * pFile;

  if ((pFile = fopen (pszFName, "rb")) == NULL)
  {
    TRACE ("Couldn't open file.\n");
  }

  try
  {
    jpeg_stdio_src (&cinfo, pFile);

    jpeg_read_header (&cinfo, TRUE);

    int w = cinfo.image_width;
    int h = cinfo.image_height;

    m_pBmp->Create (w, h, 32, FALSE);

    jpeg_start_decompress (&cinfo);

    if (cinfo.out_color_space == JCS_GRAYSCALE)
      decodeGray (w, h);
     else
      decodeRGB (w, h);
    jpeg_finish_decompress (&cinfo);
  }
  catch (CTextException)
  {
    jpeg_abort_decompress(&cinfo);
    fclose (pFile);
    throw;
  }
  fclose (pFile);
  return m_pBmp;
}
*/

void CJPEGDecoder::DoDecode
    ()
{
  try
  {
    // Initialize custom data source.
    jpeg_mem_src (&cinfo, m_pDataSrc->ReadEverything(),
                  m_pDataSrc->GetFileSize());

    jpeg_read_header (&cinfo, TRUE);

    if (m_bFast)
    {
      cinfo.do_fancy_upsampling = FALSE;
    }

    // Choose floating point DCT method.
    cinfo.dct_method = JDCT_FLOAT;

    int w = cinfo.image_width;
    int h = cinfo.image_height;

    jpeg_start_decompress (&cinfo);

    if (cinfo.out_color_space == JCS_GRAYSCALE)
      decodeGray (w, h);
     else
      decodeRGB (w, h);
    jpeg_finish_decompress (&cinfo);
  }
  catch (CTextException)
  {
    jpeg_abort_decompress(&cinfo);
    throw;
  }
}

void CJPEGDecoder::decodeRGB
    ( int w,
      int h
    )
    // Assumes IJPEG decoder is already set up.
{
  BYTE * pDst;
  int CurLine = 0;
  JSAMPARRAY ppDst = &pDst;

  CalcDestBPP (32);
  m_pBmp->Create (w, h, 32, FALSE);

  BYTE ** pLineArray = m_pBmp->GetLineArray();

  while (CurLine < h)
  {
    pDst = pLineArray[CurLine];
    jpeg_read_scanlines (&cinfo, ppDst, 1);
    CurLine++;
  }
}

void CJPEGDecoder::decodeGray
    ( int w,
      int h
    )
    // Assumes IJPEG decoder is already set up.
{
  BYTE * pDst;
  int CurLine = 0;
  BYTE * pBuf = new BYTE [w];
  JSAMPARRAY ppBuf = &pBuf;

  CalcDestBPP (8);
  m_pBmp->Create (w, h, m_DestBPP, FALSE);

  BYTE ** pLineArray = m_pBmp->GetLineArray();

  while (CurLine < h)
  {
    if (m_DestBPP == 32)
    {
      jpeg_read_scanlines (&cinfo, ppBuf, 1);

      pDst = pLineArray[CurLine];

      for (int i=0; i<w; i++)
      {
        pDst[i*4] = pBuf[i];
        pDst[i*4+1] = pBuf[i];
        pDst[i*4+2] = pBuf[i];
        pDst[i*4+3] = 0xFF;
      }
    }
    else
    {
      ppBuf = &pDst;
      *ppBuf = pLineArray[CurLine];
      jpeg_read_scanlines (&cinfo, ppBuf, 1);

    }
    CurLine++;
  }
  delete pBuf;
}
