/*
/--------------------------------------------------------------------
|
|      BMPDEC.CPP            Windows Bitmap Decoder Class
|
|      Windows bitmap file decoder. Decodes 1, 4, 8, and 24 bpp
|      bitmap files (compressed and uncompressed) and returns a 32
|      bpp DIB.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdpch.h"
#include "bmpdec.h"
#include "except.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif
#endif

IMPLEMENT_DYNAMIC (CBmpDecoder, CPicDecoder);


CBmpDecoder::CBmpDecoder
    ()
    // Creates a decoder
{
}


CBmpDecoder::~CBmpDecoder
    ()
{
}


void CBmpDecoder::DoDecode
    ()
{
  WINBITMAPINFOHEADER * pBMI = NULL;  // Pointer to bitmapinfoheader of the file.
  BYTE * pBits;

  pBMI = getInfoHeader (&pBits);

  CalcDestBPP (pBMI->biBitCount);
  m_pBmp->Create (pBMI->biWidth, pBMI->biHeight, m_DestBPP, FALSE);
  if (m_DestBPP == 8)
    m_pBmp->SetPalette (m_pPal);

  switch (pBMI->biBitCount)
  {
    case 1:
      decode1bpp (pBits);
      break;
    case 4:
      if (pBMI->biCompression == BI_RGB)
        decode4bpp (pBits);
       else
        decodeRLE4 (pBits);
      break;
    case 8:
      if (pBMI->biCompression == BI_RGB)
        decode8bpp (pBits);
       else
        decodeRLE8 (pBits);
      break;
    case 24:
      decode24bpp (pBits);
      break;
    case 32:
      decode32bpp (pBits);
      break;
    default:
      // This is not a standard bmp file.
      raiseError (ERR_FORMAT_UNKNOWN,
                  "Decoding bmp: Illegal bpp value.");
  }
  delete pBMI;
  Trace (3, "Decoding finished.\n");
}

WINBITMAPINFOHEADER * CBmpDecoder::getInfoHeader
    ( BYTE ** ppBits     // Used to return location of bitmap bits.
    )
    // Decodes the bitmap file & info headers
{
  WINBITMAPFILEHEADER BFH;
  WINBITMAPINFOHEADER * pBMI;

  BFH.bfType = ReadIWord ();
  BFH.bfSize = ReadILong ();
  BFH.bfReserved1 = ReadIWord ();
  BFH.bfReserved2 = ReadIWord ();
  BFH.bfOffBits = ReadILong ();

  // Check for bitmap file signature: First 2 bytes are 'BM'
  if (BFH.bfType != 0x4d42)
    raiseError (ERR_WRONG_SIGNATURE,
                "Bitmap decoder: This isn't a bitmap.");

  Trace (2, "Bitmap file signature found\n");

  pBMI = new WINBITMAPINFOHEADER;

  try
  {
    pBMI->biSize = ReadILong ();
    pBMI->biWidth = ReadILong ();
    pBMI->biHeight = ReadILong ();
    pBMI->biPlanes = ReadIWord ();
    pBMI->biBitCount = ReadIWord ();
    pBMI->biCompression = ReadILong ();
    pBMI->biSizeImage = ReadILong ();
    pBMI->biXPelsPerMeter = ReadILong ();
    pBMI->biYPelsPerMeter = ReadILong ();
    pBMI->biClrUsed = ReadILong ();
    pBMI->biClrImportant = ReadILong ();

    // Do sanity check
    if (pBMI->biSize != sizeof (WINBITMAPINFOHEADER))
      raiseError (ERR_FORMAT_UNKNOWN,
                  "Bitmap decoder: BITMAPINFOHEADER has wrong size.");

    Trace (2, "Bitmap header is ok.\n");

    // Read palette if 8 bpp or less.
    int PaletteSize = 0;
    if (pBMI->biBitCount <= 8)
      PaletteSize = readPalette (pBMI);
    
    *ppBits = m_pDataSrc->ReadNBytes
                   (BFH.bfSize-
                    sizeof (WINBITMAPFILEHEADER)-
                    sizeof (WINBITMAPINFOHEADER)-
                    PaletteSize);
  }
  catch (CTextException)
  {
    if (pBMI)
      delete pBMI;
    throw;
  }
  return pBMI;
}

void CBmpDecoder::decode1bpp
    ( BYTE * pBits
    )
    // Decodes a 2-color bitmap. Ignores the palette & just uses
    // black & white as 'colors' if decoding to 32 bit
{
  int i;
  int y;                           // Current row
  int x;                           // Current column

  BYTE * pSrcLine = pBits;         // Start of current row in file.
  BYTE * pDest;                    // Current destination.
  BYTE * pSrc;                     // Current position in file.
  BYTE   BTable[8];                // Table of bit masks.
  BYTE   SrcByte;                  // Source byte cache.
  int    XSize = m_pBmp->GetWidth(); // Width of bitmap in pixels.
  int    LineLen = ((XSize+7)/8 + 3) & ~3;
                                   // Width of source in bytes
                                   //   (DWORD-aligned).
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  int    OpaqueBlack = 0x00000000;
  *(((BYTE*)&OpaqueBlack)+RGBA_ALPHA) = 0xFF;

  Trace (2, "Decoding 1 bit per pixel bitmap.\n");

  // Initialize bit masks.
  for (i=0; i<8; i++)
  {
    BTable[i] = 1<<i;
  }

  for (y=0; y<m_pBmp->GetHeight(); y++)
  { // For each line...
    pSrc = pSrcLine;
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    for (x=0; x<XSize/8; x++)
    { // For each source byte...
      SrcByte = *(pSrc);
      for (i=7; i>=0; i--)
      { // For each bit...
        if (m_DestBPP == 32)
        {
          if (SrcByte & BTable[i]) // Test if bit i is set
            *((LONG *)pDest) = 0xFFFFFFFF;
           else
            *((LONG *)pDest) = OpaqueBlack;
          pDest += 4;
        }
        else
        {
          if (SrcByte & BTable[i]) // Test if bit i is set
            *pDest = 0x01;
           else
            *pDest = 0x00;
          pDest++;
        }
      }
      pSrc++;
    }

    // Last few bits in line...
    SrcByte = *(pSrc);
    for (i=7; i>7-(XSize & 7); i--)
    { // For each bit...
      if (m_DestBPP == 32)
      {
        if (SrcByte & BTable[i]) // Test if bit i is set
          *((LONG *)pDest) = 0xFFFFFFFF;
         else
          *((LONG *)pDest) = OpaqueBlack;
        pDest += 4;
      }
      else
      {
        if (SrcByte & BTable[i]) // Test if bit i is set
          *pDest = 0x01;
         else
          *pDest = 0x00;
        pDest++;
      }
    }
    pSrcLine += LineLen;
  }
}

void CBmpDecoder::decode4bpp
    ( BYTE * pBits
    )
    // Decodes an uncompressed 16-color-bitmap.
{
  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  BYTE   SrcByte;                   // Source byte cache.
  int    XSize = m_pBmp->GetWidth();// Width of bitmap in pixels.
  int    LineLen = ((XSize+1)/2 + 3) & ~3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  Trace (2, "Decoding uncompressed 4 bit per pixel bitmap.\n");

  for (y=0; y<m_pBmp->GetHeight(); y++)
  { // For each line...
    pSrc = pSrcLine;
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    for (x=0; x<XSize/2; x++)
    { // For each source byte...
      SrcByte = *(pSrc);

      if (m_DestBPP == 32)
      {
        *((RGBAPIXEL *)pDest) = m_pPal[SrcByte>>4];
        pDest += 4;
        *((RGBAPIXEL *)pDest) = m_pPal[SrcByte & 15];
        pDest += 4;
      }
      else
      {
        *pDest = SrcByte>>4;
        pDest++;
        *pDest = SrcByte & 15;
        pDest++;
      }
      pSrc++;
   }

    // Last nibble in line if line length is odd.
    if (XSize & 1)
    {
      if (m_DestBPP == 32)
      {
        *((RGBAPIXEL *)pDest) = m_pPal[(*(pSrc))>>4];
        pDest += 4;
      }
      else
      {
        *pDest = (*(pSrc))>>4;
        pDest++;
      }
    }

    pSrcLine += LineLen;
  }
}


void CBmpDecoder::decode8bpp
    ( BYTE * pBits
    )
    // Decodes an uncompressed 256-color-bitmap.
{
  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = m_pBmp->GetWidth();// Width of bitmap in pixels.
  int    LineLen = (XSize + 3) & ~3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  Trace (2, "Decoding uncompressed 8 bit per pixel bitmap.\n");

  for (y=0; y<m_pBmp->GetHeight(); y++)
  { // For each line...
    pSrc = pSrcLine;
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    for (x=0; x<XSize; x++)
    { // For each source byte...
      if (m_DestBPP == 32)
      {
        *((RGBAPIXEL *)pDest) = m_pPal[*pSrc];
        pDest += 4;
      }
      else
      {
        *pDest = *pSrc;
        pDest++;
      }
      pSrc++;
    }
    pSrcLine += LineLen;
  }
}


void CBmpDecoder::decodeRLE4
    ( BYTE * pBits
    )
    // Decodes a compressed 16-color-bitmap.
{
  int y;                              // Current row

  BYTE * pDest;                       // Current destination.
  BYTE * pSrc = pBits;                // Current position in file.
  int    XSize = m_pBmp->GetWidth();  // Width of bitmap in pixels.
  BYTE   SrcByte;                     // Source byte cache.

  BYTE   RunLength;    // Length of current run.
  BOOL   bOdd;         // TRUE if current run has odd length.

  BOOL   bEOL;         // TRUE if end of line reached.
  BOOL   bEOF=FALSE;   // TRUE if end of file reached.

  BYTE * pLineBuf;     // Current line as uncompressed nibbles.
  BYTE * pBuf;         // Current position in pLineBuf.
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  Trace (2, "Decoding RLE4-compressed bitmap.\n");

  // Allocate enough memory for DWORD alignment in original 4 bpp
  // bitmap.
  pLineBuf = new BYTE [XSize*4+28];

  for (y=0; y<m_pBmp->GetHeight() && !bEOF; y++)
  { // For each line...
    pBuf = pLineBuf;
    bEOL=FALSE;
    while (!bEOL)
    { // For each packet do
      RunLength = *pSrc;
      pSrc++;
      if (RunLength==0)
      { // Literal or escape.
        RunLength = *pSrc;
        pSrc++;
        switch (RunLength)
        {
          case 0: // End of line escape
            bEOL = TRUE;
            break;
          case 1: // End of file escape
            bEOF = TRUE;
            bEOL = TRUE;
            break;
          case 2: // Delta escape.
            // I have never seen a file using this.
            delete pLineBuf;
            raiseError (ERR_FORMAT_NOT_SUPPORTED,
                        "Encountered delta escape.");
            break;
          default:
            // Literal packet
            bOdd = (RunLength & 1);
            RunLength /= 2; // Convert pixels to bytes.
            for (int i=0; i<RunLength; i++)
            { // For each source byte...
              decode2Nibbles (pBuf, *pSrc, m_pPal);
              if (m_DestBPP == 32)
                pBuf += 8;
              else
                pBuf += 2;
              pSrc++;
            }
            if (bOdd)
            { // Odd length packet -> one nibble left over
              if (m_DestBPP == 32)
              {
                *((RGBAPIXEL *)pBuf) = m_pPal[(*(pSrc))>>4];
                pBuf += 4;
              }
              else
              {
                *pBuf = (*(pSrc))>>4;
                pBuf++;
              }
              pSrc++;
            }
            // Word alignment at end of literal packet.
            if (RunLength & 1) pSrc++;
        }
      }
      else
      { // Encoded packet:
        // RunLength 4 bpp pixels with 2 alternating
        // values.
        SrcByte = *pSrc;
        pSrc++;
        for (int i=0; i<RunLength/2; i++)
        {
          decode2Nibbles (pBuf, SrcByte, m_pPal);
          if (m_DestBPP == 32)
            pBuf += 8;
          else
            pBuf += 2;
        }
        if (RunLength & 1)
        {
          if (m_DestBPP == 32)
          {
            *((RGBAPIXEL *)pBuf) = m_pPal[(*(pSrc))>>4];
            pBuf += 4;
          }
          else
          {
            *pBuf = (*(pSrc))>>4;
            pBuf++;
          }
        }
      }
    }
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    if (m_DestBPP == 32)
      memcpy (pDest, pLineBuf, XSize*4);
    else
      memcpy (pDest, pLineBuf, XSize);
  }
  delete pLineBuf;
}


void CBmpDecoder::decodeRLE8
    ( BYTE * pBits
    )
    // Decodes a compressed 256-color-bitmap.
{
  int y;                              // Current row

  BYTE * pDest;                       // Current destination.
  BYTE * pSrc = pBits;                // Current position in file.
  int    XSize = m_pBmp->GetWidth();  // Width of bitmap in pixels.

  BYTE   RunLength;                   // Length of current run.

  BOOL   bEOL;                        // TRUE if end of line reached.
  BOOL   bEOF=FALSE;                  // TRUE if end of file reached.
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                      // Pointers to dest. lines.

  Trace (2, "Decoding RLE8-compressed bitmap.\n");

  for (y=0; y<m_pBmp->GetHeight() && !bEOF; y++)
  { // For each line...
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    bEOL=FALSE;
    while (!bEOL)
    { // For each packet do
      RunLength = *pSrc;
      pSrc++;
      if (RunLength==0)
      { // Literal or escape.
        RunLength = *pSrc;
        pSrc++;
        switch (RunLength)
        {
          case 0: // End of line escape
            bEOL = TRUE;
            break;
          case 1: // End of file escape
            bEOF = TRUE;
            bEOL = TRUE;
            break;
          case 2: // Delta escape.
            // I have never seen a file using this...
            raiseError (ERR_FORMAT_NOT_SUPPORTED,
                        "Encountered delta escape.");
            bEOL = TRUE;
            bEOF = TRUE;
            break;
          default:
            // Literal packet
            if (m_DestBPP == 32)
            {
              for (int i=0; i<RunLength; i++)
              { // For each source byte...
                *((RGBAPIXEL *)pDest) = m_pPal[*pSrc];
                pDest += 4;
                pSrc++;
              }
            }
            else
            {
              memcpy (pDest, pSrc, RunLength);
              pDest += RunLength;
              pSrc += RunLength;
            }
            // Word alignment at end of literal packet.
            if (RunLength & 1) pSrc++;
        }
      }
      else
      { // Encoded packet:
        // RunLength pixels, all with the same value
        if (m_DestBPP == 32)
        {
          RGBAPIXEL PixelVal = m_pPal[*pSrc];
          pSrc++;
          for (int i=0; i<RunLength; i++)
          {
            *((RGBAPIXEL *)pDest) = PixelVal;
            pDest += 4;
          }
        }
        else
        {
          memset (pDest, *pSrc, RunLength);
          pSrc++;
          pDest += RunLength;
        }
      }
    }
  }
}


void CBmpDecoder::decode24bpp
    ( BYTE * pBits
    )
    // Decodes true-color bitmap
{
  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = m_pBmp->GetWidth();// Width of bitmap in pixels.
  int    LineLen = (XSize*3 + 3) & ~3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  Trace (2, "Decoding 24 bit per pixel bitmap.\n");

  for (y=0; y<m_pBmp->GetHeight(); y++)
  { // For each line...
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    pSrc = pSrcLine;
    for (x=0; x<XSize; x++)
    { // For each pixel...
      *(pDest+RGBA_BLUE) = ((WINRGBQUAD *)pSrc)->rgbBlue;
      *(pDest+RGBA_GREEN) = ((WINRGBQUAD *)pSrc)->rgbGreen;
      *(pDest+RGBA_RED) = ((WINRGBQUAD *)pSrc)->rgbRed;
      *(pDest+RGBA_ALPHA) = 0xFF;
      pDest += 4;
      pSrc += 3;
    }
    pSrcLine += LineLen;
  }
}

void CBmpDecoder::decode32bpp
    ( BYTE * pBits
    )
    // Decodes true-color bitmap
{
  int y;                            // Current row

  BYTE * pSrc = pBits;              // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  int    XSize = m_pBmp->GetWidth();// Width of bitmap in pixels.
  int    LineLen = XSize*4;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = m_pBmp->GetLineArray();
                                   // Pointers to dest. lines.

  Trace (2, "Decoding 32 bit per pixel bitmap.\n");

  for (y=0; y<m_pBmp->GetHeight(); y++)
  { // For each line...
    pDest = pLineArray[m_pBmp->GetHeight()-y-1];
    pSrc += LineLen;
    memcpy (pDest, pSrc, XSize*4);
  }
}

void CBmpDecoder::decode2Nibbles
    ( BYTE * pDest,
      BYTE SrcByte,
      RGBAPIXEL * pPal
    )
    // Decodes two 4-bit pixels.
{
  if (m_DestBPP == 32)
  {
    // High nibble
    *((RGBAPIXEL *)pDest) = pPal[SrcByte>>4];
    // Low nibble
    *((RGBAPIXEL *)(pDest+4)) = pPal[SrcByte & 15];
  }
  else
  {
    *pDest = SrcByte>>4;
    *(pDest+1) = SrcByte & 15;
  }
}

int CBmpDecoder::readPalette
    ( WINBITMAPINFOHEADER * pBMI     // Pointer to bitmapinfoheader in file.
    )
    // Assumes 8 bpp or less.
{
  Trace (3, "Reading palette.\n");
  int i;

  int NumColors;
  if (pBMI->biClrUsed == 0)
    NumColors = 1<<(pBMI->biBitCount);
   else
    NumColors = pBMI->biClrUsed;

  WINRGBQUAD * pPal = (WINRGBQUAD *) m_pDataSrc->ReadNBytes
                                    (NumColors*sizeof (WINRGBQUAD));
  m_pPal = new RGBAPIXEL  [256];

  // Correct the byte ordering & copy the data.
  for (i=0; i<NumColors; i++)
  {
    *(((BYTE *)m_pPal)+i*4+RGBA_BLUE) = pPal[i].rgbBlue;
    *(((BYTE *)m_pPal)+i*4+RGBA_GREEN) = pPal[i].rgbGreen;
    *(((BYTE *)m_pPal)+i*4+RGBA_RED) = pPal[i].rgbRed;
    *(((BYTE *)m_pPal)+i*4+RGBA_ALPHA) = 0xFF;
  }

  return NumColors*4;
}
