/*  xbase.cpp

    Xbase project source code

    This file contains logic for the basic Xbase class.

    Copyright (C) 1997  Crypton Technologies, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0   10/10/97   - Initial release of software
    V 1.5   1/2/97     - Added memo field support
    V 1.6a  4/1/98     - Added expression support
    V 1.6b  4/8/98     - Numeric index keys
    V 1.7.1 5/25/98    - Big Endian support
*/

#include "stdafx.h"
#include "xbase.h"
#include <ctype.h>
#include <string.h>

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


/*************************************************************************/
XBASE::XBASE( VOID )
{
   SHORT e = 1;
   EndianType = *(CHAR *) &e;
   if( EndianType )
      EndianType = 'L';
   else
      EndianType = 'B';
   DbfList = NULL;
   FreeDbfList = NULL;
}
/*************************************************************************/
DBF * XBASE::GetDbfPtr( CHAR * Name )
{
   DbList *t;
   SHORT len;
   t = DbfList;
   len = strlen( Name );
   while( t )
   {
      if( strncmp( Name, t->DbfName, len ) == 0 )
         return t->dbf;
      t = t->NextDbf;
   }
   return NULL;
}
/*************************************************************************/
SHORT XBASE::AddDbfToDbfList( DBF * d, CHAR * DatabaseName )
{
   DbList *i, *s, *t;

   if( !FreeDbfList )
   {
      if(( i = (DbList *) malloc( sizeof( DbList ))) == NULL )
         return NO_MEMORY;
   }
   else
   {
      i = FreeDbfList;
      FreeDbfList = i->NextDbf;
   }
   memset( i, 0x00, sizeof( DbList ));

   i->DbfName  = DatabaseName;
   i->dbf      = d;

   s = NULL;
   t = DbfList;
   while( t && strcmp( t->DbfName, DatabaseName ) < 0 )
   {
      s = t;
      t = t->NextDbf;
   }
   i->NextDbf = t;
   if( s == NULL )
      DbfList = i;
   else
      s->NextDbf = i;
   return 0;
}
/***********************************************************************/
SHORT XBASE::RemoveDbfFromDbfList( DBF * d )
{
   DbList *i, *s;

   i = DbfList;
   s = NULL;

   while( i )
   {
      if( i->dbf == d )
      {
        /* remove it from current chain */
         if( s )
           s->NextDbf = i->NextDbf;
         else
           DbfList = i->NextDbf;

         /* add i to the current free chain */
         i->NextDbf = FreeDbfList;
         FreeDbfList = i;
         FreeDbfList->DbfName = NULL;
         FreeDbfList->NextDbf = NULL; 
         break;
      }
      else
      {
         s = i;
         i = i->NextDbf;
      }
   }
   return NO_ERROR;
} 

/************************************************************************/
/* This routine returns a short value from a 2 byte character stream */
SHORT XBASE::GetShort( CHAR *p )
{
   SHORT s, i;
   CHAR *sp, *tp;

   s = 0;
   tp = (CHAR *) &s;
   sp = p;
   if( EndianType == 'L' )
      for( i = 0; i < 2; i++ ) *tp++ = *sp++;
   else
   {
      sp++;
      for( i = 0; i < 2; i++ ) *tp++ = *sp--;
   }  
   return s;
}

/* This routine returns a long value from a 4 byte character stream */
LONG XBASE::GetLong( CHAR *p )
{
   LONG l;
   CHAR *sp, *tp;
   SHORT i;

   tp = (CHAR *) &l;
   sp = p;
   if( EndianType == 'L' )
      for( i = 0; i < 4; i++ ) *tp++ = *sp++;
   else
   {
      sp+=3;
      for( i = 0; i < 4; i++ )  *tp++ = *sp--;
   }  
   return l;
}

/* This routine returns a double value from an 8 byte character stream */
DOUBLE XBASE::GetDouble( CHAR *p )
{
   DOUBLE d;
   CHAR *sp, *tp;
   SHORT i;

   tp = (CHAR *) &d;
   sp = p;
   if( EndianType == 'L' )
      for( i = 0; i < 8; i++ ) *tp++ = *sp++;
   else
   {
      sp+=7;
      for( i = 0; i < 8; i++ )  *tp++ = *sp--;
   } 

   return d;
}

/* This routine puts a short value to a 2 byte character stream */
VOID XBASE::PutShort( CHAR * c, SHORT s )
{
   CHAR *sp, *tp;
   SHORT i;

   tp = c;
   sp = (CHAR *) &s;

   if( EndianType == 'L' )
   {
      for( i = 0; i < 2; i++ ) *tp++ = *sp++;
   }
   else      /* big endian */
   {
      sp++;
      for( i = 0; i < 2; i++ ) *tp++ = *sp--;
   }
   return;
}

/* This routine puts a long value to a 4 byte character stream */
VOID XBASE::PutLong( CHAR * c, LONG l )
{
   CHAR *sp, *tp;
   SHORT i;

   tp = c;
   sp = (CHAR *) &l;
   if( EndianType == 'L' )
      for( i = 0; i < 4; i++ ) *tp++ = *sp++;
   else
   {
      sp+=3;
      for( i = 0; i < 4; i++ ) *tp++ = *sp--;
   }
   return;
}

/* This routine puts a double value to an 8 byte character stream */
VOID XBASE::PutDouble( CHAR * c, DOUBLE d )
{
   CHAR *sp, *tp;
   SHORT i;

   tp = c;
   sp = (CHAR *) &d;
   if( EndianType == 'L' )
      for( i = 0; i < 8; i++ ) *tp++ = *sp++;
   else
   {
      sp+=7;
      for( i = 0; i < 8; i++ ) *tp++ = *sp--;
   }
   return;
}
/************************************************************************/
SHORT XBASE::DirectoryExistsInName( CHAR * Name )
{
   /* returns the offset in the string of the last directory slash */

   SHORT Count, Mark;
   CHAR  Delim;
   CHAR  *p;

#ifdef DOS
   Delim = '\\';
#else
   Delim = '/';
#endif

   Count = Mark = 0;
   p = Name;

   while( *p )
   {
      Count++;
      if( *p++ == Delim ) Mark = Count;
   }
   return Mark;
}
/************************************************************************/
