/*  lock.cpp 

    Xbase project source code

    This file contains the basic Xbase routines for locking Xbase files

    Copyright (C) 1997  StarTech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0    10/10/97   - Initial release of software
    V 1.5    1/2/98     - Added memo field support
    V 1.6a   4/1/98     - Added expression support
    V 1.6b   4/8/98     - Numeric index keys
    V 1.7.4b 7/3/98     - Modified Exclusive Lock to conditionally lock memo file
*/

#include "stdafx.h"
#include "xbase.h"
#include <fcntl.h>
#include <stdio.h>
#include <sys/types.h>           /* BSDI BSD/OS 3.1 */

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


#ifdef LOCKING_ON

/************************************************************************/
SHORT DBF::LockDatabase( SHORT WaitOption, SHORT LockType, ULONG LRecNo )
{
   /*   
      if WaitOption = F_SETLK  - return immediately 
                      F_SETLKW - wait until lock function executes

      if LockType   = F_RDLCK - do a read / shared lock
                      F_WRLCK - do a write / exclusive lock
                      F_UNLCK - do an unlock

      if LRecNo     = 0L  - lock the dbf header
                    = 1-n - lock appropriate record

      if locking routines fail, look in errno or do perror() for explanation
   */


   struct flock fl;

   if( LRecNo > NoOfRecs )
      return INVALID_RECORD;

   fl.l_type = LockType;

/* michael - modified code to avoid unnecessary fseek work */
 
   if( LRecNo == 0L )
   {
      fl.l_whence = SEEK_SET;
      fl.l_start = 0L;
      fl.l_len = 7L;
   }
   else if ( CurRec )
   {
      fl.l_whence = SEEK_CUR;
      fl.l_start = (HeaderLen+(RecordLen*(LRecNo-1))) - DbfTell();
      fl.l_len = 1L;
   }
   else   /* CurRec = 0 */
   {
      fl.l_whence = SEEK_SET;
      fl.l_start = HeaderLen + (RecordLen*(LRecNo-1) );
      fl.l_len = 1L;
   }

   if( fcntl( fileno( fp ), WaitOption, &fl ) == -1 )
      return LOCK_FAILED;
   else
      return NO_ERROR;
} 
/************************************************************************/
#ifdef INDEX_NDX

SHORT NDX::LockIndex( SHORT WaitOption, SHORT LockType )
{
   /*  This method locks the first 512 bytes of the index file,
       effectively locking the file from other processes that are
       using the locking protocols

      if WaitOption = F_SETLK  - return immediately 
                      F_SETLKW - wait until lock function executes

      if LockType   = F_RDLCK - do a read / shared lock
                      F_WRLCK - do a write / exclusive lock
                      F_UNLCK - do an unlock

      if locking routines fail, look in errno or do perror() for explanation
   */

   struct flock fl;

   fl.l_type = LockType;
   fl.l_whence = SEEK_SET;
   fl.l_start = 0L;
   fl.l_len = NDX_NODE_SIZE;

   if( fcntl( fileno( ndxfp ), WaitOption, &fl ) == -1 )
      return LOCK_FAILED;
   else
      return NO_ERROR;
}
#endif	/* INDEX_NDX  */
/************************************************************************/
#ifdef MEMO_FIELDS 

SHORT DBF::LockMemoFile( SHORT WaitOption, SHORT LockType )
{
   /*  This method locks the first 4 bytes of the memo file,
       effectively locking the file from other processes that are
       using the locking protocols

       The first four bytes point to the free block chain

      if WaitOption = F_SETLK  - return immediately 
                      F_SETLKW - wait until lock function executes

      if LockType   = F_RDLCK - do a read / shared lock
                      F_WRLCK - do a write / exclusive lock
                      F_UNLCK - do an unlock

      if locking routines fail, look in errno or do perror() for explanation
   */

   struct flock fl;

   fl.l_type   = LockType;
   fl.l_whence = SEEK_SET;
   fl.l_start  = 0L;
   fl.l_len    = 4L;

   if( fcntl( fileno( mfp ), WaitOption, &fl ) == -1 )
      return LOCK_FAILED;
   else
      return NO_ERROR;
}
#endif	/* MEMO_FIELDS  */
/***********************************************************************/
SHORT DBF::ExclusiveLock( SHORT LockWaitOption )
{
   /* this routine locks all files and indexes for a database file */
   /* if it fails, no locks are left on (theoretically)            */
   IxList *i;
   SHORT rc;

   AutoLockOff();
   if(( rc = LockDatabase( LockWaitOption, F_WRLCK, 0 )) != NO_ERROR )
      return rc;

   #ifdef MEMO_FIELDS
   if( MemoFieldsPresent())
      if(( rc = LockMemoFile( LockWaitOption, F_WRLCK )) != NO_ERROR )
         return rc;
   #endif

   #ifdef INDEX_NDX
   i = NdxList;
   while( i ) 
   {
      if(( rc = i->ndx->LockIndex( LockWaitOption, F_WRLCK )) != NO_ERROR )
      {
         ExclusiveUnlock();
         return rc;
      }
      i = i->NextIx;
   }
   #endif

   return NO_ERROR;
}  
/***********************************************************************/
SHORT DBF::ExclusiveUnlock( VOID )
{
   /* this routine unlocks all files and indexes for a database file */

   IxList *i;

   LockDatabase( F_SETLK, F_UNLCK, 0 );

   #ifdef MEMO_FIELDS
   if( MemoFieldsPresent())
      LockMemoFile( F_SETLK, F_UNLCK );
   #endif

   #ifdef INDEX_NDX
   i = NdxList;
   while( i ) 
   {
      i->ndx->LockIndex( F_SETLK, F_UNLCK );
      i = i->NextIx;
   }
   #endif

   AutoLockOn();
   return NO_ERROR;
}  
/***********************************************************************/
#endif  /* LOCKING_ON */
