/*  fields.cpp

    Xbase project source code

    This file contains the basic X-Base routines for reading and writing
    Xbase fields.

    Copyright (C) 1997  StarTech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://ww.startech.keller.tx.us/xbase.html
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0    10/10/97   - Initial release of software
    V 1.5    1/2/98     - Added Dbase IV Memo field support
    V 1.6a   4/1/98     - Added expression support
    V 1.6b   4/8/98     - Numeric index keys
    V 1.7.4b 7/2/98     - Changed gcvt to sprintf for portability reasons - PS
*/

#include "stdafx.h"
#include "xbase.h"
#include <stdlib.h>
#include <string.h>

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


/************************************************************************/
/* This function returns true if the data is valid logical data         */
SHORT DBF::ValidLogicalData( CHAR * buf )
{
   if( buf[0] )
     if( buf[0] == 'T' || buf[0] == 't' || buf[0] == 'F' || buf[0] == 'f' ||
         buf[0] == 'Y' || buf[0] == 'y' || buf[0] == 'N' || buf[0] == 'n' ||
         buf[0] == '?' )
       return 1;
   return 0; 
}
/************************************************************************/
/* This function returns true if the data is valid numeric data         */
SHORT DBF::ValidNumericData( CHAR * buf )
{
   char *p;

   p = buf;
   while( *p )
   {
      if( *p != '+' && *p != '-' && *p != '.' && *p != '0' && *p != '1' &&
          *p != '2' && *p != '3' && *p != '4' && *p != '5' && *p != '6' &&
          *p != '7' && *p != '8' && *p != '9' )
         return 0;
      else
         p++;
   }
   return 1;
}
/************************************************************************/
/* This function returns a fields length */

SHORT DBF::GetFieldLen( SHORT FieldNo )
{
   if( FieldNo >= 0 && FieldNo < NoOfFields )
      return SchemaPtr[FieldNo].FieldLen;
   else
      return 0;
}
/************************************************************************/
/* This function returns a fields decimal length */

SHORT DBF::GetFieldDecimal( SHORT FieldNo )
{
   if( FieldNo >= 0 && FieldNo < NoOfFields )
      return SchemaPtr[FieldNo].NoOfDecs;
   else
      return 0;
}
/************************************************************************/
/* This function returns a fields type */

CHAR DBF::GetFieldType( SHORT FieldNo )
{
   if( FieldNo >= 0 && FieldNo < NoOfFields )
      return SchemaPtr[FieldNo].Type;
   else
      return 0;
}
/************************************************************************/
/* This function returns a fields name */

CHAR * DBF::GetFieldName( SHORT FieldNo )
{
   if( FieldNo >= 0 && FieldNo < NoOfFields )
      return SchemaPtr[FieldNo].FieldName;
   else
      return 0;
}
/************************************************************************/
/* This function returns the field ID number for a given field
   or -1 if the field is not one of the fields of the database  */

SHORT DBF::GetFieldNo( CHAR * name )
{
   int i, len1, len2;

   if(( len1 = strlen( name )) > 10 )
      return -1;

   for( i = 0; i < NoOfFields; i++ )
   {
     len2 = strlen( SchemaPtr[i].FieldName );
     if( len1 == len2 )
        if( strstr( SchemaPtr[i].FieldName, name )) return i;
   }
   return -1;
}
/************************************************************************/
/* This function fills a buffer with data from the record buffer
   for a particular field number. 

   Use GetFieldNo to get a number based on a field's name 

   If successful, this function returns the field size.
*/

SHORT DBF::GetField( SHORT FieldNo, CHAR * buf, SHORT RecBufSw )
{
   if( FieldNo >= 0 && FieldNo < NoOfFields )
   {
      if( RecBufSw )
         memcpy( buf, SchemaPtr[FieldNo].Address2, SchemaPtr[FieldNo].FieldLen);
      else
	 memcpy( buf, SchemaPtr[FieldNo].Address, SchemaPtr[FieldNo].FieldLen );
      buf[SchemaPtr[FieldNo].FieldLen] = 0x00;
      return( SchemaPtr[FieldNo].FieldLen ); 
   }
   else
   {
      buf[0] = 0x00;
      return 0x00;
   }
}
/************************************************************************/
/* This function fills a field in the record buffer with data from
   a buffer for a particular field.

   Use GetFieldNo to get a number based on a field's name 

   Field type N or F is loaded as right justified, left blank filled.
   Other fields are loaded as left justified, right blank filled.

   This method does check the data's validity.

   If successful, this function returns 0, if invalid data, it returns -1
   or INVALID_FIELDNO
*/

SHORT DBF::PutField( SHORT FieldNo, CHAR * buf )
{
   SHORT len, i;
   CHAR * startpos;
   CHAR * sdp;     /*  source decimal point */
   CHAR * tp;      /*  target pointer */
   CHAR * sp;      /*  source pointer */

   if( FieldNo < 0 || FieldNo >= NoOfFields ) return INVALID_FIELDNO;

   if( DbfStatus != UPDATED )
   {
      DbfStatus = UPDATED;
      memcpy( RecBuf2, RecBuf, RecordLen );
   }
  
   if( SchemaPtr[FieldNo].Type == 'L' && !ValidLogicalData( buf ))
      return INVALID_DATA;
   else if(( SchemaPtr[FieldNo].Type == 'F' || SchemaPtr[FieldNo].Type == 'N' )
             && !ValidNumericData( buf )) 
      return INVALID_DATA;
   else if( SchemaPtr[FieldNo].Type == 'D' && !xbase->DateIsValid( buf ))
      return INVALID_DATA;

   memset( SchemaPtr[FieldNo].Address, 0x20, SchemaPtr[FieldNo].FieldLen );

   len = strlen( buf );

   if(( SchemaPtr[FieldNo].Type == 'N' || SchemaPtr[FieldNo].Type == 'F')
         && len > SchemaPtr[FieldNo].FieldLen )
      return INVALID_DATA;
   else if( len > SchemaPtr[FieldNo].FieldLen )
     len = SchemaPtr[FieldNo].FieldLen;

   if( SchemaPtr[FieldNo].Type == 'F' || SchemaPtr[FieldNo].Type == 'N' 
       || SchemaPtr[FieldNo].Type == 'M' 
     )
   {
      sdp = strchr( buf, '.' );
      len = 0;
      sp = buf;
      while( *sp && *sp != '.' ) { len++, sp++; }
      if( SchemaPtr[FieldNo].NoOfDecs > 0 )
      {
         /* do the right of decimal area */
         tp = SchemaPtr[FieldNo].Address;
         tp += SchemaPtr[FieldNo].FieldLen - SchemaPtr[FieldNo].NoOfDecs - 1;
         *tp++ = '.';
         sp = sdp;
         if( sp ) sp++;
         for( i = 0; i < SchemaPtr[FieldNo].NoOfDecs; i++ )
            if( sp && *sp ) *tp++ = *sp++; else *tp++ = '0'; 
         
         startpos= SchemaPtr[FieldNo].Address +
                   SchemaPtr[FieldNo].FieldLen -
                   SchemaPtr[FieldNo].NoOfDecs - len - 1; 
      }
      else
      {
         startpos=SchemaPtr[FieldNo].Address+SchemaPtr[FieldNo].FieldLen-len; 
      }
   }
   else
      startpos = SchemaPtr[FieldNo].Address;

   memcpy( startpos, buf, len );
   return 0;
}
/************************************************************************/
SHORT DBF::GetField( SHORT FieldNo, CHAR * buf )
{
   return GetField( FieldNo, buf, 0 );
}
/************************************************************************/
LONG DBF::GetLongField( SHORT FieldNo )
{
   CHAR buf[18];
   memset( buf, 0x00, 18 );
   GetField( FieldNo, buf );
   return atol( buf );
}
/************************************************************************/
LONG DBF::GetLongField( CHAR * FieldName )
{
  return( GetLongField( GetFieldNo( FieldName )));
}
/************************************************************************/
SHORT DBF::PutLongField( SHORT FieldNo, LONG Val )
{
   CHAR buf[18];
   memset( buf, 0x00, 18 );
   sprintf( buf, "%ld", Val );
   return( PutField( FieldNo, buf ));
}
/************************************************************************/
SHORT DBF::PutLongField( CHAR * FieldName, LONG Val )
{
  return( PutLongField( GetFieldNo( FieldName ), Val ));
}
/************************************************************************/
FLOAT DBF::GetFloatField( SHORT FieldNo )
{
   CHAR buf[21];
   memset( buf, 0x00, 21 );
   if( GetField( FieldNo, buf ) != 0 )
      return atof( buf );
   else
      return 0;
}
/************************************************************************/
FLOAT DBF::GetFloatField( CHAR * FieldName )
{
   SHORT fnum;
   if(( fnum = GetFieldNo( FieldName )) != -1 )
      return GetFloatField( fnum );
   else
      return 0;
}
/************************************************************************/
SHORT DBF::PutFloatField( SHORT FieldNo, FLOAT f )
{
   CHAR buf[25];
   memset( buf, 0x00, 25 );

// gcvt replaced w/ sprintf for portablity enhancements - PS
// gcvt( f, SchemaPtr[FieldNo].FieldLen - SchemaPtr[FieldNo].NoOfDecs, buf );

   sprintf( buf, "%f", f ); 
   return PutField( FieldNo, buf );
}
/************************************************************************/
SHORT DBF::PutFloatField( CHAR * FieldName, FLOAT f )
{
   SHORT fnum;
   if(( fnum = GetFieldNo( FieldName )) != -1 )
      return PutFloatField( fnum, f );
   else
      return 0;
}
/************************************************************************/
DOUBLE DBF::GetDoubleField( SHORT FieldNo )
{
   CHAR buf[21];
   memset( buf, 0x00, 21 );
   if( GetField( FieldNo, buf ) != 0 )
      return strtod( buf, NULL );
   else
      return 0;
}
/************************************************************************/
DOUBLE DBF::GetDoubleField( CHAR * FieldName )
{
   SHORT fnum;
   if(( fnum = GetFieldNo( FieldName )) != -1 )
      return GetDoubleField( fnum );
   else
      return 0;
}
/************************************************************************/
SHORT DBF::PutDoubleField( SHORT FieldNo, DOUBLE d )
{
   return PutFloatField( FieldNo, (FLOAT) d );
}
/************************************************************************/
SHORT DBF::PutDoubleField( CHAR * FieldName, DOUBLE d )
{
   SHORT fnum;
   if(( fnum = GetFieldNo( FieldName )) != -1 )
      return PutFloatField( fnum, (FLOAT) d );
   else
      return 0;
}
/************************************************************************/
