/*  expproc.cpp

    Xbase project source code

    Copyright (C) 1997  Crypton Technologies, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0   10/10/97   - Initial release of software
    V 1.5   1/2/97     - Added memo field support
    V 1.6a  4/1/98     - Added expression support
    V 1.7.1 5/25/98    - Enhanced expression support
*/

#include "stdafx.h"
#include "xbase.h"

#ifdef EXPRESSIONS

#include <ctype.h>
#include <math.h>
#include <string.h>

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


/*************************************************************************/
ExpNode * EXPN::GetFirstTreeNode( ExpNode * e )
{
   ExpNode * WorkNode;
   if( !e ) return e;
   WorkNode = e;
   while( WorkNode->Sibling1 )
      WorkNode = WorkNode->Sibling1;
   return WorkNode;
}
/*************************************************************************/
SHORT EXPN::ValidOperation( CHAR * Operand, CHAR Op1, CHAR Op2 )
{
   /*  Valid operation table

       operator  Field1   Field2          operator Field1   Field2

        **        N        N                 =        N        N
        *         N        N                 =        C        C
        /         N        N                 <>,#     N        N
        +         N        N                 <>,#     C        C
        +         C        C                 <=       N        N
        -         N        N                 <=       C        C
        -         C        C                 >=       N        N
        <         N        N                 >=       C        C
        <         C        C                 
        >         N        N
        >         C        C

    */


   if( Operand[0] == '*' && Operand[1] == '*' && Op1 == 'N' && Op2 == 'N' )
      return 1;

   switch( Operand[0] )
   {
      case '*': 
      case '/':
         if( Op1 == 'N' && Op2 == 'N' ) 
            return 1; 
         else 
            return 0;
//         break;

      case '+':
      case '-':
      case '<':
      case '>':
      case '=':
      case '#':
         if(( Op1 == 'N' && Op2 == 'N' ) || ( Op1 == 'C' && Op2 == 'C' ))
             return 1;
         else
             return 0;

      default:
         return 0;
   }
}
/*************************************************************************/
ExpNode * EXPN::GetNextTreeNode( ExpNode * e )
{
   if( !e->Node ) return NULL;

   /* sibling 1 && sibling 2 exists */
   if( e == e->Node->Sibling1 && e->Node->Sibling2 )
      return GetFirstTreeNode( e->Node->Sibling2 );

   /* sibling2 && sibling3 exists */
   else if( e == e->Node->Sibling2 && e->Node->Sibling3 )
      return GetFirstTreeNode( e->Node->Sibling3 );

   else
      return e->Node;
}
/*************************************************************************/
SHORT EXPN::ProcessExpression( ExpNode * e )
{
   return ProcessExpression( e, 0 );
}
/*************************************************************************/
SHORT EXPN::ProcessExpression( ExpNode * Wtree, SHORT RecBufSw )
{
   ExpNode * WorkNode;
   SHORT rc;

   if( Wtree == 0 )
      Wtree = Tree;

   /* initialize the stack - free any expnodes */
   while( GetStackDepth() > 0 )
   {
      WorkNode = (ExpNode *) Pop();
      if( !WorkNode->InTree )
         FreeExpNode( WorkNode );
   }   

   if(( WorkNode = GetFirstTreeNode( Wtree )) == NULL )
      return NO_DATA;

   while( WorkNode )
   {
      Push( (VOID *) WorkNode );
      if( WorkNode->Type == 'D' && WorkNode->dbf )
      {
         WorkNode->dbf->GetField( WorkNode->FieldNo, (CHAR *) WorkNode->Result, RecBufSw );
         if( WorkNode->dbf->GetFieldType( WorkNode->FieldNo ) == 'N' || 
            WorkNode->dbf->GetFieldType( WorkNode->FieldNo ) == 'F' )
            WorkNode->DoubResult = WorkNode->dbf->GetDoubleField( WorkNode->FieldNo );
      }
      else if( WorkNode->Type == 'O' )
      {
         if(( rc = ProcessOperator( RecBufSw )) != NO_ERROR ) return rc;
      }
      else if( WorkNode->Type == 'F' )
         if(( rc = ProcessFunction( WorkNode->NodeText )) != NO_ERROR ) return rc;

      WorkNode = GetNextTreeNode( WorkNode );
   }
   if( GetStackDepth() != 1 )    /* should only have result left in stack */
       return PARSE_ERROR;
   return NO_ERROR;
}
/*************************************************************************/
CHAR EXPN::GetOperandType( ExpNode * e )
{
   /* this routine returns
         L - logical
         N - Numeric
         C - Character
         0 - error
   */
 
   CHAR WorkType;

   if( e->Type == 'd' || e->Type == 'N' || e->Type == 'i' ) return 'N';
   if( e->Type == 'l' ) return 'L';
   if( e->Type == 's' ) return 'C';

   if( e->Type == 'C' )
   {
      if(e->NodeText[0]=='-' || e->NodeText[0]=='+' || isdigit(e->NodeText[0]))
         return 'N';
      else
         return 'C';
   }
   else if( e->Type == 'D' && e->dbf )
   {
      WorkType = e->dbf->GetFieldType( e->FieldNo );

      if( WorkType == 'C' ) return 'C';
      else if( WorkType == 'F' || WorkType == 'N' ) return 'N';
      else if( WorkType == 'L' ) return 'L';
      else return 0;
   }
   else
      return 0;
}
/*************************************************************************/
SHORT EXPN::ProcessOperator( SHORT RecBufSw )
{
   ExpNode * WorkNode;
   CHAR Operator[3];
   CHAR t;

   if( GetStackDepth() < 3 ) 
      return PARSE_ERROR;

   WorkNode = (ExpNode *) Pop();
   if( WorkNode->Len > 2 )
      return PARSE_ERROR;
   memset( Operator, 0x00, 3 );
   strncpy( Operator, WorkNode->NodeText, WorkNode->Len );
   if( !WorkNode->InTree ) 
      FreeExpNode( WorkNode );

   /* load up operand 1 */
   WorkNode = (ExpNode *) Pop();
   if(( OpType1 = GetOperandType( WorkNode )) == 0 )
      return PARSE_ERROR;
   if( OpLen1 < WorkNode->DataLen+1 && WorkNode->Type != 'd' )
   {
      if( OpLen1 > 0 ) free( Op1 );
      if(( Op1 = (CHAR *) malloc( WorkNode->DataLen+1 )) == NULL )
         return NO_MEMORY;
      OpLen1 = WorkNode->DataLen+1;
   }
   OpDataLen1 = WorkNode->DataLen;
   memset( Op1, 0x00, WorkNode->DataLen+1 );
   if( WorkNode->Type == 'D' && WorkNode->dbf )     /* database field  */
   {
      WorkNode->dbf->GetField( WorkNode->FieldNo, Op1, RecBufSw );
      t = WorkNode->dbf->GetFieldType( WorkNode->FieldNo );
      if( t == 'N' || t == 'F' )
         Opd1 = strtod( WorkNode->Result, 0 );
   }
   else if( WorkNode->Type == 'C' )     /* constant        */
      memcpy( Op1, WorkNode->NodeText, WorkNode->DataLen );
   else if( WorkNode->Type == 's' )     /* previous result */
      memcpy( Op1, WorkNode->Result, WorkNode->DataLen+1 );
   else if( WorkNode->Type == 'd' )     /* previous numeric result */   
      Opd1 = WorkNode->DoubResult;
   else if( WorkNode->Type == 'N' )     /* previous numeric result */   
      Opd1 = strtod( WorkNode->Result, 0 );
   if( !WorkNode->InTree ) 
      FreeExpNode( WorkNode );

   /* load up operand 2 */
   WorkNode = (ExpNode *) Pop();
   if(( OpType2 = GetOperandType( WorkNode )) == 0 )
      return PARSE_ERROR;
   if( OpLen2 < WorkNode->DataLen+1 && WorkNode->Type != 'd' )
   {
      if( OpLen2 > 0 ) free( Op2 );
      if(( Op2 = (CHAR *) malloc( WorkNode->DataLen+1 )) == NULL )
         return NO_MEMORY;
      OpLen2 = WorkNode->DataLen+1;
   }
   OpDataLen2 = WorkNode->DataLen;
   memset( Op2, 0x00, WorkNode->DataLen+1 );
   if( WorkNode->Type == 'D' && WorkNode->dbf )      /* database field  */
   {
      WorkNode->dbf->GetField( WorkNode->FieldNo, Op2, RecBufSw );
      t = WorkNode->dbf->GetFieldType( WorkNode->FieldNo );
      if( t == 'N' || t == 'F' )
         Opd2 = strtod( WorkNode->Result, 0 );
   }
   else if( WorkNode->Type == 'C' )     /* constant        */
      memcpy( Op2, WorkNode->NodeText, WorkNode->DataLen );
   else if( WorkNode->Type == 's' )     /* previous result */
      memcpy( Op2, WorkNode->Result, WorkNode->DataLen+1 );
   else if( WorkNode->Type == 'd' )     /* previous numeric result */
      Opd2 = WorkNode->DoubResult;
   else if( WorkNode->Type == 'N' )     /* previous numeric result */   
      Opd2 = strtod( WorkNode->Result, 0 );
   if( !WorkNode->InTree )
      FreeExpNode( WorkNode );

   if( !ValidOperation( Operator, OpType1, OpType2 ))
      return PARSE_ERROR;

   if( OpType1 == 'N' )    /* numeric procesing */
   {
      return NumericOperation( Operator );
   }
   else                    /* must be character */
      return  AlphaOperation( Operator );
}
/*************************************************************************/
SHORT EXPN::NumericOperation( CHAR * Operator )
{
   DOUBLE  Operand1, Operand2;
   ExpNode * WorkNode;
   SHORT   ResultLen;
   CHAR    SaveType;

   ResultLen = 0;

   if( Operator[0] == '=' || Operator[0] == '<' || 
       Operator[0] == '>' || Operator[0] == '#' )
      SaveType = 'l';
   else
      SaveType = 'd';

   if(( WorkNode = GetExpNode( ResultLen )) == NULL )
      return NO_MEMORY;

   WorkNode->Type = SaveType;
   WorkNode->DataLen = ResultLen;

   if( OpType1 == 'd' || OpType1 == 'N' )
      Operand1 = Opd1;
   else
      Operand1 = strtod( Op1, NULL );

   if( OpType2 == 'd' || OpType2 == 'N' )
      Operand2 =  Opd2;
   else
      Operand2 = strtod( Op2, NULL );

   if( Operator[0] == '*' && Operator[1] == '*' )
      WorkNode->DoubResult = pow( Operand2, Operand1 );
   else if( Operator[0] == '*' )
      WorkNode->DoubResult = Operand2 * Operand1;
   else if( Operator[0] == '/') 
      WorkNode->DoubResult = Operand2 / Operand1;
   else if( Operator[0] == '+' )
      WorkNode->DoubResult = Operand2 + Operand1;
   else if( Operator[0] == '-' )
      WorkNode->DoubResult = Operand2 - Operand1;

   /* = */
   else if((Operator[0]== '=' || Operator[1]== '=' ) && Operand1 == Operand2)
      WorkNode->IntResult = 1;
   else if( Operator[0] == '=' )
      WorkNode->IntResult = 0;
   /* not = */
   else if((Operator[0] == '<' && Operator[1] == '>')||Operator[0] == '#')
   {
      if( Operand1 != Operand2 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   /* less than */
   else if( Operator[0] == '<' )
   {
      if( Operand2 < Operand1 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   /* greater than */
   else if( Operator[0] == '>' )
   {
      if( Operand2 > Operand1 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   else
      return PARSE_ERROR;

   Push((VOID *) WorkNode );
   return 0;
}
/*************************************************************************/
SHORT EXPN::AlphaOperation( CHAR * Operator )
{
   SHORT ResultLen, i;
   CHAR *p, SaveType;
   ExpNode * WorkNode;

   if( Operator[0] == '=' || Operator[0] == '<' || 
       Operator[0] == '>' || Operator[0] == '#' )
   {
      ResultLen = 0;
      SaveType = 'l';
   }
   else
   {
      ResultLen = OpDataLen1 + OpDataLen2 + 1;
      SaveType = 's';
   }
   if(( WorkNode = GetExpNode( ResultLen )) == NULL )
      return NO_MEMORY;
   memset( WorkNode->Result, 0x00, ResultLen );
   WorkNode->Type = SaveType;
   if( WorkNode->Type == 'l' )
      WorkNode->DataLen = 0;
   else
      WorkNode->DataLen = ResultLen - 1;

   if( Operator[0] == '+' )
   {
      strncpy( WorkNode->Result, Op2, OpDataLen2 );
      strncat( WorkNode->Result, Op1, OpDataLen1 );
   }
   else if( Operator[0] == '-' )
   {
      strncpy( WorkNode->Result, Op2, OpDataLen2 );
      p =  WorkNode->Result;
      p += OpDataLen2-1;
      i =  OpDataLen2-1;
      while( *p == 0x20 && i > 0 )
      {
         *p = 0x00;
         p--;
         i--;
      }
      strncat( WorkNode->Result, Op1, OpDataLen1 );
      p =  WorkNode->Result;
      i = strlen( WorkNode->Result );
      p += i; 
      for( ; i < ResultLen-1; i++)
      {
         *p = 0x20;   /* pad with spaces */
         p++;
      }
   }
   /* = */
   else if((Operator[0]== '=' || Operator[1]== '=' ) && strcmp(Op1,Op2) == 0)
      WorkNode->IntResult = 1;
   else if( Operator[0] == '=' )
      WorkNode->IntResult = 0;
   /* not = */
   else if((Operator[0] == '<' && Operator[1] == '>')||Operator[0] == '#')
   {
      if( strcmp( Op1, Op2 ) != 0 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   /* less than */
   else if( Operator[0] == '<' )
   {
      if( strcmp( Op2, Op1 ) < 0 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   /* greater than */
   else if( Operator[0] == '>' )
   {
      if( strcmp( Op2, Op1 ) > 0 )
         WorkNode->IntResult = 1;
      else
         WorkNode->IntResult = 0;
   }
   else
      return PARSE_ERROR;

   Push((VOID *) WorkNode);
   return NO_ERROR;
}
/*************************************************************************/
#endif  	//	EXPRESSIONS
