/*  dbf.cpp

    Xbase project source code
   
    This file contains the basic Xbase routines for reading and writing
    Xbase .DBF files.

    Copyright (C) 1997  StarTech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0   10/10/97   - Initial release of software
    V 1.5   1/2/98     - Added memo field support
    V 1.6a  4/1/98     - Added expression support
    V 1.6b  4/8/98     - Numeric index keys
    V 1.6c  4/16/98    - Big Endian support, dBASE III + memo field support
*/

#include "stdafx.h"
#include "xbase.h"
#ifdef DOS
#include <io.h> 
#endif

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/************************************************************************/
DBF::DBF( XBASE * x )
{
   xbase = x;
   InitVars();
}
/************************************************************************/
VOID DBF::InitVars( VOID )
{
   DatabaseName    = 0x00;
   NoOfFields      = 0;
   DbfStatus       = 0;
   fp              = NULL;
   CurRec          = 0L;
   SchemaPtr       = NULL;
   RecBuf          = NULL;
   RecBuf2         = NULL;
   Version         = 0x00;
   UpdateYY        = 0x00;
   UpdateMM        = 0x00;
   UpdateDD        = 0x00;
   NoOfRecs        = 0L;
   HeaderLen       = 0x00;
   RecordLen       = 0x00;
   MdxList         = NULL;
   NdxList         = NULL;
   FreeIxList      = NULL;
   XFV             = 3;            /* Xbase file version */

#ifdef LOCKING_ON
   AutoLock        = 1;
#else
   AutoLock        = 0;
#endif

#ifdef MEMO_FIELDS
   MemoHeader.BlockSize  = DBT_BLOCK_SIZE;
   MemoHeader.Version    = 0x03;
   mfp                   = NULL;
   mbb                   = NULL;
   CurMemoBlockNo        = -1;
   mfield1               = 0;
   MStartPos             = 0;
   MFieldLen             = 0;
   NextFreeBlock         = 0L;
   FreeBlockCnt          = 0L;
   MNextBlockNo          = 0L;
   MNoOfFreeBlocks       = 0L;
#endif
}
/************************************************************************/
LONG DBF::CalcCheckSum()
{
   SHORT i;
   CHAR *p;
   LONG l = 0L;
   for( i = 0; i < RecordLen; i++ ) l+= *p++;
   return l;
}
/************************************************************************/
SHORT DBF::SetVersion( SHORT v )
{
   if( v == 0 )
      return XFV;
   else if( v == 3 )
   {
      XFV = 3;
#ifdef MEMO_FIELDS
      MemoHeader.Version = 0x03;
#endif
   }
   else if( v == 4 )
   {
      XFV = 4;
#ifdef MEMO_FIELDS
      MemoHeader.Version = 0x00;
#endif
   }
   return INVALID_OPTION;
}
/************************************************************************/
SHORT DBF::WriteHeader( SHORT PositionOption )
{
   CHAR buf[4];
   if( PositionOption ) rewind( fp );
   if( fwrite( &Version, 4, 1, fp ) != 1 ) return WRITE_ERROR;
   memset( buf, 0x00, 4 );
   xbase->PutLong( buf, NoOfRecs );
   if( fwrite( &buf, 4, 1, fp ) != 1 ) return WRITE_ERROR;
   if( PositionOption == 2 ) return NO_ERROR;   /* then it is an update */
   memset( buf, 0x00, 4 );
   xbase->PutShort( buf, HeaderLen );
   if( fwrite( &buf, 2, 1, fp ) != 1 ) return WRITE_ERROR;
   memset( buf, 0x00, 4 );
   xbase->PutShort( buf, RecordLen );
   if( fwrite( &buf, 2, 1, fp ) != 1 ) return WRITE_ERROR;
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::ReadHeader( SHORT PositionOption )
{
   CHAR buf[4];
   if( PositionOption ) rewind( fp );
   if( fread( &Version, 4, 1, fp ) != 1 ) return READ_ERROR;
   if( fread( &buf, 4, 1, fp ) != 1 ) return READ_ERROR;
   NoOfRecs = xbase->GetLong( buf );
   if( fread( &buf, 2, 1, fp ) != 1 ) return READ_ERROR;
   HeaderLen = xbase->GetShort( buf );
   if( fread( &buf, 2, 1, fp ) != 1 ) return READ_ERROR;
   RecordLen = xbase->GetShort( buf );
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::NameSuffixMissing( SHORT type, CHAR * name )
{
  /*  type 1 is DBF check
      type 2 is NDX check
      type 3 is MDX check

      Returns 0 if suffix found
              1 if suffix not found, lower case
              2 is suffix not found, upper, case
*/

SHORT len;
  
   len = strlen( name );
   if( len <= 4 )
     if( name[len-1] >= 'A' && name[len-1] <= 'Z' )
       return 2;
     else
       return 1;

   if(  type == 1          && name[len-4] == '.' &&
      ( name[len-3] == 'd' || name[len-3] == 'D' ) && 
      ( name[len-2] == 'b' || name[len-2] == 'B' ) && 
      ( name[len-1] == 'f' || name[len-1] == 'F' )
     )
      return 0;

   if(  type == 2          && name[len-4] == '.' &&
      ( name[len-3] == 'n' || name[len-3] == 'N' ) && 
      ( name[len-2] == 'd' || name[len-2] == 'D' ) && 
      ( name[len-1] == 'x' || name[len-1] == 'X' )
     )
      return 0;

   if( name[len-5] >= 'A' && name[len-5] <= 'Z' )
       return 2;
   else
       return 1;
}
/************************************************************************/
SHORT DBF::CreateDatabase( CHAR * TableName, Schema * s, SHORT Overlay )
{
/* future release - add logic to check number of fields and record length */

   SHORT    i, j, k, k2, NameLen, rc;

#ifdef MEMO_FIELDS
   SHORT MemoSw = 0;
#endif

   i = j = 0;
   DbfStatus = CLOSED;                 

   /* Get the datafile name and store it in the class */
   NameLen = strlen( TableName ) + 1;
   if(( rc = NameSuffixMissing( 1, TableName )) > 0 )
      NameLen += 4;

   if (( DatabaseName = (char *) malloc( NameLen )) == NULL )
      return NO_MEMORY;
   
   /* copy the file name into the class variable */
   strcpy( DatabaseName, TableName );

   if( rc == 1)
      strcat( DatabaseName, ".dbf" );
   else if( rc == 2 )
      strcat( DatabaseName, ".DBF" );

   /* check if the file already exists */
   if((( fp = fopen( DatabaseName, "r" )) != NULL ) && !Overlay )
   {
      free( DatabaseName );
      fclose( fp );
      return FILE_EXISTS;
   }
   else if( fp ) fclose( fp );

   if(( fp = fopen( DatabaseName, "w+b" )) == NULL )
   {
      free( DatabaseName );
      return OPEN_ERROR;
   }

   /* count the number of fields */
   i = 0;
   while( s[i].Type != 0 )
   {
      NoOfFields++;
      RecordLen += s[i].FieldLen;
      if( s[i].Type != 'C' && 
          s[i].Type != 'N' &&
          s[i].Type != 'F' &&
          s[i].Type != 'D' &&
          #ifdef MEMO_FIELDS
          s[i].Type != 'M' &&
          #endif /* MEMO_FIELDS */
          s[i].Type != 'L' )
        return UNKNOWN_FIELD_TYPE;

      #ifdef MEMO_FIELDS
      if( !MemoSw && ( s[i].Type=='M' || s[i].Type=='B' || s[i].Type=='O'))
         MemoSw++;
      #endif

      i++;
   }
   RecordLen++;                  /* add one byte for 0x0D    */

   if(( RecBuf = (CHAR *) malloc( RecordLen )) == NULL )
      return NO_MEMORY;

   if(( RecBuf2 = (CHAR *) malloc( RecordLen )) == NULL )
   { 
      free( RecBuf );
      return NO_MEMORY;
   }

   /* BlankRecord(); */
   memset( RecBuf, 0x20, RecordLen );
   memset( RecBuf2, 0x20, RecordLen );

   /* set class variables */

   if( MemoSw )
   {
      if( XFV == 3 )
         Version = '';
      else
         Version = '';
   }
   else
   {
      if( XFV == 3 )
         Version = 3;
      else
         Version = 4;
   }

/*
   if( MemoSw ) 
      Version = '';
   else
      Version = 3;
*/
   CurRec  = 0L;

   HeaderLen = 33 + NoOfFields * 32;

   UpdateYY = xbase->YearOf( xbase->Sysdate()) - 1900;
   UpdateMM = xbase->MonthOf( xbase->Sysdate());
   UpdateDD = xbase->DayOf( FMT_MONTH, xbase->Sysdate());

   /* write the header prolog */
   if(( rc = WriteHeader( 0 )) != NO_ERROR )
   {  
      free( RecBuf );
      free( RecBuf2 );
      fclose( fp );
      return WRITE_ERROR;
   }

   for( i = 0; i < 20; i++ )
   {
      if(( fwrite( "\x00", 1, 1, fp )) != 1 )
      {
         free( RecBuf );
         free( RecBuf2 );
         fclose( fp );
         return WRITE_ERROR;
      }
   }

   if((SchemaPtr = (SchemaRec *) malloc( NoOfFields * sizeof(SchemaRec))) == NULL ) 
   {
      free( RecBuf ); 
      free( RecBuf2 );
      fclose( fp );
      return NO_MEMORY;
   }
   memset( SchemaPtr, 0x00, ( NoOfFields * sizeof(SchemaRec)));

   /* write the field information into the header */
   for( i = 0, k = 1; i < NoOfFields; i++ )
   {
      strcpy( SchemaPtr[i].FieldName, s[i].FieldName );
      SchemaPtr[i].Type = s[i].Type;

      SchemaPtr[i].FieldLen = s[i].FieldLen;
      SchemaPtr[i].NoOfDecs = s[i].NoOfDecs;

      if( SchemaPtr[i].NoOfDecs > SchemaPtr[i].FieldLen )
      {
         fclose( fp );
         free( SchemaPtr );
         free( RecBuf );
         free( RecBuf2 );
         return INVALID_SCHEMA;
      }

      k2 = k;
      k += SchemaPtr[i].FieldLen;

      if(( fwrite( &SchemaPtr[i], 1, 18, fp )) != 18 )
      {
         fclose( fp );
         free( SchemaPtr );
         free( RecBuf );
         free( RecBuf2 );
         return WRITE_ERROR;
      }

      for( j = 0; j < 14; j++ )
      {
         if(( fwrite( "\x00", 1, 1, fp )) != 1 )
         {
            free( SchemaPtr );
            free( RecBuf );
            free( RecBuf2 );
            fclose( fp );
            return WRITE_ERROR;
         }
      }
      SchemaPtr[i].Address  = RecBuf  + k2;
      SchemaPtr[i].Address2 = RecBuf2 + k2;
   }

   /* write the end of header marker */
   if(( fwrite( "\x0D\x1A", 2, 1, fp )) != 1 )
   {
      fclose( fp );
      free( SchemaPtr );
      free( RecBuf );
      free( RecBuf2 );
      return WRITE_ERROR;
   }
   #ifdef MEMO_FIELDS
   if( MemoSw )
      CreateMemoFile();
   #endif

   DbfStatus = OPEN;
   return xbase->AddDbfToDbfList( this, DatabaseName );
}
/************************************************************************/
SHORT DBF::CloseDatabase( VOID )
{
#ifdef INDEX_NDX
   IxList * i;
#endif

   if( DbfStatus == CLOSED ) return NOT_OPEN;

   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )
   {
      UpdateYY = xbase->YearOf( xbase->Sysdate()) - 1900;
      UpdateMM = xbase->MonthOf( xbase->Sysdate());
      UpdateDD = xbase->DayOf( FMT_MONTH, xbase->Sysdate());

      /* update the header */
      WriteHeader( 1 );
      
      /* write eof marker */
      fseek( fp, 0L, 2 );
      fwrite( "\x0D\x1A", 1, 1, fp );
      PutRecord( CurRec );
   }

#ifdef INDEX_NDX
   i = NdxList;
   while( i )
   {
      i->ndx->CloseIndex();
      i = NdxList;
   }
#endif

   if( SchemaPtr )  free( SchemaPtr );
   if( RecBuf )     free( RecBuf ); 
   if( RecBuf2 )    free( RecBuf2 ); 

#ifdef MEMO_FIELDS
   if( mbb )        free( mbb );         /* memo block buffer */
   if( mfp )        fclose( mfp );       /* memo file pointer */
#endif

   xbase->RemoveDbfFromDbfList( this );
   if( DatabaseName ) free( DatabaseName );
   fclose( fp );
   InitVars(); 
   return NO_ERROR;
}
/************************************************************************/
/* options  1 = Print header only
            2 = Field data only
            3 = Header and Field data */

#ifdef XBASE_DEBUG
SHORT DBF::DumpHeader( SHORT Option )
{
   int i;

   if( Option < 1 || Option > 3 )
      return INVALID_OPTION;

   if( DbfStatus == CLOSED )
      return NOT_OPEN;

   cout << "\nDatabase file " << DatabaseName << "\n";

   if( Option != 2 )
   {
      cout << "\nFile header data:\n";
      if( Version == 3 )
         cout << "Dbase III file\n";
      else if ( Version == 83 )
         cout << "Dbase III file with memo fields\n";
   
      cout << "\nLast update date = " 
          << (int) UpdateMM << "/" << (int) UpdateDD << "/" << (int) UpdateYY; 

      cout << "\nHeader length    = " << HeaderLen;
      cout << "\nRecord length    = " << RecordLen;
      cout << "\nRecords in file  = " << NoOfRecs << "\n";
   }
   if( Option != 1 )
   {

      cout << "\nField Name   Type  Length  Decimals";
      cout << "\n----------   ----  ------  --------";
      for( i = 0; i <NoOfFields; i++ )
         printf( "\n%10s    %1c     %4d    %4d", SchemaPtr[i].FieldName,
                  SchemaPtr[i].Type, SchemaPtr[i]. FieldLen, SchemaPtr[i].NoOfDecs );
   }
   cout << "\n";
   return NO_ERROR;
}
#endif
/************************************************************************/
SHORT DBF::OpenDatabase( CHAR * TableName )
{
   SHORT i, j, NameLen, rc;

#ifdef MEMO_FIELDS
   SHORT MemoSw = 0;
#endif

   /* verify the file is not already open */
   if( DbfStatus != CLOSED ) return ALREADY_OPEN;

   /* Get the datafile name and store it in the class */
   NameLen = strlen( TableName ) + 1;
   if(( rc = NameSuffixMissing( 1, TableName )) > 0 )
      NameLen += 4;

   if (( DatabaseName = (char *) malloc(NameLen)) == NULL )
      return NO_MEMORY;
 
   /* copy the file name to the class variable */
   strcpy( DatabaseName, TableName );

   if( rc == 1)
      strcat( DatabaseName, ".dbf" );
   else if( rc == 2 )
      strcat( DatabaseName, ".DBF" );
   
   /* open the file */
   if(( fp = fopen( DatabaseName, "r+b" )) == NULL )
   {
      free( DatabaseName );
      return OPEN_ERROR; 
   }

   /* copy the header into memory */
   if(( rc = ReadHeader( 1 )) != NO_ERROR )
      return rc;

   /* check the version */
   if( Version == 3 || Version == '' )   	/* dBASE III+ */
   {
      XFV = 3;
      MemoHeader.Version = 0x03;
   }
   else if( Version == 4 || Version == '' )    /* dBASE IV */
   {
      XFV = 4;
      MemoHeader.Version = 0x00;
   }
   else
      return NOT_XBASE;     /* then xbase can't handle it (yet) */
      
   if (UpdateYY == 0 || UpdateMM == 0 || UpdateDD == 0 )
       return NOT_XBASE;

   /* calculate the number of fields */
   NoOfFields = ( HeaderLen - 33 ) / 32;

   if(( RecBuf = (CHAR *) malloc( RecordLen )) == NULL )
   {
      fclose( fp );
      return NO_MEMORY;
   }
   if(( RecBuf2 = (CHAR *) malloc( RecordLen )) == NULL )
   {
      fclose( fp );
      free( RecBuf );
      return NO_MEMORY;
   }
   if(( SchemaPtr = (SchemaRec *) malloc( NoOfFields * sizeof(SchemaRec))) == NULL )
   {
      free( RecBuf );
      free( RecBuf2 );
      fclose( fp );
      return NO_MEMORY;
   }
   memset( SchemaPtr, 0x00, ( NoOfFields * sizeof(SchemaRec)));

   /* copy field info into memory */
   for( i = 0, j = 1; i < NoOfFields; i++ )
   {
      fseek( fp, i*32+32, 0 );
      fread( &SchemaPtr[i].FieldName, 1, 18, fp );
      SchemaPtr[i].Address  = RecBuf + j;
      SchemaPtr[i].Address2 = RecBuf2 + j;
      j += SchemaPtr[i].FieldLen;
      #ifdef MEMO_FIELDS
         if( !MemoSw && (SchemaPtr[i].Type == 'M' || 
           SchemaPtr[i].Type == 'B' || SchemaPtr[i].Type == 'O' ))
             MemoSw++;
      #endif
   } 
   CurRec = 0L;
   BlankRecord();

#ifdef MEMO_FIELDS
   if( MemoSw )   /* does this table have memo fields ? */
      if(( rc = OpenMemoFile()) != NO_ERROR )
      {
         free( RecBuf );
         free( RecBuf2 );
         free( SchemaPtr );
         fclose( fp );
         return rc;
      }
#endif

   DbfStatus = OPEN;
   return xbase->AddDbfToDbfList( this, DatabaseName );
}
/************************************************************************/
SHORT DBF::BlankRecord( VOID )
{
   if( DbfStatus == CLOSED ) 
      return NOT_OPEN;
   memset( RecBuf, 0x20, RecordLen );
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::AppendRecord( VOID )
{
   SHORT rc;

/* check for any duplicate keys */
#ifdef INDEX_NDX
   IxList *i;
   i = NdxList;
   while( i )
   {
      if( i->ndx->UniqueIndex() )
      {
         i->ndx->CreateKey( 0, 0 );
         if( i->ndx->FindKey() == FOUND ) return KEY_NOT_UNIQUE;
      }
      i = i->NextIx;
   }
#endif

/* lock the database */
#ifdef LOCKING_ON
   if( AutoLock )
      if(( rc = LockDatabase( F_SETLKW, F_WRLCK, 0L )) != NO_ERROR) return rc;
#endif

/* lock any indexes */
#ifdef INDEX_NDX
#ifdef LOCKING_ON
   i = NdxList;
   while( i && AutoLock )
   {
      if(( rc = i->ndx->LockIndex( F_SETLKW, F_WRLCK )) != NO_ERROR ) return rc;
      i = i->NextIx;
   }
#endif               /* LOCKING_ON */

/* update the indexes */
   i = NdxList;
   while( i )
   {
      if( !i->ndx->UniqueIndex() )          /* if we didn't prepare the key */
         if(( rc = i->ndx->CreateKey( 0, 0 )) != NO_ERROR ) /* then do it before the add    */
            return rc;
      if(( rc =  i->ndx->AddKey( NoOfRecs + 1 )) != NO_ERROR ) return rc;
      i = i->NextIx;
   }
#endif              /* INDEX_NDX */

   /* write the last record */
   if( fseek( fp, ((long) HeaderLen + ( NoOfRecs * RecordLen )), 0 ) != 0 )
      return SEEK_ERROR;

   if( fwrite( RecBuf, RecordLen, 1, fp ) != 1 )
      return WRITE_ERROR;
   
   /* write the end of file marker */
   if( fwrite( "\x0d", 1, 1, fp ) != 1 )
      return WRITE_ERROR;
 
   /* calculate the latest header information */ 
   UpdateYY = xbase->YearOf( xbase->Sysdate()) - 1900;
   UpdateMM = xbase->MonthOf( xbase->Sysdate());
   UpdateDD = xbase->DayOf( FMT_MONTH, xbase->Sysdate());
   NoOfRecs++;
   CurRec = NoOfRecs;

   /* rewrite the header record */
   if(( rc = WriteHeader( 2 )) != NO_ERROR )
      return rc;

#ifdef LOCKING_ON
   if( AutoLock )
      LockDatabase( F_SETLK, F_UNLCK, 0L );

#ifdef INDEX_NDX
   i = NdxList;
   while( i && AutoLock )
   {
      i->ndx->LockIndex( F_SETLK, F_UNLCK );
      i = i->NextIx;
   }
#endif               /* INDEX_NDX  */
#endif               /* LOCKING_ON */

   DbfStatus = OPEN;
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::GetRecord( ULONG RecNo )
{
   int rc;

   if( DbfStatus == CLOSED )
      return NOT_OPEN;
 
   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )   /* update previous rec if necessary */
      if(( rc = PutRecord( CurRec )) != 0 )
         return rc;

   if( RecNo > NoOfRecs || RecNo == 0L )
      return INVALID_RECORD;

   #ifdef LOCKING_ON
   if( AutoLock )
      if(( rc = LockDatabase( F_SETLKW, F_RDLCK, RecNo )) != 0 ) return rc;
   #endif

   /* michael - modified code to avoid unecessary fseek work */

   if( !CurRec || RecNo != CurRec+1 )
   {
      if( fseek( fp, (long) HeaderLen+((RecNo-1L)*RecordLen), SEEK_SET ))
      {
         #ifdef LOCKING_ON
         LockDatabase( F_SETLK, F_UNLCK, 0L );
         #endif
         return SEEK_ERROR;
      }
   }

   if( fread( RecBuf, RecordLen, 1, fp ) != 1 )
   {
      #ifdef LOCKING_ON
      LockDatabase( F_SETLK, F_UNLCK, 0L );
      #endif
      return READ_ERROR;
   }

   DbfStatus = OPEN;
   CurRec = RecNo;
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::GetFirstRecord( VOID )
{
   SHORT rc;
   if( NoOfRecs == 0 )
      return INVALID_RECORD;

   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )  /* update previous rec if necessary */
      if(( rc = PutRecord( CurRec )) != 0 )
         return rc;

   return GetRecord( 1L );
}
/************************************************************************/
SHORT DBF::GetLastRecord( VOID )
{
   SHORT rc;
   if( NoOfRecs == 0 )
      return INVALID_RECORD;

   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )  /* update previous rec if necessary */
      if(( rc = PutRecord( CurRec )) != 0 )
         return rc;

   return GetRecord( NoOfRecs );
}
/************************************************************************/
SHORT DBF::GetNextRecord( VOID )
{
   SHORT rc;
   if( NoOfRecs == 0 )
      return INVALID_RECORD; 
   else if( CurRec >= NoOfRecs )
      return XBASE_EOF; 

   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )  /* update previous rec if necessary */
      if(( rc = PutRecord( CurRec )) != 0 )
         return rc;

   return GetRecord( ++CurRec );
}
/************************************************************************/
SHORT DBF::GetPrevRecord( VOID )
{
   SHORT rc;
   if( NoOfRecs == 0 )
      return INVALID_RECORD; 
   else if( CurRec <= 1L )
      return XBASE_BOF; 

   if( DbfStatus == UPDATED /*&& AutoUpdate*/ )  /* update previous rec if necessary */
      if(( rc = PutRecord( CurRec )) != 0 )
         return rc;

   return GetRecord( --CurRec );
}
/************************************************************************/
SHORT DBF::DumpRecord( ULONG RecNo )
{
   int i;
   char buf[241];
 
   if( RecNo == 0 || RecNo > NoOfRecs )
      return INVALID_RECORD;

   i = GetRecord( RecNo );
   if( i != NO_ERROR )
      return i;

   cout << "\nREC NUMBER " << RecNo << "\n";

   if( RecordDeleted() ) 
      cout << "\nRecord deleted...\n";

   for( i = 0; i < NoOfFields; i++ )
   {
      GetField( i, buf );
      cout << SchemaPtr[i].FieldName << " = " << buf << "\n";
   }
   cout << "\n";
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::PutRecord( ULONG RecNo )
{
   SHORT  rc;

#ifdef INDEX_NDX
   IxList *i;
#endif

   if( DbfStatus == CLOSED )
      return NOT_OPEN;
   if( RecNo > NoOfRecs || RecNo == 0L )
      return INVALID_RECORD;

#ifdef INDEX_NDX
   /* for any unique indexes that were updated, verify no unique keys exist */
   i = NdxList;
   while( i )
   {
      if( i->ndx->UniqueIndex() )
      {
         if(( i->KeyUpdated = i->ndx->KeyWasChanged()) == 1 )
            if( i->ndx->FindKey() == FOUND ) return KEY_NOT_UNIQUE;
      } 
      i = i->NextIx;
   }
#endif

/* lock the database */
#ifdef LOCKING_ON
   if( AutoLock )
   {
      if(( rc = LockDatabase( F_SETLKW, F_WRLCK, 0L )) != NO_ERROR ) return rc;
      if(( rc = LockDatabase( F_SETLKW, F_WRLCK, RecNo )) != NO_ERROR )
      {
         LockDatabase( F_SETLK, F_UNLCK, 0L );
         return rc;
      }
   }
#endif

/* lock the indexes */
#ifdef INDEX_NDX
#ifdef LOCKING_ON
   i = NdxList;
   while( i && AutoLock )
   {
      if(( rc = i->ndx->LockIndex( F_SETLKW, F_WRLCK )) != NO_ERROR ) return rc;
      i = i->NextIx;
   }
#endif               /* LOCKING_ON */

   
   /* loop through deleting old index keys and adding new index keys */
   i = NdxList;
   while( i )
   {
      if( !i->ndx->UniqueIndex() )
         i->KeyUpdated = i->ndx->KeyWasChanged();
      if( i->KeyUpdated )
      {
         if(( rc = i->ndx->AddKey( CurRec )) != NO_ERROR ) return rc;
         i->ndx->CreateKey( 1, 0 );      /* load key buf w/ old values */
         i->ndx->DeleteKey( CurRec );
      }
      i = i->NextIx;
   }
#endif                        /* INDEX_NDX */

   if( fseek( fp, (long) HeaderLen+((RecNo-1L)*RecordLen),0 ))
      return SEEK_ERROR;

   if( fwrite( RecBuf, RecordLen, 1, fp ) != 1 )
      return WRITE_ERROR;

#ifdef LOCKING_ON
   if( AutoLock )
   {
      LockDatabase( F_SETLK, F_UNLCK, 0L );
      LockDatabase( F_SETLK, F_UNLCK, RecNo );
   }

#ifdef INDEX_NDX
   i = NdxList;
   while( i && AutoLock )
   {
      i->ndx->LockIndex( F_SETLK, F_UNLCK );
      i = i->NextIx;
   }
#endif               /* INDEX_NDX  */
#endif               /* LOCKING_ON */

   CurRec = RecNo;
   DbfStatus = OPEN;
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::DeleteRecord( VOID )
{
   if( RecBuf ) 
   {
      RecBuf[0] = 0x2a;
      DbfStatus = UPDATED;
      return NO_ERROR;
   }
   else
      return INVALID_RECORD;
}
/************************************************************************/
SHORT DBF::UndeleteRecord( VOID )
{
   if( RecBuf ) 
   {
      RecBuf[0] = 0x20;
      DbfStatus = UPDATED;
      return NO_ERROR;
   }
   else
      return INVALID_RECORD;
}
/************************************************************************/
SHORT DBF::RecordDeleted( VOID )
{
   if( RecBuf[0] == 0x2a )
      return 1;
   else
      return 0;
} 
/************************************************************************/
SHORT DBF::PackDatafiles( VOID )
{
   SHORT rc, i, NameLen;
   FILE *t;
   LONG l;
   CHAR *target, *source, *TempDbfName;
   CHAR tbuf[4];

   #ifdef MEMO_FIELDS
   LONG   len, BufSize;
   CHAR   * Buf, *TempDbtName;
   CHAR   lb;
   SHORT  MemoFields;
   #endif  /* MEMO_FIELDS */

   DBF Temp( xbase );

   if(( rc = xbase->DirectoryExistsInName( DatabaseName )) > 0 )
      NameLen = rc + 13;
   else
      NameLen = 13;

   if(( TempDbfName = (CHAR *) malloc( NameLen )) == NULL )
      return NO_MEMORY;

   memset( TempDbfName, 0x00, NameLen );
   if( rc )
   {
      strncpy( TempDbfName, DatabaseName, rc );
      strcat( TempDbfName, "TMPXBASE.DBF" );
   }
   else
      strcpy( TempDbfName, "TMPXBASE.DBF" );

   if(( t = fopen( TempDbfName, "w+b" )) == NULL )
   {
      free( TempDbfName );
      return OPEN_ERROR;
   }

   /* copy file header */
   if(( rc = fseek( fp, 0, SEEK_SET )) != 0 )
   {
      free( TempDbfName );
      return SEEK_ERROR;
   }
  
   for( i = 0; i < HeaderLen; i++ )
      fputc( fgetc( fp ), t );
   fputc( 0x1a, t );

   if( fclose( t ) != 0 )
   {
      free( TempDbfName );
      return CLOSE_ERROR;
   }


   #ifdef MEMO_FIELDS
   if(( MemoFields = MemoFieldsPresent()) > 0 )
   {

      if(( TempDbtName = (CHAR *) malloc( NameLen )) == NULL )
         return NO_MEMORY;
      memset( TempDbtName, 0x00, NameLen );
      strcpy( TempDbtName, TempDbfName );
      TempDbtName[strlen(TempDbtName)-1] = 'T';

      if(( t = fopen( TempDbtName, "w+b" )) == NULL )
         return OPEN_ERROR;

      l = 1L;
      memset( tbuf, 0x00, 4 );
      xbase->PutLong( tbuf, l );
      
      if(( fwrite( &tbuf, 4, 1, t )) != 1 ) return WRITE_ERROR;
      if( MemoHeader.Version == 0x03 )
      {
         for( i = 0; i < 12; i++ ) fputc( 0x00, t );
         fputc( 0x03, t );
         for( i = 0; i < 495; i++ ) fputc( 0x00, t );
      }
      else
      {
         for( i = 0; i < 4; i++ ) fputc( 0x00, t );
         if(( fwrite( &MemoHeader.FileName,8,1,t )) != 1 ) return WRITE_ERROR;
         for( i = 0; i < 4; i++ ) fputc( 0x00, t );
         memset( tbuf, 0x00, 2 );
         xbase->PutShort( tbuf, MemoHeader.BlockSize );
         if(( fwrite( &tbuf, 2, 1, t )) != 1 ) return WRITE_ERROR;
         for( i = 22; i < MemoHeader.BlockSize; i++ ) fputc( 0x00, t );
      }
 
      if( fclose( t ) != 0 )
      {
         free( TempDbfName );
         free( TempDbtName );
         return CLOSE_ERROR;
      }
   }
#endif   /* MEMO_FIELDS */

   /* reopen as database */
   if(( rc = Temp.OpenDatabase( TempDbfName )) != NO_ERROR )
   {
      free( TempDbfName );
      #ifdef MEMO_FIELDS
      if( MemoFields ) free ( TempDbtName );
      #endif
      return rc;
   }

   Temp.ResetNoOfRecs();
   target = Temp.GetRecordBuf();
   source = GetRecordBuf();

   for( l = 1; l <= NoOfRecords(); l++ )
   {
      if(( rc = GetRecord( l )) != NO_ERROR )
      {
         free( TempDbfName );
         #ifdef MEMO_FIELDS
         if( MemoFields ) free( TempDbtName );
         #endif
         return rc;
      }

      if( !RecordDeleted())
      {
         strncpy( target, source, GetRecordLen());

         #ifdef MEMO_FIELDS
         len = BufSize = 0L;
         Buf = NULL;
         for( i = 0; i < NoOfFields; i++ )
         {
            if( GetFieldType( i ) == 'M' && MemoFieldExists( i ))
            {
               len = GetMemoFieldLen( i );
               if( len > BufSize )
               {
                  if( BufSize ) 
                     free( Buf );
                  if(( Buf = (CHAR *) malloc( len )) == NULL )
                  {
                     free( TempDbfName );
                     #ifdef MEMO_FIELDS
                     if( MemoFields ) free( TempDbtName );
                     #endif
                     return NO_MEMORY;
                  }
                  BufSize = len;
               }
               GetMemoField( i, len, Buf, -1 );
               Temp.UpdateMemoData( i, len, Buf, -1 );
            }
         }
         #endif

         if(( rc = Temp.AppendRecord()) != NO_ERROR )
         {
            free( TempDbfName );
            #ifdef MEMO_FIELDS
            if( MemoFields ) free( TempDbtName );
            #endif
            return rc;
         }
      }
   }
   Temp.CloseDatabase();
 
   if( fclose( fp ) != 0 )
   {
      free( TempDbfName );
      #ifdef MEMO_FIELDS
      if( MemoFields ) free( TempDbtName );
      #endif
      return CLOSE_ERROR;
   }

   if( remove( DatabaseName ) != 0 )
   {
      free( TempDbfName );
      #ifdef MEMO_FIELDS
      if( MemoFields ) free( TempDbtName );
      #endif
      return WRITE_ERROR;
   }

   if( rename( TempDbfName, DatabaseName ) != 0 )
   { 
      free( TempDbfName );
      #ifdef MEMO_FIELDS
      if( MemoFields ) free( TempDbtName );
      #endif
      return WRITE_ERROR;
   }
   free( TempDbfName );

#ifdef MEMO_FIELDS
   if( MemoFields )
   {
      len = strlen( DatabaseName );
      len--;
      lb = DatabaseName[len];
      if( lb == 'F' )
         DatabaseName[len] = 'T';
      else
         DatabaseName[len] = 't';
   
      if( fclose( mfp ) != 0 )        /* thanks Jourquin */
      {
         free( TempDbtName );
         return CLOSE_ERROR;
      }
   
      if( remove( DatabaseName ) != 0 )
      {
         free( TempDbtName );
         DatabaseName[len] = lb;
         return WRITE_ERROR;
      }
      if( rename( TempDbtName, DatabaseName ) != 0 )
      {
         free( TempDbtName );
         DatabaseName[len] = lb;
         return WRITE_ERROR;
      }
      free( TempDbtName );
      if(( mfp = fopen( DatabaseName, "r+b" )) == NULL )
         return OPEN_ERROR;
      DatabaseName[len] = lb;
   }

#endif /* MEMO_FIELDS */

   if(( fp = fopen( DatabaseName, "r+b" )) == NULL )
      return OPEN_ERROR;

   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::PackDatabase( SHORT LockWaitOption )
{
   SHORT rc;

   /* lock all open files and indexes */
   if(( rc = ExclusiveLock( LockWaitOption )) != NO_ERROR ) return rc;

   if(( rc = PackDatafiles()) != NO_ERROR )
   {
      ExclusiveUnlock();
      return OPEN_ERROR;
   }

   /* refresh file header */
   if(( rc = ReadHeader( 1 )) != NO_ERROR )
      return rc;

   if(( rc = RebuildAllIndices()) != NO_ERROR ) 
      return rc;

   ExclusiveUnlock();
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::CopyDbfStructure( CHAR * NewFileName, SHORT Overlay )
{
   SHORT NameLen, rc, i;
   CHAR  *ndfn;       /* new dbf file name */
   CHAR  ch, *sp, *wp;
   CHAR buf[9];
   FILE  *t;

   /* build the new file name */
   NameLen = strlen( NewFileName ) + 1;
   if(( rc = NameSuffixMissing( 1, NewFileName )) > 0 )
      NameLen += 4;
   if(( ndfn = (CHAR *) malloc( NameLen )) == NULL )
      return NO_MEMORY;

   strcpy( ndfn, NewFileName );
   if( rc == 1 )
      strcat( ndfn, ".dbf" );
   else if( rc == 2 )
      strcat( ndfn, ".DBF" );

   /* check if the file exists and Overlay is on */
   if((( t = fopen( ndfn, "r" )) != NULL ) && !Overlay )
   {
      free( ndfn );
      fclose( t );
      return FILE_EXISTS;
   }

   /* open new file */
   if(( t = fopen( ndfn, "w+b" )) == NULL )
   {
      free( ndfn );
      return OPEN_ERROR;
   }

   /* copy the file header */
   if(( rc = fseek( fp, 0, SEEK_SET )) != 0 )
      return SEEK_ERROR;
   fputc( fgetc( fp ), t );

   /* do the date */
   ch = xbase->YearOf( xbase->Sysdate()) - 1900;
   fputc( ch, t );
   ch = xbase->MonthOf( xbase->Sysdate());
   fputc( ch, t );
   ch = xbase->DayOf( FMT_MONTH, xbase->Sysdate());
   fputc( ch, t );

   /* record count */
   for( i = 0; i < 4; i++ ) fputc( 0x00, t );

   if(( rc = fseek( fp, 7L, SEEK_CUR )) != 0 )
   {
      free( ndfn );
      fclose( t );
      return SEEK_ERROR;
   }
   for( i = 0; i < 4; i++ )
      fputc( fgetc( fp ), t );

   for( i = 0; i < 17; i++ )
      fputc( 0x00, t );

   if(( rc = fseek( fp, 17L, SEEK_CUR )) != 0 )
   {
      free( ndfn );
      fclose( t );
      return SEEK_ERROR;
   }
   for( i = 29; i < HeaderLen; i++ )
      fputc( fgetc( fp ), t );

   fputc( 0x1a, t );
   fclose( t );

#ifdef MEMO_FIELDS
   if( MemoFieldsPresent())
   {
      sp = ndfn + NameLen - 2;
      if( *sp == 'F' )
         *sp = 'T';
      else if( *sp == 'f' )
         *sp = 't';
      if(( t = fopen( ndfn, "w+b" )) == NULL )
      {
         free( ndfn );
         return OPEN_ERROR;
      }
      memset( buf, 0x00, 4 );
      xbase->PutLong( buf, 1L );
      if(( fwrite( &buf, 4, 1, t )) != 1 )
      {
         free( ndfn );
         fclose( t );
         return WRITE_ERROR;
      }
      if( MemoHeader.Version == 0x03 )
      {
         for( i = 0; i < 12; i++ ) fputc( 0x00, t );
         fputc( 0x03, t );
         for( i = 0; i < 495; i++ ) fputc( 0x00, t );
      }
      else
      {
         for( i = 0; i < 4; i++ ) fputc( 0x00, t );
         memset( buf, 0x00, 9 );
         sp = ndfn;
         wp = sp;
         while( wp && *wp && *wp != '.' )
         {
#ifdef DOS
            if( *wp == '\\' )
#else
            if( *wp == '/' )
#endif
            {
               sp = wp;
               sp++;
            }
            wp++;
         }
         for( i = 0; i < 8 && *sp != '.'; i++ )
            buf[i] = *sp++;
         fwrite( &buf, 8, 1, t );
         for( i = 0; i < 4; i++ ) fputc( 0x00, t );
         memset( buf, 0x00, 2 );
         xbase->PutShort( buf, MemoHeader.BlockSize );
         if(( fwrite( &buf, 2, 1, t )) != 1 )
         {
            fclose( t );
            free( ndfn );
            return WRITE_ERROR;
         }
         for( i = 22; i < MemoHeader.BlockSize; i++ ) fputc( 0x00, t );
      }
   }
#endif   // MEMO_FIELDS
   fclose( t );
   free( ndfn );
   return NO_ERROR;
}
/************************************************************************/
#ifdef INDEX_NDX
SHORT DBF::AddNdxToIxList( NDX * n, CHAR * IndexName )
{
   IxList *i, *s, *t;

   if( !FreeIxList )
   {
      if(( i = (IxList *) malloc( sizeof( IxList ))) == NULL )
         return NO_MEMORY;
   }
   else
   {
      i = FreeIxList;
      FreeIxList = i->NextIx;
   }
   memset( i, 0x00, sizeof( IxList ));

   i->IxName  = IndexName;
   i->ndx     = n;

   s = NULL;
   t = NdxList;
   while( t && strcmp( t->IxName, IndexName ) < 0 )
   {
      s = t;
      t = t->NextIx;
   }
   i->NextIx = t;
   if( s == NULL )
      NdxList = i;
   else
      s->NextIx = i;
   return 0;
}
#endif
/************************************************************************/
SHORT DBF::RebuildAllIndices()
{
#ifdef INDEX_NDX
   SHORT rc;
   IxList *n;

   n = NdxList;
   while( n )
   {
      if(( rc = n->ndx->ReIndex()) != NO_ERROR )
      {
         ExclusiveUnlock();
         return rc;
      }
      n = n->NextIx;
   }
#endif
         
   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::DeleteAll( SHORT Option )
{
   SHORT rc;

   if(( NoOfRecords()) == 0 ) 
      return NO_ERROR;
   if(( rc = GetFirstRecord()) != NO_ERROR )
      return rc;

   if( Option == 0 )   /* delete all option */
   {
      while( 1 )
      {
         if( !RecordDeleted())
            if(( rc = DeleteRecord()) != NO_ERROR )
               return rc;
         if(( rc = GetNextRecord()) != NO_ERROR )
            break;
      }
   }
   else   /* undelete all option */
   {
      while( 1 )
      {
         if( RecordDeleted())
            if(( rc = UndeleteRecord()) != NO_ERROR )
               return rc;
         if(( rc = GetNextRecord()) != NO_ERROR )
            break;
      }
   }
   if( rc == XBASE_EOF )
      return NO_ERROR;
   else
      return rc;
}
/************************************************************************/
SHORT DBF::Zap( SHORT WaitOption )
{
#ifdef MEMO_FIELDS
   SHORT MemosExist, dbnlen;
#endif

   SHORT NameLen, rc;
   CHAR  *TempDbfName;

   if(( rc = xbase->DirectoryExistsInName( DatabaseName )) > 0 )
      NameLen = rc + 13;
   else
      NameLen = 13;

   if(( TempDbfName = (CHAR *) malloc( NameLen )) == NULL )
      return NO_MEMORY;
   memset( TempDbfName, 0x00, NameLen );

   if( rc )
   {
      strncpy( TempDbfName, DatabaseName, rc );
      strcat( TempDbfName, "TMPXBASE.DBF" );
   }
   else
      strcpy( TempDbfName, "TMPXBASE.DBF" );

   if(( rc = CopyDbfStructure( TempDbfName, 1 )) != NO_ERROR )
   {
      free( TempDbfName );
      return rc;
   }

#ifdef MEMO_FIELDS
   MemosExist = MemoFieldsPresent();
#endif

   if(( rc = ExclusiveLock( WaitOption )) != NO_ERROR )
      return rc;

   if(( rc = remove( DatabaseName )) != 0 )
   {
      ExclusiveUnlock();
      free( TempDbfName );
      return WRITE_ERROR;
   }

   if(( rc = rename( TempDbfName, DatabaseName )) != 0 )
   {
      ExclusiveUnlock();
      free( TempDbfName );
      return WRITE_ERROR;
   }

   if(( fp = fopen( DatabaseName, "r+b" )) == NULL )
   {
      ExclusiveUnlock();
      free( TempDbfName );
      return OPEN_ERROR;
   }
   ReadHeader( 1 );

#ifdef MEMO_FIELDS
   if( MemosExist )
   {
      fclose( mfp );
      dbnlen = strlen( DatabaseName ) - 1;
      if( DatabaseName[dbnlen] == 'F' )
      {
         DatabaseName[dbnlen] = 'T';    
         TempDbfName[NameLen-2] = 'T';
      }
      else if( DatabaseName[dbnlen] == 'f' )
      {
         DatabaseName[dbnlen] = 't';      
         TempDbfName[NameLen-2] = 't';
      }
            
      if(( rc = remove( DatabaseName )) != 0 )
      {
         ExclusiveUnlock();
         free( TempDbfName );
         return OPEN_ERROR;
      }
      if(( rc = rename( TempDbfName, DatabaseName )) != 0 )
      {
         ExclusiveUnlock();
         free( TempDbfName );
         return OPEN_ERROR;
      }
      if(( mfp = fopen( DatabaseName, "r+b" )) == NULL )
      {
         ExclusiveUnlock();
         free( TempDbfName );
         return OPEN_ERROR;
      }
      GetDbtHeader(1);
      if( DatabaseName[NameLen-1] == 'T' )
         DatabaseName[NameLen-1] = 'F';    
      else if( DatabaseName[NameLen-1] == 't' )
         DatabaseName[NameLen-1] = 'f';    
   }      
#endif   // MEMO_FIELDS

   free( TempDbfName );

   if(( rc = RebuildAllIndices()) != NO_ERROR )
   {
      ExclusiveUnlock();
      return rc;
   }
   ExclusiveUnlock();
   return NO_ERROR;
}
/************************************************************************/
#ifdef INDEX_NDX
SHORT DBF::RemoveNdxFromIxList( NDX * n )
{
   IxList *i, *s;

   i = NdxList;
   s = NULL;
   while( i )
   {
      if( i->ndx == n )
      {
         /* remove it from current chain */
         if( s )
           s->NextIx = i->NextIx;
         else
           NdxList = i->NextIx;

         /* add i to the current free chain */
         i->NextIx = FreeIxList;
         FreeIxList = i;
         FreeIxList->IxName = NULL;
         FreeIxList->ndx = NULL;
         break;
      }
      else
      {
         s = i;
         i = i->NextIx;
      }
   }
   return NO_ERROR;
} 
#endif
/************************************************************************/
