/*  Optimizer.h

    Product: Mentor SQL

    Copyright (C) 1998  Systementor AB, Stefan Holmberg
    Email - stefan.holmberg@systementor.se
    Web:   - http://www.systementor.se

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

///////////////////////////////////////////////////////////////////////////////
*/


/*
		The optimizer is responsible for creating an optimal plan...
		Today it is really simple.

		It is build up like a real optimizer though and
		builds up access-lists combining different join orders and
		different access methods ( filescan or index scan )
*/

#ifndef __SHOPTIMIZER_H
#define __SHOPTIMIZER_H

#include <vector>
#include "executer.h"
#include "ParseTree.h"
#include "systemcatalog.h"
#include "server.h"

class CColumnPlanInfo
{
public:
	CColumnPlanInfo( const char *szTableName, const char *szName, MENTORSQL_DATATYPES DataType, int nFieldLen )
		{
		m_strColumnName = szName;
		m_DataType = DataType;	
		m_nFieldLen = nFieldLen;
		m_strTableName = szTableName;
		}
	std::string m_strTableName;
	std::string m_strColumnName;	
	MENTORSQL_DATATYPES m_DataType;
	int m_nFieldLen;
};

typedef std::vector<CColumnPlanInfo *> CColumnPlanInfoArray;
class CPlanList;

class CTuple
{
public:
	CTuple( CColumnPlanInfoArray *pPlanInfoArray );
	~CTuple();

	void SetValue( void *pvData, int nSize, const char *szColName, const char *szTableName = NULL );

	bool GetStringValue( std::string &strRet, const char *szColName, const char *szTableName = NULL );
	bool GetStringValue( std::string &strRet, int nColNo );
	bool GetLongValue( long &lRet, const char *szColName, const char *szTableName = NULL );
	bool GetLongValue( long &lRet, int nColNo );
	bool GetDoubleValue( double &dRet, const char *szColName, const char *szTableName = NULL );
	bool GetDoubleValue( double &dRet, int nColNo );

	bool ColumnExists( const char *szColName, const char *szTableName = NULL );

	MENTORSQL_DATATYPES GetDataType( const char *szColName, const char *szTableName = NULL );
	MENTORSQL_DATATYPES GetDataType( int nColNo );
	bool GetFieldAsBuf( std::string &strRet, int nColNo );
	bool GetFieldAsBuf( std::string &strRet, const char *szColName, const char *szTableName = NULL );
protected:
	CColumnPlanInfoArray *m_pColumnPlanInfoArray;
	char *GetBuffer( int nColNo, int &nFieldSize );
	int GetColNo( const char *szColName, const char *szTableName );
	char *m_szBuf;
};

enum
{
GETROW_EOF,
GETROW_OK
};

class CPlanOperator
{
public:
	CPlanOperator()
		{
		m_pLeftChild = m_pRightChild = NULL;
		m_pTuple = NULL;
		}
	virtual ~CPlanOperator()
		{
		if ( m_pTuple )
			delete m_pTuple;
		}
	virtual bool SortOrder( std::string &strCol )
		{
		return false;
		}
	virtual long GetCost()
		{ return 0; };

	virtual int GetRow()
		{
		return GETROW_EOF;
		}
	virtual void Reset()
		{
		}

	virtual char *Name()
		{
		return "CPlanOperator";
		}
#ifdef _DEBUG
	virtual void Dump();
#endif

	CTuple *m_pTuple;

	//On every single planoperator some expressions can be applied
	void PlaceWhereConditions(  CSQLNodeArray &WhereCondArray );
	bool CanApplyCondition( CSQLNode *pNode );
	bool CondExistsInChild( CSQLNode *pNode );
	CSQLNodeArray m_WhereCondArray;
	bool ColumnExists( CSQLNode *pNode );

	bool ApplyConditions();
	bool ApplyOneCondition( CSQLNode *pNode );
	bool GetValueInTuple( CSQLNode *pNode, std::string &strRet, MENTORSQL_DATATYPES &dt1 );

	/*Expressionlist ( where-statement )*/

	CPlanOperator *m_pLeftChild; //Inner if join
	CPlanOperator *m_pRightChild;//Outer if join
	CPlanList *m_pPlanList;
};

class CSHOptimizer;

class CProjectOperator : public CPlanOperator
{
public:
	CProjectOperator( CSHOptimizer *pOpt )
		{
		m_fReset = false;
		m_pOptimizer = pOpt;
		};	
	virtual int GetRow();
	void CopyDataToTuple();
	virtual void Reset()
		{
		m_fReset = true;
		}
	bool m_fReset;
	virtual long GetCost()
		{
		return m_pLeftChild->GetCost();
		}
	virtual char *Name()
		{
		return "CProjectOperator";
		}
	CSHOptimizer *m_pOptimizer;
};

class CJoinOperator : public CPlanOperator
{
public:
	CJoinOperator()
		{
		m_fReset = false;
		}
	~CJoinOperator()
		{
		}
	
	virtual int GetRow();
	void CopyDataToTuple();
	virtual void Reset()
		{
		m_fReset = true;
		}

	virtual char *Name()
		{
		return "CJoinOperator";
		}
#ifdef _DEBUG
	virtual void Dump();
#endif
	bool m_fReset;
	virtual long GetCost()
		{ return m_pLeftChild->GetCost() * m_pRightChild->GetCost(); };
};

class CIndexPlanInfo
{
public:
	std::string m_strIndexName;
	std::string m_strColumn;
};

typedef std::vector<CIndexPlanInfo *> CIndexPlanInfoArray;

class CTablePlanInfo
{
public:
	~CTablePlanInfo();

	std::string m_strTableName;
	long m_lNrOfRows;

#ifdef _DEBUG
	void Dump();
#endif

	CColumnPlanInfoArray m_ColumnPlanInfoArray;
	CIndexPlanInfoArray m_IndexPlanInfoArray;
};

typedef std::vector<CTablePlanInfo *> CTablePlanInfoList;

class CAccessMethod : public CPlanOperator
{
public:
	CAccessMethod()
		{
		m_pTableRef = NULL;		
		}
	virtual long GetCost()
		{
		return 0;
		}
	virtual int GetRow();
	virtual char *Name()
		{
		return "CAccessMethod";
		}
#ifdef _DEBUG
	virtual void Dump();
#endif
	 CTablePlanInfo *m_pTablePlanInfo;	
	 CTable *m_pTableRef;
};


class CTableScanAccessMethod : public CAccessMethod
{
public:
	CTableScanAccessMethod()
		{
//		m_pTableRef = NULL;
		m_fReset = false;
		}
	virtual long GetCost()
		{
		return m_pTablePlanInfo->m_lNrOfRows;
		}
	virtual int GetRow();
	void CopyDataToTuple();
	virtual void Reset()
		{
		m_fReset = true;
		}
	virtual char *Name()
		{
		return "CTableScanAccessMethod";
		}
//	CTable *m_pTableRef;
	bool m_fReset;
#ifdef _DEBUG
	virtual void Dump();
#endif
};

class CIndexAccessMethod : public CAccessMethod
{
public:
	CIndexAccessMethod()
		{
//		m_pTableRef = NULL;
		m_pIndex = NULL;
		m_pIndexExpression = NULL;
		m_fIndexOnlyScan = false; 
		m_fReset = false;
		}
	virtual long GetCost();
	virtual int GetRow();
	void SetIndexExpression();

	virtual bool SortOrder( std::string &strCol );
	virtual void Reset()
		{
		m_fReset = true;
		}


	virtual char *Name()
		{
		return "CIndexAccessMethod";
		}
#ifdef _DEBUG
	void CopyDataToTuple();
	virtual void Dump();
	CIndexPlanInfo *m_pIndexPlanInfo;
//	CTable *m_pTableRef;
	CIndex *m_pIndex;
	bool m_fIndexOnlyScan;
	bool m_fReset;
	bool CanApplyCondition( CSQLNode *pNode );
//	std::string m_strCurrentValue;
#endif
protected:
	short GetFirstRecord();
	short GetNextRecord();

	CSQLNode *m_pIndexExpression; //The expression that should be used in index search
};

class CSolution
{
public:
	CSolution()
		{
		m_pPlanOperator = NULL;
		}

	~CSolution();

	void Execute();

#ifdef _DEBUG
	void Dump();
#endif

	CPlanOperator *m_pPlanOperator;
};
typedef std::vector<CSolution *> CSolutionList;
class CSHOptimizer;

class CPlanList
{
public:
	~CPlanList();
	int m_nDimension;
	CSolutionList m_SolutionList;	
	CSolution *GetBestSolution();
									 
#ifdef _DEBUG	
	std::string GetName();
#endif

	std::string m_strInner;
	CPlanList *m_pOuter;

	bool ContainsTable( const char *szTableName  );
	void AddToAvailableColumns( CTablePlanInfo * );

	CColumnPlanInfoArray m_AvailableColumns; //Available columns at this level
};
typedef std::vector<CPlanList *> CPlanListList;
typedef std::vector<CPlanOperator *> CPlanOperatorList;

class CSHOptimizer
{
public:
	CSHOptimizer();

	void OptimizeQuery( CParseTree *pTree );
	void OptimizeRowAccess( CParseTree *pTree );

	~CSHOptimizer();
	void Clear();

	void CreateOneRelationPlans( CTablePlanInfo *pPlanInfo, CParseTree *pTree, CPlanList *pPlanList  );
	void CreateJoinRelationPlans( CPlanList *pInnerPlanList, CTablePlanInfo *pPlanInfo, CParseTree *pTree, CPlanList *pPlanList  );

	int ReadSystemInfo( CSQLNode *pSelectList, CSQLNode *pWhereClause, CSQLNode *pFromClause );
	void ExtractSeparateConditions(CSQLNode *pNode, bool fParentIsAnd);

	CSolution *GetBestSolution();
	CAccessMethod *GetBestRowAccessMethod();



	void PrintOne();

	CTablePlanInfoList m_PlanInfoList;	
	CPlanListList m_PlanListList;
	CPlanOperatorList m_GlobalPlanOperatorList;
	CColumnPlanInfoArray m_ResultColumns; 
	
	CSQLNodeArray m_WhereCondArray;

protected:
	CSolution *AddProjectOperator( CSolution *pSolution );
	CPlanList *GetOneRelationList( CTablePlanInfo *pInfo );
	void BuildColumnsInfo( CTablePlanInfo *pInfo, CSQLNode *pSelectList, CSQLNode *pWhereClause, CSQLNode *pFromClause, CTable *pTable );
	void BuildIndexInfo( CTablePlanInfo *pInfo, CTable *pTable );
	bool ExistsInSchema( CTablePlanInfo *pInfo, const char *szNewColName );
	void RecursiveAddColDesc( CTablePlanInfo *pInfo, CSQLNode *pNode, CTable *pTable );


	CProjectOperator *CreateProjectOperator();
	CJoinOperator *CreateJoinOperator();
	CTableScanAccessMethod *CreateTableScanAccessMethod();
	CIndexAccessMethod *CreateIndexAccessMethod();
};

#endif





