/*  ndx.cpp  

    Xbase project source code

    NDX indexing routines for X-Base

    Copyright (C) 1997 StarTech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0     10/10/97   - Initial release of software
    V 1.3     11/30/97   - Moved GetLong and GetShort to DBF class
    V 1.5     1/2/98     - Added Dbase IV memo field support
    V 1.6a    4/1/98     - Added expression support
    V 1.6b    4/8/98     - Numeric index keys
    V 1.7.4b  6/2/98     - Big Endian fix in PutLeafNode - JAM
                         - Fix in Reindex  - PS
                         - Fix in Findkey( Tkey, DbfRec )
*/

#include "stdafx.h"
#include "xbase.h"

#ifdef INDEX_NDX
#ifdef DOS
#include <io.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


/***********************************************************************/
SHORT NDX::CloneNodeChain( VOID )
{
   NodeLink * TempNodeS;
   NodeLink * TempNodeT;
   NodeLink * TempNodeT2;

   if( CloneChain ) ReleaseNodeMemory( CloneChain );
   CloneChain = NULL;

   if( !NodeChain ) return NO_ERROR;
   TempNodeS = NodeChain;
   TempNodeT2 = NULL;

   while( TempNodeS )
   {
      if(( TempNodeT = GetNodeMemory()) == NULL )
         return NO_MEMORY;
      memcpy( TempNodeT, TempNodeS, sizeof( struct NodeLink ));
      TempNodeT->NextNode = NULL;
      TempNodeT->PrevNode = TempNodeT2;
      if( !CloneChain )
      {
         TempNodeT2 = TempNodeT;
         CloneChain = TempNodeT;
      }
      else
      {
         TempNodeT2->NextNode = TempNodeT;
         TempNodeT2 = TempNodeT2->NextNode;
      }
      TempNodeS = TempNodeS->NextNode;
   }
   return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::UncloneNodeChain( VOID )
{
   if( NodeChain )
      ReleaseNodeMemory( NodeChain );
   NodeChain = CloneChain;
   CloneChain = NULL;
   CurNode = NodeChain;
   while( CurNode->NextNode )
      CurNode = CurNode->NextNode;
   return NO_ERROR;
}
/***********************************************************************/
/* This routine dumps the node chain to stdout                         */
#ifdef XBASE_DEBUG
VOID NDX::DumpNodeChain( VOID )
{
   NodeLink  *n;
   cout << "\n*************************\n";
   cout <<   "NodeLinkCtr = " << NodeLinkCtr;
   cout << "\nReused      = " << ReusedNodeLinks << "\n";

   n = NodeChain;
   while(n)
   {
      cout << "NodeLink Chain" << n->NodeNo << "\n";
      n = n->NextNode;
   }
   n = FreeNodeChain;
   while(n)
   {
      cout << "FreeNodeLink Chain" << n->NodeNo << "\n";
      n = n->NextNode;
   }
   n = DeleteChain;
   while(n)
   {
      cout << "DeleteLink Chain" << n->NodeNo << "\n";
      n = n->NextNode;
   }
}
#endif
/***********************************************************************/
/* This routine returns a chain of one or more index nodes back to the */
/* free node chain                                                     */

VOID NDX::ReleaseNodeMemory( NodeLink * n )
{
   NodeLink * temp;

   if( !FreeNodeChain )
      FreeNodeChain = n;
   else    /* put this list at the end */
   {
      temp = FreeNodeChain;
      while( temp->NextNode )
         temp = temp->NextNode;
      temp->NextNode = n;
   }
   return;
}
/***********************************************************************/
/* This routine returns a node from the free chain if available,       */
/* otherwise it allocates new memory for the requested node             */

NodeLink * NDX::GetNodeMemory( VOID )
{
   NodeLink * temp;
   if( FreeNodeChain )
   {
      temp = FreeNodeChain;
      FreeNodeChain = temp->NextNode;
      ReusedNodeLinks++;
   }
   else
   {
      temp = (NodeLink *) malloc( sizeof( NodeLink ));
      NodeLinkCtr++;
   }
   memset( temp, 0x00, sizeof( NodeLink ));
   return temp;
}
/***********************************************************************/
#ifdef XBASE_DEBUG
VOID NDX::DumpHdrNode( VOID )
{
   cout << "\nStart node    = " << HeadNode.StartNode;
   cout << "\nTotal nodes   = " << HeadNode.TotalNodes;
   cout << "\nNo of keys    = " << HeadNode.NoOfKeys;
   cout << "\nKey Length    = " << HeadNode.KeyLen;
   cout << "\nKeys Per Node = " << HeadNode.KeysPerNode;
   cout << "\nKey type      = " << HeadNode.KeyType;
   cout << "\nKey size      = " << HeadNode.KeySize;
   cout << "\nUnknown 2     = " << HeadNode.Unknown2;
   cout << "\nUnique        = " << HeadNode.Unique;
   cout << "\nKeyExpression = " << HeadNode.KeyExpression << "\n";
}
#endif
/***********************************************************************/
NDX::NDX( DBF * pdbf ) 
{
   memset( Node, 0x00, NDX_NODE_SIZE );
   memset( &HeadNode, 0x00, sizeof( NdxHeadNode )); 
   ndxfp           = NULL;
   NodeChain       = NULL;
   CloneChain      = NULL;
   FreeNodeChain   = NULL;
   DeleteChain     = NULL;
   CurNode         = NULL;
   NdxStatus       = 0;
   NodeLinkCtr     = 0L;
   ReusedNodeLinks = 0L;
   ndx             = this;
   dbf             = pdbf;
   KeyBuf          = NULL;
   KeyBuf2         = NULL;
   IndexName       = NULL;
}
/***********************************************************************/
SHORT NDX::OpenIndex( CHAR * FileName )
{
   int NameLen, rc;

   NameLen = strlen( FileName ) + 1;
   if(( rc = dbf->NameSuffixMissing( 2, FileName )) > 0 )
      NameLen += 4;

   if(( IndexName = (char *) malloc( NameLen )) == NULL )
      return NO_MEMORY;

   /* copy the file name into the class variable */
   memset( IndexName, 0x00, NameLen ); 
   strcpy( IndexName, FileName );

   if( rc == 1 )
     strcat( IndexName, ".ndx" );
   else if ( rc == 2 )
     strcat( IndexName, ".NDX" );

   /* open the file */
   if(( ndxfp = fopen( IndexName, "r+b" )) == NULL )
   {
     free( IndexName );
     return OPEN_ERROR;
   }
   NdxStatus = 1; 
   if(( rc = GetHeadNode()) != 0) 
   { 
      free( IndexName );
      fclose( ndxfp );
      return rc;
   }

   /* parse the expression */
   if(( rc = dbf->xbase->BuildExpressionTree( HeadNode.KeyExpression,
      strlen( HeadNode.KeyExpression ), dbf )) != NO_ERROR )
         return rc;
   ExpressionTree = dbf->xbase->GetTree();
   dbf->xbase->SetTreeToNull(); 

   KeyBuf  = (CHAR *) malloc( HeadNode.KeyLen + 1 ); 
   KeyBuf2 = (CHAR *) malloc( HeadNode.KeyLen + 1);
   memset( KeyBuf,  0x00, HeadNode.KeyLen + 1 );
   memset( KeyBuf2, 0x00, HeadNode.KeyLen + 1 );

   return dbf->AddNdxToIxList( ndx, IndexName );  
}
/***********************************************************************/
SHORT NDX::CloseIndex( VOID )
{
//Added by Stefan Holmberg
   if( NodeChain )
   {
      ReleaseNodeMemory( NodeChain );
      NodeChain = NULL;
   }

   if( FreeNodeChain )
   {
      ReleaseNodeMemory( FreeNodeChain );
      FreeNodeChain = NULL;
   }
   if( CurNode )
   {
      ReleaseNodeMemory( CurNode );
      CurNode = NULL;
   }
   if( DeleteChain )
   {
      ReleaseNodeMemory( DeleteChain );
      DeleteChain = NULL;
   }
   if( CloneChain )
   {
      ReleaseNodeMemory( CloneChain );
      CloneChain = NULL;
   }


   if( IndexName ) { free ( IndexName ); IndexName = NULL; }
   if( KeyBuf )    { free ( KeyBuf );    KeyBuf = NULL;    }
   if( KeyBuf2 )   { free ( KeyBuf2 );   KeyBuf2 = NULL;   }
   dbf->RemoveNdxFromIxList( ndx );

   fclose( ndxfp );
   NdxStatus = 0;
   return 0;
}
/***********************************************************************/
SHORT NDX::GetHeadNode( VOID )
{
   CHAR *p, *q;
   SHORT i;

   if( !NdxStatus )
      return NOT_OPEN;

   if( fseek( ndxfp, 0, SEEK_SET ))
      return SEEK_ERROR;

   if(( fread( Node, NDX_NODE_SIZE, 1, ndxfp )) != 1 )
      return READ_ERROR;

   /* load the head node structure */
   p = Node;
   HeadNode.StartNode   = dbf->xbase->GetLong ( p ); p+=4;
   HeadNode.TotalNodes  = dbf->xbase->GetLong ( p ); p+=4;
   HeadNode.NoOfKeys    = dbf->xbase->GetLong ( p ); p+=4;
   HeadNode.KeyLen      = dbf->xbase->GetShort( p ); p+=2;
   HeadNode.KeysPerNode = dbf->xbase->GetShort( p ); p+=2;
   HeadNode.KeyType     = dbf->xbase->GetShort( p ); p+=2;
   HeadNode.KeySize     = dbf->xbase->GetLong ( p ); p+=4;
   HeadNode.Unknown2    = *p++;
   HeadNode.Unique      = *p++;

   q = HeadNode.KeyExpression;
   for( i = 24; i < NDX_NODE_SIZE && *p; i++ )
      *q++ = *p++;

   return 0;
}
/***********************************************************************/
/* This routine reads a leaf node from disk                            */
/*                                                                     */
/*  If SetNodeChain 2, then the node is not appended to the node chain */
/*                     but the CurNode pointer points to the node read */
/*  If SetNodeChain 1, then the node is appended to the node chain     */
/*  If SetNodeChain 0, then record is only read to Node memory         */

SHORT NDX::GetLeafNode( LONG NodeNo, SHORT SetNodeChain )
{
   NodeLink *n;

   if( !NdxStatus )
      return NOT_OPEN;

   if( fseek( ndxfp, NodeNo * NDX_NODE_SIZE, SEEK_SET ))
      return SEEK_ERROR;

   if(( fread( Node, NDX_NODE_SIZE, 1, ndxfp )) != 1 )
      return READ_ERROR;

   if( !SetNodeChain ) return 0;

   if(( n = GetNodeMemory()) == NULL )
      return NO_MEMORY;

   n->NodeNo = NodeNo;
   n->CurKeyNo = 0L;
   n->NextNode = NULL;
   n->Leaf.NoOfKeysThisNode = dbf->xbase->GetLong( Node );
   memcpy( n->Leaf.KeyRecs, Node+4, NDX_NODE_SIZE - 4 );

   /* put the node in the chain */
   if( SetNodeChain == 1 )
   {
      if( NodeChain == NULL )      /* first one ? */
      { 
         NodeChain = n;
         CurNode = n;
         CurNode->PrevNode = NULL;
      }
      else
      {
         n->PrevNode = CurNode;
         CurNode->NextNode = n;
         CurNode = n;
      }
   }
   else
      CurNode = n;
   return 0;
}
/***********************************************************************/
#ifdef XBASE_DEBUG
VOID NDX::DumpNodeRec( LONG n )
{
   CHAR *p;
   LONG NoOfKeys, LeftBranch, RecNo;
   SHORT i,j;

   GetLeafNode( n, 0 );
   NoOfKeys = dbf->xbase->GetLong( Node );
   p = Node + 4;			/* go past no of keys */
   cout << "\n--------------------------------------------------------";
   cout << "\nNode # " << n << " Number of keys = " << NoOfKeys << "\n";

   cout << "\n Key     Left     Rec      Key";
   cout << "\nNumber  Branch   Number    Data";

   for( i = 0; i < GetKeysPerNode() /*NoOfKeys*/; i++ )
   {
      LeftBranch = dbf->xbase->GetLong( p );
      p+=4;
      RecNo = dbf->xbase->GetLong( p );
      p+=4;
      cout << "\n" << i << "         " << LeftBranch << "          " << RecNo << "         ";
      if( !HeadNode.KeyType )
         for( j = 0; j < HeadNode.KeyLen; j++ ) cout << *p++;
      else
      {
         cout << dbf->xbase->GetDouble( p );
         p += 8;
      }
   }
}
#endif
/***********************************************************************/
LONG NDX::GetDbfNo( SHORT RecNo, NodeLink * n )
{
   NdxLeafNode *temp;
   CHAR *p;
   if( !n ) return 0L;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > ( temp->NoOfKeysThisNode - 1 )) return 0L;
   p = temp->KeyRecs + 4;
   p += RecNo * ( 8 + HeadNode.KeyLen );
   return( dbf->xbase->GetLong( p ));
}
/***********************************************************************/
LONG NDX::GetLeftNodeNo( SHORT RecNo, NodeLink * n )
{
   NdxLeafNode *temp;
   CHAR *p;
   if( !n ) return 0L;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > temp->NoOfKeysThisNode ) return 0L;
   p = temp->KeyRecs;
   p += RecNo * ( 8 + HeadNode.KeyLen );
   return( dbf->xbase->GetLong( p ));
}
/***********************************************************************/
CHAR * NDX::GetKeyData( SHORT RecNo, NodeLink * n )
{
   NdxLeafNode *temp;
   CHAR *p;
   if( !n ) return 0L;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > ( temp->NoOfKeysThisNode - 1 )) return 0L;
   p = temp->KeyRecs + 8;
   p += RecNo * ( 8 + HeadNode.KeyLen );
   return( p );
}
/***********************************************************************/
LONG NDX::GetTotalNodes( VOID ) 
{
   if( &HeadNode )
      return HeadNode.TotalNodes;
   else
      return 0L;
}
/***********************************************************************/
USHORT NDX::GetKeysPerNode( VOID ) 
{
   if( &HeadNode )
      return HeadNode.KeysPerNode;
   else
      return 0L;
}
/***********************************************************************/
SHORT NDX::GetFirstKey( SHORT RetrieveSw )
{
/* This routine returns 0 on success and sets CurDbfRec to the record  */
/* corresponding to the first index pointer                            */

   LONG TempNodeNo;
   SHORT rc;
   
   /* initialize the node chain */
   if( NodeChain )
   {
      ReleaseNodeMemory( NodeChain );
      NodeChain = NULL;
   }

   if(( rc = GetHeadNode()) != 0 )
   {
      CurDbfRec = 0L;
      return rc;
   }

   /* get a node and add it to the link */

   if(( rc = GetLeafNode( HeadNode.StartNode, 1 )) != 0 )
      return rc;

/* traverse down the left side of the tree */
   while( GetLeftNodeNo( 0, CurNode ))
   {
      TempNodeNo = GetLeftNodeNo( 0, CurNode );     
      if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;     
      }
      CurNode->CurKeyNo = 0;
   }
   CurDbfRec = GetDbfNo( 0, CurNode );
   if( RetrieveSw )
      return dbf->GetRecord( CurDbfRec );
   else
      return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::GetNextKey( SHORT RetrieveSw )
{
/* This routine returns 0 on success and sets CurDbfRec to the record  */
/* corresponding to the next index pointer                             */

   NodeLink * TempNodeLink;

   LONG TempNodeNo;
   SHORT rc;
    
   if( !NdxStatus )
   {
      CurDbfRec = 0L;
      return NOT_OPEN;
   }

   if( !CurNode )
      return GetFirstKey( RetrieveSw );

   /* more keys on this node ? */
   if(( CurNode->Leaf.NoOfKeysThisNode-1) > CurNode->CurKeyNo )
   {
      CurNode->CurKeyNo++;
      CurDbfRec = GetDbfNo( CurNode->CurKeyNo, CurNode );
      if( RetrieveSw )
         return dbf->GetRecord( CurDbfRec );
      else
         return NO_ERROR;
   }

   /* if head node we are at eof */
   if( CurNode->NodeNo == HeadNode.StartNode ) return XBASE_EOF;

   /* this logic assumes that interior nodes have n+1 left node no's where */
   /* n is the number of keys in the node                                  */

   /* pop up one node to the interior node level & free the leaf node      */

   TempNodeLink = CurNode;
   CurNode = CurNode->PrevNode;
   CurNode->NextNode = NULL;
   ReleaseNodeMemory( TempNodeLink );

   /* while no more right keys && not head node, pop up one node */
   while(( CurNode->CurKeyNo >= CurNode->Leaf.NoOfKeysThisNode ) &&
          ( CurNode->NodeNo != HeadNode.StartNode ))
   {
      TempNodeLink = CurNode;
      CurNode = CurNode->PrevNode;
      CurNode->NextNode = NULL;
      ReleaseNodeMemory( TempNodeLink );
   }

   /* if head node && right most key, return end-of-file */
   if(( HeadNode.StartNode == CurNode->NodeNo ) &&
      ( CurNode->CurKeyNo >= CurNode->Leaf.NoOfKeysThisNode ))
      return XBASE_EOF;

   /* move one to the right */
   CurNode->CurKeyNo++;
   TempNodeNo = GetLeftNodeNo( CurNode->CurKeyNo, CurNode );

   if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      return rc;

/* traverse down the left side of the tree */
   while( GetLeftNodeNo( 0, CurNode ))
   {
      TempNodeNo = GetLeftNodeNo( 0, CurNode );     
      if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;     
      }
      CurNode->CurKeyNo = 0;
   }
   CurDbfRec = GetDbfNo( 0, CurNode );
   if( RetrieveSw )
      return dbf->GetRecord( CurDbfRec );
   else
      return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::GetLastKey( LONG NodeNo, SHORT RetrieveSw )
{
/* This routine returns 0 on success and sets CurDbfRec to the record  */
/* corresponding to the last index pointer                             */

/* If NodeNo = 0, start at head node, otherwise start at NodeNo        */

   LONG TempNodeNo;
   SHORT rc;
  
   if( NodeNo < 0 || NodeNo > HeadNode.TotalNodes )
      return INVALID_NODE_NO;

   /* initialize the node chain */
   if( NodeChain )
   {
      ReleaseNodeMemory( NodeChain );
      NodeChain = NULL;
   }
   if( NodeNo == 0L )
      if(( rc = GetHeadNode()) != 0 )
      { 
         CurDbfRec = 0L;
         return rc;
      }

   /* get a node and add it to the link */

   if( NodeNo == 0L )
   {
      if(( rc = GetLeafNode( HeadNode.StartNode, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;
      }
   }
   else
   {
      if(( rc = GetLeafNode( NodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;
      }
   }
   CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode;

/* traverse down the right side of the tree */
   while( GetLeftNodeNo( CurNode->Leaf.NoOfKeysThisNode, CurNode ))
   {
      TempNodeNo = GetLeftNodeNo( CurNode->Leaf.NoOfKeysThisNode, CurNode );     
      if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;     
      }
      CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode;
   }
   CurNode->CurKeyNo--;           /* leaf node has one fewer ix recs */
   CurDbfRec = GetDbfNo( CurNode->Leaf.NoOfKeysThisNode-1, CurNode );
   if( RetrieveSw )
      return dbf->GetRecord( CurDbfRec );
   else
      return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::GetPrevKey( SHORT RetrieveSw )
{
/* This routine returns 0 on success and sets CurDbfRec to the record  */
/* corresponding to the previous index pointer                         */

   NodeLink * TempNodeLink;

   LONG TempNodeNo;
   SHORT rc;
    
   if( !NdxStatus )
   {
      CurDbfRec = 0L;
      return NOT_OPEN;
   }

   if( !CurNode )
   {
      CurDbfRec = 0L;
      return GetFirstKey( RetrieveSw );
   }

   /* more keys on this node ? */
   if( CurNode->CurKeyNo > 0 )
   {
      CurNode->CurKeyNo--;
      CurDbfRec = GetDbfNo( CurNode->CurKeyNo, CurNode );
      if( RetrieveSw )
         return dbf->GetRecord( CurDbfRec );
      else
         return NO_ERROR;
   }

   /* this logic assumes that interior nodes have n+1 left node no's where */
   /* n is the number of keys in the node                                  */

   /* pop up one node to the interior node level & free the leaf node      */

   if( !CurNode->PrevNode ) return XBASE_EOF;   /* michael - make sure prev node exists */

   TempNodeLink = CurNode;
   CurNode = CurNode->PrevNode;
   CurNode->NextNode = NULL;
   ReleaseNodeMemory( TempNodeLink );

   /* while no more left keys && not head node, pop up one node */
   while(( CurNode->CurKeyNo == 0 ) && 
          ( CurNode->NodeNo != HeadNode.StartNode ))
   {
      TempNodeLink = CurNode;
      CurNode = CurNode->PrevNode;
      CurNode->NextNode = NULL;
      ReleaseNodeMemory( TempNodeLink );
   }

   /* if head node && left most key, return end-of-file */
   if(( HeadNode.StartNode == CurNode->NodeNo ) &&
      ( CurNode->CurKeyNo == 0 ))
      return XBASE_EOF;

   /* move one to the left */
   CurNode->CurKeyNo--;
   TempNodeNo = GetLeftNodeNo( CurNode->CurKeyNo, CurNode );

   if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      return rc;

   if( GetLeftNodeNo( 0, CurNode ))	/* if interior node */
      CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode;
   else					/* leaf node */
      CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode - 1;

/* traverse down the right side of the tree */
   while( GetLeftNodeNo( 0, CurNode ))		/* while interior node */
   {
      TempNodeNo = GetLeftNodeNo( CurNode->Leaf.NoOfKeysThisNode, CurNode );     
      if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;     
      }
      if( GetLeftNodeNo( 0, CurNode ))	/* if interior node */
         CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode;
      else					/* leaf node */
         CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode - 1;
   }
   CurDbfRec = GetDbfNo( CurNode->Leaf.NoOfKeysThisNode - 1, CurNode );
   if( RetrieveSw )
      return dbf->GetRecord( CurDbfRec );
   else
      return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::CompareKey( CHAR * Key1, CHAR * Key2, SHORT Klen )
{
/*   if key1 = key2  --> return 0      */
/*   if key1 > key2  --> return 1      */
/*   if key1 < key2  --> return 2      */

CHAR   *k1, *k2;
SHORT  i;
DOUBLE d1, d2;

   if( Klen > HeadNode.KeyLen ) Klen = HeadNode.KeyLen;

   if( HeadNode.KeyType == 0 )
   {
      k1 = Key1;
      k2 = Key2;
      for( i = 0; i < Klen; i++ )
      {
         if( *k1 > *k2 ) return 1;
         if( *k1 < *k2 ) return 2;
         k1++;
         k2++;
      }
      return 0;
   }
   else		/* key is numeric */
   {
      d1 = dbf->xbase->GetDouble( Key1 );
      d2 = dbf->xbase->GetDouble( Key2 );
      if( d1 == d2 ) return 0;
      else if( d1 > d2 ) return 1;
      else return 2;
   }
}
/***********************************************************************/
LONG NDX::GetLeafFromInteriorNode( CHAR * Tkey, SHORT Klen )
{
   /* This function scans an interior node for a key and returns the   */
   /* correct interior leaf node no                                    */

   SHORT p;

   /* if Tkey > any keys in node, return right most key */
   p = CurNode->Leaf.NoOfKeysThisNode - 1;
   if( CompareKey( Tkey, GetKeyData( p, CurNode ), Klen ) == 1 )
   {
      CurNode->CurKeyNo = CurNode->Leaf.NoOfKeysThisNode;
      return GetLeftNodeNo( CurNode->Leaf.NoOfKeysThisNode, CurNode );
   }

   /* otherwise, start at the beginning and scan up */
   p = 0;
   while( p < CurNode->Leaf.NoOfKeysThisNode &&
          ( CompareKey( Tkey, GetKeyData( p, CurNode ), Klen ) == 1 ))
      p++;

   CurNode->CurKeyNo = p;
   return GetLeftNodeNo( p, CurNode );
}
/***********************************************************************/
SHORT NDX::KeyExists( DOUBLE d )
{
   char buf[9];
   memset( buf, 0x00, 9 );
   dbf->xbase->PutDouble( buf, d );
   return FindKey( buf, 8, 0 );
}
/***********************************************************************/
SHORT NDX::FindKey( DOUBLE d )
{
   char buf[9];
   memset( buf, 0x00, 9 );
   dbf->xbase->PutDouble( buf, d );
   return FindKey( buf, 8, 1 );
}
/***********************************************************************/
SHORT NDX::FindKey( CHAR * Key )
{
   return FindKey( Key, strlen( Key ), 1 );
}
/***********************************************************************/
SHORT NDX::FindKey( CHAR * Tkey, LONG DbfRec )
{
   /* find a key with a specifc DBF record number */
   SHORT rc;

   LONG CurDbfRecNo;
   LONG CurNdxDbfNo;

   /* if we are already on the correct key, return FOUND */
   if( CurNode )
   {
    CurDbfRecNo = dbf->GetCurRecNo();
    CurNdxDbfNo = GetDbfNo( CurNode->CurKeyNo, CurNode );
    if( CurDbfRecNo == CurNdxDbfNo && 
      (strncmp(Tkey, GetKeyData( CurNode->CurKeyNo, CurNode ),
         HeadNode.KeyLen ) == 0 ))
          return FOUND;
   }

   rc =  FindKey( Tkey, HeadNode.KeyLen, 0 );

   while( rc == 0 || rc == FOUND )
   {
      if( strncmp( Tkey, GetKeyData( CurNode->CurKeyNo, CurNode ),
          HeadNode.KeyLen ) == 0 )
      {
         if( DbfRec == GetDbfNo( CurNode->CurKeyNo, CurNode ))
            return FOUND;
         else
            rc = GetNextKey( 0 );
      }
      else
         return NOT_FOUND;
   }
   return NOT_FOUND;
}
/***********************************************************************/
SHORT NDX::FindKey( VOID )
{
   /* if no paramaters given, use KeyBuf */
   return( FindKey( KeyBuf, HeadNode.KeyLen, 0 ));
}
/***********************************************************************/
SHORT NDX::FindKey( CHAR * Tkey, SHORT Klen, SHORT RetrieveSw )
{
   /* This routine sets the current key to the found key */
 
   /* if RetrieveSw is true, the method positions the dbf record */

   SHORT rc,i;
   LONG TempNodeNo;

   if( NodeChain )
   {
      ReleaseNodeMemory( NodeChain );
      NodeChain = NULL;
   }

   if(( rc = GetHeadNode()) != 0 )
   {
      CurDbfRec = 0L;
      return rc;
   }

   /* load first node */
   if(( rc = GetLeafNode( HeadNode.StartNode, 1 )) != 0 )
   {
      CurDbfRec = 0L;
      return rc;
   }

   /* traverse down the tree until it hits a leaf */
   while( GetLeftNodeNo( 0, CurNode ))	/* while interior node */
   {
      TempNodeNo = GetLeafFromInteriorNode( Tkey, Klen );
      if(( rc = GetLeafNode( TempNodeNo, 1 )) != 0 )
      {
         CurDbfRec = 0L;
         return rc;
      }
   }
   
   /* leaf level */
   for( i = 0; i < CurNode->Leaf.NoOfKeysThisNode; i++ )
   {
      rc = CompareKey(  Tkey, GetKeyData( i, CurNode ), Klen );
      if( rc == 0 ) 
      {
         CurNode->CurKeyNo = i;
         CurDbfRec = GetDbfNo( i, CurNode );
         if( RetrieveSw ) dbf->GetRecord( CurDbfRec );
         return FOUND;
      }
      else if( rc == 2 )
      {
         CurNode->CurKeyNo = i;
         CurDbfRec = GetDbfNo( i, CurNode );
         if( RetrieveSw ) dbf->GetRecord( CurDbfRec );
         return NOT_FOUND;
      }
   }
   CurNode->CurKeyNo = i;
   CurDbfRec = GetDbfNo( i, CurNode );
   if( RetrieveSw ) dbf->GetRecord( CurDbfRec );
   return NOT_FOUND;
}
/***********************************************************************/
SHORT NDX::CalcKeyLen( VOID )
{
   SHORT rc;
   ExpNode * TempNode;
   CHAR FieldName[11];
   CHAR Type;

   TempNode = dbf->xbase->GetFirstTreeNode( ExpressionTree );

   if( !TempNode ) return 0;
   if( TempNode->Type == 'd' ) return -8;
   if( TempNode->Type == 'D' )
   {
      memset( FieldName, 0x00, 11 );
      memcpy( FieldName, TempNode->NodeText, TempNode->Len );
      Type = dbf->GetFieldType( dbf->GetFieldNo( FieldName ));
      if( Type == 'N' || Type == 'F' )
         return -8;
   }

   if(( rc = dbf->xbase->ProcessExpression( ExpressionTree )) != NO_ERROR )
      return 0;

   TempNode = (ExpNode *) dbf->xbase->Pop();
   if( !TempNode ) return 0;
   rc = TempNode->DataLen;

   if( !TempNode->InTree ) dbf->xbase->FreeExpNode( TempNode );
   return rc;
}
/***********************************************************************/
SHORT NDX::CreateIndex(CHAR * IxName, CHAR * Exp, SHORT Unique, SHORT Overlay )
{
   SHORT i, NameLen, KeyLen, rc;

   NdxStatus = CLOSED;
   if( strlen( Exp ) > 488 ) return INVALID_KEY_EXPRESSION;
   if( dbf->GetDbfStatus() == 0 ) return NOT_OPEN;

   /* Get the index file name and store it in the class */
   NameLen = strlen( IxName ) + 1;

   if(( rc = dbf->NameSuffixMissing( 2, IxName )) > 0 )
      NameLen +=4;

   if(( IndexName = (char *) malloc( NameLen )) == NULL )
      return NO_MEMORY;

   memset( IndexName, 0x00, NameLen ); 
   /* copy the name to the class variable */
   strcpy( IndexName, IxName );
   
   if( rc == 1 )
      strcat( IndexName, ".ndx" );
   else if( rc == 2 )
      strcat( IndexName, ".NDX" );

   /* check if the file already exists */
   if (((ndxfp = fopen( IndexName, "r" )) != NULL ) && !Overlay )
   {
      free( IndexName );
      fclose( ndxfp );
      return FILE_EXISTS;
   }
   else if( ndxfp ) fclose( ndxfp );

   if(( ndxfp = fopen( IndexName, "w+b" )) == NULL )
   {
      free( IndexName );
      return OPEN_ERROR;
   }

   /* parse the expression */
   if(( rc = dbf->xbase->BuildExpressionTree( Exp, strlen( Exp ), dbf )) != NO_ERROR )
      return rc;
   ExpressionTree = dbf->xbase->GetTree();
   dbf->xbase->SetTreeToNull(); 

   /* build the header record */
   memset( &HeadNode, 0x00, sizeof( NdxHeadNode ));
   HeadNode.StartNode  = 1L;
   HeadNode.TotalNodes = 2L;
   HeadNode.NoOfKeys   = 1L;

   KeyLen = CalcKeyLen();

   if( KeyLen == 0 || KeyLen > 100 )       /* 100 byte key length limit */
      return INVALID_KEY;
   else if( KeyLen == -8 )
   { 
      HeadNode.KeyType = 1;                /* numeric key */
      HeadNode.KeyLen = 8; 
   }
   else
   { 
      HeadNode.KeyType = 0;                /* character key */
      HeadNode.KeyLen = KeyLen; 
   }

   HeadNode.KeysPerNode = (USHORT) ( NDX_NODE_SIZE - (2*sizeof( LONG ))) /
      (HeadNode.KeyLen + 8 );

   HeadNode.KeySize = HeadNode.KeyLen + 8;
   while(( HeadNode.KeySize % 4 ) != 0 ) HeadNode.KeySize++;  /* multiple of 4*/
   HeadNode.Unique = Unique;
   strncpy( HeadNode.KeyExpression, Exp, 488 );
   KeyBuf  = (CHAR *) malloc( HeadNode.KeyLen + 1 ); 
   KeyBuf2 = (CHAR *) malloc( HeadNode.KeyLen + 1 ); 
   memset( KeyBuf,  0x00, HeadNode.KeyLen + 1 );
   memset( KeyBuf2, 0x00, HeadNode.KeyLen + 1 );

   if(( rc = PutHeadNode( &HeadNode, ndxfp, 0 )) != 0 ) return rc;
   /* write node #1 all 0x00 */
   for( i = 0; i < NDX_NODE_SIZE; i++ )
   {
      if(( fwrite( "\x00", 1, 1, ndxfp )) != 1 )
      {
         fclose( ndxfp );
         return WRITE_ERROR;
      }
   }
   NdxStatus = OPEN;

   return dbf->AddNdxToIxList( ndx, IndexName );  
}
/***********************************************************************/
SHORT NDX::PutLeftNodeNo( SHORT RecNo, NodeLink *n, LONG NodeNo )
{
   /* This routine sets n node's leftnode number */
   NdxLeafNode *temp;
   CHAR *p;
   if( !n ) return INVALID_NODELINK;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > HeadNode.KeysPerNode)
      return INVALID_KEY;
   p = temp->KeyRecs;
   p+= RecNo * ( 8 + HeadNode.KeyLen );
   dbf->xbase->PutLong( p, NodeNo );
   return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::PutDbfNo( SHORT RecNo, NodeLink *n, LONG DbfNo )
{
   /* This routine sets n node's dbf number */

   NdxLeafNode *temp;
   CHAR *p;
   if( !n ) return INVALID_NODELINK;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > (HeadNode.KeysPerNode-1))
      return INVALID_KEY;
   p = temp->KeyRecs + 4;
   p+= RecNo * ( 8 + HeadNode.KeyLen );
   dbf->xbase->PutLong( p, DbfNo );
   return NO_ERROR;
}
/************************************************************************/
SHORT NDX::PutLeafNode( LONG l, NodeLink *n )
{
   if(( fseek( ndxfp, l * NDX_NODE_SIZE , SEEK_SET )) != 0 )
   {
      fclose( ndxfp );
      return( SEEK_ERROR );
   }

   dbf->xbase->PutLong( Node, n->Leaf.NoOfKeysThisNode );

   if(( fwrite( Node, 4, 1, ndxfp )) != 1 )
   {
      fclose( ndxfp );
      return WRITE_ERROR;
   }

   if(( fwrite( &n->Leaf.KeyRecs, NDX_NODE_SIZE-4, 1, ndxfp )) != 1 )
   {
      fclose( ndxfp );
      return WRITE_ERROR;
   }
   return 0;   
}
/************************************************************************/
SHORT NDX::PutHeadNode( NdxHeadNode * Head, FILE * f, SHORT UpdateOnly )
{
   CHAR buf[4];
 
   if(( fseek( f, 0L, SEEK_SET )) != 0 )
   {
      fclose( f );
      return( SEEK_ERROR );
   }

   memset( buf, 0x00, 4 );
   dbf->xbase->PutLong( buf, Head->StartNode );
   if(( fwrite( &buf, 4, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   memset( buf, 0x00, 4 );
   dbf->xbase->PutLong( buf, Head->TotalNodes );
   if(( fwrite( &buf, 4, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   memset( buf, 0x00, 4 );
   dbf->xbase->PutLong( buf, Head->NoOfKeys );
   if(( fwrite( &buf, 4, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }
   if( UpdateOnly )
      return NO_ERROR;

   memset( buf, 0x00, 2 );
   dbf->xbase->PutLong( buf, Head->KeyLen );
   if(( fwrite( &buf, 2, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   memset( buf, 0x00, 2 );
   dbf->xbase->PutLong( buf, Head->KeysPerNode );
   if(( fwrite( &buf, 2, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   memset( buf, 0x00, 2 );
   dbf->xbase->PutLong( buf, Head->KeyType );
   if(( fwrite( &buf, 2, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   memset( buf, 0x00, 4 );
   dbf->xbase->PutLong( buf, Head->KeySize );
   if(( fwrite( &buf, 4, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }

   if(( fwrite( &Head->Unknown2, 490, 1, f )) != 1 )
   {
      fclose( f );
      return WRITE_ERROR;
   }
   return 0;   
}
/************************************************************************/
SHORT NDX::PutKeyData( SHORT RecNo, NodeLink *n )
{
   /* This routine copies the KeyBuf data into NodeLink n */
   NdxLeafNode *temp;
   CHAR *p;
   SHORT i;
   if( !n ) return INVALID_NODELINK;
   temp = &n->Leaf;
   if( RecNo < 0 || RecNo > (HeadNode.KeysPerNode-1))
      return INVALID_KEY;
   p = temp->KeyRecs + 8;
   p+= RecNo * ( 8 + HeadNode.KeyLen );
   for( i = 0; i < HeadNode.KeyLen; i++ )
   {
      *p = KeyBuf[i];
      p++;
   }
   return NO_ERROR;
}
/************************************************************************/
SHORT NDX::PutKeyInNode( NodeLink * n, SHORT pos, LONG d, LONG l, SHORT w )
{
   SHORT i;

   /* check the node */
   if( !n ) return INVALID_NODELINK;
   if( pos < 0 || pos > HeadNode.KeysPerNode ) return INVALID_RECORD;
   if( n->Leaf.NoOfKeysThisNode >= HeadNode.KeysPerNode ) return NODE_FULL;

   /* if key movement, save the original key */
   if( pos < n->Leaf.NoOfKeysThisNode )
      memcpy( KeyBuf2, KeyBuf, HeadNode.KeyLen + 1);

   /* if interior node, handle the right most left node no */
   if( GetLeftNodeNo( 0, n ))
      PutLeftNodeNo( n->Leaf.NoOfKeysThisNode+1, n,
         GetLeftNodeNo( n->Leaf.NoOfKeysThisNode, n ));

   for( i = n->Leaf.NoOfKeysThisNode; i > pos; i-- )
   {
      memcpy( KeyBuf, GetKeyData(i-1,n), HeadNode.KeyLen );  
      PutKeyData( i, n );
      PutDbfNo( i, n, GetDbfNo(i-1,n));
      PutLeftNodeNo(i, n, GetLeftNodeNo(i-1,n));
   }
   /* put new key in node */

   if( pos < n->Leaf.NoOfKeysThisNode )
      memcpy( KeyBuf, KeyBuf2, HeadNode.KeyLen + 1);

   PutKeyData( pos, n );
   PutDbfNo( pos, n, d );
   PutLeftNodeNo( pos, n, l );
   n->Leaf.NoOfKeysThisNode++;
   if( w )
      return PutLeafNode( n->NodeNo, n );
   else
      return 0;
}
/************************************************************************/
SHORT NDX::SplitLeafNode( NodeLink *n1, NodeLink *n2, SHORT pos, LONG d )
{
   SHORT i,j,rc;

   if( !n1 || !n2 ) return INVALID_NODELINK;
   if( pos < 0 || pos > HeadNode.KeysPerNode ) return INVALID_RECORD;

   if( pos < HeadNode.KeysPerNode ) /* if it belongs in node */
   {
      /* save the original key */
      memcpy( KeyBuf2, KeyBuf, HeadNode.KeyLen + 1);
      PutKeyData( HeadNode.KeysPerNode, n2 ); 
      for( j = 0,i = pos; i < n1->Leaf.NoOfKeysThisNode; j++,i++ )
      {
         memcpy( KeyBuf, GetKeyData( i, n1 ), HeadNode.KeyLen );
         PutKeyData   ( j, n2 );
         PutDbfNo     ( j, n2, GetDbfNo  ( i, n1 ));
         n2->Leaf.NoOfKeysThisNode++; 
      }

      /* restore original key */
      memcpy( KeyBuf, KeyBuf2, HeadNode.KeyLen + 1);

      /* update original leaf */
      PutKeyData( pos, n1 );
      PutDbfNo  ( pos, n1, d );
      n1->Leaf.NoOfKeysThisNode = pos+1;
   }         
   else    /* put the key in a new node because it doesn't fit in the CurNode*/
   {
      PutKeyData   ( 0, n2 );
      PutDbfNo     ( 0, n2, d );
      n2->Leaf.NoOfKeysThisNode++; 
   }
   if(( rc = PutLeafNode( n1->NodeNo, n1 )) != 0 )
       return rc;
   if(( rc = PutLeafNode( n2->NodeNo, n2 )) != 0 )
       return rc;
   return 0;
}
/************************************************************************/
SHORT NDX::SplitINode( NodeLink *n1, NodeLink *n2, LONG t )
                   /* parent, tempnode, tempnodeno */
{
   SHORT i,j,rc;
   NodeLink * SaveNodeChain;
   NodeLink * SaveCurNode;

   /* if not at the end of the node shift everthing to the right */
   if( n1->CurKeyNo+1 < HeadNode.KeysPerNode )   /* this clause appears to work */
   {
      if( CurNode->NodeNo == HeadNode.StartNode ) cout << "\nHead node ";
   
      for( j = 0,i = n1->CurKeyNo+1; i < n1->Leaf.NoOfKeysThisNode; i++,j++ ) 
      {
         memcpy( KeyBuf, GetKeyData( i, n1 ), HeadNode.KeyLen ); 
         PutKeyData( j, n2 );
         PutLeftNodeNo( j, n2, GetLeftNodeNo( i, n1 ));
      }
      PutLeftNodeNo( j, n2, GetLeftNodeNo( i, n1 ));
   
      n2->Leaf.NoOfKeysThisNode = n1->Leaf.NoOfKeysThisNode -
         n1->CurKeyNo - 1;
      n1->Leaf.NoOfKeysThisNode = n1->Leaf.NoOfKeysThisNode - 
         n2->Leaf.NoOfKeysThisNode;

      /* attach the new leaf to the original parent */
      SaveNodeChain = NodeChain;
      NodeChain = NULL;
      SaveCurNode = CurNode;
      GetLastKey( CurNode->NodeNo, 0 );
      memcpy( KeyBuf, GetKeyData( CurNode->CurKeyNo, CurNode ),HeadNode.KeyLen);
      ReleaseNodeMemory( NodeChain );
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;
      PutKeyData( n1->CurKeyNo, n1 );
      PutLeftNodeNo( n1->CurKeyNo + 1, n1, t );
   }
   else if( n1->CurKeyNo + 1 == HeadNode.KeysPerNode )
   {
      SaveNodeChain = NodeChain;
      NodeChain = NULL;
      SaveCurNode = CurNode;
      GetLastKey( t, 0 );
      memcpy( KeyBuf,GetKeyData(CurNode->CurKeyNo,CurNode), HeadNode.KeyLen ); 
      PutKeyData( 0, n2 );
      PutLeftNodeNo( 0, n2, t ); 
      PutLeftNodeNo( 1, n2, GetLeftNodeNo( n1->Leaf.NoOfKeysThisNode, n1 ));
      ReleaseNodeMemory( NodeChain );
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;
      n2->Leaf.NoOfKeysThisNode = 1;
      n1->Leaf.NoOfKeysThisNode--;
   }
   else /* pos = HeadNode.KeysPerNode */
   {
      SaveNodeChain = NodeChain;
      NodeChain = NULL;
      SaveCurNode = CurNode;
      GetLastKey( CurNode->NodeNo, 0 );
      memcpy( KeyBuf, GetKeyData( CurNode->CurKeyNo, CurNode ), HeadNode.KeyLen );
      ReleaseNodeMemory( NodeChain );
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;

      PutKeyData( 0, n2 );
      PutLeftNodeNo( 0, n2, CurNode->NodeNo );
      PutLeftNodeNo( 1, n2, t );
      n2->Leaf.NoOfKeysThisNode = 1;
      n1->Leaf.NoOfKeysThisNode--;
   }
   n2->NodeNo = HeadNode.TotalNodes++; 
   if((rc = PutLeafNode( n1->NodeNo,n1 )) != 0) return rc;
   if((rc = PutLeafNode( n2->NodeNo,n2 )) != 0) return rc;
   return 0;
}
/************************************************************************/
SHORT NDX::CreateKey( SHORT RecBufSw, SHORT KeyBufSw )
{ 
   /* RecBufSw   0   Use RecBuf    */
   /*            1   Use RecBuf2   */
   /* KeyBufSw   0   Use KeyBuf    */
   /*            1   Use KeyBuf2   */

   SHORT rc;
   ExpNode * TempNode;

   if(( rc = dbf->xbase->ProcessExpression( ExpressionTree, RecBufSw )) != NO_ERROR )
      return rc;
   TempNode = (ExpNode *) dbf->xbase->Pop();
   if( !TempNode ) return INVALID_KEY;

   if( KeyBufSw )
   {
      if( HeadNode.KeyType == 1 )    /* numeric key   */
         dbf->xbase->PutDouble( KeyBuf2, TempNode->DoubResult );
      else                           /* character key */
      {
         memset( KeyBuf2, 0x00, HeadNode.KeyLen + 1 );
         memcpy( KeyBuf2, TempNode->Result, TempNode->DataLen );
      }
   }
   else
   {
      if( HeadNode.KeyType == 1 )    /* numeric key   */
         dbf->xbase->PutDouble( KeyBuf, TempNode->DoubResult );
      else                           /* character key */
      {
         memset( KeyBuf, 0x00, HeadNode.KeyLen + 1 );
         memcpy( KeyBuf, TempNode->Result, TempNode->DataLen );
      }
   }
   if( !TempNode->InTree ) dbf->xbase->FreeExpNode( TempNode );
   return 0;
}
/************************************************************************/
SHORT NDX::AddKey( LONG DbfRec )
{
 /* This routine assumes KeyBuf contains the contents of the index to key */

   SHORT i,rc;
   NodeLink * TempNode;   
   NodeLink * Tparent;
   LONG TempNodeNo = 0L;          /* new, unattached leaf node no */
   NodeLink * SaveNodeChain;
   NodeLink * SaveCurNode;

   /* find node key belongs in */
   rc = FindKey( KeyBuf, HeadNode.KeyLen, 0 );
   if( rc == FOUND && HeadNode.Unique )
      return KEY_NOT_UNIQUE;

   /* 1.02 next statement moves to key match w/ space in nodes to reduce splits */
   if( !HeadNode.Unique && rc == FOUND &&
        HeadNode.KeysPerNode == CurNode->Leaf.NoOfKeysThisNode )
   {
      if(( rc = CloneNodeChain()) != NO_ERROR ) return rc;
      if(( rc = GetNextKey( 0 )) != NO_ERROR )
         UncloneNodeChain();
      while( HeadNode.KeysPerNode == CurNode->Leaf.NoOfKeysThisNode &&
             rc == NO_ERROR && 
             (CompareKey( KeyBuf, GetKeyData( CurNode->CurKeyNo, CurNode ),
                HeadNode.KeyLen ) == 0 )
           )
         if(( rc = GetNextKey( 0 )) != NO_ERROR )
            UncloneNodeChain();
   }

   /* update header node */
   HeadNode.NoOfKeys++;

   /************************************************/
   /* section A - if room in node, add key to node */
   /************************************************/

   if( CurNode->Leaf.NoOfKeysThisNode < HeadNode.KeysPerNode )
   {
      if(( rc = PutKeyInNode( CurNode,CurNode->CurKeyNo,DbfRec,0L,1)) != 0)
         return rc;
      if(( rc = PutHeadNode( &HeadNode, ndxfp, 1 )) != 0)
         return rc;
      return NO_ERROR;
   }   

   /***********************************************************************/
   /* section B - split leaf node if full and put key in correct position */
   /***********************************************************************/

   TempNode = GetNodeMemory();
   TempNode->NodeNo = HeadNode.TotalNodes++;

   rc = SplitLeafNode( CurNode, TempNode, CurNode->CurKeyNo, DbfRec );
   if( rc ) return rc;

   TempNodeNo = TempNode->NodeNo;
   ReleaseNodeMemory( TempNode );

   /*****************************************************/
   /* section C go up tree splitting nodes as necessary */
   /*****************************************************/

   Tparent = CurNode->PrevNode;

   while( Tparent && 
          Tparent->Leaf.NoOfKeysThisNode >= HeadNode.KeysPerNode )
   {
      TempNode = GetNodeMemory();
      if( !TempNode ) return NO_MEMORY;

      rc = SplitINode( Tparent, TempNode, TempNodeNo );
      if( rc ) return rc;

      TempNodeNo = TempNode->NodeNo;
      ReleaseNodeMemory( TempNode );
      ReleaseNodeMemory( CurNode );
      CurNode = Tparent;
      CurNode->NextNode = NULL;
      Tparent = CurNode->PrevNode;
   }

   /************************************************************/
   /* Section D  if CurNode is split root, create new root     */
   /************************************************************/

   /* at this point
       CurNode = The node that was just split
       TempNodeNo = The new node split off from CurNode */

   if(CurNode->NodeNo == HeadNode.StartNode ) 
   {
      TempNode = GetNodeMemory();
      if( !TempNode ) return NO_MEMORY;

      SaveNodeChain = NodeChain;
      NodeChain = NULL;
      SaveCurNode = CurNode;
      GetLastKey( CurNode->NodeNo, 0 );
      memcpy( KeyBuf, GetKeyData( CurNode->CurKeyNo,CurNode ),HeadNode.KeyLen );

      ReleaseNodeMemory( NodeChain );
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;

      PutKeyData( 0, TempNode );
      PutLeftNodeNo( 0, TempNode, CurNode->NodeNo );
      PutLeftNodeNo( 1, TempNode, TempNodeNo );
      TempNode->NodeNo = HeadNode.TotalNodes++;
      TempNode->Leaf.NoOfKeysThisNode++;
      HeadNode.StartNode = TempNode->NodeNo;
      rc = PutLeafNode( TempNode->NodeNo, TempNode );
      if( rc ) return rc;
      rc = PutHeadNode( &HeadNode, ndxfp, 1 );
      if( rc ) return rc;
      ReleaseNodeMemory( TempNode );
      return NO_ERROR;
   }
   /**********************************/
   /* Section E  make room in parent */
   /**********************************/
   for( i = Tparent->Leaf.NoOfKeysThisNode; i > Tparent->CurKeyNo; i-- )
   {
      memcpy( KeyBuf, GetKeyData( i-1, Tparent ), HeadNode.KeyLen );
      PutKeyData( i, Tparent );
      PutLeftNodeNo( i+1, Tparent, GetLeftNodeNo( i, Tparent ));
   }

   /* put key in parent */

   SaveNodeChain = NodeChain;
   NodeChain = NULL;
   SaveCurNode = CurNode;
   GetLastKey( CurNode->NodeNo, 0 );

   memcpy( KeyBuf,GetKeyData( CurNode->CurKeyNo, CurNode ), HeadNode.KeyLen );

   ReleaseNodeMemory( NodeChain );
   NodeChain = SaveNodeChain;
   CurNode = SaveCurNode;

   PutKeyData( i, Tparent );
   PutLeftNodeNo( i+1, Tparent, TempNodeNo );
   Tparent->Leaf.NoOfKeysThisNode++;
   rc = PutLeafNode( Tparent->NodeNo, Tparent );
   if( rc ) return rc;
   rc = PutHeadNode( &HeadNode, ndxfp, 1 );
   if( rc ) return rc;
 
   return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::RemoveKeyFromNode( SHORT pos, NodeLink *n )
{
   SHORT i;

   /* check the node */
   if( !n ) return INVALID_NODELINK;
   if( pos < 0 || pos > HeadNode.KeysPerNode ) return INVALID_KEY;

   for( i = pos; i < n->Leaf.NoOfKeysThisNode-1; i++ )
   {
      memcpy( KeyBuf, GetKeyData( i+1, n), HeadNode.KeyLen );
   
      PutKeyData( i, n );
      PutDbfNo( i, n, GetDbfNo( i+1, n ));
      PutLeftNodeNo( i, n, GetLeftNodeNo( i+1, n ));
   }   
   PutLeftNodeNo( i, n, GetLeftNodeNo( i+1, n ));
   n->Leaf.NoOfKeysThisNode--;
   /* if last key was deleted, decrement CurKeyNo */
   if( n->CurKeyNo > n->Leaf.NoOfKeysThisNode )
      n->CurKeyNo--;
   return PutLeafNode( n->NodeNo, n );
}   
/***********************************************************************/
SHORT NDX::UpdateParentKey( NodeLink * n )
{
/* this routine goes backwards thru the node chain looking for a parent
   node to update */

   NodeLink * TempNode;
   
   if( !n ) return INVALID_NODELINK;
   if( !GetDbfNo( 0, n ))
   {
      cout << "Fatal index error - Not a leaf node" << n->NodeNo << "\n";
      exit(0);
      return NOT_LEAFNODE;
   }

   TempNode = n->PrevNode;

   while( TempNode )
   {
      if( TempNode->CurKeyNo < TempNode->Leaf.NoOfKeysThisNode )
      {
         memcpy(KeyBuf,GetKeyData(n->Leaf.NoOfKeysThisNode-1,n),HeadNode.KeyLen);
         PutKeyData( TempNode->CurKeyNo, TempNode );
         return PutLeafNode( TempNode->NodeNo, TempNode );
      }
      TempNode = TempNode->PrevNode;
   }
   return NO_ERROR;
}
/***********************************************************************/
/* This routine queues up a list of nodes which have been emptied      */
VOID NDX::UpdateDeleteList( NodeLink *n )
{
   n->NextNode = DeleteChain;
   DeleteChain = n;
}
/***********************************************************************/
/* Delete nodes from the node list - for now we leave the empty nodes  */
/* dangling in the file. Eventually we will remove nodes from the file */

VOID NDX::ProcessDeleteList( VOID )
{
   if( DeleteChain )
   {
      ReleaseNodeMemory( DeleteChain );
      DeleteChain = NULL;
   }
}
/***********************************************************************/
SHORT NDX::KeyWasChanged( VOID )
{
   CreateKey( 0, 0 );            /* use KeyBuf,  RecBuf    */
   CreateKey( 1, 1 );            /* use KeyBuf2, RecBuf2   */
   if( CompareKey( KeyBuf, KeyBuf2, HeadNode.KeyLen ) != 0 )
      return 1;
   else
      return 0;
}
/***********************************************************************/
NodeLink * NDX::LeftSiblingHasSpace( NodeLink * n )
{
   NodeLink * TempNode;
   NodeLink * SaveCurNode;

   /* returns a Nodelink to NodeLink n's left sibling if it has space */

   /* if left most node in parent return NULL */
   if( n->PrevNode->CurKeyNo == 0 )  
      return NULL;

   SaveCurNode = CurNode;
   GetLeafNode( GetLeftNodeNo( n->PrevNode->CurKeyNo-1, n->PrevNode ), 2 ); 
   if( CurNode->Leaf.NoOfKeysThisNode < HeadNode.KeysPerNode )
   {
      TempNode = CurNode;  
      CurNode = SaveCurNode;
      TempNode->PrevNode = n->PrevNode;
      return TempNode;
   }
   else  /* node is already full */
   {
      ReleaseNodeMemory( CurNode );
      CurNode = SaveCurNode;
      return NULL;
   }
}
/***********************************************************************/
NodeLink * NDX::RightSiblingHasSpace( NodeLink * n )
{
   /* returns a Nodelink to NodeLink n's right sibling if it has space */

   NodeLink * TempNode;
   NodeLink * SaveCurNode;

   /* if left most node in parent return NULL */
   if( n->PrevNode->CurKeyNo >= n->PrevNode->Leaf.NoOfKeysThisNode )  
      return NULL;

   SaveCurNode = CurNode;
   /* point curnode to right sib*/
   GetLeafNode( GetLeftNodeNo( n->PrevNode->CurKeyNo+1, n->PrevNode ), 2 ); 

   if( CurNode->Leaf.NoOfKeysThisNode < HeadNode.KeysPerNode )
   {
      TempNode = CurNode;  
      CurNode = SaveCurNode;
      TempNode->PrevNode = n->PrevNode;
      return TempNode;
   }
   else /* node is already full */
   {
      ReleaseNodeMemory( CurNode );
      CurNode = SaveCurNode;
      return NULL;
   }
}
/***********************************************************************/
SHORT NDX::MoveToRightNode( NodeLink * n, NodeLink * Right )
{
   SHORT j;
   NodeLink * TempNode;
   NodeLink * SaveCurNode;
   NodeLink * SaveNodeChain;

   if( n->CurKeyNo == 0 )
   {
      j = 1;
      SaveNodeChain = NodeChain;
      SaveCurNode = CurNode;
      NodeChain = NULL;
      GetLastKey( n->NodeNo, 0 ); 
      memcpy( KeyBuf, GetKeyData( CurNode->CurKeyNo, CurNode),HeadNode.KeyLen);
      ReleaseNodeMemory( NodeChain );
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;
   }
   else
   {
      j = 0;
      memcpy( KeyBuf, GetKeyData( j, n ), HeadNode.KeyLen);
   }
   PutKeyInNode( Right, 0, 0L, GetLeftNodeNo( j, n ), 1 );
   ReleaseNodeMemory( Right );
   TempNode = n;
   CurNode = n->PrevNode;
   n = n->PrevNode;
   n->NextNode = NULL;
   UpdateDeleteList( TempNode );
   DeleteSibling( n );
   return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::MoveToLeftNode( NodeLink * n, NodeLink * Left )
{
   SHORT j, rc;
   NodeLink * SaveNodeChain;
   NodeLink * TempNode;

   if( n->CurKeyNo == 0 )
      j = 1;
   else
      j = 0;
   
   /* save the original node chain */
   SaveNodeChain = NodeChain;
   NodeChain = NULL;

   /* determine new right most key for left node */
   GetLastKey( Left->NodeNo, 0 );
   memcpy( KeyBuf, GetKeyData( CurNode->CurKeyNo, CurNode ), HeadNode.KeyLen);
   ReleaseNodeMemory( NodeChain );
   NodeChain = NULL;			/* for next GetLastKey */
   PutKeyData( Left->Leaf.NoOfKeysThisNode, Left); 
   PutLeftNodeNo( Left->Leaf.NoOfKeysThisNode+1, Left, GetLeftNodeNo( j,n ));
   Left->Leaf.NoOfKeysThisNode++;
   Left->CurKeyNo = Left->Leaf.NoOfKeysThisNode;
   if(( rc = PutLeafNode( Left->NodeNo, Left )) != 0 ) return rc;

   n->PrevNode->NextNode = NULL;
   UpdateDeleteList( n );

   /* get the new right most key for left to update parents */
   GetLastKey( Left->NodeNo, 0 );
   
   /* assemble the chain */
   TempNode = Left->PrevNode;
   TempNode->CurKeyNo--;
   NodeChain->PrevNode = Left->PrevNode;
   UpdateParentKey( CurNode );
   ReleaseNodeMemory( NodeChain );
   ReleaseNodeMemory( Left );
   CurNode = TempNode;
   NodeChain = SaveNodeChain;
   TempNode->CurKeyNo++;    
   DeleteSibling( TempNode );
   return NO_ERROR;
}
/***********************************************************************/
SHORT NDX::DeleteSibling( NodeLink * n )
{
   NodeLink * Left;
   NodeLink * Right;
   NodeLink * SaveCurNode;
   NodeLink * SaveNodeChain;
   NodeLink * TempNode;
   SHORT  rc;

   /* this routine deletes sibling CurRecNo out of NodeLink n */
   if( n->Leaf.NoOfKeysThisNode > 1 )
   {  
      RemoveKeyFromNode( n->CurKeyNo, n );
      if( n->CurKeyNo == n->Leaf.NoOfKeysThisNode )
      {
         SaveNodeChain = NodeChain;
         SaveCurNode = CurNode;
         NodeChain = NULL;
         GetLastKey( n->NodeNo, 0 );
         /* assemble the node chain */
         TempNode = NodeChain->NextNode;
         NodeChain->NextNode = NULL;
         ReleaseNodeMemory( NodeChain );
         TempNode->PrevNode = n;
         UpdateParentKey( CurNode );
         /* take it back apart */
         ReleaseNodeMemory( TempNode );
         NodeChain = SaveNodeChain;
         CurNode = SaveCurNode;
      }
   }
   else if( n->NodeNo == HeadNode.StartNode )
   {
      /* get here if root node and only one child remains */
      /* make remaining node the new root */
      if( n->CurKeyNo == 0 )   
         HeadNode.StartNode = GetLeftNodeNo( 1, n );
      else
         HeadNode.StartNode = GetLeftNodeNo( 0, n );
      UpdateDeleteList( n );
      NodeChain = NULL;
      CurNode = NULL;
   }
   else if (( Left = LeftSiblingHasSpace( n )) != NULL )
   {
      return MoveToLeftNode( n, Left );
   }
   else if (( Right = RightSiblingHasSpace( n )) != NULL )
   {
      return MoveToRightNode( n, Right );
   }
   /* else if left sibling exists */   
   else if( n->PrevNode->CurKeyNo > 0 )
   {
      /* move right branch from left sibling to this node */  
      SaveCurNode = CurNode;
      SaveNodeChain = NodeChain;
      NodeChain = NULL;
      GetLeafNode( GetLeftNodeNo( n->PrevNode->CurKeyNo-1, n->PrevNode ), 2 );
      Left = CurNode;
      Left->PrevNode = SaveCurNode->PrevNode;
      GetLastKey( Left->NodeNo, 0 );
      
      strncpy( KeyBuf, GetKeyData( CurNode->CurKeyNo,CurNode),HeadNode.KeyLen );
      if( n->CurKeyNo == 1 )
         PutLeftNodeNo( 1, n, GetLeftNodeNo( 0, n ));
      PutKeyData( 0, n );
      PutLeftNodeNo( 0, n, GetLeftNodeNo( Left->Leaf.NoOfKeysThisNode, Left ));
      if(( rc = PutLeafNode( n->NodeNo, n )) != NO_ERROR ) return rc;
      SaveCurNode = n->PrevNode;
      SaveCurNode->NextNode = NULL;
      ReleaseNodeMemory( n );
      Left->Leaf.NoOfKeysThisNode--;
      if(( rc = PutLeafNode( Left->NodeNo, Left )) != NO_ERROR ) return rc;
      /* rebuild left side of tree */
      GetLastKey( Left->NodeNo, 0 );
      NodeChain->PrevNode = SaveCurNode;
      SaveCurNode->CurKeyNo--;
      UpdateParentKey( CurNode );
      ReleaseNodeMemory( NodeChain );
      ReleaseNodeMemory( Left );
      CurNode = SaveCurNode;
      NodeChain = SaveNodeChain;
   }
   /* right sibling must exist */
   else if( n->PrevNode->CurKeyNo <= n->PrevNode->Leaf.NoOfKeysThisNode )
   {
      /* move left branch from left sibling to this node */   
      SaveCurNode = CurNode;
      SaveNodeChain = NodeChain;
      NodeChain = NULL;

      /* move the left node number one to the left if necessary */
      if( n->CurKeyNo == 0 )
      {
         PutLeftNodeNo( 0, n, GetLeftNodeNo( 1, n ));
         GetLastKey( GetLeftNodeNo( 0, n ), 0 );
         memcpy(KeyBuf,GetKeyData(CurNode->CurKeyNo,CurNode),HeadNode.KeyLen);
         PutKeyData( 0, n );
         ReleaseNodeMemory( NodeChain );
         NodeChain = NULL;
      }
      GetLeafNode( GetLeftNodeNo( n->PrevNode->CurKeyNo+1, n->PrevNode ), 2 );
      
      /* put leftmost node number from right node in this node */
      PutLeftNodeNo( 1, n, GetLeftNodeNo( 0, CurNode ));      
      if(( rc = PutLeafNode( n->NodeNo, n )) != NO_ERROR ) return rc;

      /* remove the key from the right node */
      RemoveKeyFromNode( 0, CurNode );
      if(( rc = PutLeafNode( CurNode->NodeNo, CurNode )) != NO_ERROR ) return rc;
      ReleaseNodeMemory( CurNode );

      /* update new parent key value */
      GetLastKey( n->NodeNo, 0 );
      NodeChain->PrevNode = n->PrevNode;
      UpdateParentKey( CurNode );
      ReleaseNodeMemory( NodeChain );
      
      NodeChain = SaveNodeChain;
      CurNode = SaveCurNode;
   }
   else
   {
      /* this should never be true-but could be if 100 byte limit is ignored*/
      cout << "Fatal index error\n";
      exit(0);
   }
   return NO_ERROR;   
}
/***********************************************************************/
SHORT NDX::DeleteKey( LONG DbfRec )
{
/* this routine assumes the key to be deleted is in KeyBuf */

   NodeLink * TempNode;
   SHORT rc;

   if( HeadNode.KeyType != 0x00 ) return INVALID_KEY_TYPE;
   if(( rc = FindKey( KeyBuf, DbfRec )) != FOUND )
      return rc;

   /* found the record to delete at this point */
   HeadNode.NoOfKeys--;

   /* delete the key from the node                                    */
   if(( rc = RemoveKeyFromNode( CurNode->CurKeyNo, CurNode )) != 0 )
      return rc;

   /* if root node, we are done */
   if( !( CurNode->NodeNo == HeadNode.StartNode ))
   {
      /* if leaf node now empty */
      if( CurNode->Leaf.NoOfKeysThisNode == 0 )
      {
         TempNode = CurNode->PrevNode;
         TempNode->NextNode = NULL;
         UpdateDeleteList( CurNode );
         CurNode = TempNode;
         DeleteSibling( CurNode );
         ProcessDeleteList();
      }

      /* if last key of leaf updated, update key in parent node */
      /* this logic updates the correct parent key              */

      else if( CurNode->CurKeyNo == CurNode->Leaf.NoOfKeysThisNode )
      {
         UpdateParentKey( CurNode );
      }
   }
   if(( rc = PutHeadNode( &HeadNode, ndxfp, 1 )) != 0 )
      return rc;
   return NO_ERROR;
}
/************************************************************************/
#ifdef XBASE_DEBUG
SHORT NDX::CheckNdxIntegrity( SHORT option )
{
   /* if option = 1, print out some stats */

   SHORT rc;
   LONG ctr = 1L;

   rc = dbf->GetRecord( ctr );
   while( ctr < dbf->NoOfRecords() )
   {
      ctr++;
      if( option ) cout << "\nChecking Record " << ctr;
      if(!dbf->RecordDeleted())      
      {
         CreateKey( 0, 0 );
         rc = FindKey( KeyBuf, dbf->GetCurRecNo());
         if( rc != FOUND )
         {
            if( option )
            {
               cout << "\nRecord number " << dbf->GetCurRecNo() <<  " Not Found\n";
               cout << "Key = " << KeyBuf << "\n";
            }
            return rc;
         }
      }
      if(( rc = dbf->GetRecord( ctr )) != NO_ERROR )
         return rc;
   }
   cout << "Exiting with rc = " << rc << "\n";
   if( option )
      cout << "\nTotal records checked = " << ctr << "\n";

   return NO_ERROR;
}
#endif
/***********************************************************************/
SHORT NDX::ReIndex( VOID )
{
   /* this method assumes the index has been locked in exclusive mode */

   LONG l;
   SHORT rc, i, NameLen;
   NdxHeadNode TempHead;
   FILE *t;
   CHAR * TempName;

   memcpy( &TempHead, &HeadNode, sizeof( struct NdxHeadNode ));

   TempHead.NoOfKeys = 1L;
   TempHead.TotalNodes = 2L;
   TempHead.StartNode = 1L;

   if(( rc = dbf->xbase->DirectoryExistsInName( IndexName )) > 0 )
      NameLen = rc + 13;
   else
      NameLen = 13;

   if(( TempName = (CHAR *) malloc( NameLen )) == NULL )
      return NO_MEMORY;

   memset( TempName, 0x00, NameLen );
   if( rc )
   {
      strncpy( TempName, IndexName, rc );
      strcat( TempName, "TEMPFILE.NDX" );
   }
   else
   {
      strcpy( TempName, "TEMPFILE.NDX" );
   }
 
   if(( t = fopen( TempName, "w+b" )) == NULL )
   {
      free( TempName );
      return OPEN_ERROR;
   }
   free( TempName );               /*  ???? */

   if(( rc = PutHeadNode( &TempHead, t, 0 )) != 0 )
   {
      fclose( t );
      remove( "TEMPFILE.NDX" );
      return rc;
   }

   for( i = 0; i < NDX_NODE_SIZE; i++ )
   {
      if(( fwrite( "\x00", 1, 1, t )) != 1 )
      {
         fclose( t );
         remove( "TEMPFILE.NDX" );
         return WRITE_ERROR;
      }
   }

   if( fclose( ndxfp ) != 0 )
      return CLOSE_ERROR;

   if( fclose( t ) != 0 )
      return CLOSE_ERROR;

   if( remove( IndexName ) != 0 )
      return CLOSE_ERROR;
    
   if( rename( "TEMPFILE.NDX", IndexName ) != 0 )
      return WRITE_ERROR;

   if(( ndxfp = fopen( IndexName, "r+b" )) == NULL )
      return OPEN_ERROR;

   dbf->AutoLockOff();

   for( l = 1; l <= dbf->NoOfRecords(); l++ )
   {
      if(( rc = dbf->GetRecord(l)) != NO_ERROR )
         return rc;

      /* Create the key */
      CreateKey( 0, 0 );

      /* add key to index */
      if(( rc = AddKey( l )) != NO_ERROR )
         return rc;
   } 
   dbf->AutoLockOn();
   return NO_ERROR;
}
#endif     /* NDX */
