/*  memo.cpp 

    Xbase project source code

    This file contains the basic Xbase routines for handling
    dBASE III+ and dBASE IV style memo .dbt files

    Copyright (C) 1997  StarTech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0   10/10/97   - Initial release of software
    V 1.5   1/2/98     - Added memo field support
    V 1.6a  4/1/98     - Added expression support
    V 1.6b  4/8/98     - Numeric index keys
    V 1.7.1 5/25/98    - Added support for dBase III+ type memo files
*/

#include "stdafx.h"
#include <stdio.h>
#include "xbase.h"

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif


#ifdef MEMO_FIELDS

/************************************************************************/
SHORT DBF::SetMemoBlockSize( SHORT BlockSize )
{
   if( MemoHeader.Version == 0x03 ) return NO_ERROR;
   if( BlockSize % 512 != 0 ) return INVALID_BLOCK_SIZE;
   MemoHeader.BlockSize = BlockSize;
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::GetDbtHeader( SHORT Option )
{
   CHAR *p;
   SHORT i;
   CHAR MemoBlock[24];

   /*  Option = 0  -->  read only first four bytes
                1  -->  read the entire thing  */

   if( !mfp ) 
      return NOT_OPEN;

   if( fseek( mfp, 0, SEEK_SET )) 
      return SEEK_ERROR;

   if(( fread( MemoBlock, 24, 1, mfp )) != 1 )
      return READ_ERROR;

   p = MemoBlock;
   MemoHeader.NextBlock = xbase->GetLong( p ); 
   if( MemoHeader.Version == 0x03 || Option == 0 )
      return NO_ERROR;
 
   /* version IV stuff follows */
   p+=8;
   for( i = 0; i < 8; i++, p++ ) 
      MemoHeader.FileName[i] = *p;
   MemoHeader.Version  = *p;
   p+=4;
   MemoHeader.BlockSize = xbase->GetShort( p ); 
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::OpenMemoFile( VOID )
{
   LONG  Size, NewSize, l;
   SHORT len, rc;
   CHAR  lb;     /* last byte */

   len = strlen( DatabaseName );
   len--;
   lb = DatabaseName[len];
   if( lb == 'F' )
      DatabaseName[len] = 'T';
   else if( lb == 'f' )
      DatabaseName[len] = 't';
   else
      return INVALID_NAME;

   if(( mfp = fopen( DatabaseName, "r+b" )) == NULL )
   {
      DatabaseName[len] = lb;
      return OPEN_ERROR;
   }

   DatabaseName[len] = lb;
   if(( rc = GetDbtHeader(1)) != 0 )
   {
      fclose( mfp );
      return rc;
   }

   len = GetMemoBlockSize();
   if( len == 0 || ((len % 512) != 0 ))
   {
      fclose( mfp );
      return INVALID_BLOCK_SIZE;
   }

   /* logic to verify file size is a multiple of block size */
   if(( rc = fseek( mfp, 0, SEEK_END )) != 0 )
   {
      fclose( mfp );
      return SEEK_ERROR;
   }

   /* if the file is not a multiple of block size, fix it, append nulls */
   Size = ftell( mfp );
   if(( Size % MemoHeader.BlockSize ) != 0 )
   {
      NewSize = ( Size / MemoHeader.BlockSize + 1) * MemoHeader.BlockSize;
      for( l = Size; l < NewSize; l++ )
        fputc( 0x00, mfp );
   }

   if(( mbb = (VOID *) malloc(len)) == NULL )
   {
      fclose( mfp );
      return NO_MEMORY;
   }
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::CreateMemoFile( VOID )
{
   SHORT len,i;
   CHAR  *sp, *wp;
   CHAR  lb;
   CHAR  buf[4];

   len = GetMemoBlockSize();
   if( len == 0 || len % 512 != 0 )
      return INVALID_BLOCK_SIZE;

   sp = DatabaseName;
   wp = sp;
   while( wp && *wp && *wp != '.' )
   {
#ifdef DOS
      if( *wp == '\\' )
#else
      if( *wp == '/' )
#endif
      {
         sp = wp;
         sp++;
      }
      wp++;
   }
   memset( MemoHeader.FileName, 0x00, 8 );
   for( i = 0; i < 8 && *sp != '.'; i++ )
      MemoHeader.FileName[i] = *sp++;

   len = strlen( DatabaseName );
   len--;
   lb = DatabaseName[len];
   if( lb == 'F' )
      DatabaseName[len] = 'T';
   else if( lb == 'f' )
      DatabaseName[len] = 't';
   else
      return INVALID_NAME;

   /* Initialize the variables */
   MemoHeader.NextBlock = 1L;

   if(( mfp = fopen( DatabaseName, "w+b" )) == NULL )
   {
      DatabaseName[len] = lb;
      return OPEN_ERROR;
   }
   DatabaseName[len] = lb;

   if(( fseek( mfp, 0L, SEEK_SET )) != 0 )
   {
      fclose( mfp );
      return SEEK_ERROR;
   }

   memset( buf, 0x00, 4 );
   xbase->PutLong( buf, MemoHeader.NextBlock );
   if(( fwrite( &buf, 4, 1, mfp )) != 1 ) 
   {
      fclose( mfp );
      return WRITE_ERROR;
   }

   if( MemoHeader.Version == 0x03 )   /* dBASE III+ */
   {
      for( i = 0; i < 12; i++ )  fputc( 0x00, mfp );
      fputc( 0x03, mfp );
      for( i = 0; i < 495; i++ ) fputc( 0x00, mfp );
   }
   else
   {
      for( i = 0; i < 4; i++ )  fputc( 0x00, mfp );
      fwrite( &MemoHeader.FileName,  8, 1, mfp );
      for( i = 0; i < 4; i++ )  fputc( 0x00, mfp );
      memset( buf, 0x00, 2 );
      xbase->PutShort( buf, MemoHeader.BlockSize );
      if(( fwrite( &buf, 2, 1, mfp )) != 1 ) 
      {
         fclose( mfp );
         return WRITE_ERROR;
      }
      for( i = 22; i <  MemoHeader.BlockSize; i++ ) fputc( 0x00, mfp );
   }
        
   if(( mbb = (VOID *) malloc( MemoHeader.BlockSize )) == NULL )
   {
      fclose( mfp );
      return NO_MEMORY;
   }

   return NO_ERROR;
}
/***********************************************************************/
/* Option = 0 - 1st Block of a set of valid data blocks, load buckets  */
/* Option = 1 - subsequant block of data in a multi block set or db III*/
/* Option = 2 - 1st block of a set of free blocks, load buckets        */                     
/* Option = 3 - read 8 bytes of a block, don't load any buckets        */

SHORT DBF::ReadMemoBlock( LONG BlockNo, SHORT Option )
{
   LONG ReadSize;

   CurMemoBlockNo = -1;

   if( BlockNo < 1L )
      return INVALID_BLOCK_NO;

   if( fseek( mfp,(LONG) BlockNo * MemoHeader.BlockSize, SEEK_SET ))
      return SEEK_ERROR;

   if( Option ==  0 || Option == 1 )
      ReadSize = MemoHeader.BlockSize;
   else
      ReadSize = 8L;

   if(fread( mbb, ReadSize, 1, mfp ) != 1 )
      return READ_ERROR;

   if( Option == 0 )
   {
      mfield1   = xbase->GetShort( (CHAR *) mbb );
      MStartPos = xbase->GetShort( (CHAR *) mbb+2 );
      MFieldLen = xbase->GetLong ( (CHAR *) mbb+4 );
   }
   else if( Option == 2 )
   {
      NextFreeBlock = xbase->GetLong( (CHAR *) mbb );
      FreeBlockCnt  = xbase->GetLong( (CHAR *) mbb+4 );
   }
   
   if( Option ==  0 || Option == 1 )
      CurMemoBlockNo = BlockNo;

   return NO_ERROR;
}
/************************************************************************/
SHORT DBF::WriteMemoBlock( LONG BlockNo, SHORT Option )
{
/* Option = 0 - 1st Block of a set of valid data blocks, set buckets    */
/* Option = 1 - subsequant block of data in a multi block set or db III */
/* Option = 2 - 1st block of a set offree blocks, set buckets           */                     

   LONG WriteSize;

   if( BlockNo < 1L )
      return INVALID_BLOCK_NO;

   CurMemoBlockNo = -1;

   if( Option == 0 )
   {
      xbase->PutShort( (CHAR *) mbb, mfield1 );
      xbase->PutShort( (CHAR *) mbb+2, MStartPos );
      xbase->PutLong ( (CHAR *) mbb+4, MFieldLen );
      WriteSize = MemoHeader.BlockSize;
   }
   else if( Option == 2 )
   {
      xbase->PutLong((CHAR *) mbb, NextFreeBlock );
      xbase->PutLong((CHAR *) mbb+4, FreeBlockCnt );
      WriteSize = 8L;
   }
   else
      WriteSize = MemoHeader.BlockSize;

   if( fseek( mfp,(LONG) BlockNo * MemoHeader.BlockSize, SEEK_SET ))
      return SEEK_ERROR;

   if(( fwrite( mbb, WriteSize, 1, mfp )) != 1 ) 
      return WRITE_ERROR;

   if( Option == 0 || Option == 1 ) 
      CurMemoBlockNo = BlockNo;

   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::GetMemoField( SHORT FieldNo, LONG len, CHAR * Buf, SHORT LockOpt )
{
   LONG BlockNo, Tcnt, Scnt;
   CHAR *tp, *sp;           /* target and source pointers */
   SHORT rc;
   SHORT Vswitch;

   if( FieldNo < 0 || FieldNo > ( NoOfFields - 1 ))
      return INVALID_FIELDNO;
   if( GetFieldType( FieldNo ) != 'M' )
      return NOT_MEMO_FIELD;

   #ifdef LOCKING_ON
   if( LockOpt != -1 )
      if(( rc = LockMemoFile( LockOpt, F_RDLCK )) != NO_ERROR )
         return LOCK_FAILED;
   #endif

   if(( BlockNo = GetLongField( FieldNo )) == 0 )
   {
      #ifdef LOCKING_ON
      if( LockOpt != -1 )
         LockMemoFile( F_SETLK, F_UNLCK );
      #endif
      return NO_MEMO_DATA;
   }

   if( MemoHeader.Version == 0x03 )
      Vswitch = 1;
   else 
      Vswitch = 0;

   if((  rc = ReadMemoBlock( BlockNo, Vswitch )) != 0 )
   {
      #ifdef LOCKING_ON
      if( LockOpt != -1 )
         LockMemoFile( F_SETLK, F_UNLCK );
      #endif
      return rc;
   }

   tp = Buf;
   sp = (CHAR *) mbb;

   if( MemoHeader.Version == 0x00 )
   {
      sp+=8;
      Scnt = 8L;  
   }
   else
      Scnt = 0L;

   Tcnt = 0L;
   while( Tcnt < len )
   {
      *tp++ = *sp++;
      Scnt++;
      Tcnt++;
      if( Scnt >= MemoHeader.BlockSize )
      {
         BlockNo++;
         if((  rc = ReadMemoBlock( BlockNo, 1 )) != 0 )
            return rc;
         Scnt = 0;
         sp = (CHAR *) mbb;
      }
   } 
   #ifdef LOCKING_ON
   if( LockOpt != -1 )
      LockMemoFile( F_SETLK, F_UNLCK );
   #endif

   return NO_ERROR;
}
/***********************************************************************/
LONG DBF::GetMemoFieldLen( SHORT FieldNo )
{
   LONG  BlockNo, ByteCnt;
   SHORT scnt, NotDone;
   CHAR *sp, *spp;

   if(( BlockNo = GetLongField( FieldNo )) == 0L )
      return 0L;

   if( MemoHeader.Version == 0x00 )   /* dBASE IV */
   {
      if( BlockNo == CurMemoBlockNo && CurMemoBlockNo != -1 )
         return MFieldLen;
      if( ReadMemoBlock( BlockNo, 0 ) != NO_ERROR )
         return 0L;
      return MFieldLen - MStartPos;
   }
   else  /* version 0x03 dBASE III+ */
   {
      ByteCnt = 0L;
      sp = spp = NULL;
      NotDone = 1;
      while( NotDone )
      {
         if( ReadMemoBlock( BlockNo++, 1 ) != NO_ERROR )
            return 0L;
         scnt = 0;
         sp = (CHAR *) mbb;
         while( scnt < 512 && NotDone )
         {
            if( *sp == 0x1a && *spp == 0x1a )
               NotDone = 0;
            else
            {
               ByteCnt++; scnt++; spp = sp; sp++;
            }
         }
      }
      if( ByteCnt > 0 ) ByteCnt--;
      return ByteCnt;
   }
}
/***********************************************************************/
SHORT DBF::MemoFieldsPresent( VOID )
{
   SHORT i;
   for( i = 1; i < NoOfFields; i++ )
      if( GetFieldType( i ) == 'M' ) 
         return 1;

   return 0;
}
/***********************************************************************/
SHORT DBF::DeleteMemoField( SHORT FieldNo )
{
   LONG SBlockNo, SNoOfBlocks, SNextBlock;
   LONG LastFreeBlock, LastFreeBlockCnt, LastDataBlock;
   SHORT rc;

   NextFreeBlock    = 0L;
   LastFreeBlockCnt = 0L;
   LastFreeBlock    = 0L;
   SNextBlock       = 0L;

   if( MemoHeader.Version == 0x03 )    /* type III */
   {
      PutField( FieldNo, "          " );
      return NO_ERROR;
   }

   /* Get Block Number */
   if(( SBlockNo = GetLongField( FieldNo )) == 0 )
      return INVALID_BLOCK_NO;

   /* Load the first block */
   if(( rc = ReadMemoBlock( SBlockNo, 0 )) != NO_ERROR )
      return rc;

   /* Determine number of blocks to delete */
   SNoOfBlocks = MFieldLen / MemoHeader.BlockSize + 1;

   /* Determine last good data block */
   if(( rc = fseek( mfp, 0, SEEK_END )) != 0 )
      return SEEK_ERROR;

   LastDataBlock = ftell( mfp ) / MemoHeader.BlockSize;
   
   /* position to correct location in chain */
   NextFreeBlock = MemoHeader.NextBlock;

   while( SBlockNo > NextFreeBlock && SBlockNo < LastDataBlock )
   {
      LastFreeBlock    = NextFreeBlock;
      if(( rc = ReadMemoBlock( NextFreeBlock, 2 )) != NO_ERROR ) return rc;
      LastFreeBlockCnt = FreeBlockCnt;
   }

   /* if next block should be concatonated onto the end of this set */
   if((SBlockNo+SNoOfBlocks) == NextFreeBlock && NextFreeBlock < LastDataBlock )
   {
      if(( rc = ReadMemoBlock( NextFreeBlock, 2 )) != NO_ERROR )
         return NO_ERROR;
      SNoOfBlocks += FreeBlockCnt;
      SNextBlock = NextFreeBlock;
   }
   else if( LastFreeBlock == 0L )
      SNextBlock = MemoHeader.NextBlock;
   else
      SNextBlock = NextFreeBlock;
 
   /* if this is the first set of free blocks */
   if( LastFreeBlock == 0L )
   {
      /* 1 - write out the current block */
      /* 2 - update header block         */
      /* 3 - write header block          */
      /* 4 - update data field           */

      NextFreeBlock = SNextBlock;
      FreeBlockCnt = SNoOfBlocks;
      if(( rc = WriteMemoBlock( SBlockNo, 2 )) != NO_ERROR )
         return rc;
      
      MemoHeader.NextBlock = SBlockNo;
      if(( rc = UpdateHeadNextNode()) != NO_ERROR ) return rc;
      PutField( FieldNo, "          " );
      return NO_ERROR;
   }

   /* determine if this block set should be added to the previous set */
   if(( LastFreeBlockCnt + LastFreeBlock ) == SBlockNo )
   {
      if(( rc = ReadMemoBlock( LastFreeBlock, 2 )) != NO_ERROR )
         return rc;
      NextFreeBlock = SNextBlock;
      FreeBlockCnt += SNoOfBlocks;
      if(( rc = WriteMemoBlock( LastFreeBlock, 2 )) != NO_ERROR )
         return rc;
      PutField( FieldNo, "          " );
      return NO_ERROR;
   }

   /* insert into the chain */
   /* 1 - set the next bucket on the current node         */
   /* 2 - write this node                                 */
   /* 3 - go to the previous node                         */
   /* 4 - insert this nodes id into the previous node set */
   /* 5 - write previous node                             */

   FreeBlockCnt = SNoOfBlocks;
   if(( rc = WriteMemoBlock( SBlockNo, 2 )) != NO_ERROR )
      return rc;
   if(( rc = ReadMemoBlock( LastFreeBlock, 2 )) != NO_ERROR )
      return rc;
   NextFreeBlock = SBlockNo;
   if(( rc = WriteMemoBlock( LastFreeBlock, 2 )) != NO_ERROR )
      return rc;
   PutField( FieldNo, "          " );
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::AddMemoData( SHORT FieldNo, LONG Len, CHAR * Buf )
{
   SHORT rc;
   LONG  BlocksNeeded, LastDataBlock, SaveBlock, CurBlock, NextFreeBlock2;
   LONG  HeadBlock;

   BlocksNeeded = Len / MemoHeader.BlockSize + 1;

   /* look for free spot in the chain */
   if(( rc = fseek( mfp, 0, SEEK_END )) != 0 ) return SEEK_ERROR;
   LastDataBlock = ftell( mfp ) / MemoHeader.BlockSize;

   if( LastDataBlock == MemoHeader.NextBlock )  /* no free space */
   {
      MemoHeader.NextBlock += BlocksNeeded;
      if(( rc = PutMemoData( LastDataBlock, BlocksNeeded, Len, Buf )) != NO_ERROR )
         return rc;
      HeadBlock = LastDataBlock;
      if(( rc = UpdateHeadNextNode()) != NO_ERROR ) return rc;
   }
   else
   {
      SaveBlock = MemoHeader.NextBlock;
      CurBlock  = MemoHeader.NextBlock;
      while( CurBlock < LastDataBlock && BlocksNeeded > FreeBlockCnt )
      {
         SaveBlock = CurBlock;
         if(( rc = ReadMemoBlock( CurBlock, 2 )) != NO_ERROR ) return rc;
         CurBlock = NextFreeBlock;
      }

      /* data fits in here ? */
      if( SaveBlock < LastDataBlock && FreeBlockCnt >= BlocksNeeded )
      {
         if( BlocksNeeded == FreeBlockCnt )  /* takes em all */
         {
            NextFreeBlock2 = NextFreeBlock;
            if(( rc = PutMemoData( SaveBlock, BlocksNeeded, Len, Buf )) != NO_ERROR )
               return rc;
            HeadBlock = SaveBlock;

 /* next line was changed but not tested 1/27/98 when ported to dos */
            if( SaveBlock == MemoHeader.NextBlock )  /* update header */
            {
               MemoHeader.NextBlock = NextFreeBlock2;
               if(( rc = UpdateHeadNextNode()) != NO_ERROR )
                  return rc;
            }
            else
            {
               if(( rc = ReadMemoBlock( SaveBlock, 2 )) != NO_ERROR ) return rc;
               NextFreeBlock = NextFreeBlock2;
               if(( rc = WriteMemoBlock( SaveBlock, 2 )) != NO_ERROR ) return rc;
            }
         }   
         else   /* calculate new starting position with in the block set */
         {
            FreeBlockCnt -= BlocksNeeded;
            if(( rc = WriteMemoBlock( SaveBlock, 2 )) != NO_ERROR ) return rc;
            SaveBlock = SaveBlock + FreeBlockCnt;
            if(( rc = PutMemoData( SaveBlock, BlocksNeeded, Len, Buf )) != NO_ERROR )
               return rc;
            HeadBlock = SaveBlock;
         }
      }
      else /* append to the end */
      {
         if(( rc = PutMemoData( CurBlock, BlocksNeeded, Len, Buf )) != NO_ERROR )
            return rc;
         HeadBlock = CurBlock;
         if(( rc = ReadMemoBlock( SaveBlock, 2 )) != NO_ERROR )
            return rc;
         NextFreeBlock += BlocksNeeded;
         if(( rc = WriteMemoBlock( SaveBlock, 2 )) != NO_ERROR ) return rc;
      }               
   }
   PutLongField( FieldNo, HeadBlock );
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::UpdateHeadNextNode( VOID )
{      
   CHAR buf[4];
   memset( buf, 0x00, 4 );
   xbase->PutLong( buf, MemoHeader.NextBlock );
   if(( fseek( mfp, 0L, SEEK_SET )) != 0 ) return SEEK_ERROR;
   if(( fwrite( &buf, 4, 1, mfp )) != 1 ) return WRITE_ERROR;
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::PutMemoData( LONG StartBlock, LONG BlocksNeeded, LONG Len, 
          CHAR * Buf)
{
   SHORT i, rc, Qctr, Tctr, flen;
   LONG  CurBlock;
   CHAR *tp, *sp;

   CurBlock = StartBlock;
   tp = (CHAR *) mbb;
   sp = Buf;
   Qctr = 0;   /* total length processed */

   if( MemoHeader.Version == 0x03 )
   {
      flen = Len - 2;
      Tctr = 0;
   }
   else  /* dBASE IV */
   {
      tp += 8;
      Tctr = 8;
   }

   for( i = 0; i < BlocksNeeded; i++ )
   {
      while( Tctr < MemoHeader.BlockSize && Qctr < Len )
      {
         if( Qctr >= flen && MemoHeader.Version == 0x03 )
            *tp++ = 0x1a;    /* end of data marker */
         else
            *tp++ = *sp++;  
         Tctr++; Qctr++;
      }

      if( i == 0 && MemoHeader.Version == 0x00 )
      {
         mfield1 = -1;
         MStartPos = 8;
         MFieldLen = Len + MStartPos;
         if(( rc = WriteMemoBlock( CurBlock++, 0 )) != NO_ERROR )
            return rc;
      }
      else 
      {
         if(( rc = WriteMemoBlock( CurBlock++, 1 )) != NO_ERROR )
            return rc;
      }
      Tctr = 0;
      tp = (CHAR *) mbb;
   }
   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::UpdateMemoData( SHORT FieldNo, LONG Len, CHAR * Buf, SHORT LockOpt )
{
   SHORT rc;
   LONG  BlocksNeeded, BlocksAvailable;

#ifdef LOCKING_ON
   if( LockOpt != -1 )
      if(( rc = LockMemoFile( LockOpt, F_WRLCK )) != NO_ERROR )
         return LOCK_FAILED;
#endif

   if( Len != 0L && MemoHeader.Version == 0x03 ) Len += 2;   
   if( Len == 0L )   /* handle delete */
   {
      if( MemoFieldExists( FieldNo ) ) 
      {
         if(( rc = DeleteMemoField( FieldNo )) != NO_ERROR )
         {
            #ifdef LOCKING_ON
            LockMemoFile( F_SETLK, F_UNLCK );
            #endif
            return rc;
         }
      }
   }
   else if((GetMemoFieldLen(FieldNo)==0L) || (MemoHeader.Version==0x03))
   {
      if(( rc = AddMemoData( FieldNo, Len, Buf )) != NO_ERROR )
      {
         #ifdef LOCKING_ON
         LockMemoFile( F_SETLK, F_UNLCK );
         #endif
         return rc;
      }
   }
   else   /* version IV type files, reuse unused space */
   {
      BlocksNeeded = Len / MemoHeader.BlockSize + 1L;
      if(( rc = ReadMemoBlock( GetLongField( FieldNo ), 3 )) != NO_ERROR ) 
      {
         #ifdef LOCKING_ON
         LockMemoFile( F_SETLK, F_UNLCK );
         #endif
         return rc;
      }

      MFieldLen = xbase->GetLong(( CHAR *) mbb+4 );
      BlocksAvailable = MFieldLen / MemoHeader.BlockSize + 1L;

      if( BlocksNeeded == BlocksAvailable )
      {
         if(( rc = PutMemoData( GetLongField( FieldNo ), BlocksNeeded, Len, Buf )) != NO_ERROR )
         {
            #ifdef LOCKING_ON
            LockMemoFile( F_SETLK, F_UNLCK );
            #endif
            return rc;
         }
      }
      else
      {
         if(( rc = DeleteMemoField( FieldNo )) != NO_ERROR )  
         {
            #ifdef LOCKING_ON
            LockMemoFile( F_SETLK, F_UNLCK );
            #endif
            return rc;
         }
         if(( rc = AddMemoData( FieldNo, Len, Buf )) != NO_ERROR )
         {
            #ifdef LOCKING_ON
            LockMemoFile( F_SETLK, F_UNLCK );
            #endif
            return rc;
         }
      }
   }

#ifdef LOCKING_ON
   if( LockOpt != -1 )
      if(( rc = LockMemoFile( F_SETLK, F_UNLCK )) != NO_ERROR )
         return LOCK_FAILED;
#endif

   return NO_ERROR;
}
/***********************************************************************/
SHORT DBF::MemoFieldExists( SHORT FieldNo )
{
   if( GetLongField( FieldNo ) == 0L )
      return 0;
   else
      return 1;
}
/***********************************************************************/
#ifdef XBASE_DEBUG
VOID DBF::DumpMemoHeader( VOID )
{
   SHORT i;
   cout << "\n*********************************";
   cout << "\nMemo header data...";
   cout << "\nNext Block " << MemoHeader.NextBlock;
   if( MemoHeader.Version == 0x00 )
   {
      cout << "\nFilename   ";
      for( i = 0; i < 8; i++ )
         cout << MemoHeader.FileName[i];
   }
   cout << "\nBlocksize  " << MemoHeader.BlockSize;
   return;
}
/***********************************************************************/
SHORT DBF::DumpMemoFreeChain( VOID )
{
   SHORT rc;
   LONG  CurBlock, LastDataBlock;

   if(( rc = GetDbtHeader(1)) != NO_ERROR )
      return rc;
   if( fseek( mfp, 0, SEEK_END ) != 0 )
      return SEEK_ERROR;
   LastDataBlock = ftell( mfp ) / MemoHeader.BlockSize;
   CurBlock = MemoHeader.NextBlock;
   cout << "\nTotal blocks in file = " << LastDataBlock;
   cout << "\nHead Next Block = " << CurBlock;
   while( CurBlock < LastDataBlock )
   {
      if(( rc = ReadMemoBlock( CurBlock, 2 )) != NO_ERROR )
         return rc;
      cout << "\n**********************************";
      cout << "\nThis Block = " << CurBlock;
      cout << "\nNext Block = " << NextFreeBlock;
      cout << "\nNo Of Blocks = " << FreeBlockCnt << "\n";
      CurBlock = NextFreeBlock;
   }
   return NO_ERROR;
}
/***********************************************************************/
VOID DBF::DumpMemoBlock( VOID )
{
   SHORT i;
   CHAR  *p;
   p = (CHAR *) mbb;
   if( MemoHeader.Version == 0x03 )
   {
      for( i = 0; i < 512; i++ )
         cout << *p++;
   }
   else
   {
      cout << "\nField1     => " << mfield1;
      cout << "\nStart Pos  => " << MStartPos;
      cout << "\nField Len  => " << MFieldLen;
      cout << "\nBlock data => ";
      p += 8;
      for( i = 8; i < MemoHeader.BlockSize; i++ )
         cout << *p++;
   }
   return;
}
#endif  /* XBASE_DEBUG */
/***********************************************************************/
#endif  /* MEMO_FIELD */
