/*  expfunc.cpp

    Xbase project source code

    This file contains logic for handling Xbase expressions.

    Copyright (C) 1997  Startech, Gary A. Kunkel   
    email - xbase@startech.keller.tx.us
    www   - http://www.startech.keller.tx.us/xbase.html

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    V 1.0    10/10/97   - Initial release of software
    V 1.5    1/2/97     - Added memo field support
    V 1.6a   4/1/98     - Added expression support
    V 1.6b   4/8/98     - Numeric index keys
    V 1.7.1  5/25/98    - Enhanced expression logic
    V 1.7.4b 7/2/98     - changed gcvt to sprintf for portability reasons
*/

#include "stdafx.h"
#include "xbase.h"
#ifdef EXPRESSIONS

#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "exp.h"

#ifdef _DEBUG
#include "shalloc.h"
#define malloc(a) DebugMalloc(a, __FILE__, __LINE__ )
#endif

/*************************************************************************/
SHORT EXPN::ProcessFunction( CHAR * Func )
{
/* 1 - pop function from stack
   2 - verify function name and get no of parms needed 
   3 - verify no of parms >= remainder of stack
   4 - pop parms off stack
   5 - execute function
   6 - push result back on stack
*/

   CHAR *buf;
   ExpNode *p1, *p2, *p3, *WorkNode, *FuncNode;
   SHORT ParmsNeeded,len;
   CHAR ptype;  /* process type s=string, l=logical, d=double */
   DOUBLE DoubResult;
   LONG IntResult;

   FuncNode = (ExpNode *) Pop();
   ParmsNeeded = GetFuncInfo( Func, 1 );
   if( ParmsNeeded == -1 ) 
      return INVALID_FUNCTION;
   if( ParmsNeeded > GetStackDepth() )
      return INSUFFICIENT_PARMS;

   p1 = p2 = p3 = NULL;
   if( ParmsNeeded > 2 ) p3 = (ExpNode *) Pop(); 
   if( ParmsNeeded > 1 ) p2 = (ExpNode *) Pop(); 
   if( ParmsNeeded > 0 ) p1 = (ExpNode *) Pop(); 

   if( strncmp( Func, "ABS", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = ABS( GetDoub( p1 ));
   }
   else if( strncmp( Func, "ASC", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = ASC( p1->Result );
   }
   else if( strncmp( Func, "AT", 2 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = AT( p1->Result, p2->Result );
   }
   else if( strncmp( Func, "CDOW", 4 ) == 0 )
   {  
      ptype = 's';
      buf = CDOW( p1->Result );
   }
   else if( strncmp( Func, "CHR", 3 ) == 0 )
   {  
      ptype = 's';
      buf = CHR( GetInt( p1 ));
   }
   else if( strncmp( Func, "CMONTH", 6 ) == 0 )
   {  
      ptype = 's';
      buf = CMONTH( p1->Result );
   }
   else if( strncmp( Func, "DATE", 4 ) == 0 )
   {  
      ptype = 's';
      buf = DATE();
   }
   else if( strncmp( Func, "DAY", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = DAY( p1->Result );
   }
   else if( strncmp( Func, "DOW", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = DOW( p1->Result );
   }
   else if( strncmp( Func, "DTOC", 4 ) == 0 )
   {  
      ptype = 's';
      buf = DTOC( p1->Result );
   }
   else if( strncmp( Func, "EXP", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = EXP( GetDoub( p1 ));
   }
   else if( strncmp( Func, "INT", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = INT( GetDoub( p1 ));
   }
   else if( strncmp( Func, "ISALPHA", 7 ) == 0 )
   {  
      ptype = 'l';
      IntResult = ISALPHA( p1->Result );
   }
   else if( strncmp( Func, "ISLOWER", 7 ) == 0 )
   {  
      ptype = 'l';
      IntResult = ISLOWER( p1->Result );
   }
   else if( strncmp( Func, "ISUPPER", 7 ) == 0 )
   {  
      ptype = 'l';
      IntResult = ISUPPER( p1->Result );
   }
   else if( strncmp( Func, "LEN", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = LEN( p1->Result );
   }
   else if( strncmp( Func, "LEFT", 4 ) == 0 )
   {  
      ptype = 's';
      buf = LEFT( p1->Result, INT( p2->DoubResult ));   
   }
   else if( strncmp( Func, "LTRIM", 5 ) == 0 )
   {  
      ptype = 's';
      buf = LTRIM( p1->Result );
   }  
   else if( strncmp( Func, "LOG", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = LOG( GetDoub( p1 ));
   }  
   else if( strncmp( Func, "LOWER", 5 ) == 0 )
   {  
      ptype = 's';
      buf = LOWER( p1->Result );
   }  
   else if( strncmp( Func, "MAX", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = MAX( GetDoub( p1 ), GetDoub( p2 ));
   }  
   else if( strncmp( Func, "MIN", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = MIN( GetDoub( p1 ), GetDoub( p2 ));
   }  
   else if( strncmp( Func, "MONTH", 5 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = MONTH( p1->Result );
   }  
   else if( strncmp( Func, "REPLICATE", 9 ) == 0 )
   {
      ptype = 's';
      buf = REPLICATE( p1->Result, GetInt( p2 ));
   }
   else if( strncmp( Func, "RIGHT", 5 ) == 0 )
   {
      ptype = 's';
      buf = RIGHT( p1->Result, GetInt( p2 ));
   }
   else if( strncmp( Func, "RTRIM", 5 ) == 0 )
   {  
      ptype = 's';
      buf = RTRIM( p1->Result );
   }  
   else if( strncmp( Func, "SPACE", 5 ) == 0 )
   {  
      ptype = 's';
      buf = SPACE( INT( GetDoub( p1 )));
   }
   else if( strncmp( Func, "SQRT", 4 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = SQRT( GetDoub( p1 ));
   }
   else if( strncmp( Func, "STR", 3 ) == 0 )
   {
      ptype = 's';
      buf = STR( p1->Result );
   }   
   else if( strncmp( Func, "SUBSTR", 4 ) == 0 )
   {
      ptype = 's';
      buf = SUBSTR( p1->Result, GetInt( p2 ), GetInt( p3 )); 
   }
   else if( strncmp( Func, "TRIM", 4 ) == 0 )
   {  
      ptype = 's';
      buf = TRIM( p1->Result );
   }  
   else if( strncmp( Func, "UPPER", 5 ) == 0 )
   {  
      ptype = 's';
      buf = UPPER( p1->Result );
   }  
   else if( strncmp( Func, "VAL", 3 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = VAL( p1->Result );
   }  
   else if( strncmp( Func, "YEAR", 4 ) == 0 )
   {  
      ptype = 'd';
      DoubResult = YEAR( p1->Result );
   }  

   if( p1 && !p1->InTree ) FreeExpNode( p1 );
   if( p2 && !p2->InTree ) FreeExpNode( p2 );
   if( p3 && !p3->InTree ) FreeExpNode( p3 );
   if( !FuncNode->InTree ) FreeExpNode( FuncNode );

   len = strlen( buf );
   if(( WorkNode = GetExpNode( len+1 )) == NULL )
      return NO_MEMORY;

   switch( ptype )
   {
      case 's':
      WorkNode->DataLen = len;
      WorkNode->ExpressionType = 'C';
      WorkNode->Type = 's';
      strcpy( WorkNode->Result, buf );
      break;

      case 'd':
      WorkNode->DataLen = 0;
      WorkNode->ExpressionType = 'N';
      WorkNode->Type = 'd';
      WorkNode->DoubResult = DoubResult;
      break;

      case 'l':
      WorkNode->DataLen = 0;
      WorkNode->ExpressionType = 'L';
      WorkNode->Type = 'l';
      WorkNode->IntResult = IntResult;
      break;

      default:
      cout << "\nInternal error. " << ptype;
      break;
   }

   Push( (VOID *) WorkNode );
   return NO_ERROR;
}

/*************************************************************************/
CHAR * EXPN::GetCharResult( VOID )
{
   CHAR * s;
   ExpNode * e;
   if( GetStackDepth() < 1 ) return NULL;
   e = (ExpNode *) Pop();
   s = e->Result;
   Push( (VOID *) e );
   return s;
}
/*************************************************************************/
LONG EXPN::GetIntResult( VOID )
{
   LONG l;
   ExpNode * e;
   if( GetStackDepth() < 1 ) return 0L;
   e = (ExpNode *) Pop();
   l = e->IntResult;
   Push( (VOID *) e );
   return l;
}
/*************************************************************************/
/*************************************************************************/
DOUBLE EXPN::GetDoubleResult( VOID )
{
   DOUBLE d;
   ExpNode * e;
   if( GetStackDepth() < 1 ) return (DOUBLE) 0;
   e = (ExpNode *) Pop();
   d = e->DoubResult;
   Push( (VOID *) e );
   return d;
}
/*************************************************************************/
DOUBLE EXPN::GetDoub( ExpNode * p )
{
   if( p->Type == 'd' )
      return p->DoubResult;
   else if( p->Type == 'N' || p->Type == 's' )
      return( strtod( p->Result, NULL ));
   else if( p->Type == 'D' )
      return( p->dbf->GetDoubleField( p->FieldNo ));
   else
      return 0;
}
/*************************************************************************/
LONG EXPN::GetInt( ExpNode *p )
{
   if( p->Type == 'l' || p->Type == 'i' )
      return p->IntResult;
   else if( p->Type == 'N' || p->Type == 's' )
      return atoi( p->Result );
   else if( p->Type == 'D' )
      return p->dbf->GetLongField( p->FieldNo );
   else
      return 0L;
}
/*************************************************************************/
/*************************************************************************/
DOUBLE EXPN::ABS( DOUBLE d )
{
   if( d < (DOUBLE) 0 )
      return d * -1;
   else
      return d;
}
/*************************************************************************/
LONG EXPN::ASC( CHAR * String )
{
   return *String;
}
/*************************************************************************/
LONG EXPN::AT( CHAR * s1, CHAR *s2 )
{
   /* looks for s1 in s2 */
   LONG cnt;
   CHAR *p1, *p2;

   if( strlen( s1 ) > strlen( s2 )) return 0;
   if(( p1 = strstr( s2, s1 )) == NULL )
      return 0;
   cnt = 1;
   p2 = s2;
   while( p2++ != p1 ) cnt++;
   return cnt;
}
/*************************************************************************/
CHAR * EXPN::CDOW( CHAR * Date8 )
{
   static CHAR buf[10];
   SHORT len,i;
   strcpy( buf, FormatDate( "DDDD", Date8 )); 
   len = strlen( buf );
   for( i = len; i < 9; i++ ) buf[i] = 0x20;
   buf[9] = 0x00;
   return buf;
}
/*************************************************************************/
CHAR * EXPN::CHR( LONG l )
{
   static CHAR buf[2];
   SHORT i;
   i = (SHORT) l;
   buf[0] = i;
   buf[1] = 0x00;
   return buf;
}
/*************************************************************************/
CHAR * EXPN::CMONTH( CHAR * Date8 )
{
   static CHAR buf[10];
   SHORT len,i;
   strcpy( buf, FormatDate( "MMMM", Date8 ));
   len = strlen( buf );
   for( i = len; i < 9; i++ ) buf[i] = 0x20;
   buf[9] = 0x00;
   return buf;
}
/*************************************************************************/
LONG EXPN::DAY( CHAR * Date8 )
{
   return DayOf( FMT_MONTH, Date8 );
}
/*************************************************************************/
LONG EXPN::DOW( CHAR * Date8 )
{
   return DayOf( FMT_WEEK, Date8 );
}
/*************************************************************************/
CHAR * EXPN::DTOC( CHAR * Date8 )
{
   strcpy( WorkBuf, FormatDate( "MM/DD/YY", Date8 ));
   return WorkBuf;
}
/*************************************************************************/
DOUBLE EXPN::EXP( DOUBLE d )
{
   return exp( d );
}
/*************************************************************************/
LONG EXPN::INT( DOUBLE d )
{
   return (LONG) d;
}
/*************************************************************************/
LONG EXPN::ISALPHA( CHAR * String )
{
   if( isalpha(*String) ) return 1;
   else return 0;
}
/*************************************************************************/
LONG EXPN::ISLOWER( CHAR * String )
{
   if( islower(*String) ) return 1;
   else return 0;
}
/*************************************************************************/
LONG EXPN::ISUPPER( CHAR * String )
{
   if( isupper(*String) ) return 1;
   else return 0;
}
/*************************************************************************/
LONG EXPN::LEN( CHAR * String )
{
   LONG len;
   len = strlen( String );
   len--;
   while( len >= 0 && String[len] == 0x20 ) len--;
   return ++len;
}
/*************************************************************************/
CHAR * EXPN::LEFT( CHAR * String, SHORT Len )
{
   SHORT i;
   for( i = 0; i < Len && i < 100; i++ )
      WorkBuf[i] = String[i];

   WorkBuf[i] = 0x00;
   return WorkBuf;
}
/*************************************************************************/
/* This method removes any leading spaces from String */
CHAR * EXPN::LTRIM( CHAR * String )
{
   CHAR *sp;
   SHORT i;
   sp = String;
   i = 0;
   while( sp && *sp == 0x20 ) sp++;
   while( sp && i < 100 )
   {
      WorkBuf[i++] = *sp;
      sp++;
   }
   strcpy( String, WorkBuf ); 
   return WorkBuf;
}
/*************************************************************************/
DOUBLE EXPN::LOG( DOUBLE d )
{
   return log( d );
}
/*************************************************************************/
CHAR * EXPN::LOWER( CHAR * String )
{
   CHAR *sp;
   SHORT i;
   sp = String;
   i = 0;
   while( sp && i < 100 )
   {
      WorkBuf[i++] = tolower( *sp );
      sp++;
   }
   return WorkBuf;
}
/*************************************************************************/
DOUBLE EXPN::MAX( DOUBLE d1, DOUBLE d2 )
{
   if( d1 > d2 ) 
      return d1;
   else
      return d2;
}
/*************************************************************************/
DOUBLE EXPN::MIN( DOUBLE d1, DOUBLE d2 )
{
   if( d1 < d2 ) 
      return d1;
   else
      return d2;
}
/*************************************************************************/
LONG EXPN::MONTH( CHAR * Date8 )
{
   return MonthOf( Date8 );
}
/*************************************************************************/
CHAR * EXPN::REPLICATE( CHAR * String, SHORT Cnt )
{
   SHORT len, i;
   len = strlen( String );
   if(( len * Cnt ) > 100 ) return NULL;
   memset( WorkBuf, 0x00, len+1 );
   for( i = 0; i < Cnt; i++ )
      strcat( WorkBuf, String );
   return WorkBuf;
}
/*************************************************************************/
CHAR * EXPN::RIGHT( CHAR * String, SHORT cnt )
{
   SHORT len;
   CHAR *p;
   len = strlen( String );
   if( len < cnt ) return String;
   len = LEN( String );
   if( len < cnt ) return String;
   p = String + len - cnt;
   return p;
}
/*************************************************************************/
CHAR * EXPN::RTRIM( CHAR * String )
{
   return TRIM( String );
}
/*************************************************************************/
CHAR * EXPN::SPACE( SHORT Cnt )
{ 
   if( Cnt > 100 ) return NULL;
   memset( WorkBuf, 0x20, Cnt );
   WorkBuf[Cnt] = 0x00;
   return WorkBuf;
} 
/*************************************************************************/
DOUBLE EXPN::SQRT( DOUBLE d )
{
   return sqrt( d );
}
/*************************************************************************/
CHAR * EXPN::STR( DOUBLE d )
{
   SHORT len, i;
// gcvt replaced with sprintf for portability reasons
// gcvt( d, 10, WorkBuf );
   sprintf( WorkBuf, "%f", d );
   len = strlen( WorkBuf );
   if( len > 10 )
      strcpy( WorkBuf, "**********" );
   else if( len < 10 )
   {
      for( i = len; i < 10; i++ )
         WorkBuf[i] = 0x20;
      WorkBuf[10] = 0x00;
   }
   return WorkBuf;
}
/*************************************************************************/
CHAR * EXPN::STR( CHAR * String )
{
   SHORT len, i;
   len = strlen( String );
   if( len > 10 )
      strcpy( WorkBuf, "**********" );
   else 
   {
      strcpy( WorkBuf, String );
      if( len < 10 )
         for( i = len; i < 10; i++ )
            WorkBuf[i] = 0x20;
      WorkBuf[10] = 0x00;
   }
   return WorkBuf;
}
/*************************************************************************/
CHAR * EXPN::SUBSTR( CHAR * String, SHORT StartPos, SHORT Len )
{
   SHORT i;
   CHAR *s;
   if( StartPos < 1 ) return NULL;
   s = String + StartPos - 1;
   for( i = 0; i < Len; i++ )
      WorkBuf[i] = *s++;
   WorkBuf[i] = 0x00;
   return WorkBuf;
}
/*************************************************************************/
CHAR * EXPN::DATE()
{
   strcpy( WorkBuf, Sysdate());
   return WorkBuf;
}
/*************************************************************************/
CHAR * EXPN::TRIM( CHAR * String )
{
   CHAR *sp;
   SHORT i, len;
   len = strlen( String );
   sp = String;
   sp += len - 1;
   for( i = len; i > 0; i-- )
   {
      if( *sp == 0x20 ) 
      {
         *sp = 0x00;
         sp--;
      }
      else
         i = 0;
   }
   return String;
}
/*************************************************************************/
CHAR * EXPN::UPPER( CHAR * String )
{
   CHAR *sp;
   SHORT i;
   sp = String;
   i = 0;
   while( sp && i < 100 )
   {
      WorkBuf[i++] = toupper( *sp );
      sp++;
   }
   return WorkBuf;
}
/*************************************************************************/
LONG EXPN::VAL( CHAR * String )
{
   return atoi( String );
}
/*************************************************************************/
LONG EXPN::YEAR( CHAR * Date8 )
{
   return YearOf( Date8 );
}
/*************************************************************************/
#endif     // EXPRESSIONS
