/*  Product: Mentor SQL

    Copyright (C) 1998  Systementor AB, Stefan Holmberg
    Email - stefan.holmberg@systementor.se
    Web:   - http://www.systementor.se

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

///////////////////////////////////////////////////////////////////////////////
*/
#include "FlexLexer.h"
#include "executer.h"
#include "error.h"
#include "systemcatalog.h"
#include "server.h"
#include "general.h"
#include "optimizer.h"
#include "shalloc.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#endif


CExecuter::CExecuter( CParseTree *pTree )
{
m_pTree = pTree;
m_lResultSize = 0;
m_szResult = NULL;
m_pResultSet = NULL;
}

CExecuter::~CExecuter()
{
if ( m_szResult )
	delete []m_szResult;
if ( m_pResultSet )
	delete m_pResultSet;
}


int CExecuter::Execute()
{
if ( m_szResult )
	delete []m_szResult;
m_lResultSize = 0;
if ( m_pResultSet )
	delete m_pResultSet;

switch( m_pTree->m_pFirst->m_nType )
	{
	case NodeType_DropTable:
	case NodeType_CreateTable:
	case NodeType_CreateIndex:
	case NodeType_DropIndex:
		return ExecuteUtility();
		break;
	case NodeType_Insert:
		return ExecuteInsert();			
	case NodeType_Delete:
		return ExecuteDelete();
	case NodeType_Select:
		return ExecuteQuery();
	case NodeType_Update:
		return ExecuteUpdate();
	default:
		return ERROR_NUM_UNDEFINED_COMMAND;
	}
return ERROR_NUM_OK;
}

long CExecuter::ResultSize()
{
return m_lResultSize;
}

void CExecuter::CopyToBuf( char *szBuf )
{
if ( !m_pResultSet )
	return;
m_pResultSet->CopyToBuf( szBuf, m_lResultSize );
}


int CExecuter::ExecuteUtility()
{
CSQLNode *pNode = m_pTree->m_pFirst;
if ( !pNode )
	return ERROR_NUM_INTERNAL_PARSE_ERROR;

switch( pNode->m_nType )
	{
	case NodeType_CreateTable:
		{
		CSQLNode *pNameIdent = m_pTree->GetList( "NameIdent" );
		CSQLNode *pColDataTypeList = m_pTree->GetList( "ColDataTypeList" );

		int nColCount = 0;
		while ( pColDataTypeList )
			{
			nColCount++;
			pColDataTypeList = pColDataTypeList->m_pNext;
			}

		CColumnDesc *pSchema = new CColumnDesc[ nColCount ];	

		pColDataTypeList = m_pTree->GetList( "ColDataTypeList" );
		nColCount = 0;
		while ( pColDataTypeList )
			{
			CSQLNode *pDataTypeNode = pColDataTypeList->m_pLeftChild;
			strcpy( pSchema[nColCount].m_szName ,pColDataTypeList->m_pSQLToken->m_strText.c_str() );
			switch( pDataTypeNode->m_nType )
				{
				case NodeType_CharDataType:
					pSchema[nColCount].m_FieldLen = atoi(pDataTypeNode->m_pLeftChild->m_pSQLToken->m_strText.c_str());
					pSchema[nColCount].m_DataType = MENTORSQL_DATATYPE_CHAR;
					break;
				case NodeType_IntegerDataType:
					pSchema[nColCount].m_FieldLen = 9;
					pSchema[nColCount].m_DataType = MENTORSQL_DATATYPE_INT;
					break;
				case NodeType_FloatDataType:
					pSchema[nColCount].m_FieldLen = 9;
					pSchema[nColCount].m_DataType = MENTORSQL_DATATYPE_FLOAT;
					break;
				default:
					break;
				}
			nColCount++;
			pColDataTypeList = pColDataTypeList->m_pNext;
			}
		int nRet = GetServer()->GetSystemCatalog()->CreateTable((const char *)pNameIdent->m_pSQLToken->m_strText.c_str(), pSchema, nColCount );
		delete []pSchema;
		return nRet;
		}
		break;
	case NodeType_DropTable:
		{
		CSQLNode *pNameIdent = m_pTree->GetList( "NameIdent" );
		int nRet = GetServer()->GetSystemCatalog()->DropTable((const char *)pNameIdent->m_pSQLToken->m_strText.c_str() );
		return nRet;
		}
	case NodeType_CreateIndex:
		{
		CSQLNode *pIndexNameIdent = m_pTree->GetList( "IndexNameIdent" );
		CSQLNode *pTableNameIdent = m_pTree->GetList( "TableNameIdent" );
		CSQLNode *pColNameIdent = m_pTree->GetList( "ColNameIdent" );
		
		if ( _CheckTableNames( pTableNameIdent ) == false )
			return ERROR_NUM_SELECTNOTABLE;
		
		int nColCount;
		int nRet = _CheckAndQualifyColumnNames( pColNameIdent, pTableNameIdent, nColCount );
		if ( nRet )
			return nRet;
		nRet = GetServer()->GetSystemCatalog()->CreateIndex(pTableNameIdent->m_pSQLToken->m_strText.c_str(), pIndexNameIdent->m_pSQLToken->m_strText.c_str(), pColNameIdent->m_pRightChild->m_pSQLToken->m_strText.c_str() );
		return nRet;
		}
	case NodeType_DropIndex:
		{
		//CSQLNode *pIndexNameIdent = m_pTree->GetList( "IndexNameIdent" );
		//int nRet = GetServer()->GetSystemCatalog()->DropIndex((const char *)pIndexNameIdent->m_pSQLToken->m_strText.c_str() );
		//return nRet;
		}
	default:
		return ERROR_NUM_UNDEFINED_COMMAND;
	}
}

bool CExecuter::_CheckTableNames( CSQLNode *pFromClause )
{
CSQLNode *pTableNode = pFromClause;
while ( pTableNode )
	{
	CTable *pTable = GetServer()->GetSystemCatalog()->GetTable( pTableNode->m_pSQLToken->m_strText.c_str() );
	if ( !pTable )
		return false;
	pTableNode = pTableNode->m_pNext;	
	}
return true;
}

int CExecuter::_QualifyColName( CSQLNode *pNode, CSQLNode *pFromClause )
{
if ( pNode->m_nType != NodeType_Dot )
	return -1;

const char *szColName = pNode->m_pRightChild->m_pSQLToken->m_strText.c_str();
if ( pNode->m_pLeftChild )
	return 0;
int nFoundCount = 0;


CSQLNode *pTableNode = pFromClause;

while ( pTableNode )
	{
	CTable *pTempTable = GetServer()->GetSystemCatalog()->GetTable( pTableNode->m_pSQLToken->m_strText.c_str() );
	if ( pTempTable->GetColumnInfo( szColName ))
		{
		if ( nFoundCount == 0 )
			{
			pNode->m_pLeftChild = m_pTree->ManuallyAddIdentifier( pTempTable->GetName().c_str(), "Id" );
			}
		else
			return ERROR_NUM_AMBIGOUS_COLNAMES;
		nFoundCount++;					
		}
	pTableNode = pTableNode->m_pNext;
	}
if ( nFoundCount != 1 )
	return ERROR_NUM_NOSUCHCOLUMN;
return 0;
}

int CExecuter::_RecursiveCheck( CSQLNode *pNode, CSQLNode *pFromClause )
{
if ( pNode == NULL )
	return 0;
switch( pNode->m_nType )
	{
	case NodeType_Equals:
	case NodeType_More:
	case NodeType_Less:
	case NodeType_MoreEquals:
	case NodeType_LessEquals:
		{
		int n1 = _RecursiveCheck( pNode->m_pLeftChild, pFromClause ); // col
		if ( n1 )
			return n1;
		n1 = _RecursiveCheck( pNode->m_pRightChild, pFromClause ); //identifier
		return n1;
		}
	case NodeType_Dot:
		{
		return _QualifyColName( pNode, pFromClause );
		}
	case NodeType_Integer_Value:
	case NodeType_String_Value:
		{
		return 0;
		}
	case NodeType_Identifier:
		{
		return _QualifyColName( pNode, pFromClause ) ;
		}
	case NodeType_And:
	case NodeType_Or: //We should perform check on all childs
		{
		int n1 = _RecursiveCheck( pNode->m_pLeftChild, pFromClause ); // cond 1
		if ( n1 )
			return n1;
		n1 = _RecursiveCheck( pNode->m_pRightChild, pFromClause ); //cond 2
		return n1;
		}
	}
return -1;
}


int CExecuter::_CheckAndValidateOrderNames( CSQLNode *pTableIdent, CSQLNode *pOrderClause )
{
if ( pOrderClause->m_nType == NodeType_Empty )
	return 0;

CSQLNode *pColNode = pOrderClause->m_pLeftChild;
//This is the first actual node, for example AND
return _RecursiveCheck( pColNode, pTableIdent );
}

int CExecuter::_CheckAndQualifyWhereNames( CSQLNode *pFromClause, CSQLNode *pWhereClause )
{
if ( pWhereClause->m_nType == NodeType_Empty )
	return 0;

CSQLNode *pColNode = pWhereClause->m_pLeftChild;
//This is the first actual node, for example AND
return _RecursiveCheck( pColNode, pFromClause );
}


int CExecuter::_CheckAndQualifyUpdateList( CSQLNode *pTableIdent, CSQLNode *pUpdateList )
{
//Check all columns, and fully qualify them
CSQLNode *pColNode = pUpdateList;

CTable *pTempTable = GetServer()->GetSystemCatalog()->GetTable( pTableIdent->m_pSQLToken->m_strText.c_str() );
//For columnnames leftchild is table and rightchild is column
while ( pColNode )
	{
	if ( pColNode->m_pLeftChild->m_pLeftChild == NULL )
		{
		int rc = _QualifyColName( pColNode->m_pLeftChild, pTableIdent );
		if ( rc )
			return rc;
		
		//OK now we need to see if right datatype is used...
		const char *szColName = pColNode->m_pLeftChild->m_pRightChild->m_pSQLToken->m_strText.c_str();
		CColumnDesc *pSchemaRec = pTempTable->GetColumnInfo( szColName );
		int nRet = Compare( pSchemaRec, pTempTable, pColNode->m_pRightChild );
		if ( nRet == ERROR_NUM_WRONG_DATATYPE )
			return nRet;

		//Right datatype so it is no problem
		}
	pColNode = pColNode->m_pNext;
	}
return 0;
}


int CExecuter::_CheckAndQualifyColumnNames( CSQLNode *pSelectClause, CSQLNode *pFromClause, int &nColCount )
{
//Check all columns
nColCount = 0;
if ( pSelectClause->m_nType == NodeType_Star )
	{
	//All columns, so just count all columns in them...
	CSQLNode *pTableNode = pFromClause;
	CSQLNode *pNode = NULL;
	while ( pTableNode )
		{
		CTable *pTempTable = GetServer()->GetSystemCatalog()->GetTable( pTableNode->m_pSQLToken->m_strText.c_str() );
		for ( int nCount = 0; nCount < pTempTable->GetFieldCount(); nCount++ )
			{
			//We need to add these manually

			CSQLNode *pColNode = m_pTree->ManuallyAddDot( pTempTable->GetName().c_str(), pTempTable->GetFieldName(nCount) );
			if ( pNode )
				pNode->AddLast( pColNode );
			else
				pNode = pColNode;
			}
		nColCount += pTempTable->GetFieldCount();
		pTableNode = pTableNode->m_pNext;
		}
	m_pTree->SetList( "ColNameList", pNode );
	//
	}
else
	{
	//Check all columns, and fully qualify them
	CSQLNode *pColNode = pSelectClause;
	//For columnnames leftchild is table and rightchild is column
	while ( pColNode )
		{
		if ( pColNode->m_pLeftChild == NULL )
			{
			int rc = _QualifyColName( pColNode, pFromClause );
			if ( rc )
				return rc;
			// Check to see if any table contains the column
			const char *szColName = pColNode->m_pRightChild->m_pSQLToken->m_strText.c_str();
			int nFoundCount = 0;
			CSQLNode *pTableNode = pFromClause;
			while ( pTableNode )
				{
				CTable *pTempTable = GetServer()->GetSystemCatalog()->GetTable( pTableNode->m_pSQLToken->m_strText.c_str() );
				if ( pTempTable->GetColumnInfo( szColName ))
					{
					if ( nFoundCount == 0 )
						{
						pColNode->m_pLeftChild = m_pTree->ManuallyAddIdentifier( pTempTable->GetName().c_str(), "Id" );
						}
					else
						return ERROR_NUM_AMBIGOUS_COLNAMES;
					nFoundCount++;					
					}
				pTableNode = pTableNode->m_pNext;
				}
			if ( nFoundCount == 0 )
				{
				return ERROR_NUM_NOSUCHCOLUMN;
				}
			}
		else
			{
			const char *szColName = pColNode->m_pRightChild->m_pSQLToken->m_strText.c_str();
			const char *szTableName = pColNode->m_pLeftChild->m_pSQLToken->m_strText.c_str();
			
			//And check to see so the table specified is in the fromlist 
			CSQLNode *pTableNode2 = pFromClause;
			bool fFound = false;
			while ( pTableNode2 )
				{
				if ( !pTableNode2->m_pSQLToken->m_strText.compare( szTableName ) )
					{
					fFound = true;
					break;
					}
				pTableNode2 = pTableNode2->m_pNext;
				}
			if ( !fFound )
				return ERROR_NUM_SELECTTABLENOTINFROMCLAUSE;

			CTable *pTempTable = GetServer()->GetSystemCatalog()->GetTable( szTableName );
			if ( !pTempTable->GetColumnInfo( szColName ))
				{
				return ERROR_NUM_NOSUCHCOLUMN;
				}
			}
		pColNode = pColNode->m_pNext;
		nColCount++;
		}
	}
return 0;
}




int CExecuter::ExecuteQuery()
{
//Can only be select
CSQLNode *pSelectClause = m_pTree->GetList( "ColNameList" );
CSQLNode *pFromClause = m_pTree->GetList( "NameIdent" );
CSQLNode *pWhereClause = m_pTree->GetList( "WhereClause" );
CSQLNode *pOrderByClause = m_pTree->GetList( "OrderColList" );

if ( !_CheckTableNames( pFromClause ) )
	return ERROR_NUM_SELECTNOTABLE;

int nColCount;
int nRet = _CheckAndQualifyColumnNames( pSelectClause, pFromClause, nColCount );
if ( nRet )
	return nRet;
if ( pSelectClause->m_nType == NodeType_Star )
	pSelectClause = m_pTree->GetList( "ColNameList" );

nRet = _CheckAndQualifyWhereNames( pFromClause, pWhereClause );
if ( nRet )
	return nRet;

nRet = _CheckAndValidateOrderNames( pFromClause, pOrderByClause );
CSHOptimizer Optimizer;
Optimizer.OptimizeQuery( m_pTree );
Optimizer.GetBestSolution()->Execute();
return ERROR_NUM_OK;
}
		  


int CExecuter::_RecordUpdate( CTable *pTable, CSQLNode *pDotColNode, CSQLNode *pDataNode, int nColCount )
{
CColumnDesc *pSchemaRec;
int nIndex;
if ( nColCount != -1 )
	{
	nIndex = nColCount;
	}
else
	{
	const char *szColName = pDotColNode->m_pRightChild->m_pSQLToken->m_strText.c_str();
	nIndex = pTable->GetFieldNo( (char *)szColName );
	}

pSchemaRec = pTable->GetColumnInfo( nIndex );

switch( pSchemaRec->m_DataType )
	{
	case MENTORSQL_DATATYPE_CHAR:
		{
		if ( pDataNode->m_nType != NodeType_String_Value )
			return ERROR_NUM_WRONG_DATATYPE;

		std::string strTemp = pDataNode->m_pSQLToken->m_strText;
		strTemp = strTemp.substr( 1, strTemp.length()-2);
		pTable->PutField( nIndex, (char *)strTemp.c_str() );
		}
		//pTable->m_pDBF->PutLongField( 
		break;
	case MENTORSQL_DATATYPE_INT:
		if ( pDataNode->m_nType != NodeType_Integer_Value )
			return ERROR_NUM_WRONG_DATATYPE;
		pTable->PutLongField( nIndex, atol(pDataNode->m_pSQLToken->m_strText.c_str()) );
		break;
	case MENTORSQL_DATATYPE_FLOAT:
		if ( pDataNode->m_nType != NodeType_Float_Value )
			return ERROR_NUM_WRONG_DATATYPE;
		pTable->PutDoubleField( nIndex, atof(pDataNode->m_pSQLToken->m_strText.c_str()) );
		break;
	}
return 0;
}

int CExecuter::ExecuteUpdate()
{
CSQLNode *pTableIdent = m_pTree->GetList( "NameIdent" );
CSQLNode *pWhereClause = m_pTree->GetList( "WhereClause" );
CSQLNode *pUpdateList = m_pTree->GetList( "UpdateList");

std::string strTable = pTableIdent->m_pSQLToken->m_strText;

CTable *pTable = GetServer()->GetSystemCatalog()->GetTable( strTable.c_str() );
if ( !pTable )
	return ERROR_NUM_INVALIDTABLE;

//Check and qualify Updatenames
int nRet = _CheckAndQualifyUpdateList( pTableIdent, pUpdateList );
if ( nRet )
	return nRet;

//We are gonna loop through all records and applying the where clause...
nRet = _CheckAndQualifyWhereNames( pTableIdent, pWhereClause );
if ( nRet )
	return nRet;

CSHOptimizer Opt;
Opt.OptimizeRowAccess( m_pTree );
CAccessMethod *pAccessMethod = Opt.GetBestRowAccessMethod();


int nAffected = 0;
while ( pAccessMethod->GetRow() == GETROW_OK )
	{
	//The plan operator should be an CAccessMethod
	CSQLNode *pValueList = pUpdateList;
	while ( pValueList )
		{
		nRet = _RecordUpdate( pAccessMethod->m_pTableRef, pValueList->m_pLeftChild, pValueList->m_pRightChild );
		if ( nRet )
			return nRet;
		pValueList = pValueList->m_pNext;
		}
	pAccessMethod->m_pTableRef->PutRecord( pAccessMethod->m_pTableRef->GetCurRecNo() );
	nAffected++;
	}
printf("Affected count %d\n", nAffected );
pAccessMethod->m_pTableRef->GetFirstRecord(); //Flush the table...
return 0;
}


//Here we make use of the optimizer in deteriming which rows should be deleted
//first though a check if every record should be deleted
int CExecuter::ExecuteDelete()
{
CSQLNode *pTableIdent = m_pTree->GetList( "NameIdent" );
CSQLNode *pWhereClause = m_pTree->GetList( "WhereClause" );

std::string strTable = pTableIdent->m_pSQLToken->m_strText;

CTable *pTable = GetServer()->GetSystemCatalog()->GetTable( strTable.c_str() );
if ( !pTable )
	return ERROR_NUM_INVALIDTABLE;

if ( pWhereClause->m_nType == NodeType_Empty )
	{
	//All records should be deleted...
	pTable->DeleteAllRecords();
	pTable->GetFirstRecord(); //Flush the table...
	}
else
	{
	//Some parsing to do...
	//
	//However, first check to see if columns are ok
	int nRet = _CheckAndQualifyWhereNames( pTableIdent, pWhereClause );
	if ( nRet )
		return nRet;

	// Here we should call the opptimizer - we want it to optimize one table access
	// 
	CSHOptimizer Opt;
	Opt.OptimizeRowAccess( m_pTree );
	CAccessMethod *pAccessMethod = Opt.GetBestRowAccessMethod();
	

	int nAffected = 0;
	while ( pAccessMethod->GetRow() == GETROW_OK )
		{
		pAccessMethod->m_pTableRef->DeleteRecord();
		nAffected++;
		}
	printf("Deleted count %d\n", nAffected );
	}
return 0;
}


//Insert is pretty simple - nothing to optimize - at least yet ( as long as 
// INSERT INTO FROM is not supported )
int CExecuter::ExecuteInsert()
{
CSQLNode *pTableIdent = m_pTree->GetList( "NameIdent" );
CSQLNode *pValueTypeList = m_pTree->GetList( "InsertValueList" );

std::string strTable = pTableIdent->m_pSQLToken->m_strText;

CTable *pTable = GetServer()->GetSystemCatalog()->GetTable( strTable.c_str() );

pTable->BlankRecord();

int nColCount = 0;
while ( pValueTypeList )
	{
	int nRet = _RecordUpdate( pTable, pValueTypeList, pValueTypeList, nColCount );
	if ( nRet )
		return nRet;
	pValueTypeList = pValueTypeList->m_pNext;
	nColCount++;
	}
pTable->AppendRecord();
pTable->GetLastRecord();
return 0;
}			



/*
void Dump( CTempTable *pSet , long lRecNo )
{
   int i;
   char buf[241];
 
   for( i = 0; i < pSet->GetFieldCount(); i++ )
   {
      pSet->GetField( i, buf );
      printf ("%s\t", buf );
   }
   printf("\n");
}

void CExecuter::DumpTable()
{
//
for ( int i = 0; i < m_pResultSet->GetFieldCount(); i++ )
	{
	printf( m_pResultSet->GetFieldName(i) );
	printf( "\t");
	}
printf( "\n");
printf( "\n");
short rc = m_pResultSet->GetFirstRecord();
while ( rc == ERROR_NUM_OK )
	{
	Dump( m_pResultSet, m_pResultSet->GetCurRecNo() );
	rc = m_pResultSet->GetNextRecord();
	}
}
*/
