/*	scr2com.c

A PC text screen compiler.
Execute with the name of the output file, .COM assumed.
SCR2COM reads the attributes and characters
from the video buffer, then writes a .COM file
which will redisplay them when it is executed.

A portion of the screen may be captured instead of the default
full screen.  Specify the limits on the command line as follows:

-t  Top row (default 0)
-b  Bottom row (default 24)
-l  Left column (default 0)
-r  Right column (default 79)

For any limit that is not specified, the default is used.

SCR2COM uses the Zortech disp_ package for video IO.
Must be compiled with -b switch.

This program uses a very simple technique of #including and
compiling a HEX representation of a .COM program that will
perform the redisplay.  The .COM program (COM.C, COM.COM) was
scanned "by hand" to determine the offsets for the video buffer
integer size limits.  These are hard coded into this source, and
must be modified if COM.C is changed.

Written by Francis X. Guidry
Released to the public domain

*/

#include <stdio.h>
#include <string.h>
#include <disp.h>

unsigned char prog[] = {
#include "com.hex"
};

unsigned progsize = 9392;
unsigned buffoffs = 0x1384;
unsigned *tp = (unsigned *) (prog + 0x137c);
unsigned *bp = (unsigned *) (prog + 0x137e);
unsigned *lp = (unsigned *) (prog + 0x1380);
unsigned *rp = (unsigned *) (prog + 0x1382);

unsigned t = 0;
unsigned b = 24;
unsigned l = 0;
unsigned r = 79;

/* -----------------------------------------------------------
	Test command line.  Drive the remaining processing.
   ----------------------------------------------------------- */

main(int argc, char** argv)
{
	FILE *f;
	FILE *outfile(char *);
	char *parse(int,char**);
	char *name;
	
	if(argc < 2)
		errexit("Must specify output file");
	name = parse(argc,argv);
	f = outfile(name);
	readvid();
	writecom(f);
	return 0;
}


/* -----------------------------------------------------------
	Exit due to an error condition.
   ----------------------------------------------------------- */

errexit(char *s)
{
	puts(s);
	exit(1);
}


/* -----------------------------------------------------------
	Parse the command line.  Return the file name.
   ----------------------------------------------------------- */

char *
parse(int argc,char **argv)
{
	int i, v;
	char *s, *n;
	
	for(i = 1; i < argc; i++) {
		if(**(argv + i) == '-') {
			s = *(argv + i);
			s++;
			if(*s == 't' || *s == 'T')  {
				if(*(++s))
					v = atoi(s);
				else v = atoi(*(argv + (++i)));
				if(v >= 0 && v <= 24)
					t = v;
				else errexit("-t parameter must be 0 through 24");
			} else if(*s == 'l' || *s == 'L') {
				if(*(++s))
					v = atoi(s);
				else v = atoi(*(argv + (++i)));
				if(v >= 0 && v <= 79)
					l = v;
				else errexit("-l parameter must be 0 through 79");
			} else if(*s == 'b' || *s == 'B') {
				if(*(++s))
					v = atoi(s);
				else v = atoi(*(argv + (++i)));
				if(v >= 0 && v <= 24)
					b = v;
				else errexit("-b parameter must be 0 through 24");
			} else if(*s == 'r' || *s == 'R') {
				if(*(++s))
					v = atoi(s);
				else v = atoi(*(argv + (++i)));
				if(v >= 0 && v <= 79)
					r = v;
				else errexit("-r parameter must be 0 through 79");
			} else errexit("Invalid option");
		} else n = *(argv + i);
	}
	if(r <= l)
		errexit("r value must be greater than l value");
	if(b <= t)
		errexit("b value must be greater than t value");
	return n;
}


/* -----------------------------------------------------------
	Open the output file.  Add .COM if not present in filename.
   ----------------------------------------------------------- */

FILE *
outfile(char *s)
{
	FILE *f;
	char name[256];
	
	strlwr(s);
	strcpy(name,s);
	if(!strstr(name,".com"))
		strcat(name,".com");
	f = fopen(name,"wb");
	if(!f)
		errexit("Unable to open output file.");
	return f;
}


/* -----------------------------------------------------------
	Copy the screen into the .COM file buffer.
   ----------------------------------------------------------- */

readvid()
{
	disp_open();
	disp_peekbox((unsigned *) (prog + buffoffs),t,l,b,r);
	disp_close();
	*tp = t;
	*lp = l;
	*bp = b;
	*rp = r;
}


/* -----------------------------------------------------------
	Write the .COM file
   ----------------------------------------------------------- */

writecom(FILE *f)
{
	fwrite(prog,1,progsize,f);
	fclose(f);
}



