*.............................................................................
*
*   Program Name: GLLIBR.PRG       Created By: Global Technologies Corporation
*   Date Created: 06/05/90           Language: Clipper 5.0
*   Time Created: 11:27:44             Author: Bill French
*
*   The Graphics Language - Copyright (c) 1990,1991 - Bits Per Second Ltd.
*            In Association With Global Technologies Corporation
*
*.............................................................................
#include "gllibr.ch"

static _screens_[MaxScreens][6]                  // declare the screen array
static _handles_[MaxHandles][10]                 // declare the object array
static _eshadow_ := "n+/b"                       // declare the default shadow color
static _icnfile_ := ""                           // current icon file
static _dgepath_ := ""                           // declare the dge resources path
static _icnwidt_ 
static _icnheig_ 

// __SetGraphics() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Initialize graphics mode and establish system variables
// Mapped Command: SET GRAPHICS
FUNCTION __SetGraphics(mode)
   local screen, handle
   mode := if(mode == NIL, FALSE, mode)
   if mode                                       // is it on or off? (TRUE = on)
      sethires(0)                                // graphics mode
      for screen := 1 to MaxScreens              // establish a blank screen array
         _screens_[screen,1] := NullInteger      // upper left row
         _screens_[screen,2] := NullInteger      // upper left column
         _screens_[screen,3] := NullInteger      // lower right row
         _screens_[screen,4] := NullInteger      // lower right column
         _screens_[screen,5] := NullInteger      // dGE handle
         _screens_[screen,6] := NullString       // GL memvar
      next                                       // for n := 1 to MaxHandles
      for handle := 1 to MaxHandles              // establish a blank object array
         _handles_[handle,1] := NullInteger      // upper left row
         _handles_[handle,2] := NullInteger      // upper left column
         _handles_[handle,3] := NullInteger      // lower right row
         _handles_[handle,4] := NullInteger      // lower right column
         _handles_[handle,5] := NullString       // object text
         _handles_[handle,6] := NullInteger      // object type
         _handles_[handle,7] := ShadowOff        // shadow
         _handles_[handle,8] := NullString       // object name
         _handles_[handle,9] := InactiveObject   // status (inactive)
      next                                       // for n := 1 to MaxHandles
      _icnwidt_ := getfontinf(2)/PointsPerColumn // get the icon width
      _icnheig_ := getfontinf(3)/PointsPerLine   // get the icon height
   else
      settext()                                  // text mode
   endif                                         // if off                                        // if were leaving
RETURN(Void)

// __SetVideo() ---------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Set the dGE video mode for EGA of VGA
// Mapped Command: SET VIDEO TO
FUNCTION __SetVideo(video)
   do case
   case upper(video) == "EGA"                    // ega mode
      setvideo(6)
   case upper(video) == "VGA"                    // vga mode
      setvideo(7)
   otherwise                                     // default to ega mode
      setvideo(6)
   endcase
RETURN(Void)

// __SetResources() -----------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Set the dGE resource search path
// Mapped Command: SET DGE RESOURCES TO
FUNCTION __SetResources(path)
   path := if(empty(path),"",path + "\")
   path := if(empty(path),getenv("DGE") + "\",path)
   _dgepath_ := path
RETURN(_dgepath_)

// __SetPalette() -------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Set the graphics screen background color
// Mapped Command: SET PALETTE BACKGROUND
FUNCTION __SetPalette(color,bright)
   setcolor(setcolor())
   setpal(__PalWordToColor(bright + color),0,0)  // set the palette background
RETURN(Void)

// __ClearGScreen() -----------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Clear the graphics screen
// Mapped Command: CLEAR GRAPHICS SCREEN
FUNCTION __ClearGScreen()
   clrscreen()                                   // clear the graphics screen
RETURN(Void)

// __ClearGWindow() -----------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Clear a window area in the graphics screen
// Mapped Command: CLEAR GRAPHICS WINDOW
FUNCTION __ClearGWindow(Pos1_a,Pos1_b,Pos2_a,Pos2_b,bevel)
   if bevel
      clrwin(__XdGE(Pos1_b-.325),__YdGE(Pos2_a+.15),__XdGE(Pos2_b+.325),__YdGE(Pos1_a-.15))
   else
      clrwin(__XdGE(Pos1_b),__YdGE(Pos2_a),__XdGE(Pos2_b),__YdGE(Pos1_a))
   endif                                         // if bevel
RETURN(Void)

// __ResetGArray() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Reset the dGE internal array
// Mapped Command: RESET GRAPHICS ARRAY
FUNCTION __ResetGArray()
   datareset()                                   // reset the dGE data array
RETURN(Void)

// __ScaleGArray() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Adjust the scale of data
// Mapped Command: SCALE GRAPHICS ARRAY
FUNCTION __ScaleGArray(percent)
   datapc(percent)                               // scale the dGE data array
RETURN(Void)

// __SetDrawArea() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Restrict drawing to a window area
// Mapped Command: SET DRAWING AREA
FUNCTION __SetDrawArea(Pos1a,Pos1b,Pos2a,Pos2b)
   if Pos1a == NIL
      clipwin(0,0,1350,1000)
   else
      clipwin(__XdGE(Pos1b),__YdGE(Pos2a),__XdGE(Pos2b),__YdGE(Pos1a))
   endif                                         // if pos1a == nil
RETURN(Void)

// __SaveGScreen() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Save an area of the graphics screen
// Mapped Command: SAVE GRAPHICS SCREEN
FUNCTION __SaveGScreen(label,Pos1a,Pos1b,Pos2a,Pos2b)
   local handle
   local screen := __UnusedScreen(label)
   if screen > 0
      handle = snapcopy(__XdGE(Pos1b),__YdGE(Pos2a),__XdGE(Pos2b),__YdGE(Pos1a),0)
      if handle != 0
         _screens_[screen,1] := Pos1a            // upper left row
         _screens_[screen,2] := Pos1b            // upper left column
         _screens_[screen,3] := Pos2a            // lower right row
         _screens_[screen,4] := Pos2b            // lower right column
         _screens_[screen,5] := handle           // dGE video handle
         _screens_[screen,6] := label            // screen label
      else
         __HandleError(NoMemoryLeft,label)
      endif
   else
      __HandleError(NoHandlesLeft,label)
   endif
RETURN(screen)

// __RestGScreen() ------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Restore a saved area of the graphics screen
// Mapped Command: RESTORE GRAPHICS SCREEN
FUNCTION __RestGScreen(label)
   local Pos1a, Pos1b, Pos2a, Pos2b, Handle
   local screen := __ScanScreens(label)
   if screen > 0
      Pos1a  := _screens_[screen,1]           // upper left row
      Pos1b  := _screens_[screen,2]           // upper left column
      Pos2a  := _screens_[screen,3]           // upper left row
      Pos2b  := _screens_[screen,4]           // upper left column
      handle := _screens_[screen,5]           // dGE handle
      if handle != 0
         snappaste(__XdGE(Pos1b),__YdGE(Pos2a),handle)
         snapkill(handle)
         _screens_[screen,1] := NullInteger   // upper left row
         _screens_[screen,2] := NullInteger   // upper left column
         _screens_[screen,3] := NullInteger   // lower right row
         _screens_[screen,4] := NullInteger   // lower right column
         _screens_[screen,5] := NullInteger   // dGE handle
         _screens_[screen,6] := NullString    // GL memvar
      else
         __HandleError(NoHandlesLeft,screen)
      endif
   else
      __HandleError(NoSuchHandle,label)
   endif
RETURN(Void)

// __UnusedScreen() -----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Find a free screen handle
// Mapped Command: 
FUNCTION __UnusedScreen(label)
   local n
   for n := 1 to MaxScreens
      if empty(_screens_[n,6])
         retu(n)
      endif                                      // if _handles_[n,8] := object
   next                                          // for n := 1 to MaxHandles
RETURN(0)

// __ScanScreens() ------------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Find the handle of a specified screen label
// Mapped Command: 
FUNCTION __ScanScreens(label)
   local n
   for n := 1 to MaxScreens
      if _screens_[n,6] == label
         retu(n)
      endif                                      // if _handles_[n,8] := object
   next                                          // for n := 1 to MaxHandles
RETURN(0)

// __ShadeArea() --------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Fill an enclosed area
// Mapped Command: SHADE AREA AT
FUNCTION __ShadeArea(x,y,pattern)
   shade(__XdGE(y),__YdGE(x),if(pattern == NIL,0,pattern),__DgeColor(setcolor()))
RETURN(Void)

// __DrawFrame() --------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Draw box
// Mapped Command: DRAW BOX FROM
FUNCTION __DrawFrame(x1,y1,x2,y2,pattern,bevel)
   pattern := if(pattern == NIL,64,pattern)
   if bevel
      __DrawBevel(x1,y1,x2-x1,y2-y1,pattern)
   else
      boxfill(__XdGE(y1),__YdGE(x2),__XdGE_(y2-y1),__YdGE_(x2-x1),pattern,__DgeColor(setcolor()))
   endif                                         // if bevel
RETURN(Void)

// __DrawCircle() -------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Draw a circle
// Mapped Command: DRAW CIRCLE AT
FUNCTION __DrawCircle(x,y,radius)
   drawcircle(__XdGE(y),__YdGE(x),__XdGE_(radius),0,360,0,0,__DgeColor(setcolor()))
RETURN(Void)

// __DrawLine() ---------------------------------------------------------------
// TecGuide-> {Function Ref::Graphical Functions::UDF} {SOURCE}
//    Description: Draw a line
// Mapped Command: DRAW LINE FROM
FUNCTION __DrawLine(Pos1_a,Pos1_b,Pos2_a,Pos2_b,style)
   drawline(__XdGE(Pos1_b),__YdGE(Pos1_a),__XdGE(Pos2_b),__YdGE(Pos2_a),0,if(style == NIL,0,style),__DgeColor(setcolor()))
RETURN(Void)

// __SetCSet() ----------------------------------------------------------------
// TecGuide-> {Function Ref::String Functions::UDF} {SOURCE}
//    Description: Set the current character set
// Mapped Command: SET CHARACTER SET
FUNCTION __SetCSet(type,size)
   type := upper(substr(type,1,4))               // get the character type
   size := upper(substr(size,1,4))               // get the character size
   do case                                       // evaluate the type
   case type == "SYST"                           // standard dge character sets
      do case
      case size == "SMAL" .and. file(_dgepath_+"DGE1EGA.CHR")
         loadcset(0,_dgepath_+"DGE1EGA.CHR")
      case (size == "LARG" .or. size == "STAN" .or. size == "STD") .and. file(_dgepath_+"DGE0EGA.CHR")
         loadcset(0,_dgepath_+"DGE0EGA.CHR")
      case size == "0906" .and. file(_dgepath_+"DGE0906.STX")
         loadcset(0,_dgepath_+"DGE0906.STX")
      case size == "1106" .and. file(_dgepath_+"DGE1106.STX")
         loadcset(0,_dgepath_+"DGE1106.STX")
      case size == "1108" .and. file(_dgepath_+"DGE1108.STX")
         loadcset(0,_dgepath_+"DGE1108.STX")
      case size == "1608" .and. file(_dgepath_+"DGE1608.STX")
         loadcset(0,_dgepath_+"DGE1608.STX")
      case size == "1609" .and. file(_dgepath_+"DGE1609.STX")
         loadcset(0,_dgepath_+"DGE1609.STX")
      endcase
   case type == "ROMA"                           // roman character sets
      do case
      case size == "1628" .and. file(_dgepath_+"RMN1628.STX")
         loadcset(0,_dgepath_+"RMN1628.STX")
      case size == "1914" .and. file(_dgepath_+"RMN1914.STX")
         loadcset(0,_dgepath_+"RMN1914.STX")
      case size == "2828" .and. file(_dgepath_+"RMN2828.STX")
         loadcset(0,_dgepath_+"RMN2828.STX")
      case size == "3828" .and. file(_dgepath_+"RMN3828.STX")
         loadcset(0,_dgepath_+"RMN3828.STX")
      case size == "5742" .and. file(_dgepath_+"RMN5742.STX")
         loadcset(0,_dgepath_+"RMN5742.STX")
      endcase
   case type == "SWIS"                           // swiss character sets
      do case
      case size == "1425" .and. file(_dgepath_+"SWI1425.STX")
         loadcset(0,_dgepath_+"SWI1425.STX")
      case size == "1713" .and. file(_dgepath_+"SWI1713.STX")
         loadcset(0,_dgepath_+"SWI1713.STX")
      case size == "2525" .and. file(_dgepath_+"SWI2525.STX")
         loadcset(0,_dgepath_+"SWI2525.STX")
      case size == "3325" .and. file(_dgepath_+"SWI3325.STX")
         loadcset(0,_dgepath_+"SWI3325.STX")
      case size == "4937" .and. file(_dgepath_+"SWI4937.STX")
         loadcset(0,_dgepath_+"SWI4937.STX")
      endcase
   endcase
RETURN(Void)

// __DrawText() ---------------------------------------------------------------
// TecGuide-> {Function Ref::String Functions::UDF} {SOURCE}
//    Description: Draw graphical text
// Mapped Command: DRAW <string> AT
FUNCTION __DrawText(text,x,y,type,size,vertical,center,rightjust)
   local mode
   vertical  := if(vertical == NIL,0,vertical)   // determine positioning
   center    := if(center == NIL,0,center)       // horizontal positioning (center)
   rightjust := if(rightjust == NIL,0,rightjust) // horizontal positioning (right just)
   mode      := vertical + center + rightjust    // calculate the display mode
   __SetCSet(if(type == NIL,"",type),if(size == NIL,"",size))
   saystring(__XdGE(y),__YdGE(x),4,mode,__DgeColor(setcolor()),text)
RETURN(Void)

// __SetDelimiter() -----------------------------------------------------------
// TecGuide-> {Function Ref::String Functions::UDF} {SOURCE}
//    Description: Set the string input delimiters
// Mapped Command: SET PROMPT DELIMITER
FUNCTION __SetDelimiter(chr)
   setdelim(chr)                                 // set the get delimiter
RETURN(Void)

// __SetIcon() ----------------------------------------------------------------
// TecGuide-> {Function Ref::Icon Functions::UDF} {SOURCE}
//    Description: Set the current icon file
// Mapped Command: SET ICON
FUNCTION __SetIcon(iconfile)
   if iconfile == NIL                            // if no file name was passed
      loadicon("")                               // clear the icon file in dGE
      _icnfile_ := ""                            // reset the static variable
   else                                          // otherwise...
      loadicon(_dgepath_+iconfile)               // load the file that was specified and set the static variable
      _icnfile_ := if(len(_dgepath_) > 0,_dgepath_ + iconfile,iconfile)
   endif                                         // if iconfile == nil
RETURN(_icnfile_)

// __DrawStdIcon() ------------------------------------------------------------
// TecGuide-> {Function Ref::Icon Functions::UDF} {SOURCE}
//    Description: Draw internal icon
// Mapped Command: DRAW STD ICON <icon>
FUNCTION __DrawStdIcon(icon,x,y,vector,xor)
   local mode
   vector  := if(vector == NIL,FALSE,vector)
   xor     := if(xor == NIL,FALSE,xor)
   mode    := 0                                  // establish cartesion drawing mode
   mode    := mode + if(vector,1,0)              // vector drawing mode
   mode    := mode + if(xor,16,0)                // vector drawing mode
   drawicon(__XdGE(y),__YdGE(x),mode,icon,__DgeColor(setcolor()))
RETURN(Void)

// __DrawSuperIcon() ----------------------------------------------------------
// TecGuide-> {Function Ref::Icon Functions::UDF} {SOURCE}
//    Description: Draw super icon
// Mapped Command: DRAW SUPER ICON <icon>
FUNCTION __DrawSuperIcon(icon,x,y,vector,replace,or,black,inverse,composite,p1,p2,p3,p4)
   local mode := 0                               // establish cartesian drawing mode
   vector  := if(vector == NIL,FALSE,vector)
   or      := if(or == NIL,FALSE,or)
   black   := if(black == NIL,FALSE,black)
   inverse := if(inverse == NIL,FALSE,inverse)
   mode    := mode + if(vector,1,0)              // vector drawing mode
   mode    := mode + if(or,8,0)                  // xor mode
   mode    := mode + if(black,32,0)              // black mode
   mode    := mode + if(inverse,64,0)            // inverse mode
   icon    := icon + 16
   do case
   case composite == TRUE
      replace := if(replace == NIL,FALSE,replace)
      mode    := mode + if(replace,4,0)          // replace mode
      drawicon(__XdGE(y-(_icnwidt_/2)),__YdGE(x+(_icnheig_/2)),mode,icon+0,__DgeColor(setcolor()))
      drawicon(__XdGE(y-(_icnwidt_/2)),__YdGE(x-(_icnheig_/2)),mode,icon+1,__DgeColor(setcolor()))
      drawicon(__XdGE(y+(_icnwidt_/2)),__YdGE(x+(_icnheig_/2)),mode,icon+2,__DgeColor(setcolor()))
      drawicon(__XdGE(y+(_icnwidt_/2)),__YdGE(x-(_icnheig_/2)),mode,icon+3,__DgeColor(setcolor()))
   case p1 != NIL
      replace := if(replace == NIL,FALSE,replace)
      mode    := mode + 4                        // replace mode
      drawicon(__XdGE(y),__YdGE(x),mode,icon+0,p1)
      drawicon(__XdGE(y),__YdGE(x),mode,icon+1,p2)
      drawicon(__XdGE(y),__YdGE(x),mode,icon+2,p3)
      drawicon(__XdGE(y),__YdGE(x),mode,icon+3,p4)
   otherwise
      replace := if(replace == NIL,FALSE,replace)
      mode    := mode + if(replace,4,0)          // replace mode
      drawicon(__XdGE(y),__YdGE(x),mode,icon,__DgeColor(setcolor()))
   endcase
RETURN(Void)

// __SetPrintDevice() ---------------------------------------------------------
// TecGuide-> {Function Ref::Printer Functions::UDF} {SOURCE}
//    Description: Establish the print device and channel
// Mapped Command: SET GRAPHICS PRINT
FUNCTION __SetPrintDevice(lpt1,lpt2,lpt3,com1,com2)
   do case
   case lpt1                                     // lpt1
      prndev(0,1)
   case lpt2                                     // lpt2
      prndev(0,2)
   case lpt3                                     // lpt3
      prndev(0,3)
   case com1                                     // com1
      prndev(1,1)
   case com2                                     // com2
      prndev(1,2)
   endcase
RETURN(Void)

// __PrintMatrix() ------------------------------------------------------------
// TecGuide-> {Function Ref::Printer Functions::UDF} {SOURCE}
//    Description: Print screen to a matrix printer
//    dGE functions: printscr()
// Mapped Command: PRINT IMAGE TO MATRIX
FUNCTION __PrintMatrix()
   printscrn()
RETURN(Void)

// __PrintLaser() -------------------------------------------------------------
// TecGuide-> {Function Ref::Printer Functions::UDF} {SOURCE}
//    Description: Print screen to a laser printer
// Mapped Command: PRINT IMAGE TO LASER
FUNCTION __PrintLaser(reset,formfeed,aspect,paintjet,bwpaintjet,landscape,reverse,hoffset,voffset,density)
   local mode := reset+formfeed+aspect+paintjet+bwpaintjet+landscape+reverse
   hoffset    := if(hoffset == NIL,0,hoffset)
   voffset    := if(voffset == NIL,0,voffset)
   density    := if(density == NIL,0,density)
   printpcl(mode,hoffset,voffset,density)
RETURN(Void)

// __PrintPostScript() --------------------------------------------------------
// TecGuide-> {Function Ref::Printer Functions::UDF} {SOURCE}
//    Description: Print screen to a postscript printer
// Mapped Command: PRINT IMAGE TO POSTSCRIPT
FUNCTION __PrintPostScript(landscape,reverse,hoffset,voffset,hscale,vscale,density)
   local mode := landscape + reverse
   hoffset    := if(hoffset == NIL,0,hoffset)
   voffset    := if(voffset == NIL,0,voffset)
   hscale     := if(hscale  == NIL,0,hscale )
   vscale     := if(vscale  == NIL,0,vscale )
   density    := if(density == NIL,0,density)
   printps(mode,hoffset,voffset,hscale,vscale,density)
RETURN(Void)

// __SetVectorPrint() ----------------------------------------------------------
// TecGuide-> {Function Ref::Printer Functions::UDF} {SOURCE}
//    Description: Toggle vector printing ON or OFF
// Mapped Command: SET VECTOR PRINT
FUNCTION __SetVectorPrint(command,hoffset,voffset,hlength,units,vscale,orient,postscript,window,color,pattern,noeject)
   local mode
   command := if(command == NIL,2,command)
   if command == 1
      hoffset := if(hoffset == NIL,0,hoffset)    // horizontal offset
      voffset := if(voffset == NIL,0,voffset)    // vertical offset
      hlength := if(hlength == NIL,1350,hlength) // default to 1350 pixels
      units   := if(units == NIL,"MMS",upper(units))  // default to mms
      do case                                    // convert units to integer
      case units == "MMS"
         units := 0
      case units == "POIN" .or. units == "1/72"
         units := 1
      case units == "1/100"
         units := 2
      endcase
      vscale  := if(vscale == NIL,100,vscale)    // default to no change in scale
      mode    := 1                               // pcl5 (default)
      mode    := mode + postscript               // postscript
      mode    := mode + window                   // clipping window
      mode    := mode + color                    // color printing
      mode    := mode + pattern                  // pattern priority
      vpon(hoffset,voffset,hlength,units,vscale,orient,mode)  // issue the print off function
   else
      vpoff(noeject)                             // issue the print off function
   endif
RETURN(Void)

// __SetGMouse() --------------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Initialize the mouse and set the cursor type
// Mapped Command: SET MOUSE
FUNCTION __SetGMouse(status,cursor)
   do case                                       // evaluate the requested cursor type
   case cursor == NIL                            // if no cursor was specified
      if status                                  // if ON
         if mreset() > 0                         // mouse reset, return number of buttons
            mcuron()                             // display the mouse cursor
         else
            __RunTimeError(NoMouseDriver,"SET MOUSE ON","__SetGMouse()")
         endif                                   // if mreset() > 0                         // mouse reset, return number of buttons
      else                                       // otherwise
         mcuroff()                               // hide the mouse cursor
      endif                                      // if status (SET MOUSE ON)
   case status == NIL                            // if no status was selected
      mcurtype(cursor)                           // assume the cursor type is being selected
   endcase
RETURN(Void)

// __DefineMouseWindow() ------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Set the area where the mouse can freely move
// Mapped Command: DEFINE MOUSE WINDOW FROM
FUNCTION __DefineMouseWindow(Pos1_a,Pos1_b,Pos2_a,Pos2_b)
   msetwin(__XdGE(Pos1_b),__YdGE(Pos2_a),__XdGE(Pos2_b-1),__YdGE(Pos1_a-1))
RETURN(Void)

// __FixMousePosition() -------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Move the mouse cursor to a new position
//    dGE functions: mfixpos()
// Mapped Command: FIX MOUSE POSITION AT
FUNCTION __FixMousePosition(x,y)
   mfixpos(__XdGE(y),__YdGE(x))                  // establish a specific mouse position
RETURN(Void)

// __SetEventShadow() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Set objct shadow color
// Mapped Command: SET EVENT SHADOW TO <color>
FUNCTION __SetEventShadow(color)
   _eshadow_ := if(color == NIL,"w/n",color)     // set the object shadow color
RETURN(Void)

// __DefEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Define a click region object
// Mapped Command: DEFINE EVENT <label> FROM
FUNCTION __DefEventRegion(label,Pos1_a,Pos1_b,Pos2_a,Pos2_b,activate)
   local handle := __ScanObjects(label)          // see if we can find the object
   handle := if(handle == 0,__FindUnusedHandle(label),handle)
   if __HandleInRange(handle) > 0                // if the handle is valid
      _handles_[handle,01] := Pos1_a
      _handles_[handle,02] := Pos1_b
      _handles_[handle,03] := Pos2_a
      _handles_[handle,04] := Pos2_b
      _handles_[handle,05] := NullString         // n/a in this object type
      _handles_[handle,06] := EventRegionObject  // object type
      _handles_[handle,07] := ShadowOff          // shadow
      _handles_[handle,08] := label              // object name
      _handles_[handle,09] := InactiveObject     // status
      _handles_[handle,10] := NullInteger        // dGE icon number (0 through 7)
      if activate
         __ActEventRegion(label)
      endif
   else                                          // otherwise handle was invalid
      __HandleError(NoHandlesLeft,label)         // branch to handle error routine
   endif                                         // if handle > 0 .and. handle <= maxobjects      // if successful in gettong a get area
RETURN(Void)

// __ActEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Toggles event region to active status
// Mapped Command: ACTIVATE EVENT <label>
FUNCTION __ActEventRegion(label)
   local handle := __ScanObjects(label)          // get a handle if possible
   if handle > 0                                 // find out if the button exists
      _handles_[handle,9] := ActiveObject        // status (active)
      do case
      case _handles_[handle,06] == EventRegionObject
         msethot(handle, ;
            __XdGE(_handles_[handle,2]), ;
            __YdGE(_handles_[handle,3]), ;
            __XdGE_((_handles_[handle,4] - _handles_[handle,2])), ;
            __YdGE_((_handles_[handle,3] - _handles_[handle,1])))
      case _handles_[handle,06] == IconButtonObject
         __ActIconButton(label)
      case _handles_[handle,06] == TextButtonObject
         * ...
      endcase
   else                                          // otherwise the button doesn't exists
      __HandleError(NoSuchLabel,label)           // process the error
   endif
RETURN(Void)

// __FlaEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Redraw an event object for flash effect (default activates)
// Mapped Command: FLASH EVENT <label>
FUNCTION __FlaEventRegion(label)
   local handle := __ScanObjects(label)          // if the button does indeed exits
   if __HandleInRange(handle) > 0                // if we have a valid handle
      do case
      case _handles_[handle,06] == EventRegionObject
         msethot(handle,0,0,0,0)                 // clear the mouse hot region
      case _handles_[handle,06] == IconButtonObject
         __ClrIconButton(handle)                 // clear the icon from the screen
         __ActIconButton(label)                  // redisplay the icon
      case _handles_[handle,06] == TextButtonObject
         * ...
      endcase
   else                                          // apparently there is no object by that name
      __HandleError(NoSuchLabel,label)           // branch to the handle error routine
   endif                                         // if handle > 0 .and. handle <= maxobjects      // if successful in gettong a get area
RETURN(Void)

// __MovEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Moves, activates and redisplays the specified event object
// Mapped Command: MOVE EVENT <label>
FUNCTION __MovEventRegion(label,Pos1,Pos2,activate,deactivate)
   local handle := __ScanObjects(label)          // get a handle if possible
   local PrevPos1, PrevPos2, currcolor
   if handle > 0                                 // find out if the button exists
      PrevPos1 := _handles_[handle,1]            // save the old position
      PrevPos2 := _handles_[handle,2]            // save the old position
      _handles_[handle,1] := Pos1                // status (active)
      _handles_[handle,2] := Pos2                // status (active)
      _handles_[handle,9] := if(activate == NIL,_handles_[handle,9],ActiveObject)
      _handles_[handle,9] := if(deactivate == NIL,_handles_[handle,9],InactiveObject)
      do case
      case _handles_[handle,06] == EventRegionObject
         _handles_[handle,3] := _handles_[handle,3] + (Pos1 - PrevPos1)
         _handles_[handle,4] := _handles_[handle,4] + (Pos2 - PrevPos2)
      case _handles_[handle,06] == IconButtonObject
         msethot(handle, ;
            __XdGE(_handles_[handle,2] - (_icnwidt_/2)), ;
            __YdGE((_handles_[handle,1] + _icnheig_) - (_icnheig_/2)), ;
            __XdGE_(_icnwidt_), ;
            __YdGE_(_icnheig_))
         if _handles_[handle,7]                  // if a shadow has been selected, display shadow
            currcolor := setcolor()              // save the current color
            setcolor(_eshadow_)                  // set color to the shadow color and draw the shadow box
            loadicon(_dgepath_ + "gllibr.ico")
            __DrawSuperIcon(0,_handles_[handle,1]+IconShadowOffsetD,_handles_[handle,2]+IconShadowOffsetR)
            loadicon(_icnfile_)
            setcolor(currcolor)                  // restore the Clipper color
         endif                                   // if shadow
         __DrawSuperIcon(_handles_[handle,10],_handles_[handle,1],_handles_[handle,2])
      case _handles_[handle,06] == TextButtonObject
         * ...
      endcase
   else                                          // otherwise the button doesn't exists
      __HandleError(NoSuchLabel,label)           // process the error
   endif
RETURN(Void)

// __DeaEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Toggles event region to inactive status
// Mapped Command: DEACTIVATE EVENT <label>
FUNCTION __DeaEventRegion(label,clr)
   local handle := __ScanObjects(label)          // get a handle if possible
   if handle > 0                                 // if the object does indeed exist
      _handles_[handle,9] := InactiveObject      // status (inactive)
      do case
      case _handles_[handle,06] == EventRegionObject
         msethot(handle,0,0,0,0)                 // clear the mouse hot region
      case _handles_[handle,06] == IconButtonObject
         __DeaIconButton(label,clr)
      case _handles_[handle,06] == TextButtonObject
         * ...
      endcase
   else                                          // otherwise it's an invalid object
      __HandleError(NoSuchLabel,label)           // branch to the handle error routine
   endif
RETURN(Void)

// __RelEventRegion() ---------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: 
// Mapped Command: RELEASE EVENT <label>
FUNCTION __RelEventRegion(label)
   local handle := __ScanObjects(label)          // if the button does indeed exits
   if handle > 0                                 // if we have a valid handle ID
      do case
      case _handles_[handle,06] == EventRegionObject
         * do nothing...                         // no need to clear anything
      case _handles_[handle,06] == IconButtonObject
         __ClrIconButton(handle)                 // clear the icon from the screen
      case _handles_[handle,06] == TextButtonObject
         * __ClrTextButton(handle)               // clear the text from the screen
      endcase
      _handles_[handle,01] := 0                  // upper left row
      _handles_[handle,02] := 0                  // upper left column
      _handles_[handle,03] := 0                  // lower right row
      _handles_[handle,04] := 0                  // lower right column
      _handles_[handle,05] := NullString         // object text
      _handles_[handle,06] := 0                  // object type
      _handles_[handle,07] := ShadowOff          // shadow
      _handles_[handle,08] := NullString         // object name
      _handles_[handle,09] := InactiveObject     // status (inactive)
      _handles_[handle,10] := NullInteger        // dGE icon number (0 through 7)
   else                                          // apparently there is no object by that name
      __HandleError(NoSuchLabel,label)           // branch to the handle error routine
   endif                                         // if handle > 0 .and. handle <= maxobjects      // if successful in gettong a get area
RETURN(Void)

// __DefIconButton() ----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Define and optionally activate a super icon button
// Mapped Command: DEFINE EVENT <label> AT
FUNCTION __DefIconButton(label,Pos1,Pos2,icon,activate,shadow)
   local handle := __ScanObjects(label)          // see if we can find the object
   handle := if(handle == 0,__FindUnusedHandle(label),handle)
   if __HandleInRange(handle) > 0                // if the handle is valid
      _handles_[handle,01] := (Pos1)
      _handles_[handle,02] := (Pos2)
      _handles_[handle,03] := 0
      _handles_[handle,04] := 0
      _handles_[handle,05] := NullString         // n/a in this object type
      _handles_[handle,06] := IconButtonObject   // object type
      _handles_[handle,07] := shadow             // shadow
      _handles_[handle,08] := label              // object name
      _handles_[handle,09] := InactiveObject     // status
      _handles_[handle,10] := icon               // dGE icon number (0 through 7)
      if activate
         __ActIconButton(label)
      endif
   else                                          // otherwise handle was invalid
      __HandleError(NoHandlesLeft,label)         // branch to handle error routine
   endif                                         // if handle > 0 .and. handle <= maxobjects      // if successful in gettong a get area
RETURN(Void)

// __ActIconButton() ----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Toggles the button to active and displays it
// Mapped Command:
FUNCTION __ActIconButton(label)
   local currcolor
   local handle := __ScanObjects(label)          // get a handle if possible
   if handle > 0                                 // find out if the button exists
      _handles_[handle,9] := ActiveObject        // status (active)
      msethot(handle, ;
         __XdGE(_handles_[handle,2] - (_icnwidt_/2)), ;
         __YdGE((_handles_[handle,1] + _icnheig_) - (_icnheig_/2)), ;
         __XdGE_(_icnwidt_), ;
         __YdGE_(_icnheig_))
      if _handles_[handle,7]                     // if a shadow has been selected, display shadow
         currcolor := setcolor()                 // save the current color
         setcolor(_eshadow_)                     // set color to the shadow color and draw the shadow box
         loadicon(_dgepath_ + "gllibr.ico")
         __DrawSuperIcon(0,_handles_[handle,1]+IconShadowOffsetD,_handles_[handle,2]+IconShadowOffsetR)
         loadicon(_icnfile_)
         setcolor(currcolor)                     // restore the Clipper color
      endif                                      // if shadow
      __DrawSuperIcon(_handles_[handle,10],_handles_[handle,1],_handles_[handle,2])
   else                                          // otherwise the button doesn't exists
      __HandleError(NoSuchLabel,label)           // process the error
   endif
RETURN(Void)

// __DeaIconButton() ----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Toggles a button off
// Mapped Command:
FUNCTION __DeaIconButton(label,clr)
   local handle := __ScanObjects(label)          // get a handle if possible
   if handle > 0                                 // if the object does indeed exist
      _handles_[handle,9] := InactiveObject      // status (inactive)
      msethot(handle,0,0,0,0)                    // clear the mouse hot region
      if clr                                     // deactivate and clear from the array
         __ClrIconButton(handle)                 // clear the icon from the screen
      endif
   else                                          // otherwise it's an invalid object
      __HandleError(NoSuchLabel,label)           // branch to the handle error routine
   endif
RETURN(Void)

// __ClrIconButton() ----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Clear an icon from the screen given the handle ID
// Mapped Command: 
FUNCTION __ClrIconButton(handle)
   clrwin(__XdGE(_handles_[handle,2])-__XdGE_(_icnwidt_/2),;
      __YdGE(_handles_[handle,1])-__YdGE_((_icnheig_/2)+IconShadowOffsetD+.1),;
      __XdGE(_handles_[handle,2])+__XdGE_((_icnwidt_/2)+IconShadowOffsetR+.1),;
      __YdGE(_handles_[handle,1])+__YdGE_(_icnheig_/2))
RETURN(Void)

// __WaitForEvent() -----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Get a mouse click and return the handle number
// Mapped Command: WAIT EVENT TO
FUNCTION __WaitForEvent(flash)
   local handle, label
   flash  := if(flash == NIL,TRUE,flash)         // are we going to flash the object on selection
   do while TRUE                                 // loop until the mouse has been clicked
      do while TRUE                              // loop until the mouse has been clicked
         if mstatus() == 1                       // if the mouse has been clicked
            exit                                 // exit from the loop
         endif                                   // mstatus() == 1
      enddo                                      // continue looping
      handle := mgethot()                        // get the handle where it was clicked (may be zero)
      if handle > 0                              // if the click was in a hot region
         if _handles_[handle,9] == ActiveObject  // if the object selected is active
            label := __FindObject(handle)        // determine the object name of the handle that was clicked
            if flash                             // if a flash has been requested on selection
               __FlaEventRegion(label)           // flash the object with the shadow
            endif                                // if flash
            retu(label)                          // return the handle label
         endif
      endif                                      // if _handles_[handle,?]
   enddo                                         // do while true                                 // loop until the mouse has been clicked
RETURN("")                                       // return a blank label

// __WaitForClick() -----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Get a mouse click from a specified object area
// Mapped Command: WAIT EVENT <label>
FUNCTION __WaitForClick(label,deactivate,release,noflash)
   local handle  := __ScanObjects(label)         // get the handle for this object
   if __HandleInRange(handle) > 0                // if the handle is valid
      noflash := if(noflash == NIL,FALSE,noflash)  // are we going to flash the object on selection
      do while TRUE                              // loop until the region specified was clicked in
         if mstatus() == 1 .and. handle == mgethot()
            exit                                 // exit when the region is clicked
         endif                                   // if mstatus() == 1 .and. region == mgethot()
      enddo                                      // continue looping
      if deactivate
         __DeaEventRegion(label,FALSE)
      endif                                      // if deactivate
      if release
         __RelEventRegion(label)
      endif                                      // if release
   else
      __Handleerror(NoSuchLabel,label,procname())
   endif                                         // if __handleinrange()
RETURN("")                                       // return a blank label

// __HandleError() ------------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Display handle error and quit
// Mapped Command: 
FUNCTION __HandleError(error,label,procname)
   procname := if(procname == NIL,"Unknown Proc",procname)
   settext()
   clear screen
   do case
   case error == NoSuchLabel
      ? procname + ": No such label: " + label + "!"
   case error == NoLabelsLeft
      ? procname + ": No handles left to create label: " + label + "!"
   case error == NoMemoryLeft
      ? procname + ": No video memory left to create screen save: " + label + "!"
   endcase
   quit
RETURN(Void)

// __HandleInRange() ----------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Determine if handle number is in valid range
// Mapped Command: 
FUNCTION __HandleInRange(handle)
RETURN(if(handle >=1 .and. handle <= MaxHandles,1,0))

// __FindUnusedHandle() -------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Find the next free handle
// Mapped Command: 
FUNCTION __FindUnusedHandle()
   local n
   for n := 1 to MaxHandles
      if empty(_handles_[n,8])
         retu(n)
      endif                                      // if _handles_[n,8] := object
   next                                          // for n := 1 to MaxHandles
RETURN(0)

// __ScanObjects() ------------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Find the handle of a specified object
// Mapped Command: 
FUNCTION __ScanObjects(object)
   local n
   for n := 1 to MaxHandles
      if _handles_[n,8] == object
         retu(n)
      endif                                      // if _handles_[n,8] := object
   next                                          // for n := 1 to MaxHandles
RETURN(0)

// __FindObject() -------------------------------------------------------------
// TecGuide-> {Function Ref::Event Functions::UDF} {SOURCE}
//    Description: Find the object of a specified handle
// Mapped Command: 
FUNCTION __FindObject(handle)
RETURN(if(handle>0 .and. handle<=MaxHandles,_handles_[handle,8],""))

// __DrawBarChart() -----------------------------------------------------------
// TecGuide-> {Function Ref::Business Functions::UDF} {SOURCE}
//    Description: Draw a bar chart
// Mapped Command: DRAW BAR CHART AT
FUNCTION __DrawBarChart(Pos1,Pos2,dbf,field,label,width,height,division,filter,solid,dotted,dashed,box,pat,color,three_d,horiz)
   local n, select_, xlabeltxt, ylabeltxt, maxvalue, divisions, scalefact, increment
   local gmode := three_d + horiz                // calculate the chart mode
   local amode := solid + dotted + dashed + box  // calculate the axis mode
   local pattern := 1                            // establish a pattern increment
   label     := if(label == NIL,"",label)        // establish X axes label default
   width     := if(width == NIL,BarChartWidth,width)  // establish chart width
   height    := if(height == NIL,BarChartHeight,height)  // establish chart height
   color     := if(color == NIL,"",color)
   if " " $ color .and. "BRIG" $ upper(color)
      color := substr(color,at("BRIG",upper(color)))
      color := ltrim(substr(color,at(" ",color)))
      color := "bright " + trim(substr(color,1,at(" ",color)))
   else
      if " " $ color
         color := trim(substr(color,1,at(" ",color)))
      endif                                      // if " " $ color
   endif                                         // if " " $ color .and. "brig" $ upper(color)
   select_   := select()                         // save the current area
   xlabeltxt := ylabeltxt := ""                  // establish the label text memvar
   use &dbf new                                  // open the plot database
   datareset()                                   // clear the dGE data array
   if filter != NIL                              // are we filtering the dbf?
      set filter to &filter                      // establish a filter
      go top                                     // reset the database pointer
   endif                                         // if filter != nil
   maxvalue := &field                            // start with the first value
   n := 1                                        // establish a bar counter
   do while .not. eof()                          // loop through all the valid records
      maxvalue := if(&field > maxvalue,&field,maxvalue)  // get the max value
      skip                                       // next valid record
      n ++                                       // increment the bar counter
   enddo
   maxvalue  := 1.10 * maxvalue                  // increase the max by 10%
   division  := if(division == NIL,maxvalue/4,division)
   divisions := int(maxvalue/division)           // establish default dependent value
   scalefact := __YdGE_(height+2)/maxvalue
   for n := 1 to divisions - 1                   // create the y label text
      ylabeltxt := ylabeltxt + str(division * n,5)
   next
   n := 1                                        // establish a bar counter
   go top
   do while .not. eof()                          // loop through all the valid records
      datastore(scalefact * &field,if(pat,pattern,0),0,if(empty(color),__DgeColor(setcolor()),__WordToColor(color)))
      pattern := if(pattern == 20,1,pattern+1)   // increment the pattern
      if len(label) > 0
         xlabeltxt := xlabeltxt + &label         // accumulate the label string
      endif
      n ++                                       // increment the bar counter
      skip                                       // next valid record
   enddo
   increment := __XdGE(width)/n                  // calculate the increment
   xyaxes(__XdGE(Pos2-2),__YdGE(Pos1+.5),__XdGE_(width),__YdGE_(height+2),n,divisions,amode,__DgeColor(setcolor()))
   labelx(__XdGE(Pos2+.75),__YdGE(Pos1+2),increment,if(len(label)>0,len(&label),0),0,BarXLabels,__DgeColor(setcolor()),xlabeltxt)
   labely(__XdGE(Pos2-(5 + 2.5)),__YdGE(Pos1-1),__YdGE_(height+2)/divisions,5,0,0,__DgeColor(setcolor()),ylabeltxt)
   bargraph(__XdGE(Pos2),__YdGE(Pos1),increment,gmode,1)  // display the bar chart
   use                                           // close plot database
   select(select_)                               // restore area
RETURN(Void)

// __DrawXYChart() ------------------------------------------------------------
// TecGuide-> {Function Ref::Business Functions::UDF} {SOURCE}
//    Description: Draw an XY chart
// Mapped Command: DRAW XY CHART AT
FUNCTION __DrawXYChart(Pos1,Pos2,dbf,field,label,width,height,division,filter,solid,dotted,dashed,box,col)
   local n, maxvalue, divisions, scalefact, increment
   local amode := solid + dotted + dashed + box  // calculate the axis mode
   local select_ := select()                     // save the current area  
   local color := 1                              // establish acolor increment
   local xlabeltxt := ""                         // establish the xlabel text memvar
   local ylabeltxt := ""                         // establish the ylabel text memvar
   label     := if(label == NIL,"",label)        // establish X axes label default
   width     := if(width == NIL,BarChartWidth,width)  // establish chart width
   height    := if(height == NIL,BarChartHeight,height)  // establish chart height
   use &dbf new                                  // open the plot database
   datareset()                                   // clear the dGE data array
   if filter != NIL                              // are we filtering the dbf?
      set filter to &filter                      // establish a filter
      go top                                     // reset the database pointer
   endif                                         // if filter != nil
   maxvalue := &field                            // start with the first value
   n := 1                                        // establish a bar counter
   do while .not. eof()                          // loop through all the valid records
      maxvalue := if(&field > maxvalue,&field,maxvalue)  // get the max value
      skip                                       // next valid record
      n ++                                       // increment the bar counter
   enddo
   maxvalue  := 1.10 * maxvalue                  // increase the max by 10%
   division  := if(division == NIL,maxvalue/4,division)
   divisions := int(maxvalue/division)           // establish default dependent value
   scalefact := __YdGE_(height+2)/maxvalue
   for n := 1 to divisions - 1                   // create the y label text
      ylabeltxt := ylabeltxt + str(division * n,5)
   next
   n := 1                                        // establish a bar counter
   go top
   do while .not. eof()                          // loop through all the valid records
      datastore(scalefact * &field,0,0,0)
      color := if(color == 20,1,if(color == 7,color+2,color+1))
      if len(label) > 0
         xlabeltxt := xlabeltxt + &label         // accumulate the label string
      endif
      n ++                                       // increment the bar counter
      skip                                       // next valid record
   enddo
   increment := __XdGE(width)/n                  // calculate the increment
   xyaxes(__XdGE(Pos2-2),__YdGE(Pos1+.5),__XdGE_(width),__YdGE_(height+2),n,divisions,amode,__DgeColor(setcolor()))
   labelx(__XdGE(Pos2+.75),__YdGE(Pos1+2),increment,if(len(label)>0,len(&label),0),0,BarXLabels,__DgeColor(setcolor()),xlabeltxt)
   labely(__XdGE(Pos2-(5 + 2.5)),__YdGE(Pos1-1),__YdGE_(height+2)/divisions,5,0,0,__DgeColor(setcolor()),ylabeltxt)
   xygraph(__XdGE(Pos2),__YdGE(Pos1),increment,0,__DgeColor(setcolor()))  // display the bar chart
   use                                           // close plot database
   select(select_)                               // restore area
RETURN(Void)

// __DrawPieChart() -----------------------------------------------------------
// TecGuide-> {Function Ref::Business Functions::UDF} {SOURCE}
//    Description: Draw a pie chart
// Mapped Command: DRAW PIE CHART AT
FUNCTION __DrawPieChart(Pos1,Pos2,dbf,field,filter,pat,col,label,offset,slice,radius,percent,noconnect)
   local n, maxvalue, divisions, scalefact, increment
   local pattern := 1                            // establish the beginning pattern
   local color   := 2                            // establish the beginning color
   local select_ := select()                     // save the current area 
   local labeltxt:= ""                           // establish a blank label accumulator
   label   := if(label == NIL,"",label)          // get the specified label (not sure if this has to be a field)
   offset  := if(offset == NIL,PieLabelOffSet,offset)  // set the offset if not specified
   slice   := if(slice == NIL,0,slice)           // pie slice to explode
   radius  := if(radius == NIL,PieChartRadius,radius)  // determine the radius, default to 20
   use &dbf new                                  // open the plot database
   datareset()                                   // clear the dGE daya array
   if filter != NIL                              // is there a filter statement?
      set filter to &filter                      // set the requested filter
      go top                                     // reset the database pointer
   endif                                         // if filter != nil
   maxvalue := &field                            // start with the first value in the plot field
   n := 1                                        // establish a bar counter
   do while .not. eof()                          // loop through all the valid records
      maxvalue := if(&field > maxvalue,&field,maxvalue)  // get the max value of each slice
      skip                                       // next valid record
      n ++                                       // increment the slice counter
   enddo                                         // keep doing it 'till the eof()
   go top                                        // back to the first record
   n := 1                                        // establish a slice counter
   do while .not. eof()                          // loop through the valid records
      datastore(if(&field<0,0,&field*(1000/maxvalue)),if(pat,pattern,20),if(n == slice,1,0),if(col,color,__DgeColor(setcolor())))
      color := if(color == 20,1,if(color == 7,color+2,color+1))
      pattern := if(pattern == 20,1,pattern+1)   // increment the pattern
      if percent == 0                            // if percentages are not being used for labels
         labeltxt := labeltxt + &label           // accumulate the label string
      endif                                      // if percent == 0
      n ++                                       // increment the pie slice counter (always = n-1)
      skip                                       // next valid record
   enddo                                         // do while .not. eof()                          // loop through the valid records
   piechart(__XdGE(Pos2),__YdGE(Pos1),__XdGE_(radius))  // draw the pie chart
   do case                                       // evaluate label style
   case percent > 0                              // percentage labels
      labelpie(__XdGE_(offset),__XdGE_(radius*if(slice > 0,1.35,1)),0,0,percent+noconnect,__dGEColor(setcolor()),"")
   case .not. empty(label)                       // text labels
      labelpie(__XdGE_(offset),__XdGE_(radius*if(slice > 0,1.35,1)),len(&label),0,noconnect,__dGEColor(setcolor()),labeltxt)
   endcase
   use                                           // close plot database
   select(select_)                               // restore area
RETURN(Void)

// __XdGE_() ------------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert @SAY Y value to dGE X value
// Mapped Command: 
FUNCTION __XdGE_(value)
RETURN(PointsPerColumn * if(value < 0,0,value))  // return the X length in dGE coordinates

// __YdGE_() ------------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert @SAY X value to dGE Y value
// Mapped Command: 
FUNCTION __YdGE_(value)
RETURN(PointsPerLine * if(value < 0,0,value))    // return the Y length in dGE coordinates

// __XdGE() -------------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert @SAY Y coordinate to dGE X coordinate
// Mapped Command: 
FUNCTION __XdGE(value)
RETURN(PointsPerColumn * if(value < 0,0,value))  // return the X location in dGE coordinates

// __YdGE() -------------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert @SAY X coordinate to dGE Y coordinate
// Mapped Command: 
FUNCTION __YdGE(value)
RETURN(1000-(PointsPerLine * if(value < 0,0,value)))  // return the Y location in dGE coordinates

// __DgeColor() ---------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert dBase color string to dGE numeric value
// Mapped Command: 
FUNCTION __DgeColor(colorstr)
   local fg, fg_bright
   if at("/",colorstr) > 0                       // check to make sure we have a color string
      fg := upper(substr(colorstr,1,at("/",colorstr)-1))  // get the foreground color from the passed string
   endif                                           
   fg_bright := if("+" $ fg,8,0)                 // if it's a bright color establish a memvar
   do case                                       // evaluate the color string
   case substr(fg,1,1) == "N" .or. fg == " "     // and return the integer value
      retu(0+fg_bright)
   case substr(fg,1,1) == "W"                    // if white is present in the string
      retu(7+fg_bright)
   otherwise                                     // otherwise
      retu(fg_bright + if('R' $ fg,4,0) + if('G' $ fg,2,0) + if('B' $ fg,1,0))  // added - PMF
   endcase
RETURN(Void)

// __WordToColor() ------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert color word to dGE numeric equivalent
// Mapped Command:
FUNCTION __WordToColor(color)
   do case                                       // evaluate the color word passed
   case upper(color) == "BLACK"                  // and return the integer value
      retu(00)
   case upper(color) == "BLUE"                   // cyan
      retu(01)
   case upper(color) == "GREEN"                  // magenta
      retu(02)
   case upper(color) == "CYAN"                   // white
      retu(03)
   case upper(color) == "RED"                    // red
      retu(04)
   case upper(color) == "MAGENTA"                // magenta
      retu(05)
   case upper(color) == "BROWN"                  // brown
      retu(06)
   case upper(color) == "WHITE"
      retu(07)
   case upper(color) == "GREY" .or. upper(color) == "GRAY"
      retu(08)
   case upper(color) == "BRIGHT BLUE"
      retu(09)
   case upper(color) == "BRIGHT GREEN"
      retu(10)
   case upper(color) == "BRIGHT CYAN"
      retu(11)
   case upper(color) == "BRIGHT RED"
      retu(12)
   case upper(color) == "BRIGHT MAGENTA"
      retu(13)
   case upper(color) == "YELLOW"
      retu(14)
   case upper(color) == "BRIGHT WHITE"
      retu(15)
   otherwise                                     // if non of the words match, assume white
      retu(7)
   endcase
RETURN(Void)

// __PalWordToColor() ------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Convert color word to dGE numeric equivalent for setpal()
// Mapped Command: 
FUNCTION __PalWordToColor(color)
   do case                                       // evaluate the color word passed
   case upper(color) == "BLACK"                  // and return the integer value
      retu(00)
   case upper(color) == "BLUE"                   // cyan
      retu(01)
   case upper(color) == "GREEN"                  // magenta
      retu(02)
   case upper(color) == "CYAN"                   // white
      retu(03)
   case upper(color) == "RED"                    // red
      retu(04)
   case upper(color) == "MAGENTA"                // magenta
      retu(05)
   case upper(color) == "BROWN"                  // brown
      retu(06)
   case upper(color) == "WHITE"
      retu(07)
   case upper(color) == "GREY" .or. upper(color) == "GRAY"
      retu(56)
   case upper(color) == "BRIGHT BLUE"
      retu(09)
   case upper(color) == "BRIGHT GREEN"
      retu(18)
   case upper(color) == "BRIGHT CYAN"
      retu(27)
   case upper(color) == "BRIGHT RED"
      retu(36)
   case upper(color) == "BRIGHT MAGENTA"
      retu(45)
   case upper(color) == "YELLOW"
      retu(54)
   case upper(color) == "BRIGHT WHITE"
      retu(63)
   otherwise                                     // if non of the words match, assume white
      retu(7)
   endcase
RETURN(Void)

// __ActiveObjects() ----------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Determine the number of active objects in the region array
// Mapped Command: 
FUNCTION __ActiveObjects()
   local n
   local k := 0                                  // establish an active object counter
   for n := 1 to MaxHandles                      // loop through the object array
      k := if(_handles_[n,9] > 0,k++,k)          // if it's an active object in the get array, increment the counter
   next                                          // for n := 1 to MaxHandles
RETURN(k)                                        // return the number of objects that are active

// __DrawBevel() --------------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Display bevel graphics around a box
// Mapped Command: 
FUNCTION __DrawBevel(x,y,depth,width,pattern)
   local currcolor := setcolor()                 // save the current Clipper color
   set color to BevelFrameColor
   draw box from x-.15,y-.325 to x+depth+.15,y+width+.325 pattern 20
   set color to "w/"
   draw line from x+depth-.15,y+width-.325 to x+depth+.15,y+width+.325
   set color to LowerRightBevelColor
   draw line from x-.15,y-.325 to x+.15,y+.325
   draw line from x-.15,y+width+.325 to x+.15,y+width-.325
   draw line from x+depth-.15,y+.325 to x+depth+.15,y-.325
   set color to BevelSurfaceColor
   draw box from x+.15,y+.325 to x+depth-.15,y+width-.325 pattern pattern
   set color to UpperLeftBevelColor
   shade area at x-.05,y+.4
   shade area at x+.4,y-.1
   set color to LowerRightBevelColor
   shade area at x+.2,y+width-.2
   shade area at x+depth,y+.35
   setcolor(currcolor)                           // restore the Clipper color
RETURN(Void)

// __RunTimeError() -----------------------------------------------------------
// TecGuide-> {Function Ref::Supporting Functions::UDF} {SOURCE}
//    Description: Display run time error and quit
// Mapped Command: 
FUNCTION __RunTimeError(error,label,procname)
   procname := if(procname == NIL,"Unknown Proc",procname)
   settext()
   clear screen
   do case
   case error == NoMouseDriver
      ? procname + ": No mouse driver present: " + label + "!"
   endcase
   quit
RETURN(Void)
