/****************************************************************************\
*                                                                            *
*  FADE.C                                                                    *
*                                                                            *
*  This program demonstrates how to perform a smooth palette fade with       *
*  Fastgraph.  This example assumes a 256-color video mode with 6-bit DAC    *
*  values (i.e., between 0 and 63).  These values are defined at the top of  *
*  this file, so you can change them easily.                                 *
*                                                                            *
*  The fadein and fadeout routines in this program were originally developed *
*  by John Wagner, author of the IMPROCES image processing program.          *
*                                                                            *
*  To compile this program and link it with Fastgraph:                       *
*                                                                            *
*     BCC -ms FADE.C FGS.LIB    (Borland C++)                                *
*                                                                            *
*     CL /AS FADE.C /link FGS   (Microsoft C)                                *
*                                                                            *
*     QCL /AS FADE.C /link FGS  (Microsoft QuickC)                           *
*                                                                            *
*     PC /ms FADE               (Power C)                                    *
*     PCL FADE [,,0] ;FGS                                                    *
*                                                                            *
*     TCC -ms FADE.C FGS.LIB    (Turbo C and C++)                            *
*                                                                            *
*  This program also can be linked with Fastgraph/Light if you replace the   *
*  FGS library references with FGLS.                                         *
*                                                                            *
*  Fastgraph (tm) and Fastgraph/Light (tm) are graphics libraries published  *
*  by Ted Gruber Software.  For more info, please call, write, or FAX.       *
*                                                                            *
*  Ted Gruber Software                           orders/info (702) 735-1980  *
*  PO Box 13408                                          FAX (702) 735-4603  *
*  Las Vegas, NV  89112                                  BBS (702) 796-7134  *
*                                                                            *
\****************************************************************************/

#include <fastgraf.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef __TURBOC__
#include <mem.h>
#else
#include <memory.h>
#endif

/* function prototypes */

void main(void);
void fadein(char *,int);
void fadeout(int);

/* these values can be changed for different video modes */

#define NDACS 256
#define NCOLORS 64
#define VIDEO_MODE 19

/* this typedef struct is a clean way to do DACs */

typedef struct
{
   unsigned char r, g, b;
}
RGB;

/* these global arrays hold two complete sets of DAC values */

RGB dacs1[NDACS], dacs2[NDACS];

/* main program */

void main()
{
   int delay;
   int old_mode;

   /* make sure the requested graphics mode is available */

   if (fg_testmode(VIDEO_MODE,2) == 0)
   {
      printf("This program requires a %d-color graphics mode.\n",NDACS);
      exit(1);
   }

   /* calculate the base delay between DAC updates */

   delay = fg_measure() / 128;

   /* initialize Fastgraph for the requested video mode */

   old_mode = fg_getmode();
   fg_setmode(VIDEO_MODE);
   fg_allocate(1);

   /* for each PCX file, fade it in and then back out */

   fadein("TGS.PCX",delay);
   fg_waitfor(36);
   fadeout(delay);
   fg_waitfor(18);

   fadein("BALLOONS.PCX",delay*2);
   fg_waitfor(36);
   fadeout(delay*2);
   fg_waitfor(18);

   fadein("MOUSE.PCX",delay*4);
   fg_waitfor(36);
   fadeout(delay*4);

   /* restore the original video mode and screen attributes */

   fg_freepage(1);
   fg_setmode(old_mode);
   fg_reset();
}

/****************************************************************************\
*                                                                            *
*  fadein                                                                    *
*                                                                            *
*  Display an image by gradually increasing each DAC's RGB components to     *
*  their original values.                                                    *
*                                                                            *
\****************************************************************************/

void fadein(PCXfile,delay)
char *PCXfile;
int delay;
{
   register int i, j;
   int target;

   /* erase the visual page */

   fg_setpage(0);
   fg_erase();

   /* display the PCX file on the hidden page */

   fg_setpage(1);
   fg_move(0,0);
   fg_disppcx(PCXfile,0);
   fg_setpage(0);

   /* get its DAC values */

   fg_getdacs(0,NDACS,(unsigned char *)dacs1);

   /* zero all of the DACs */

   memset(dacs2,NDACS*3,0);
   fg_setdacs(0,NDACS,(unsigned char *)dacs2);

   /* display the blacked-out image on the visual page */

   fg_copypage(1,0);

   /* cycle through the DACs, gradually increasing them to their old values */

   for (j = 0; j < NCOLORS; j++)
   {
      /* increment each RGB component if it is below its old value */

      target = NCOLORS - j;

      for (i = 0; i < NDACS; i++)
      {
         if (dacs1[i].r > target && dacs2[i].r < dacs1[i].r) dacs2[i].r++;
         if (dacs1[i].g > target && dacs2[i].g < dacs1[i].g) dacs2[i].g++;
         if (dacs1[i].b > target && dacs2[i].b < dacs1[i].b) dacs2[i].b++;
      }

      /* update the DACs each time through the loop */

      fg_stall(delay);
      fg_setdacs(0,NDACS,(unsigned char *)dacs2);
   }
}

/****************************************************************************\
*                                                                            *
*  fadeout                                                                   *
*                                                                            *
*  Erase an image by gradually fading each DAC's RGB components to black.    *
*                                                                            *
\****************************************************************************/

void fadeout(delay)
int delay;
{
   register int i, j;

   /* load the dacs1 and dacs2 arrays with the current DAC values */

   fg_getdacs(0,NDACS,(unsigned char *)dacs1);
   memcpy(dacs2,dacs1,NDACS*3);

   /* cycle through the DACs, gradually reducing them to 0 (black) */

   for (j = 0; j < NCOLORS; j++)
   {
      /* decrement each RGB component if it is above 0 */

      for (i = 0; i < NDACS; i++)
      {
         if (dacs2[i].r) dacs2[i].r--;
         if (dacs2[i].g) dacs2[i].g--;
         if (dacs2[i].b) dacs2[i].b--;
      }

      /* update the DACs each time through the loop */

      fg_stall(delay);
      fg_setdacs(0,NDACS,(unsigned char *)dacs2);
   }
}
