(*----------------------------------------------------------------------*)
(*        Display_Help  --- Display help screen for PibCat              *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_Help;

BEGIN (* Display_Help *)

   WRITELN;
   WRITELN('Program: PIBCAT --- Catalog files on a disk.');
   WRITELN('Author:  Philip R. Burns.');
   WRITELN('Version: 1.7  April 1, 1989.');
   WRITELN('Usage:   PIBCAT v /a /c /e=filespec /f=filespec /i=indent /l /m=margin /n');
   WRITELN('                  /o=filename /p=pagesize /s=filespec /t=timezone /x /2');
   WRITELN;
   WRITELN('                v               volume (drive letter) to catalog');
   WRITELN('                                (default is current drive)');
   WRITELN('                                If given as ?, this text is displayed.');
   WRITELN('                /a              Requests that PibCat output be appended to');
   WRITELN('                                the file specified by "/o=".  Default is to');
   WRITELN('                                overwrite the the output file.');
   WRITELN('                /c=format       Produce condensed listing suitable for');
   WRITELN('                                input to a database manager or sorting program.');
   WRITELN('                                Two formats are available:  a columnar format');
   WRITELN('                                specified as "/c=column" or a comma-delimited');
   WRITELN('                                format specified as "/c=comma".  Specifying');
   WRITELN('                                "/c" without a format produces the columnar');
   WRITELN('                                format.  This option overrides all other');
   WRITELN('                                formatting options.');
   WRITELN('                /e=filespec     DOS file spec to match when listing');
   WRITELN('                                entries in library files');
   WRITELN('                                (default is *.* -- list all entries).');
   WRITELN(' ');

   WRITE  ('Hit <Enter> to continue: ');
   READLN;

   WRITELN;
   WRITELN('                /f=filespec     DOS file spec to match when listing files');
   WRITELN('                                (default is *.* -- list all files)');
   WRITELN('                /i=indent       # columns to space for library entries');
   WRITELN('                                (default is 0)');
   WRITELN('                /l              display long file names');
   WRITELN('                                (default is to display short file');
   WRITELN('                                names only)');
   WRITELN('                /m=margin       left margin to leave (default is 0)');
   WRITELN('                /n              list contents of libraries at end of each');
   WRITELN('                                subdirectory (default is list contents');
   WRITELN('                                following library file name)');
   WRITELN('                /o=filename     write catalog listing to file "filename"');
   WRITELN('                                (default is "CATALOG.LIS")');
   WRITELN('                /p=pagesize     paginate listing using "pagesize" lines');
   WRITELN('                                (default is no pagination)');
   WRITELN('                /s=filename     write status information to file "filename"');
   WRITELN('                                (default is DOS standard output)');
   WRITELN('                /t=timezone     number of hours local time lags/leads Greenwich');
   WRITELN('                                Mean Time [GMT] (default is 7)');
   WRITELN('                /x              don''t list library file contents');
   WRITELN('                                (default is to list contents)');
   WRITELN(' ');

   WRITE  ('Hit <Enter> to continue: ');
   READLN;

   WRITELN;
   WRITELN('                /2              Opens files without SHARE for DOS v2.x');
   WRITELN('                                compatibility (default is to open files');
   WRITELN('                                with share for DOS v3.1 and above)');
   WRITELN;
   WRITELN('Aborting:  Hit ^C to abort catalog listing.');
   WRITELN;
   WRITELN('Formats:   PibCat understands the contents of .ARC, .DWC, .LBR,');
   WRITELN('           .LZH, .LZS, .MD, .PAK, .ZIP, and .ZOO files.');
   WRITELN;

END   (* Display_Help *);

(*----------------------------------------------------------------------*)
(*             Initialize --- Initialize PibCat program                 *)
(*----------------------------------------------------------------------*)

FUNCTION Initialize : BOOLEAN;

VAR
   S           : AnyStr;
   S2          : AnyStr;
   I           : INTEGER;
   J           : INTEGER;
   IErr        : INTEGER;
   Dos_Version : WORD;

(* STRUCTURED *) CONST
   Legit_Drives : SET OF CHAR = ['A'..'Z','?'];

BEGIN (* Initialize *)
                                   (* --- Set defaults --- *)

                                   (* Drive to catalog is current drive *)
   GetDir( 0 , S );
   Cat_Drive         := UpCase( S[ 1 ] );

                                   (* Default output file is CATALOG.LIS *)

   Output_File_Name  := 'CATALOG.LIS';

                                   (* Default status file is standard output *)

   Status_File_Name  := '';

                                   (* Don't produce paginated listing file *)
   Do_Printer_Format := FALSE;
   Page_Size         := 0;
                                   (* No extra spaces at left margin *)
   Left_Margin       := 0;
                                   (* No extra indent for libraries *)
   Library_Indent    := 0;
                                   (* List contents of library files *)
   Expand_Libs       := TRUE;
                                   (* Expand libraries after main listing *)
   Expand_Libs_In    := TRUE;
                                   (* No ^C hit yet terminating cataloguing *)
   User_Break        := FALSE;
                                   (* Catalog all files by default *)
   Find_Spec         := '*.*';
                                   (* Catalog all library entries by default *)
   Entry_Spec        := '*.*';
                                   (* We start on first page *)
   Page_Number       := 1;
                                   (* Lots of lines left on this page *)
   Lines_Left        := 32767;
                                   (* No files yet *)
   File_Count        := 0;
   Total_Files       := 0;
   Total_Space       := 0;
   Total_Entries     := 0;
   Total_ESpace      := 0;
   Total_Dirs        := 0;
                                   (* No titles yet *)
   Volume_Title      := '';
   Subdir_Title      := '';
   File_Title        := '';
                                   (* Not help mode only *)
   Help_Only         := FALSE;
                                   (* Only short file names by default *)
   Show_Long_File_Names := FALSE;
                                   (* Assume we are using SHARE    *)
   Use_Share         := TRUE;                                     
                                   (* # of seconds local time leads/lags *)
                                   (* Greenwich Mean Time (GMT)          *)

   GMT_Difference       := 8 * 3600;

                                   (* Assume we do daylight savings adjustment *)
   Use_Daylight_Savings := TRUE;

                                   (* We aren't producing condensed output *)
   Do_Condensed_Listing := FALSE;

                                   (* But if we were, it would be columnar *)

   Condensed_Listing_Type := Condensed_Columnar;

                                   (* Open output file for overwrite *)
   Open_For_Append := FALSE;

                                   (* Grab command line parameters *)
   FOR I := 1 TO ParamCount DO
      BEGIN

         S := UpperCase( ParamStr( I ) );

         IF ( S[ 1 ] = '/' ) THEN
            BEGIN

               IF ( S[3] = '=' ) THEN
                  S2 := COPY( S, 4, LENGTH( S ) - 3 )
               ELSE
                  S2 := '';

               CASE UpCase( S[2] ) OF

                                   (* Open output for append  *)

                  'A':  Open_For_Append := TRUE;

                                   (* Write condensed listing *)

                  'C':  BEGIN
                           Do_Condensed_Listing := TRUE;
                           IF ( S2 = 'COMMA' ) THEN
                              Condensed_Listing_Type := Condensed_Comma
                           ELSE IF ( S2 = 'COLUMN' ) THEN
                              Condensed_Listing_Type := Condensed_Columnar;
                        END;   

                                   (* Match entry within libraries *)
                  'E':  BEGIN
                           IF ( S2 <> '' ) THEN
                              Entry_Spec := S2;
                           FOR J := 1 TO LENGTH( S2 ) DO
                              Entry_Spec[ J ] := UpCase( Entry_Spec[ J ] );
                        END;
                                   (* Match this file spec *)
                  'F':  BEGIN
                           IF ( S2 <> '' ) THEN
                              Find_Spec := S2;
                        END;
                                   (* # of space to indent when listing *)
                                   (* contents of libraries             *)
                  'I':  BEGIN
                           VAL( S2, J, IErr );
                           IF ( IErr = 0 ) THEN
                              Library_Indent := J;
                        END;

                                   (* If long file names should be listed *)

                  'L':  Show_Long_File_Names := TRUE;

                                   (* # of space in left margin of output *)
                  'M':  BEGIN
                           VAL( S2, J, IErr );
                           IF ( IErr = 0 ) THEN
                              Left_Margin := J;
                        END;

                                   (* Expand libraries after all files *)
                                   (* listed in a subdirectory         *)
                  'N':  BEGIN
                           Expand_Libs_In   := FALSE;
                           Expand_Libs      := TRUE;
                        END;
                                   (* Output file name *)
                                   
                  'O':  Output_File_Name := S2;

                                   (* Page size for printing *)

                  'P':  BEGIN

                           VAL( S2, J, IErr );

                           IF ( IErr = 0 ) THEN
                              BEGIN
                                 Page_Size  := J;
                                 Lines_Left := J;
                              END;

                           Do_Printer_Format := ( Page_Size > 0 );

                        END;

                                   (* Status file name *)
                                   
                  'S':  Status_File_Name := S2;

                                   (* Number of hours or minutes local time *)
                                   (* leads/lags Greenwich Mean Time        *)
                  'T':  BEGIN

                           IF ( LENGTH( S2 ) > 0 ) THEN 
                              BEGIN

                                 J := LENGTH( S2 );
                                 
                                 IF ( S2[ J ] = 'A' ) THEN
                                    BEGIN
                                       DELETE( S2 , J , 1 );
                                       Use_Daylight_Savings := FALSE;
                                    END;
                                    
                                 VAL( S2, J, IErr );

                                 IF ( IErr = 0 ) THEN
                                    GMT_Difference := J;

                                 IF ( ABS ( GMT_Difference ) <= 12 ) THEN
                                    GMT_Difference := GMT_Difference * 3600
                                 ELSE
                                    GMT_Difference := GMT_Difference * 60;   

                                 IF ( ABS( GMT_Difference  ) > ( 12 * 3600 ) ) THEN
                                    GMT_Difference := 8 * 3600;    

                              END;

                        END;
                                   (* If library contents should be expanded *)

                  'X':  Expand_Libs := FALSE;

                                   (* If SHARE to be used when opening files *)

                  '2':  Use_Share   := FALSE;

                  ELSE;

               END (* CASE *);

            END
         ELSE
            IF ( S[ 1 ] IN Legit_Drives ) THEN
               Cat_Drive := S[ 1 ];
      END;
                                   (* Resolve output style parameter       *)
                                   (* conflicts.  "/c" overrides others.   *)
   IF Do_Condensed_Listing THEN
      Do_Printer_Format := FALSE;

                                   (* If the drive was a "?" then we have  *)
                                   (* a help request.  Display help info   *)
                                   (* and quit.                            *)
   IF ( Cat_Drive = '?' ) THEN
      BEGIN
         Display_Help;
         Initialize := FALSE;
         Help_Only  := TRUE;
         EXIT;
      END;
                                   (* Fix up entry spec for comparisons    *)
                                   (* later on.  If '*.*', then don't      *)
                                   (* bother with entry spec checks later. *)

   Check_Entry_Spec( Entry_Spec, Entry_Name, Entry_Ext, Use_Entry_Spec );

                                   (* Get string of blanks for left margin *)

   Left_Margin_String := DUPL( ' ' , Left_Margin );

                                   (* Get DOS version and set open *)
                                   (* file modes accordingly.      *)

   I           := DosVersion;
   Dos_Version := LO( I ) * 10 + HI( I );

   IF ( ( Dos_Version >= 31 ) AND Use_Share ) THEN
      BEGIN
         Read_Open_Mode  := 64;
         Write_Open_Mode := 66;
      END
   ELSE
      BEGIN
         Read_Open_Mode  := 0;
         Write_Open_Mode := 2;
      END;
                                   (* Open status file *)
   FileMode := Write_Open_Mode;

   ASSIGN ( Status_File , Status_File_Name );
   REWRITE( Status_File );

   FileMode := 2;
                                   (* Continue if we got it *)
   IF ( IOResult <> 0 ) THEN
      BEGIN
         WRITELN;
         WRITELN( 'Can''t open status file ', Status_File_Name );
         WRITELN;
         Initialize := FALSE;
         EXIT;
      END;
                                   (* Open output file *)
   FileMode := Write_Open_Mode;

   ASSIGN    ( Output_File , Output_File_Name );
   SetTextBuf( Output_File , Output_File_Buffer );

   IF Open_For_Append THEN
      APPEND    ( Output_File )
   ELSE
      REWRITE   ( Output_File );

   FileMode := 2;
                                   (* Continue if we got it *)
   IF ( IOResult <> 0 ) THEN
      BEGIN
         WRITELN( Status_File );
         WRITELN( Status_File , 'Can''t open output file ', Output_File_Name );
         WRITELN( Status_File );
         Initialize := FALSE;
         EXIT;
      END;
                                   (* Prevent heap allocation death *)

   HeapError := @Heap_Error_Handler;

                                   (* See how many file segments we can get *)

   Stack_Alloc := PRED( ( MemAvail - 8192 ) DIV SIZEOF( File_Stack_Type ) );

                                   (* If we can't allocate even one segment *)
                                   (* then report error and quit.           *)
   IF ( Stack_Alloc < 0 ) THEN
      BEGIN
         WRITELN( Status_File );
         WRITELN( Status_File , 'Not enough memory to process file directories ' );
         WRITELN( Status_File );
         Initialize := FALSE;
         EXIT;
      END
   ELSE
                                   (* Otherwise, allocate the segments *)
      FOR I := 0 TO Stack_Alloc DO
         NEW( File_Stack[ I ] );

                                   (* Get bracketing Unix dates for   *)
                                   (* daylight savings time calcs.    *)
   Get_Daylight_Savings_Times;

                                   (* Clear condensed output line     *)

   Condensed_Output_Line := DUPL( ' ' , 130 );

                                   (* Indicate initialization went OK *)
   Initialize := TRUE;

END   (* Initialize *);

(*----------------------------------------------------------------------*)
(*     Display_Volume_Label   ---  Display volume label of disk         *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_Volume_Label;

VAR
   Vol_Time     : LONGINT;
   STime        : STRING[10];
   SDate        : STRING[10];

BEGIN (* Display_Volume_Label *)

                                   (* Blank out volume title line *)

   Volume_Title := DUPL( ' ' , 80 );

                                   (* Get volume label from DOS *)

   Dir_Get_Volume_Label( Cat_Drive, Volume_Label, Vol_Time );

                                   (* If condensed listing, stuff volume *)
                                   (* label in output line and exit.     *)

   IF Do_Condensed_Listing THEN EXIT;

                                   (* If not condensed, make sure output *)
                                   (* starts on a new line.              *)
   WRITELN( Output_File );
                                   (* If no volume label, don't output it. *)

   IF ( Volume_Label = '' ) THEN
      BEGIN

         Volume_Title := Left_Margin_String              +
                         ' Contents of volume on drive ' +
                         Cat_Drive                       +
                         ' as of '                       +
                         DateString                      +
                         '  '                            +
                         TimeOfDayString;

         IF Do_Printer_Format THEN
            BEGIN
               WRITELN( Output_File , FF_Char );
               WRITE  ( Output_File , Volume_Title );
               WRITELN( Output_File , '     Page ', Page_Number );
            END
         ELSE
            WRITELN( Output_File , Volume_Title );

         DEC( Lines_Left );

      END
   ELSE
                                   (* If volume label, output it along with *)
                                   (* its creation time and date.           *)
      BEGIN

         Volume_Title := Left_Margin_String        +
                         ' Contents of volume '    +
                         Volume_Label              +
                         ' as of '                 +
                         DateString                +
                         '  '                      +
                         TimeOfDayString;

         IF Do_Printer_Format THEN
            BEGIN
               WRITELN( Output_File , FF_Char );
               WRITE  ( Output_File , Volume_Title );
               WRITELN( Output_File , '     Page ', Page_Number );
            END
         ELSE
            WRITELN( Output_File , Volume_Title );

         Volume_Label := Volume_Label + DUPL( ' ' , 12 - LENGTH( Volume_Label ) );

         Dir_Convert_Date_And_Time( Vol_Time , SDate , STime );

         WRITELN( Output_File );
         WRITE  ( Output_File , Left_Margin_String,
                  ' Volume: ',Volume_Label );

         IF ( SDate <> '         ' ) THEN
            WRITE  ( Output_File , ' Created: ', SDate, '  ', STime );

         DEC( Lines_Left , 3 );

      END;

   WRITELN( Output_File );
                                   (* Count lines left on page *)
   DEC( Lines_Left , 2 );

END   (* Display_Volume_Label *);

(*----------------------------------------------------------------------*)
(*     Display_Page_Titles  ---  Display page titles at top of page     *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_Page_Titles;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Display_Page_Titles;                                   *)
(*                                                                      *)
(*    Purpose:   Displays page headers for paginated output file        *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Display_Page_Titles;                                           *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Display_Page_Titles *)
                                   (* No titles if condensed listing   *)

   IF Do_Condensed_Listing THEN EXIT;

                                   (* Skip to top of new page using FF *)

   WRITELN( Output_File , FF_Char );

                                   (* Reset lines left to page size    *)
   Lines_Left  := Page_Size;
                                   (* Increment page count             *)

   INC( Page_Number );

                                   (* Display extant titles            *)
                                   (*   -- Volume title                *)

   WRITELN( Output_File );
   WRITELN( Output_File , Volume_Title , '     Page ', Page_Number );
   WRITELN( Output_File );
                                   (*   -- Subdirectory title          *)

   WRITELN( Output_File , Subdir_Title );
   WRITELN( Output_File );

   DEC( Lines_Left , 5 );

   IF ( File_Title <> '' ) THEN
      BEGIN
                                   (*   -- File title          *)

         WRITELN( Output_File , File_Title );
         WRITELN( Output_File );

         DEC( Lines_Left , 2 );

      END;

END   (* Display_Page_Titles *);

(*----------------------------------------------------------------------*)
(*     Write_Condensed_Line --- Write summary line to output file       *)
(*----------------------------------------------------------------------*)

PROCEDURE Write_Condensed_Line( VAR File_Name : AnyStr;
                                    File_Size : LONGINT;
                                    File_Time : LONGINT;
                                VAR Lib_Name  : AnyStr;
                                VAR File_Path : AnyStr  );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Write_Condensed_Line                                   *)
(*                                                                      *)
(*    Purpose:   Writes one summary line for a file/entry               *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Write_Condensed_Line( VAR File_Name : AnyStr;                  *)
(*                                 File_Size : LONGINT;                 *)
(*                                 File_Time : LONGINT;                 *)
(*                             VAR Lib_Name  : AnyStr;                  *)
(*                             VAR File_Path : AnyStr  );               *)
(*                                                                      *)
(*    Remarks:                                                          *)
(*                                                                      *)
(*       Here is the format of each summary line:                       *)
(*                                                                      *)
(*          Columns   Contents                                          *)
(*          =======   ========                                          *)
(*                                                                      *)
(*           1 - 12   File name                                         *)
(*          14 - 22   File size in bytes                                *)
(*          24 - 31   Date in YY/MM/DD format                           *)
(*          33 - 37   Time in 24 hour HH:MM format                      *)
(*          39 - 50   Library Name  (if file is member of a library)    *)
(*          52 - 63   Volume label                                      *)
(*          65 - 130  Path (without trailing file name)                 *)
(*                                                                      *)
(*       If a file is a member of a library, then the path is that of   *)
(*       the library file itself, NOT any path stored in the library.   *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   SSize : STRING[ 10 ];
   STime : STRING[ 10 ];
   SDate : STRING[ 10 ];

BEGIN (* Write_Condensed_Line *)

                                   (* Blank output line beyond volume label *)

   FillChar( Condensed_Output_Line[ 1 ] , 130 , ' ' );

                                   (* Convert file size to characters *)

   STR( File_Size : 8 , SSize );

                                   (* Convert file time and date    *)

   Dir_Convert_Date_And_Time_2( File_Time , SDate , STime );

                                   (* Move file info to output line *)

   MOVE( File_Name[ 1 ]   , Condensed_Output_Line[  1 ], LENGTH( File_Name ) );
   MOVE( SSize[ 1 ]       , Condensed_Output_Line[ 14 ], LENGTH( SSize ) );
   MOVE( SDate[ 1 ]       , Condensed_Output_Line[ 24 ], LENGTH( SDate ) );
   MOVE( STime[ 1 ]       , Condensed_Output_Line[ 33 ], LENGTH( STime ) );
   MOVE( Lib_Name [ 1 ]   , Condensed_Output_Line[ 39 ], LENGTH( Lib_Name  ) );
   MOVE( Volume_Label[ 1 ], Condensed_Output_Line[ 52 ], LENGTH( Volume_Label ) );
   MOVE( File_Path[ 1 ]   , Condensed_Output_Line[ 65 ], LENGTH( File_Path ) );

                                   (* Insert commas if needed *)

   IF ( Condensed_Listing_Type = Condensed_Comma ) THEN
      BEGIN
         Condensed_Output_Line[ 13 ] := ',';
         Condensed_Output_Line[ 23 ] := ',';
         Condensed_Output_Line[ 32 ] := ',';
         Condensed_Output_Line[ 38 ] := ',';
         Condensed_Output_Line[ 51 ] := ',';
         Condensed_Output_Line[ 64 ] := ',';
      END;
                                   (* Write summary line to output file *)

   WRITELN( Output_File , Condensed_Output_Line );

END   (* Write_Condensed_Line *);

(*----------------------------------------------------------------------*)
(*     Display_Error --- Display an error message                       *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_Error( Error_Message : AnyStr );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Display_Error                                          *)
(*                                                                      *)
(*    Purpose:   Displays an error message.                             *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Display_Error( Error_Message : AnyStr );                       *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Display_Error *)

   IF Do_Condensed_Listing THEN
      WRITELN( Status_File , Left_Margin_String, '  >>> ' , Error_Message )
   ELSE
      BEGIN
                                   (* See if there's room on this page *)
                                   (* for error message.               *)

         IF ( Lines_Left < 1 ) THEN
            Display_Page_Titles;
                                   (* Write out error message *)

         WRITELN( Output_File , Left_Margin_String, '  >>> ' , Error_Message );

                                   (* Count lines left on page *)

         IF Do_Printer_Format THEN
            BEGIN

               DEC( Lines_Left );

               IF ( Lines_Left < 1 ) THEN
                  Display_Page_Titles;

            END;

      END;

END   (* Display_Error *);

(*----------------------------------------------------------------------*)
(* Start_Contents_Listing --- Initialize listing of library contents     *)
(*----------------------------------------------------------------------*)

FUNCTION Start_Contents_Listing(    File_Desc : AnyStr;
                                    File_Name : AnyStr;
                                VAR Lib_File  : FILE;
                                VAR File_Pos  : LONGINT;
                                VAR Ierr      : INTEGER  ) : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Function:  Start_Contents_Listing                                 *)
(*                                                                      *)
(*    Purpose:   Initializes listing of library file contents           *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       OK := Start_Contents_Listing(     File_Desc : AnyStr;          *)
(*                                         File_Name : AnyStr;          *)
(*                                     VAR Lib_File  : FILE;            *)
(*                                     VAR File_Pos  : LONGINT;         *)
(*                                     VAR Ierr      : INTEGER  );      *)
(*                                                                      *)
(*          File_Desc --- Description of library file type              *)
(*          File_Name --- Library file name                             *)
(*          Lib_File  --- Library file handle                           *)
(*          File_Pos  --- Initial position in library file              *)
(*          Ierr      --- = 0 if file opened OK; <> 0 if error          *)
(*                                                                      *)
(*          OK        --- TRUE if library file opened OK.               *)
(*                                                                      *)
(*    Calls:                                                            *)
(*                                                                      *)
(*       DUPL                                                           *)
(*       Display_Page_Titles                                            *)
(*       Open_File                                                      *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Start_Contents_Listing *)

                                   (* Set left margin spacing *)

   Left_Margin_String := Left_Margin_String + DUPL( ' ' , Library_Indent );

                                   (* Set file title *)

   File_Title := Left_Margin_String + File_Desc + File_Name;

                                   (* Display library file's name *)
   IF Do_Printer_Format THEN
      IF ( Lines_Left < 3 ) THEN
         Display_Page_Titles;
                                   (* If we're listing contents at end  *)
                                   (* of directory, print library name. *)
                                   (* Do_Blank_Line flags whether we    *)
                                   (* need to print blank line in entry *)
                                   (* lister subroutine.  If listing    *)
                                   (* inline, then it's true for the    *)
                                   (* first file; otherwise it's false. *)
                                   (* This is to prevent unnecessary    *)
                                   (* blank lines in output listing     *)
                                   (* when no files are selected from   *)
                                   (* a given library.                  *)

   IF ( NOT Do_Condensed_Listing ) THEN
      IF ( NOT Expand_Libs_In ) THEN
         BEGIN
            WRITELN( Output_File ) ;
            WRITE  ( Output_File , File_Title );
            DEC( Lines_Left , 2 );
            Do_Blank_Line := FALSE;
         END
      ELSE
         Do_Blank_Line := TRUE;
                                   (* Make sure drive attached to file name *)

   IF ( POS( ':' , File_Name ) = 0 ) THEN
      File_Name := Cat_Drive + ':' + File_Name;

                                   (* No entries displayed yet *)
   Current_Entries := 0;

                                   (* Try opening library file for processing *)

   Open_File( File_Name, Lib_File, File_Pos, Ierr );

                                   (* Set flag indicating if open went OK *)

   Start_Contents_Listing := ( Ierr = 0 );

                                   (* If open fails, write error message *)
   IF ( Ierr <> 0 ) THEN
      Display_Error( 'Cannot open file ' + File_Name )

   ELSE IF ( NOT Expand_Libs_In ) THEN
      IF ( NOT Do_Condensed_Listing ) THEN
         BEGIN

            WRITELN( Output_File );
            WRITELN( Output_File );

                                   (* Count lines left on page *)

            IF Do_Printer_Format THEN
               DEC( Lines_Left );

         END;

END   (* Start_Contents_Listing *);

(*----------------------------------------------------------------------*)
(*   End_Contents_Listing --- Finish listing of library contents        *)
(*----------------------------------------------------------------------*)

PROCEDURE End_Contents_Listing( VAR Lib_File : FILE;
                                    Error    : INTEGER );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: End_Contents_Listing                                   *)
(*                                                                      *)
(*    Purpose:   Finishes listing of library file contents              *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       End_Contents_Listing( VAR Lib_File : FILE;                     *)
(*                                 Error    : INTEGER );                *)
(*                                                                      *)
(*          Lib_File  --- Library file handle                           *)
(*          Error     --- Error type                                    *)
(*                                                                      *)
(*    Calls:                                                            *)
(*                                                                      *)
(*       DUPL                                                           *)
(*       Close_File                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* End_Contents_Listing *)

                                   (* Display error message if necessary   *)

   IF ( Current_Entries > 0 ) THEN
      CASE Error OF
         Format_Error  : Display_Error( 'Bad file format' );
         Too_Many_Subs : Display_Error( 'Too many nested subdirectories' );
         ELSE;
      END (* CASE *);
                                   (* Print blank line after last entry    *)
                                   (* in library, if we're expanding       *)
                                   (* contents inline, but only if library *)
                                   (* had any entries listed.              *)
   IF ( Expand_Libs_In AND
        ( NOT Do_Blank_Line ) AND ( NOT Do_Condensed_Listing ) ) THEN
      BEGIN

         WRITELN( Output_File );

         IF Do_Printer_Format THEN
            DEC( Lines_Left );

      END;
                                   (* Close library file *)
   Close_File( Lib_File );
                                   (* Restore previous left margin spacing *)

   Left_Margin_String := DUPL( ' ' , Left_Margin );

                                   (* No file title *)
   File_Title := '';

END   (* End_Contents_Listing *);

(*----------------------------------------------------------------------*)
(*     Display_One_Entry --- Write contents line to output file         *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_One_Entry( VAR File_Name : AnyStr;
                                 File_Size : LONGINT;
                                 File_Time : LONGINT;
                             VAR Lib_Name  : AnyStr;
                             VAR File_Path : AnyStr;
                             VAR Long_Name : AnyStr  );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Display_One_Entry                                      *)
(*                                                                      *)
(*    Purpose:   Writes information about one entry in library file     *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Display_One_Entry( VAR File_Name : AnyStr;                     *)
(*                              File_Size : LONGINT;                    *)
(*                              File_Time : LONGINT;                    *)
(*                          VAR Lib_Name  : AnyStr;                     *)
(*                          VAR File_Path : AnyStr;                     *)
(*                          VAR Long_Name : AnyStr  );                  *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   STime : STRING[ 10 ];
   SDate : STRING[ 10 ];
   I     : INTEGER;

BEGIN (* Display_One_Entry *)
                                   (* Write condensed-style line if *)
                                   (* requested                     *)
   IF Do_Condensed_Listing THEN
      Write_Condensed_Line( File_Name, File_Size, File_Time, Lib_Name,
                            File_Path )
   ELSE
      BEGIN                        (* Write normal style listing    *)

                                   (* Make sure room on current page *)
                                   (* for this entry name.           *)
                                   (* If enough room, print blank    *)
                                   (* line if requested.  This will  *)
                                   (* only happen for first file.    *)
         IF Do_Blank_Line THEN
            BEGIN
               IF ( Lines_Left < 2 ) THEN
                  Display_Page_Titles
               ELSE
                  BEGIN
                     WRITELN( Output_File );
                     DEC( Lines_left );
                  END;
               Do_Blank_Line := FALSE;
            END
         ELSE
            IF ( Lines_Left < 1 ) THEN
               Display_Page_Titles;

                                   (* Add '. ' to front if we're     *)
                                   (* expanding ARCs in main listing *)
         IF Expand_Libs_In THEN
            File_Name := '. ' + File_Name;

                                   (* Convert date and time to displayable form *)

         Dir_Convert_Date_And_Time( File_Time , SDate , STime );

                                   (* Write out file name, length, date, time *)

         WRITE( Output_File , Left_Margin_String, '      ' , File_Name );

         FOR I := LENGTH( File_Name ) TO 14 DO
            WRITE( Output_File , ' ' );

         WRITE  ( Output_File , File_Size:8, '  ' );
         WRITE  ( Output_File , SDate, '  ' );
         WRITE  ( Output_File , STime );

                                   (* See if we're to write out *)
                                   (* long file names.  If so,  *)
                                   (* write out subdirectory    *)
                                   (* path followed by file     *)
                                   (* name.                     *)

         IF Show_Long_File_Names THEN
            WRITE( Output_File , '  ' , Long_Name );

                                   (* Finish output line       *)

         WRITELN( Output_File );

                                   (* Count lines left on page *)
         IF Do_Printer_Format THEN
            DEC( Lines_Left );

      END;
                                   (* Increment total entry count *)
   INC( Total_Entries );
                                   (* Increment entry count this library *)
   INC( Current_Entries );
                                   (* Increment total space used  *)

   Total_ESpace := Total_ESpace + File_Size;

END   (* Display_One_Entry *);
