(*----------------------------------------------------------------------*)
(*   Display_MD_Contents --- Display contents of library (.MD) file     *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_MD_Contents( MDFileName : AnyStr );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Display_MD_Contents                                    *)
(*                                                                      *)
(*    Purpose:   Displays contents of a library file (.MD file)         *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Display_MD_Contents( MDFileName : AnyStr );                    *)
(*                                                                      *)
(*          MDFileName --- name of .MD file whose contents              *)
(*                         are to be listed.                            *)
(*                                                                      *)
(*    Calls:                                                            *)
(*                                                                      *)
(*       Aside from internal subroutines, these routines are required:  *)
(*                                                                      *)
(*          Dir_Convert_Date_And_Time                                   *)
(*                            --- convert DOS packed date/time to string*)
(*          Open_File         --- open a file                           *)
(*          Close_File        --- close a file                          *)
(*          Entry_Matches     --- Perform wildcard match                *)
(*          Display_Page_Titles                                         *)
(*                            --- Display titles at top of page         *)
(*          DUPL              --- Duplicate a character into a string   *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

TYPE
   Array4 = ARRAY[1..4] OF CHAR;
   
(* STRUCTURED *) CONST
   ValidSig : Array4 = 'MDmd'      (* Signature to verify mdcd file     *);

(*----------------------------------------------------------------------*)
(*                  Map of MD file entry header                         *)
(*----------------------------------------------------------------------*)

TYPE
   MD_Entry_Type = RECORD             (* Header for each compressed file *)

                      Signature    : Array4              (* file/header signature (MDmd)      *);
                      ReleaseLevel : BYTE                (* compress version                  *);
                      HeaderType   : BYTE                (* header type. only type 1 for now  *);
                      HeaderSize   : WORD                (* size of this header in bytes      *);
                      UserInfo     : WORD                (* any user info desired             *);
                      Reserved1    : WORD                (* future use and upward compatablty *);
                      Reserved2    : LONGINT             (* future use and upward compatablty *);
                      Reserved3    : ARRAY[1..8] OF BYTE (* future use and upward compatablty *);
                      CompressType : BYTE                (* type of compression               *);
                      OrigFileSize : LONGINT             (* original file size in bytes       *);
                      CompFileSize : LONGINT             (* compressed file size in bytes     *);
                      FileAttr     : WORD                (* original file attribute           *);
                      FileDate     : LONGINT             (* original file date/time           *);
                      FileCRC      : WORD                (* file crc                          *);
                      FileName     : STRING[12]          (* file name                         *);
                      PathName     : DirStr              (* original drive\path               *);

                  END;

VAR
   MDFile        : FILE                 (* MD file to be read             *);
   MD_Entry      : MD_Entry_Type        (* Header for one file in MD file *);
   MD_Pos        : LONGINT              (* Current byte offset in MD file *);
   Bytes_Read    : INTEGER              (* # bytes read from MD file      *);
   Ierr          : INTEGER              (* Error flag                     *);
   Long_Name     : AnyStr               (* Long file name                 *);

(*----------------------------------------------------------------------*)
(*        Get_Next_MD_Entry --- Get next header entry in MD file        *)
(*----------------------------------------------------------------------*)

FUNCTION Get_Next_MD_Entry( VAR MDEntry : MD_Entry_Type;
                            VAR Error   : INTEGER ) : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Function:  Get_Next_MD_Entry                                      *)
(*                                                                      *)
(*    Purpose:   Gets header information for next file in MD file       *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       OK := Get_Next_MD_Entry( VAR MDEntry : MD_Entry_Type;          *)
(*                                VAR Error   : INTEGER );              *)
(*                                                                      *)
(*          MDEntry --- Header data for next file in MD file            *)
(*          Error   --- Error flag                                      *)
(*          OK      --- TRUE if header successfully found, else FALSE   *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Get_Next_MD_Entry *)
                                   (* Assume no error to start *)
   Error := 0;
                                   (* Except first time, move to     *)
                                   (* next supposed header record in *)
                                   (* MD file.                       *)

   IF ( MD_Pos <> 0 ) THEN
      Seek( MDFile, MD_Pos );

                                   (* Read in the file header entry. *)

   BlockRead( MDFile, MDEntry, SizeOf( MD_Entry ), Bytes_Read );
   Error := 0;
                                   (* If we didn't read enough, assume     *)
                                   (* it's the end of the file.            *)

   IF ( Bytes_Read <  SizeOf( MD_Entry ) ) THEN
      Error := End_Of_File
                                   (* Check signature.  If wrong, then    *)
                                   (* file is bad or not an MD file at    *)
                                   (* all.                                *)

   ELSE IF ( MDEntry.Signature <> ValidSig ) THEN
      Error := Format_Error
   ELSE                            (* Header looks ok -- we got    *)
                                   (* the entry data.  Position to *)
                                   (* next header.                 *)
      WITH MDEntry DO
         MD_Pos := MD_Pos + HeaderSize + CompFileSize;

                                    (* Report success/failure to calling *)
                                    (* routine.                          *)

   Get_Next_MD_Entry := ( Error = 0 );

END   (* Get_Next_MD_Entry *);

(*----------------------------------------------------------------------*)
(*         Display_MD_Entry --- Display MD file header entry            *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_MD_Entry( MD_Entry : MD_Entry_Type );

VAR
   FName      : AnyStr;

BEGIN (* Display_MD_Entry *)

   WITH MD_Entry DO
      BEGIN
                                   (* Pick up file name              *)
         FName := FileName;
                                   (* See if this file matches the   *)
                                   (* entry spec wildcard.  Exit if  *)
                                   (* not.                           *)

         IF Use_Entry_Spec THEN
            IF ( NOT Entry_Matches( FName ) ) THEN
               EXIT;
                                   (* See if we're to write out *)
                                   (* long file names.  If so,  *)
                                   (* get subdirectory path     *)
                                   (* followed by file name.    *)

         IF Show_Long_File_Names THEN
            Long_Name := PathName + FileName
         ELSE
            Long_Name := '';
                                   (* Display info for this entry *)

         Display_One_Entry( FName, OrigFileSize, FileDate, MDFileName,
                            Current_Subdirectory, Long_Name );

      END;

END (* Display_MD_Entry *);

(*----------------------------------------------------------------------*)

BEGIN (* Display_MD_Contents *)
                                   (* Open MD file and initialize display *)

   IF Start_Contents_Listing( ' MD file: ',
                              Current_Subdirectory + MDFileName, MDFile,
                              MD_Pos, Ierr ) THEN
      BEGIN
                                   (* Loop over entries in MD file *)

         WHILE( Get_Next_MD_Entry( MD_Entry , Ierr ) ) DO
            Display_MD_Entry( MD_Entry );

                                   (* Close MD file *)

         End_Contents_Listing( MDFile , Ierr );

      END;

END   (* Display_MD_Contents *);
