(*----------------------------------------------------------------------*)
(*     Display_LZH_Contents --- Display contents of archive file    *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_LZH_Contents( LZHFileName : AnyStr );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Procedure: Display_LZH_Contents                                   *)
(*                                                                      *)
(*    Purpose:   Displays contents of an LHARC (.LZH file)              *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       Display_LZH_Contents( LZHFileName : AnyStr );                  *)
(*                                                                      *)
(*          LZHFileName --- name of LZH file whose contents             *)
(*                          are to be listed.                           *)
(*                                                                      *)
(*    Calls:                                                            *)
(*                                                                      *)
(*          Dir_Convert_Date_And_Time                                   *)
(*          Start_Library_Listing                                       *)
(*          End_Library_Listing                                         *)
(*          Display_Page_Titles                                         *)
(*          Entry_Matches                                               *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

(*----------------------------------------------------------------------*)
(*                  Map of LZH file entry header                        *)
(*----------------------------------------------------------------------*)

TYPE
   Char5           = ARRAY[ 1 .. 5  ] OF CHAR;

   LZH_Entry_Bytes = ARRAY[ 0 .. 21 ] OF BYTE;

   LZH_Entry_Type  = RECORD
                        RecLen   : BYTE      (* Header record length     *);
                        CheckSum : BYTE      (* Checksum of header bytes *);
                        Compress : Char5     (* Compression type         *);
                        CSize    : LONGINT   (* Compressed size          *);
                        OSize    : LONGINT   (* Original size            *);
                        Time     : WORD      (* Packed time              *);
                        Date     : WORD      (* Packed date              *);
                        Attr     : WORD      (* File attributes          *);
                        FNameLen : BYTE      (* Length of file name      *);
                     END;

VAR
   LZHFile       : FILE                 (* LZH file to be read            *);
   LZH_Entry     : LZH_Entry_Type       (* Header for one file in library *);
   LZH_Pos       : LONGINT              (* Current byte offset in library *);
   Bytes_Read    : INTEGER              (* # bytes read from library file *);
   Ierr          : INTEGER              (* Error flag                     *);
   Display_Entry : BOOLEAN              (* TRUE to display this entry     *);
   FName         : AnyStr               (* Short file name                *);
   Long_Name     : AnyStr               (* Long file name                 *);
   DirS          : DirStr               (* Directory name                 *);
   FExt          : ExtStr               (* Extension of file name         *);
   CheckSum      : INTEGER              (* Header checksum                *);

(*----------------------------------------------------------------------*)
(*   Get_Next_LZH_Entry --- Get next header entry in library            *)
(*----------------------------------------------------------------------*)

FUNCTION Get_Next_LZH_Entry( VAR LZHEntry      : LZH_Entry_Type;
                             VAR Display_Entry : BOOLEAN;
                             VAR Error         : INTEGER ) : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*    Function:  Get_Next_LZH_Entry                                     *)
(*                                                                      *)
(*    Purpose:   Gets header information for next file in library       *)
(*                                                                      *)
(*    Calling sequence:                                                 *)
(*                                                                      *)
(*       OK := Get_Next_LZH_Entry( VAR LZHEntry :                       *)
(*                                     LZH_Entry_Type;                  *)
(*                                 VAR Display_Entry : BOOLEAN;         *)
(*                                 VAR Error    : INTEGER ) :           *)
(*                                 BOOLEAN;                             *)
(*                                                                      *)
(*          LZHEntry      --- Header data for next file in library      *)
(*          Display_Entry --- TRUE to display this entry                *)
(*          Error         --- Error flag                                *)
(*          OK            --- TRUE if header successfully found         *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   I         : INTEGER;
   LZHBuffer : LZH_Entry_Bytes ABSOLUTE LZH_Entry;

BEGIN (* Get_Next_LZH_Entry *)
                                   (* Assume no error to start       *)
   Error         := 0;
                                   (* Assume we don't display this   *)
                                   (* entry.                         *)
   Display_Entry      := FALSE;
   Get_Next_LZH_Entry := FALSE;
                                   (* Except first time, move to     *)
                                   (* next supposed header record in *)
                                   (* library.                       *)
   IF ( LZH_Pos <> 0 ) THEN
      Seek( LZHFile, LZH_Pos );
                                   (* Check for I/O error            *)
   IF ( IOResult <> 0 ) THEN
      BEGIN
         Error := Format_Error;
         EXIT;
      END;
                                   (* Read in the file header entry. *)

   BlockRead( LZHFile, LZHEntry, SIZEOF( LZHEntry ), Bytes_Read );

                                   (* Check for I/O error            *)
   IF ( IOResult <> 0 ) THEN
      BEGIN
         Error := Format_Error;
         EXIT;
      END;
                                   (* If wrong size read, or header marker *)
                                   (* is incorrect, report library format  *)
                                   (* error.                               *)

   IF ( Bytes_Read <> SIZEOF( LZHEntry ) ) THEN
      BEGIN
         IF ( LZHEntry.RecLen = 0 ) THEN
            Error := End_Of_File
         ELSE
            Error := Format_Error;
      END
   ELSE                            (* Header looks ok. *)
      WITH LZHEntry DO
         BEGIN
                                   (* Pick up file name. *)

            BlockRead( LZHFile, Long_Name[ 1 ], LZHEntry.FNameLen, Bytes_Read );

                                   (* Check for I/O error            *)

            IF ( IOResult <> 0 ) THEN
               BEGIN
                  Error := Format_Error;
                  EXIT;
               END;
                                   (* Set length of file name *)

            Long_Name[ 0 ] := CHR( Bytes_Read );

                                   (* Position to next header. *)

            LZH_Pos := LZH_Pos + LZHEntry.CSize + SIZEOF( LZHEntry ) +
                       Bytes_Read + 2;

                                   (* Compute checksum of header *)
            CheckSum := 0;

            FOR I := 1 TO 21 DO
               CheckSum := ( CheckSum + LZHBuffer[ I ] ) AND 255;

            FOR I := 1 TO Bytes_Read DO
               CheckSum := ( CheckSum + ORD( Long_Name[ I ] ) ) AND 255;

                                    (* If checksum wrong, quit.          *)

            IF ( CheckSum <> LZH_Entry.CheckSum ) THEN
               Error := Format_Error;

         END;
                                    (* Report success/failure to calling *)
                                    (* routine.                          *)

   Display_Entry      := ( Error = 0 );
   Get_Next_LZH_Entry := Display_Entry;

END   (* Get_Next_LZH_Entry *);

(*----------------------------------------------------------------------*)
(*      Display_LZH_Entry --- Display file entry info                   *)
(*----------------------------------------------------------------------*)

PROCEDURE Display_LZH_Entry( LZH_Entry : LZH_Entry_Type );

VAR
   TimeDate  : LONGINT;
   TimeDateW : ARRAY[1..2] OF WORD ABSOLUTE TimeDate;

BEGIN (* Display_LZH_Entry *)

   WITH LZH_Entry DO
      BEGIN
                                   (* Extract short file name from   *)
                                   (* long file name.                *)

         FSplit( Long_Name, DirS, FName, FExt );

         FName := FName + FExt;

                                   (* See if this file matches the   *)
                                   (* entry spec wildcard.  Exit if  *)
                                   (* not.                           *)
         IF Use_Entry_Spec THEN
            IF ( NOT Entry_Matches( FName ) ) THEN
               EXIT;
                                   (* Get date and time of creation *)
         TimeDateW[ 1 ] := Time;
         TimeDateW[ 2 ] := Date;

                                   (* Zap long file name if same  *)
                                   (* as short file name.         *)

         IF ( Long_Name = FName ) THEN
            Long_Name := '';

                                   (* Display info for this entry *)

         Display_One_Entry( FName, OSize, TimeDate, LZHFileName,
                            Current_Subdirectory, Long_Name );

      END;

END (* Display_LZH_Entry *);

(*----------------------------------------------------------------------*)

BEGIN (* Display_LZH_Contents *)

                                   (* Note if LZH or LZS type.         *)

   FSplit( LZHFileName, DirS, FName, FExt );

   IF ( LENGTH( FExt ) > 1 ) THEN
      IF ( FExt[ 1 ] = '.' ) THEN
         DELETE( FExt, 1, 1 );
                                   (* Open library file and initialize *)
                                   (* contents display.                *)

   IF Start_Contents_Listing( ' ' + FExt + ' file: ',
                              Current_Subdirectory + LZHFileName, LZHFile,
                              LZH_Pos, Ierr ) THEN
      BEGIN
                                   (* Loop over entries in library file *)

         WHILE( Get_Next_LZH_Entry( LZH_Entry , Display_Entry , Ierr ) ) DO
            IF Display_Entry THEN
               Display_LZH_Entry( LZH_Entry );

                                   (* Close library files, complete display *)

         End_Contents_Listing( LZHFile , Ierr );

      END;

END   (* Display_LZH_Contents *);
