-- ADA-TUTR.ADA   Ver. 1.00   25-MAR-1988
-- Copyright 1988 John J. Herro
-- Software Innovations Technology
-- 1083 Mandarin Drive NE, Palm Bay, FL 32905-4706   (407)951-0233
--
-- Introduction:
--
-- ADA-TUTR provides interactive instruction in the Ada programming language,
-- allowing the student to learn at his own pace.  On a PC, it requires a hard
-- disk or a 3.5-inch disk.  Access to an Ada compiler is helpful, but not
-- required.  The user can exit this program at any time by striking X (or
-- control-BREAK), and later resume the session exactly where he left off.
--
-- ADA-TUTR presents a screenful of information at a time.  Screens are read
-- in 64-byte blocks from the random access file ADA-TUTR.DAT, using DIRECT_IO.
-- For most screens, ADA-TUTR waits for the user to strike one character to
-- determine which screen to show next.  Screens are numbered starting with
-- 101; each screen has a three-digit number.  Screens 101 through 106 have
-- special uses, as follows:
--
-- 101 - This screen is presented when the user completes the Ada course.  It
--       contains a congratulatory message.  After this screen is shown,
--       control returns directly to the operating system; the program doesn't
--       wait for the user to strike a character.
-- 102 - This screen is presented when the user exits ADA-TUTR before
--       completing the course.  It displays the number of the screen where the
--       student left off, so he can later resume the session at the same
--       place (see screen 105).  After this screen is shown, control returns
--       directly to the operating system; the program doesn't wait for the
--       user to strike a character.
-- 103 - This screen is shown whenever the student strikes X or control-BREAK.
--       It asks if the user wants to exit the program.  If he strikes Y,
--       screen 102 is shown and control returns to the operating system.  If
--       he types N (perhaps because he struck X in error), ADA-TUTR returns to
--       the previous screen.  From screen 103, the student can also strike M
--       to see the main menu (screen 106).
-- 104 - This is the opening screen.  It asks if the student has used ADA-TUTR
--       before.  If he strikes N, an introductory screen is presented and the
--       course begins.  If he strikes Y, screen 105 is shown.
-- 105 - This screen says "Welcome back!"  It then asks the user to type the
--       number where he earlier left off (see screen 102).  For this screen,
--       instead of striking one character, the user types a three-digit number
--       and presses ENTER.  Any number from 104 through 596 (the highest
--       possible screen number) is accepted.  That screen is then shown, so
--       the student can resume learning where he left off.  If the user has
--       forgotten the number, or if he wants to resume at some other place, he
--       can request screen number 106 to see the main menu.  Also, if he types
--       the number of a non-existent screen, screen 106 is shown.
-- 106 - This screen contains the main menu of topics covered in ADA-TUTR.
--       When the user selects a main topic, an appropriate sub-menu is shown.
--

-- Format of the Data File:
--
-- ADA-TUTR.DAT is a random access file containing 64-byte blocks.  The first
-- block is referred to as block 1.  Data for the screens are stored starting
-- in block 32.  Blocks 1 through 31 contain the numbers of the starting blocks
-- for each of the 496 possible screens (101 through 596).  Four bytes are
-- allocated for each starting block number.  These numbers are stored in
-- ASCII, and 1000 is added to each block number so that it will have exactly
-- four digits.  For example, screen 101 starts at block 32, so the first four
-- bytes of block 1 contain "1032" in ASCII.  The next four bytes refer to
-- screen 102, etc.  Not all 496 possible screens exist.  For any unused
-- screens, the starting block number of screen 106 (the main menu) is stored.
--
-- The first 64-byte block of each screen is a "control block."  Thus, block
-- 32 of ADA-TUTR.DAT is the control block for screen 101.  The control block
-- is followed by enough 64-byte blocks to contain the text for that screen.
--
-- The first two bytes of the control block contain, in ASCII, the number of
-- text blocks that follow.  A constant 10 is added to this number so that it
-- will always have exactly two digits.  The next two bytes contain, in ASCII,
-- the number of significant text characters in the last 64-byte block, plus
-- ten.  This number can range from 11 to 74.  For example, if a screen has 130
-- characters, three text blocks will be needed, and the last block will have
-- two significant characters.  Thus, the first four bytes of the control block
-- will contain "1312" in ASCII.
--
-- Starting with byte 5, the control block contains a list of characters that
-- the user might strike after seeing this screen.  Each character is followed
-- by the three-digit number of the next screen to be shown when that character
-- is struck.  The list is terminated by $.  There is enough room for 14
-- characters and screen numbers, and the terminating $, with three bytes to
-- spare.  For example, in the control block for screen 104, bytes 5 through 13
-- might contain "Y105N118$".  This means that, if the user strikes Y, screen
-- 105 will be shown, and if he strikes N, screen 118 will be shown.  Striking
-- any other character will simply cause a beep (except that CR will be
-- ignored).  A "screen number" of 100 following a character simply means "go
-- back to the previous screen."  This feature is used in screen 103.
--
-- If the list of characters begins with '#', the user is prompted to type the
-- next screen number.  This feature is used in screen 105.
--

-- Before Running ADA-TUTR on a PC:
--
-- ADA-TUTR uses ANSI escape sequences for highlighting, cursor positioning,
-- and reverse video.  Therefore, on a PC the device driver ANSI.SYS must be
-- installed before ADA-TUTR will work correctly.  To install ANSI.SYS, do the
-- following:
--
-- 1.  If there is a file CONFIG.SYS in the root directory of the disk from
--     which you boot, type it and look for a line saying "DEVICE=ANSI.SYS"
--     (without the quotes), in either upper or lower case.  If that line is
--     not present, add it to CONFIG.SYS anywhere in the file, using an
--     ordinary text editor or word processor in the non-document mode.  If
--     there is no CONFIG.SYS file, create one containing the single line
--     "DEVICE=ANSI.SYS" (without the quotes).
--
-- 2.  If there is no file ANSI.SYS in your root directory, copy ANSI.SYS from
--     from your system distribution diskette to the root directory of the disk
--     from which you boot.
--
-- 3.  Reboot the computer.  ADA-TUTR should then work correctly.
--
--
--
--
-- Note:  This program is compiled with version 1.30 Beta of Artek Ada (tm,
--        Artek Corp.)  Due to minor bugs in that beta-test version, the form
--        feeds must be removed from a copy of this file before compilation.
--        Also, the two subprograms cannot be made separate, which would have
--        been preferred.
--
--
--

with QGET, CON_IO, DIRECT_IO; use CON_IO;
   -- The procedure QGET and the package CON_IO (console I/O) come with Artek
   -- Ada.  QGET gets one character from the keyboard.  CON_IO contains PUT,
   -- PUT_LINE, and NEW_LINE, similar to TEXT_IO.  It also contains GET to get
   -- a string (with editing capability), and CLS to clear the screen.
procedure ADA_TUTR is
   subtype BLOCK_SUBTYPE is STRING(1 .. 64);
   package RANDOM_IO is new DIRECT_IO(BLOCK_SUBTYPE); use RANDOM_IO;
   DATA_FILE         : FILE_TYPE;      -- The file from which screens are read.
   BLOCK,                               -- A 64-byte block read from data file.
   CONTROL_BLOCK     : BLOCK_SUBTYPE;  -- Control info. for the current screen.
   PLACE,                                -- Index to search list of characters.
   WHERE,                              -- Location of ctrl. block in data file.
   NUM_OF_BLOCKS,                       -- # of blocks of text for this screen.
   LEN_OF_LAST_BLOCK : INTEGER;           -- # of signif. chars. in last block.
   SN, OLD_SN        : INTEGER := 104;     -- Screen #.  Opening screen is 104.
   INDX              : STRING(1 .. 1984);    -- Starting block numbers, + 1000.
   INPUT             : STRING(1 .. 4);              -- Input typed by the user.
   OK                : BOOLEAN;            -- True when user response is valid.
   LEGAL_NOTE        : constant STRING(1 .. 30) :=
        " Copyright 1988 John J. Herro ";
                         -- LEGAL_NOTE isn't used by the program, but it causes
                         -- the compiler to place this string in the .EXE file.

   procedure USER_TYPES_NEXT_SCREEN_NUMBER is
   begin
      OK := FALSE;
      while not OK loop            -- Keep trying until user response is valid.
         PUT("# ");                       -- Prompt user to type screen number.
         INPUT := "    ";  GET(INPUT);  NEW_LINE; -- Get screen num. from user.
         if INPUT(1) = 'x' or INPUT(1) = 'X' or INPUT(1) = ASCII.ETX then
            SN := 103;             -- Show screen 103 if user types X or BREAK.
            OK := TRUE;                      -- X or BREAK is a valid response.
         else
            begin                            -- Convert ASCII input to screen
               SN := INTEGER'VALUE(INPUT);   -- number.  If in range, set OK to
               OK := SN in 104 .. 596;       -- TRUE.  If input could not be
            exception                        -- converted to a number (e.g.,
               when others => null;          -- illegal character), leave OK =
            end;                             -- FALSE so user can try again.
         end if;
         if not OK then
            PUT_LINE("Incorrect number.  Please try again.");
         end if;
      end loop;
   end USER_TYPES_NEXT_SCREEN_NUMBER;

   procedure USER_TYPES_ONE_CHARACTER is
   begin
      PUT(">");                         -- Prompt user to strike one character.
      OK := FALSE;
      while not OK loop            -- Keep trying until user response is valid.
         QGET(INPUT(1));                        -- Get one character from user.
         if INPUT(1) in 'a' .. 'z' then        -- Force upper case to simplify.
            INPUT(1) := CHARACTER'VAL(CHARACTER'POS(INPUT(1)) - 32);
         end if;
         if INPUT(1) = 'X' or INPUT(1) = ASCII.ETX then
            SN := 103;             -- Show screen 103 if user types X or BREAK.
            OK := TRUE;                      -- X or BREAK is a valid response.
         end if;
         PLACE := 5;        -- Search list of valid characters for this screen.
         while not OK and CONTROL_BLOCK(PLACE) /= '$' loop  -- $ ends the list.
            if INPUT(1) = CONTROL_BLOCK(PLACE) then
                 -- Typed char. found in list; get screen # from control block.
               SN := INTEGER'VALUE(CONTROL_BLOCK(PLACE + 1 .. PLACE + 3));
               OK := TRUE;   -- Characters in the list are all valid responses.
            end if;
            PLACE := PLACE + 4; -- A 3-digit number follows each char. in list.
         end loop;
         if not OK and INPUT(1) /= ASCII.CR then         -- Beep if response is
            PUT(ASCII.BEL);                              -- not valid, but
         end if;                                         -- ignore CRs quietly.
      end loop;
   end USER_TYPES_ONE_CHARACTER;

begin
   OPEN(DATA_FILE, MODE => IN_FILE, NAME => "ADA-TUTR.DAT");
   for I in 1 .. 31 loop                -- Read list of starting block numbers.
      READ(DATA_FILE, ITEM => BLOCK, FROM => COUNT(I));
      INDX(64*I - 63 .. 64*I) := BLOCK;
   end loop;
   while OLD_SN >= 103 loop    -- Exit program after showing screen 101 or 102.
      WHERE := INTEGER'VALUE(INDX(SN*4 - 403 .. SN*4 - 400)) - 1000;
                                 -- Get location of control block in data file.
      READ(DATA_FILE, ITEM => CONTROL_BLOCK, FROM => COUNT(WHERE));
      NUM_OF_BLOCKS := INTEGER'VALUE(CONTROL_BLOCK(1 .. 2)) - 10;
      LEN_OF_LAST_BLOCK := INTEGER'VALUE(CONTROL_BLOCK(3 .. 4)) - 10;
      CLS;                           -- Clear the display and show this screen.
      for I in 1 .. NUM_OF_BLOCKS - 1 loop
         READ(DATA_FILE, ITEM => BLOCK, FROM => COUNT(WHERE + I));
         if SN = 102 and I = 3 then  -- Screen 102 needs # where user left off.
            BLOCK(35 .. 38) := INTEGER'IMAGE(OLD_SN);
         end if;
         PUT(BLOCK);
      end loop;
      READ(DATA_FILE, ITEM => BLOCK, FROM => COUNT(WHERE + NUM_OF_BLOCKS));
      PUT(BLOCK(1 .. LEN_OF_LAST_BLOCK));  -- Show only significant characters.
      if SN /= 103 then      -- Screen 103 means user typed X or BREAK to exit.
         OLD_SN := SN;       -- Save any other screen number, so user can go
      end if;                -- back to it from 103.  Don't go "back" to 103!
      if SN >= 103 then              -- If this screen doesn't end the program,
         if CONTROL_BLOCK(5) = '#' then      -- set SN to # of the next screen.
            USER_TYPES_NEXT_SCREEN_NUMBER;
         else
            USER_TYPES_ONE_CHARACTER;
         end if;
      end if;
      if SN = 100 then -- Screen number "100" means go back to the last screen.
         SN := OLD_SN;
      end if;
   end loop;
exception
   when NAME_ERROR =>
      PUT_LINE("I'm sorry.  The file ADA-TUTR.DAT seems to be missing.");
end ADA_TUTR;
