Unit Multi;

{$I Sys75.Inc}

Interface

Const
  MaxLockRetries : Byte = 10;

  NormalMode = $02; { ---- 0010 }
  ReadOnly   = $00; { ---- 0000 }
  WriteOnly  = $01; { ---- 0001 }
  ReadWrite  = $02; { ---- 0010 }
  DenyAll    = $10; { 0001 ---- }
  DenyWrite  = $20; { 0010 ---- }
  DenyRead   = $30; { 0011 ---- }
  DenyNone   = $40; { 0100 ---- }
  NoInherit  = $70; { 1000 ---- }

Type
  Taskers = (NoTasker, DesqView, DoubleDOS, Windows, OS2, NetWare);

Var
  MultiTasking: Boolean;
  MultiTasker : Taskers;
  MultiVersion: Word;
  VideoSeg    : Word;
  VideoOfs    : Word;

Const
  MultiString: Array [NoTasker..NetWare] of String [9] =
    ('Dos', 'DesqView', 'DoubleDOS', 'Windoze', 'OS/2', 'NetWare');

Procedure SetFileMode (Mode: Word);
  {- Set filemode for typed/untyped files }

Procedure ResetFileMode;
  {- Reset filemode to ReadWrite (02h) }

Procedure LockFile (Var F);
  {- Lock file F }

Procedure UnLockFile (Var F);
  {- Unlock file F }

Procedure LockBytes (Var F;  Start, Bytes: LongInt);
  {- Lock Bytes bytes of file F, starting with Start }

Procedure UnLockBytes (Var F;  Start, Bytes: LongInt);
  {- Unlock Bytes bytes of file F, starting with Start }

Procedure LockRecords (Var F;  Start, Records: LongInt);
  {- Lock Records records of file F, starting with Start }

Procedure UnLockRecords (Var F;  Start, Records: LongInt);
  {- Unlock Records records of file F, starting with Start }

Function  TimeOut: Boolean;
  {- Check for LockRetry timeout }

Procedure TimeOutReset;
  {- Reset internal LockRetry counter }

Function  InDos: Boolean;
  {- Is DOS busy? }

Procedure GiveTimeSlice;
  {- Give up remaining CPU time slice }

Procedure BeginCrit;
  {- Enter critical region }

Procedure EndCrit;
  {- End critical region }

Implementation

Uses
  DOS;

Var
  InDosFlag: ^Word;
  LockRetry: Byte;

  {=============================================================================}

  Procedure FLock (Handle: Word; Pos, Len: LongInt);
  Inline (
  $B8 / $00 / $5C /    {  mov   AX,$5C00        ;DOS FLOCK, Lock subfunction}
  $8B / $5E / $04 /    {  mov   BX,[BP + 04]    ;Place file handle in Bx register}
  $C4 / $56 / $06 /    {  les   DX,[BP + 06]    ;Load position in ES:DX}
  $8C / $C1 /        {  mov   CX,ES           ;Move ES pointer to CX register}
  $C4 / $7E / $08 /    {  les   DI,[BP + 08]    ;Load length in ES:DI}
  $8C / $C6 /        {  mov   SI,ES           ;Move ES pointer to SI register}
  $CD / $21);       {  int   $21             ;Call DOS}

  {-----------------------------------------------------------------------------}

  Procedure FUnlock (Handle: Word; Pos, Len: LongInt);
  Inline (
  $B8 / $01 / $5C /    {  mov   AX,$5C01        ;DOS FLOCK, Unlock subfunction}
  $8B / $5E / $04 /    {  mov   BX,[BP + 04]    ;Place file handle in Bx register}
  $C4 / $56 / $06 /    {  les   DX,[BP + 06]    ;Load position in ES:DX}
  $8C / $C1 /        {  mov   CX,ES           ;Move ES pointer to CX register}
  $C4 / $7E / $08 /    {  les   DI,[BP + 08]    ;Load length in ES:DI}
  $8C / $C6 /        {  mov   SI,ES           ;Move ES pointer to SI register}
  $CD / $21);       {  int   $21             ;Call DOS}

  {=============================================================================}

  Procedure SetFileMode (Mode: Word);
Begin
  FileMode := Mode;
End;    { SetFileMode }

{-----------------------------------------------------------------------------}

Procedure ResetFileMode;
Begin
  FileMode := NormalMode;
End;    { ResetFileMode }

{-----------------------------------------------------------------------------}

Procedure LockFile (Var F);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, 0, FileSize (File (F) ) );
End;    { LockFile }

{-----------------------------------------------------------------------------}

Procedure UnLockFile (Var F);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, 0, FileSize (File (F) ) );
End;    { UnLockFile }

{-----------------------------------------------------------------------------}

Procedure LockBytes (Var F;  Start, Bytes: LongInt);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, Start, Bytes);
End;    { LockBytes }

{-----------------------------------------------------------------------------}

Procedure UnLockBytes (Var F;  Start, Bytes: LongInt);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, Start, Bytes);
End;    { UnLockBytes }

{-----------------------------------------------------------------------------}

Procedure LockRecords (Var F;  Start, Records: LongInt);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, Start * FileRec (F).RecSize, Records * FileRec (F).RecSize);
End;    { LockBytes }

{-----------------------------------------------------------------------------}

Procedure UnLockRecords (Var F;  Start, Records: LongInt);
Begin
  If Not MultiTasking Then
    Exit;

  While InDos Do
    GiveTimeSlice;

  FLock (FileRec (F).Handle, Start * FileRec (F).RecSize, Records * FileRec (F).RecSize);
End;    { UnLockBytes }

{-----------------------------------------------------------------------------}

Function  TimeOut: Boolean;
Begin
  GiveTimeSlice;
  TimeOut := True;

  If MultiTasking And (LockRetry < MaxLockRetries) Then
  Begin
    TimeOut := False;
    Inc (LockRetry);
  End;  { If }
End;    { TimeOut }

{-----------------------------------------------------------------------------}

Procedure TimeOutReset;
Begin
  LockRetry := 0;
End;    { TimeOutReset }

{-----------------------------------------------------------------------------}

Function  InDos: Boolean;
Begin   { InDos }
  InDos := InDosFlag^ > 0;
End;    { InDos }

{-----------------------------------------------------------------------------}

Procedure GiveTimeSlice;  Assembler;
Asm     { GiveTimeSlice }
  cmp   MultiTasker, DesqView
  je    @DVwait
  cmp   MultiTasker, DoubleDOS
  je    @DoubleDOSwait
  cmp   MultiTasker, Windows
  je    @WinOS2wait
  cmp   MultiTasker, OS2
  je    @WinOS2wait
  cmp   MultiTasker, NetWare
  je    @Netwarewait

  @Doswait:
  Int   $28
  jmp   @WaitDone

  @DVwait:
  mov   AX, $1000
  Int   $15
  jmp   @WaitDone
  
  @DoubleDOSwait:
  mov   AX, $EE01
  Int   $21
  jmp   @WaitDone
  
  @WinOS2wait:
  mov   AX, $1680
  Int   $2F
  jmp   @WaitDone
  
  @Netwarewait:
  mov   BX, $000A
  Int   $7A
  jmp   @WaitDone
  
  @WaitDone:
End;    { TimeSlice }

{----------------------------------------------------------------------------}

Procedure BeginCrit;  Assembler;
Asm     { BeginCrit }
  cmp   MultiTasker, DesqView
  je    @DVCrit
  cmp   MultiTasker, DoubleDOS
  je    @DoubleDOSCrit
  cmp   MultiTasker, Windows
  je    @WinCrit
  jmp   @EndCrit
  
  @DVCrit:
  mov   AX, $101B
  Int   $15
  jmp   @EndCrit
  
  @DoubleDOSCrit:
  mov   AX, $EA00
  Int   $21
  jmp   @EndCrit
  
  @WinCrit:
  mov   AX, $1681
  Int   $2F
  jmp   @EndCrit
  
  @EndCrit:
End;    { BeginCrit }

{----------------------------------------------------------------------------}

Procedure EndCrit;  Assembler;
Asm     { EndCrit }
  cmp   MultiTasker, DesqView
  je    @DVCrit
  cmp   MultiTasker, DoubleDOS
  je    @DoubleDOSCrit
  cmp   MultiTasker, Windows
  je    @WinCrit
  jmp   @EndCrit
  
  @DVCrit:
  mov   AX, $101C
  Int   $15
  jmp   @EndCrit
  
  @DoubleDOSCrit:
  mov   AX, $EB00
  Int   $21
  jmp   @EndCrit
  
  @WinCrit:
  mov   AX, $1682
  Int   $2F
  jmp   @EndCrit

  @EndCrit:
End;    { EndCrit }

{============================================================================}

Var
  H: Byte;
Begin { Share }
  {- Init }
  LockRetry := 0;

  Asm
    @CheckDV:
    mov   AX, $2B01
    mov   CX, $4445
    mov   DX, $5351
    Int   $21
    cmp   AL, $FF
    je    @CheckWindows
    mov   MultiTasker, DesqView
    jmp   @CheckDone

    @CheckWindows:
    mov   AX, $1600
    Int   $2F
    cmp   AL, $00
    je    @CheckDoubleDOS
    cmp   AL, $80
    je    @CheckDoubleDOS
    mov   MultiTasker, Windows
    jmp   @CheckDone

    @CheckDoubleDOS:
    mov   AX, $E400
    Int   $21
    cmp   AL, $00
    je    @CheckOS2
    mov   MultiTasker, DoubleDOS
    jmp   @CheckDone

    @CheckOS2:
    mov   AX, $3001
    Int   $21
    cmp   AL, $0A
    je    @InOS2
    cmp   AL, $14
    jne   @CheckNetware
    @InOS2:
    mov   MultiTasker, OS2
    jmp   @CheckDone

    @CheckNetware:
    mov   AX, $7A00
    Int   $2F
    cmp   AL, $FF
    jne   @NoTasker
    mov   MultiTasker, NetWare
    jmp   @CheckDone

    @NoTasker:
    mov   MultiTasker, NoTasker

    @CheckDone:
    {-Set MultiTasking }
    cmp   MultiTasker, NoTasker
    mov   VideoSeg, $B800
    mov   VideoOfs, $0000
    je    @NoMultiTasker
    mov   MultiTasking, $01
    {-Get video address }
    mov   AH, $FE
    mov   BX, VideoSeg
    mov   ES, BX
    mov   BX, VideoOfs
    mov   DI, BX
    xor   BX, BX
    Int   $10
    mov   VideoSeg, ES
    mov   VideoOfs, DI
    jmp   @Done

    @NoMultiTasker:
    mov   MultiTasking, $00

    @Done:
    {-Get InDos flag }
    mov   AH, $34
    Int   $21
    mov   Word Ptr InDosFlag, BX
    mov   Word Ptr InDosFlag + 2, ES
  End;  { asm }

  Case MultiTasker of
    NoTasker: MultiVersion := DosVersion;
    DesqView: Asm
                Mov AX, 2B01h
                Mov CX, 4445h
                Mov DX, 5351h
                Int 21h
                Mov MultiVersion, BX
                Mov AX, 0DE0Bh
                Mov BX, 0200h
                Int 15h
              End;
    DoubleDOS: MultiVersion := DosVersion;
    Windows: Asm
               Mov AX, 1600h
               Int 2Fh
               Mov MultiVersion, BX
             End;
    OS2:
         Begin
           H := Hi (DosVersion) Div 10;
           MultiVersion := H Shl 8 + Lo (DosVersion);
         End;
    NetWare: MultiVersion := DosVersion;
  End;
End.  { Share }
