{$S-,R-,V-,I-,B-,F+,O+,A+}

{Conditional defines that may affect this unit}
{$I APDEFINE.INC}

{*********************************************************}
{*                    APFOSSIL.PAS 1.12                  *}
{*     Copyright (c) TurboPower Software 1991.           *}
{*                 All rights reserved.                  *}
{*********************************************************}
{Special thanks to Steve Gabrilowitz for his initial work on this unit}

unit ApFossil;
  {-Provides serial I/O using FOSSIL services}

interface

uses
  Dos,
  {$IFDEF DPMI}                                                        {!!.11}
  Dpmi,                                                                {!!.11}
  WinApi,                                                              {!!.11}
  {$ENDIF}                                                             {!!.11}
  {$IFDEF UseOpro}
  OpInline,
  OpRoot,
  {$ENDIF}
  {$IFDEF UseTpro}
  TpInline,
  TpMemChk,
  {$ENDIF}
  ApMisc,
  ApPort,
  ApTimer;

type
  {Holds driver information from FOSSIL GetDriverInfo call}
  DriverInfo = record
    diSize      : Word;
    diSpec      : Byte;
    diRev       : Byte;
    diID        : Pointer;
    diInSize    : Word;
    diInFree    : Word;
    diOutSize   : Word;
    diOutFree   : Word;
    diSWidth    : Byte;
    diSHeight   : Byte;
    diBaudMask  : Byte;
  end;

const
  {---- FOSSIL options ----}
  ptHandleFossilBug = $8000; {Set to handle OutBuffUsed bug in FOSSIL driver}
  ptTrueOutBuffFree = $4000; {Set to return true OutBuffFree value}
  ptPutCharWait     = $2000; {Set to use "wait" transmit}

  {---- Default options ----}
  DefFossilOptionsSimple  = ptHandleFossilBug;
  DefFossilOptions : Word = DefFossilOptionsSimple;

  {Custom constants}
  FossilSignature = $1954;       {Fossil Signature}
  FossilInt       = $14;         {Standard Fossil interrupt}

  ValidLineStatus  : Byte = $03; {Mask for valid line status bits}
  ValidModemStatus : Byte = $80; {Mask for valid modem status bits}

{#Z+}
procedure fInitPort(var P : PortRecPtr; ComName : ComNameType;
                    Baud : LongInt;
                    Parity : ParityType; DataBits : DataBitType;
                    StopBits : StopBitType;
                    InSize, OutSize : Word;
                    Options : Word);
  {-Open fossil port}

procedure fInitPortKeep(var P : PortRecPtr; ComName : ComNameType;
                    InSize, OutSize : Word);
  {-Open fossil port (without changing line params)}

procedure fDonePort(var P : PortRecPtr);
  {-Closes fossil port ComName}

procedure GetDriverVersion(P: PortRecPtr; var name, hiver, lover: byte);
  {-Gets FOSSIL driver version nuumber and attempts to identify driver type.
       Name  FOSSIL Driver Installed
          0  Unrecognized driver
          1  X00
          2  BNU                                                          }

procedure fSetUart(ComName : ComNameType; NewBase : Word;
                   NewIrq, NewVector : Byte);
  {-Dummy routine required by high-level routines}

procedure fSetLine(P : PortRecPtr; Baud : LongInt; Parity : ParityType;
                   DataBits : DataBitType; StopBits : StopBitType);
  {-Sets the fossil and the port record with the new values}

procedure fGetLine(P : PortRecPtr; var Baud : LongInt;
                   var Parity : ParityType;
                   var DataBits : DataBitType;
                   var StopBits : StopBitType;
                   FromHardware : Boolean);
  {-Gets the line params directly from the fossil}

procedure fSetModem(P : PortRecPtr; DTR, RTS : Boolean);
  {-Sets the port record with the new values}

procedure fGetModem(P : PortRecPtr; var DTR, RTS : Boolean);
  {-Gets the DTR,RTS settings directly from the fossil}

procedure fGetChar(P : PortRecPtr; var C : Char);
  {-Returns C (sets error if none available)}

procedure fPeekChar(P : PortRecPtr; var C : Char; PeekAhead : Word);
  {-Looks ahead PeekAhead chars (with 1 being the next character)}

procedure fPutChar(P : PortRecPtr; C : Char);
  {-Adds char to xmit buffer or outputs in directly}

procedure fStartTransmitter(P : PortRecPtr);
  {-Does nothing (but required by some high-level routines)}

function fCharReady(P : PortRecPtr) : Boolean;
  {-Returns True if fossil status call shows a character waiting}

function fTransReady(P : PortRecPtr) : Boolean;
  {-Returns True if fossil status call shows room in output buffer}

procedure fSendBreak(P : PortRecPtr);
  {-Sends a serial line break}

procedure fActivatePort(P : PortRecPtr; Restore : Boolean);
  {-Initializes the fossil port}

procedure fDeactivatePort(P : PortRecPtr; Restore : Boolean);
  {-Deactivates the fossil port}

procedure fSavePort(P : PortRecPtr; var PSR);
  {-Does nothing }

procedure fRestorePort(P : PortRecPtr; var PSR);
  {-Does nothing }

procedure fGotError(P : PortRecPtr; StatusCode : Word);
  {-Called when an error occurs (GotError calls the optional ErrorHandler)}

function fUpdateLineStatus(P : PortRecPtr) : Byte;                     {!!.11}
  {-Returns line status register value}                                {!!.11}

function fUpdateModemStatus(P : PortRecPtr) : Byte;                    {!!.11}
  {-Returns modem status register value}                               {!!.11}

{$IFDEF UseHWFlow}
procedure fHWFlowSet(P : PortRecPtr; Enable : Boolean;                 {!!.11}
                     BufferFull, BufferResume : Word;                  {!!.11}
                     Options : Word);                                  {!!.11}
  {-Enables/disables hardware flow control}                            {!!.11}

function fHWFlowGet(P : PortRecPtr) : FlowState;                       {!!.11}
  {-Returns hardware flow control state}                               {!!.11}
{$ENDIF}

{$IFDEF UseSWFlow}
procedure fSWFlowSet(P : PortRecPtr; Enable : Boolean;                 {!!.11}
                     BufferFull, BufferResume : Word;                  {!!.11}
                     Options : Word);                                  {!!.11}
  {-Enables/disables software flow control}                            {!!.11}

function fSWFlowGet(P : PortRecPtr) : FlowState;                       {!!.11}
  {-Returns software flow control state}                               {!!.11}

procedure fSWFlowCtl(P : PortRecPtr; OnChar, OffChar : Char;           {!!.11}
                     Resume : Boolean);                                {!!.11}
  {-Sets software flow control characters and/or resumes transmits}    {!!.11}
{$ENDIF}

procedure fBufferStatus(P : PortRecPtr;                                {!!.11}
                        var InFree, OutFree, InUsed, OutUsed : Word);  {!!.11}
  {-Returns various buffer values}                                     {!!.11}

procedure fBufferFlush(P : PortRecPtr; FlushIn, FlushOut: Boolean);    {!!.11}
  {-Flushes input/output buffers}                                      {!!.11}

{#Z-}

procedure ActivateApFossil;
  {-Registers this unit as the active "device layer"}

  {=====================================================================}

implementation

var
  Regs : Registers;

{!!.11 new}
procedure FossilIntr(var Regs : Registers);
  {-Normal int in rmode, virtualized int in pmode}
{$IFDEF DPMI}
var
  DRegs : DPMIRegisters;
{$ENDIF}
begin
  {$IFDEF DPMI}
  FillChar(DRegs, SizeOf(DRegs), 0);
  with DRegs do begin
    AX := Regs.AX;
    BX := Regs.BX;
    CX := Regs.CX;
    DX := Regs.DX;
    ES := Regs.ES;
    DI := Regs.DI;
    if SimulateRealModeInt(FossilInt, DRegs) = 0 then ;
    Regs.AX := AX;
  end;
  {$ELSE}
  Intr(FossilInt, Regs);
  {$ENDIF}
end;

procedure fUpdateLineAndModemStatus(P : PortRecPtr);
  {-Update LineStatus and ModemStatus fields from Fossil}
begin
  with P^, Regs do begin
    AH := $03;
    DX := Ord(PortName);
    FossilIntr(Regs);                                                  {!!.11}

    {Refresh status values}
    ModemStatus := AL and ValidModemStatus;                            {!!.12}
    LineStatus := AH and ValidLineStatus;                              {!!.12}
  end;
end;

{!!.11 modified}
procedure fUpdateDriverInfo(P : PortRecPtr; var Info : DriverInfo);
  {-Return current driver information from the fossil driver}

{$IFDEF DPMI}
type
  DosMemRec = record
    Sele, Segm : Word;
  end;
var
  Status : Word;
  M : DosMemRec;
{$ENDIF}

begin
  {$IFDEF DPMI}
  FillChar(Info, SizeOf(Info), 0);
  LongInt(M) := GlobalDosAlloc(SizeOf(Info));
  if LongInt(M) = 0 then
    Exit;
  Regs.ES := M.Segm;
  Regs.DI := 0;
  {$ELSE}
  Regs.ES := Seg(Info);
  Regs.DI := Ofs(Info);
  {$ENDIF}

  with Regs do begin
    AH := $1B;
    CX := SizeOf(Info);
    DX := Ord(P^.PortName);
    FossilIntr(Regs);
    {$IFDEF DPMI}
    Move(Mem[M.Sele:0], Info, SizeOf(Info));
    Status := GlobalDosFree(M.Sele);
    {$ENDIF}
  end;
end;

procedure GetDriverVersion(P: PortRecPtr; var name, hiver, lover: byte);
var
  Info : DriverInfo;
  sc,sd:word;
{$IFDEF DPMI}
type
  DosMemRec = record
    Sele, Segm : Word;
  end;
var
  Status : Word;
  M : DosMemRec;
{$ENDIF}
begin
  {$IFDEF DPMI}
  FillChar(Info, SizeOf(Info), 0);
  LongInt(M) := GlobalDosAlloc(SizeOf(Info));
  if LongInt(M) = 0 then
    Exit;
  Regs.ES := M.Segm;
  Regs.DI := 0;
  {$ELSE}
  Regs.ES := Seg(Info);
  Regs.DI := Ofs(Info);
  {$ENDIF}

  with Regs do begin
    AH := $1B;
    CX := SizeOf(Info);
    DX := Ord(P^.PortName);
    FossilIntr(Regs);
    {$IFDEF DPMI}
    Move(Mem[M.Sele:0], Info, SizeOf(Info));
    Status := GlobalDosFree(M.Sele);
    {$ENDIF}
    sc:=cx;
    sd:=dx;
    if (CX = 12376) and (DX = 8240) Then
      name := 1
    else if (CX = 19) and (DX = 2) Then
      name := 2
    else
      name := 0;
    with info do begin
      hiver := diSpec;
      lover := diRev;
    end;
  end;
end;

procedure fInitPortKeep(var P : PortRecPtr; ComName : ComNameType;
                        InSize, OutSize : Word);
  {-Fossil open port procedure}
var
  x00NewFunc, Found : Boolean;
  I : Byte;
begin
  AsyncStatus := ecOk;

  {For Fossil, ComName must be in Com1..Com4}
  if ComName > Com4 then begin
    AsyncStatus := ecBadPortNumber;
    Exit;
  end;

  {Init Fossil}
  with Regs do begin
    x00NewFunc := True;
    AH :=$1C;
    BX := 0;
    DX := Ord(ComName);
    FossilIntr(Regs);                                                  {!!.11}
    if Regs.AX <> FossilSignature then begin
      x00NewFunc := False;
      AH :=$04;
      BX := 0;
      DX := Ord(ComName);
      FossilIntr(Regs);                                                  {!!.11}
      if Regs.AX <> FossilSignature then begin
        AsyncStatus := ecNoFossil;
        Exit;
      end;
    end;
  end;

  {Allocate Port record}
  if not GetMemCheck(P, SizeOf(PortRec)) then begin
    AsyncStatus := ecOutOfMemory;
    Exit;
  end;

  with P^ do begin
    {Check for an available port slot}
    Found := False;
    I := 1;
    while not Found and (I <= MaxActivePort) do
      if ActiveComPort[I] = nil then begin
        CurrentPort := I;
        ActiveComPort[I] := P;
        Found := True;
      end else
        Inc(I);

    {Can't open port if no slots available}
    if not Found then begin
      FreeMemCheck(P, SizeOf(PortRec));
      AsyncStatus := ecNoMorePorts;
      Exit;
    end;

    {Store the port name}
    PortName := ComName;

    {No control over the modem, set to zero for now}
    ModemControl := 0;

    {No flow control}
    SWFState := 0;                                                     {!!.12}
    SWFGotXoff := False;
    SWFSentXoff := False;
    SWFOnChar := DefaultXonChar;
    SWFOffChar := DefaultXoffChar;
    HWFTransHonor := 0;
    HWFRecHonor := 0;
    HWFRemoteOff := False;
    LastXmitError := 0;

    {Misc other inits}
    Flags := DefPortOptions + DefFossilOptions;
    Buffered := False;
    BreakReceived := False;
    TxReady := True;
    TxInts := False;
    LineStatus := 0;
    DoneProc := fDonePort;
    ErrorProc := NoErrorProc;
    ErrorData := nil;
    UserAbort := NoAbortProc;
    ProtocolActive := False;

    {Zero out buffer stuff (prevents errors if buffer routines are called)}
    InBuff := nil;
    InHead := nil;
    InBuffEnd := nil;
    InBuffLen := 65535;
    InBuffCount := 0;
    OutBuff := nil;
    OutHead := nil;
    OutBuffEnd := nil;
    OutBuffLen := 65535;
    OutBuffCount := 0;
    UseStatusBuffer := False;
    StatBuff := nil;
    StatHead := nil;
    StatTail := nil;

    {Used to identify X00's enhanced FOSSIL features}
    NewFunc := x00NewFunc;

    {Get what line info we can from the FOSSIL driver}
    fGetLine(P, CurBaud, CurParity, CurDataBits, CurStopBits, True);
  end;
end;

procedure fInitPort(var P : PortRecPtr; ComName : ComNameType;
                    Baud : LongInt;
                    Parity : ParityType; DataBits : DataBitType;
                    StopBits : StopBitType;
                    InSize, OutSize : Word;
                    Options : Word);
  {-Fossil open port procedure}
begin
  AsyncStatus := ecOk;

  {Allocate the port record and do inits}
  fInitPortKeep(P, ComName, InSize, OutSize);
  if AsyncStatus <> ecOk then
    Exit;

  with P^ do begin
    {Set the line parameters}
    fSetLine(P, Baud, Parity, DataBits, StopBits);
    if AsyncStatus <> ecOk then begin
      {Failed, release memory and free slot}
      FreeMemCheck(P, SizeOf(PortRec));
      ActiveComPort[CurrentPort] := nil;
    end;

    {Save the desired options}
    Flags := Options;
  end;
end;

procedure fDonePort(var P : PortRecPtr);
  {-Closes ComName}
var
  I : Word;                                                            {!!.12}
begin
  AsyncStatus := ecOk;

  if P = nil then
    Exit;

  I := P^.CurrentPort;                                                 {!!.12}

  {Deinit the fossil}
  If P^.NewFunc Then
    Regs.AH := $1D
  Else
    Regs.AH := $05;
  Regs.DX := Ord(P^.PortName);
  FossilIntr(Regs);                                                    {!!.11}

  {Release the heap space}
  FreeMemCheck(P, SizeOf(PortRec));                                    {!!.12}
  P := nil;

  {!!.12 moved down}
  {Show port slot as now available}
  ActiveComPort[I] := nil;
end;

procedure fSetUart(ComName : ComNameType; NewBase : Word;
                   NewIrq, NewVector : Byte);
  {-Dummy routine required by high-level routines}
begin
  AsyncStatus := epNonFatal+ecNotSupported;
end;

function BaudMask(Baud : LongInt; newfunc: boolean; var Mask : Byte) : Boolean;
  {-Convert Baud to Mask, return False if invalid Baud}
begin
  BaudMask := True;
  if not newfunc then case Baud div 100 of
    3  : Mask := $02;
    6  : Mask := $03;
    12 : Mask := $04;
    24 : Mask := $05;
    48 : Mask := $06;
    96 : Mask := $07;
    192: Mask := $00;
    384: Mask := $01;
    else begin
      Mask := 0;
      BaudMask := False;
    end
  end else case Baud div 10 of
       11: Mask := $00;
       15: Mask := $01;
       30: Mask := $02;
       60: Mask := $03;
      120: Mask := $04;
      240: Mask := $05;
      480: Mask := $06;
      960: Mask := $07;
     1920: Mask := $08;
     2880: Mask := $80;
     3840: Mask := $81;
     5760: Mask := $82;
     7680: Mask := $83;
    11520: Mask := $84;
    else begin
      Mask := 0;
      BaudMask := False;
    end;
  end;
end;

procedure fSetLine(P : PortRecPtr; Baud : LongInt; Parity : ParityType;
                  DataBits : DataBitType; StopBits : StopBitType);
  {-sets the port record with the new values}
var
  BaudCode,
  ParityCode,
  DataCode,
  StopCode : Byte;
  SaveAX : Word;
begin
  AsyncStatus := ecOk;

  with Regs, P^ do begin
    if Baud = 0 then
      {Set mask with known baud}
      if BaudMask(CurBaud, NewFunc, BaudCode) then else
    else
      if not BaudMask(Baud, NewFunc, BaudCode) then begin
        fGotError(P, epFatal+ecInvalidBaudRate);
        Exit;
      end;

    {Set Parity code}
    if not newfunc then case Parity of
      NoParity : ParityCode := 0;
      OddParity : ParityCode := 1;
      EvenParity : ParityCode := 3;
      else begin
        fGotError(P, epFatal+ecInvalidParity);
        Exit;
      end;
    end else case Parity of
      NoParity : ParityCode := 0;
      OddParity : ParityCode := 1;
      EvenParity : ParityCode := 2;
      MarkParity : ParityCode := 3;
      SpaceParity : ParityCode := 4;
      else begin
        fGotError(P, epFatal+ecInvalidParity);
        Exit;
      end;
    end;

    {Set databit and stopbit codes}
    StopCode := StopBits - 1;
    DataCode := DataBits - 5;

    If NewFunc Then Begin
      AH := $1E;
      AL := $00;
      BH := ParityCode;
      BL := StopCode;
      CH := DataCode;
      CL := BaudCode;
    End Else Begin
      AH := $00;
      {Assemble the option byte and try to set the options}
      AL := (BaudCode shl 5) + (ParityCode shl 3) +
            (StopCode shl 2) + DataCode;
    End;
    DX := Ord (PortName) and $07;
    SaveAX := AX;
    FossilIntr(Regs);
                                                                       {!!.11}
    {If AH is unchanged then the port doesn't exist}
    if SaveAX = AX then begin
      fGotError(P, epFatal+ecBadPortNumber);
      Exit;
    end;

    {Save line parameters in CurXxx}
    with P^ do begin
      if Baud <> 0 then
        CurBaud := Baud;
      CurParity := Parity;
      CurDataBits := DataBits;
      CurStopBits := StopBits;
    end;
  end;
end;

procedure fGetLine(P : PortRecPtr; var Baud : LongInt;
                   var Parity : ParityType;
                   var DataBits : DataBitType;
                   var StopBits : StopBitType;
                   FromHardware : Boolean);
  {-Get line parameters from internal record}
var
  Info : DriverInfo;
begin
  with P^ do
    if not FromHardware then begin
      {Return current field values}
      Baud := CurBaud;
      Parity := CurParity;
      DataBits := CurDataBits;
      StopBits := CurStopBits;
    end else begin
      {Get what info we can from the FOSSIL driver}
      fUpdateDriverInfo(P, Info);
      with Info do
        case (diBaudMask shr 5) of
          $02  : Baud := 300;
          $03  : Baud := 600;
          $04  : Baud := 1200;
          $05  : Baud := 2400;
          $06  : Baud := 4800;
          $07  : Baud := 9600;
          $00  : Baud := 19200;
          $01  : Baud := 38400;
        end;
      Parity := NoParity;
      DataBits := 8;
      StopBits := 1;
    end;
end;

procedure fSetModem(P : PortRecPtr; DTR, RTS : Boolean);
  {-Can only set DTR}
begin
  with P^, Regs do begin
    AH := $06;
    AL := Ord(DTR);
    DX := Ord(PortName);
    FossilIntr(Regs);                                                  {!!.11}
  end;
end;

procedure fGetModem(P : PortRecPtr; var DTR, RTS : Boolean);
  {-Does nothing (can't get modem params from FOSSIL)}
begin
  if not p^. newfunc then begin
    fGotError(P, epNonFatal+ecNotSupported);
    DTR := False;
    RTS := False;
  end else with regs, P^ do begin
    AH := $1F;
    AL := 00;
    DX := Ord(PortName);
    FossilIntr (Regs);
    RTS := False;
    DTR := Odd (BL);
  end;
end;

procedure fGetChar(P : PortRecPtr; var C : Char);
  {-Calls FOSSIL to check for and return C}
begin
  AsyncStatus := ecOk;

  with P^, Regs do begin
    {Call status to see if char is ready}
    if fCharReady(P) then begin
      AH := $02;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}
      if (AH and $07) = $07 then begin
        {Timeout waiting for char, report error}
        C := #$FF;
        fGotError(P, epNonFatal+ecTimeout);
      end else begin
        {Get char and error bits}
        Byte(C) := AL;
        LineStatus := AH and ValidLineStatus;                          {!!.12}
        (*if LineStatus <> 0 then                                      {!!.12}
          fGotError(P, epNonFatal+ecUartError);*)                      {!!.12}

        {$IFDEF Tracing}
        if TracingOn then
          AddTraceEntry('R', C);
        {$ENDIF}
      end;
    end else
      fGotError(P, epNonFatal+ecBufferIsEmpty);
  end;
end;

{!!.12 modified}
procedure fPeekChar(P : PortRecPtr; var C : Char; PeekAhead : Word);
begin
  if PeekAhead <> 1 then begin
    C := #$FF;
    fGotError(P, epNonFatal+ecNotBuffered);
  end else with P^, Regs do begin
    AH := $0C;
    DX := Ord(PortName);
    FossilIntr(Regs);
    if AH = $FF then
      fGotError(P, ecBufferIsEmpty)
    else begin
      AsyncStatus := ecOk;                                             {!!.12}
      LineStatus := AH and ValidLineStatus;
      C := Char(AL);
    end;
  end;
end;

procedure fPutChar(P : PortRecPtr; C : Char);
  {-Puts a char to FOSSIL}
begin
  AsyncStatus := ecOk;

  {Call Fossil to send a char}
  with P^, Regs do begin
    AL := Byte(C);

    {Specify wait or nowait as requested}
    if FlagIsSet(Flags, ptPutCharWait) then                            {!!.11}
      AH := $01
    else
      AH := $0B;

    {Call the FOSSIL}
    DX := Ord(PortName);
    FossilIntr(Regs);                                                  {!!.11}

    if not FlagIsSet(Flags, ptPutCharWait) then                        {!!.12}
      {Check for buffer full error}
      if AX = 0 then begin
        fGotError(P, epNonFatal+ecBufferIsFull);
        Exit;
      end;
    (*!!.12
    else begin
      {Check for timeout error}
      if (AH and $07) = $07 then
        fGotError(P, epNonFatal+ecTransmitFailed)
      else
        LineStatus := AH;
    end;*)

    ModemStatus := AL and ValidModemStatus;                            {!!.12}
    LineStatus := AH and ValidLineStatus;                              {!!.12}

    {$IFDEF Tracing}
    if TracingOn then
      AddTraceEntry('T', C);
    {$ENDIF}
  end;
end;

procedure fStartTransmitter(P : PortRecPtr);
  {-Dummy procedure required by high-level routines}
begin
  {nothing to do}
end;

function fCharReady(P : PortRecPtr) : Boolean;
  {-Returns True if FOSSIL status call has DataReady set}
begin
  with P^, Regs do begin
    AH := $03;
    DX := Ord(PortName);
    FossilIntr(Regs);                                                  {!!.11}

    {Refresh status values, set function result}
    ModemStatus := AL and ValidModemStatus;                            {!!.12}
    LineStatus := AH and ValidLineStatus;                              {!!.12}
    fCharReady := Odd(AH);
  end;
end;

function fTransReady(P : PortRecPtr) : Boolean;
  {-Returns True if fossil has room for another character }
begin
  with P^, Regs do begin
    AH := $03;
    DX := Ord(PortName);
    FossilIntr(Regs);                                                  {!!.11}

    {Refresh status values, set function result}
    ModemStatus := AL and ValidModemStatus;                            {!!.12}
    LineStatus := AH and ValidLineStatus;                              {!!.12}
    fTransReady := (AH and $20) = $20
  end;
end;

procedure fSendBreak(P : PortRecPtr);
begin
  with regs, p^ do begin
    AH := $1A;
    DX := Ord (PortName);
    AL := 1;
    FossilIntr (Regs);
    Delay (100);
    AH := $1A;
    DX := Ord (PortName);
    AL := 0;
    FossilIntr (Regs);
  end;
end;

procedure fActivatePort(P : PortRecPtr; Restore : Boolean);
  {-Does nothing -- FOSSIL uses polled I/O}
begin
  {nothing to do}
end;

procedure fDeactivatePort(P : PortRecPtr; Restore : Boolean);
  {-Does nothing -- FOSSIL uses polled I/O}
begin
  {nothing to do}
end;

procedure fSavePort(P : PortRecPtr; var PSR);
  {-Does nothing -- FOSSIL uses polled I/O}
begin
  {nothing to do}
end;

procedure fRestorePort(P : PortRecPtr; var PSR);
  {-Does nothing -- FOSSIL uses polled I/O}
begin
  {nothing to do}
end;

procedure fGotError(P : PortRecPtr; StatusCode : Word);
  {-Called when an error occurs (GotError calls the optional ErrorHandler)}
begin
  AsyncStatus := StatusCode;
  with P^ do
    if @ErrorProc <> @NoErrorProc then begin
      ErrorProc(ErrorData, StatusCode);
      if ProtocolActive then
        {Remove error class on protocol errors}
        AsyncStatus := AsyncStatus mod 10000;
    end;
end;

{!!.11}
function fUpdateLineStatus(P : PortRecPtr) : Byte;
  {-Returns line status register value}
begin
  fUpdateLineAndModemStatus(P);
  fUpdateLineStatus := P^.LineStatus;
end;

{!!.11}
function fUpdateModemStatus(P : PortRecPtr) : Byte;
  {-Returns modem status register value}
begin
  fUpdateLineAndModemStatus(P);
  fUpdateModemStatus := P^.ModemStatus;
end;

{!!.11}
{$IFDEF UseHWFlow}
procedure fHWFlowSet(P : PortRecPtr; Enable : Boolean;
                     BufferFull, BufferResume : Word;
                     Options : Word);
  {-Enables/disables hardware flow control}
begin
  AsyncStatus := ecOk;

  with P^, Regs do begin
    if Enable then begin
      {Turning flow control on...}
      if (Options and (hfUseRTS+hfRequireCTS)) <> (hfUseRTS+hfRequireCTS) then begin
        fGotError(P, ecInvalidArgument);
        Exit;
      end;

      AH := $0F;
      AL := $02;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}

      {Say it's on}
      HWFRecHonor := 1;

    end else begin
      {Turning flow control off...}
      AH := $0F;
      AL := $00;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}

      {Say it's off}
      HWFRecHonor := 0;
    end;
  end;
end;

{!!.11}
function fHWFlowGet(P : PortRecPtr) : FlowState;
  {-Returns hardware flow control state, on or off only}
begin
  with P^ do begin
    if HWFRecHonor = 1 then
      fHWFlowGet := fsClear
    else
      fHWFlowGet := fsOff;
  end;
end;
{$ENDIF}

{!!.11}
{$IFDEF UseSWFlow}
procedure fSWFlowSet(P : PortRecPtr; Enable : Boolean;
                     BufferFull, BufferResume : Word;
                     Options : Word);
  {-Enables/disables software flow control}
begin
  AsyncStatus := ecOk;

  with P^, Regs do begin
    if Enable then begin
      if FlagIsSet(Options, sfTransmitFlow) then                       {!!.12}
        AL := $01                                                      {!!.12}
      else                                                             {!!.12}
        AL := $00;                                                     {!!.12}
      if FlagIsSet(Options, sfReceiveFlow) then                        {!!.12}
        AL := AL or $08;                                               {!!.12}
      AH := $0F;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}

      {Say it's on }
      SWFState := Options;                                             {!!.12}
    end else begin
      AH := $0F;
      AL := $00;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}

      {Say it's off}
      SWFState := 0;                                                   {!!.12}
    end;
  end;
end;

{!!.11}
function fSWFlowGet(P : PortRecPtr) : FlowState;
  {-Returns software flow control state}
begin
  with P^ do begin
    if SWFState <> 0 then                                              {!!.12}
      fSWFlowGet := fsClear
    else
      fSWFlowGet := fsOff;
  end;
end;

{!!.11}
procedure fSWFlowCtl(P : PortRecPtr; OnChar, OffChar : Char;
                     Resume : Boolean);
  {-Sets software flow control characters and/or resumes transmits}
begin
  fGotError(P, epNonFatal+ecNotSupported);
end;
{$ENDIF}

{!!.11}
procedure fBufferStatus(P : PortRecPtr;
                        var InFree, OutFree, InUsed, OutUsed : Word);
  {-Returns various buffer values}
var
  Info : DriverInfo;
  Used : Word;
  Free : Word;
  PercentUsed : Word;
  PercentFree : Word;
begin
  fUpdateDriverInfo(P, Info);
  with P^, Info do begin
    InFree := diInFree;
    if (Flags and ptTrueOutBuffFree) = ptTrueOutBuffFree then begin
      {Return actual value}
      Free := diOutFree;
      if FlagIsSet(Flags, ptHandleFossilBug) and (Free = diOutSize-1) then
        Free := diOutSize;
      OutFree := Free;
    end else begin
      {Make a guess about whether to return 0 or 65535}
      PercentFree := (diOutFree * LongInt(100)) div diOutSize;
      if PercentFree > 90 then
        OutFree := 65535
      else
        OutFree := 0;
    end;

    InUsed := diInSize - diInFree;
    Used := diOutSize - diOutFree;
    if FlagIsSet(Flags, ptHandleFossilBug) and (Used = 1) then
      Used := 0;
    OutUsed := Used;
  end;
end;

{!!.11}
procedure fBufferFlush(P : PortRecPtr; FlushIn, FlushOut: Boolean);
  {-Flushes input/output buffers}
begin
  with P^, Regs do begin

    if FlushIn then begin
      {Flush the input buffer}
      AH := $0A;
      DX := Ord(PortName);
      FossilIntr(Regs);                                                {!!.11}
    end;

    if FlushOut then begin
      {Flush the output buffer}
      AH := $09;
      DX := Ord(PortName);
      FossilIntr(Regs);
    end;
  end;
end;

procedure ActivateApFossil;
  {-Registers this unit as the active "device layer"}
begin
  {$IFNDEF UseOOP}
  InitPort := fInitPort;
  InitPortKeep := fInitPortKeep;
  DonePort := fDonePort;
  SetLine := fSetLine;
  GetLine := fGetLine;
  SetModem := fSetModem;
  GetModem := fGetModem;
  GetChar := fGetChar;
  PutChar := fPutChar;
  CharReady := fCharReady;
  TransReady := fTransReady;
  SendBreak := fSendBreak;
  ActivatePort := fActivatePort;
  DeactivatePort := fDeactivatePort;
  SavePort := fSavePort;
  RestorePort := fRestorePort;
  GotError := fGotError;

  {!!.11 new}
  UpdateLineStatus := fUpdateLineStatus;
  UpdateModemStatus := fUpdateModemStatus;
  {$IFDEF UseHWFlow}
  HWFlowSet := fHWFlowSet;
  HWFlowGet := fHWFlowGet;
  {$ENDIF}
  {$IFDEF UseSWFlow}
  SWFlowSet := fSWFlowSet;
  SWFlowGet := fSWFlowGet;
  SWFlowCtl := fSWFlowCtl;
  {$ENDIF}
  BufferStatus := fBufferStatus;
  BufferFlush := fBufferFlush;

  {$ENDIF}
  SetUart := fSetUart;
end;

begin
  {$IFDEF AutoDeviceInit}
  ActivateApFossil;
  {$ENDIF}

  {Set ANSI output hook to use this device layer}
  AnsiOutput := fPutChar;
end.
