{                         P32 - 32bit Pascal Compiler                        }
{ -------------------------------------------------------------------------- }
{                       Johan Prins - jprins@knoware.nl                      }
{ ========================================================================== }
{ First pass, high-level optimizer                                           }
{                                                   (c) Copyright 1996-1998  }
{                                                                            }
{ -------------------------------------------------------------------------- }

unit P32_opt;

interface

uses P32_err, P32_prep, P32_asm, P32_scan, P32_symb, P32_tree;

procedure OptimizeTree(var p: pTree);
procedure InitOptimizer;

implementation

var
   current_proc : pSymbolRecord;
   realtyp,
   stringtyp,
   booleantyp,
   s32bit_typ   : pTypeRecord;

function CalculateConstants(var p: pTree):boolean;

var sym:pSymbolRecord;

begin
  CalculateConstants:=FALSE;
  if (p^.left^.op=__ordconst) and (p^.right^.op=__ordconst) then
    begin
      p^.return:=NIL;
      if (p^.op=__slash) then
        begin
          p^.d := p^.left^.n / p^.right^.n;
          DestroyTree(p^.left);
          DestroyTree(p^.right);
          p^.op:=__realconst;
          CalculateConstants:=TRUE;
          exit; {not a good solution but it works...}
        end;
      case p^.op of
      __add : p^.n := p^.left^.n + p^.right^.n;
      __sub : p^.n := p^.left^.n - p^.right^.n;
      __mul : p^.n := p^.left^.n * p^.right^.n;
      __div : p^.n := p^.left^.n div p^.right^.n;
      __mod : p^.n := p^.left^.n mod p^.right^.n;
      __shl : p^.n := p^.left^.n shl p^.right^.n;
      __shr : p^.n := p^.left^.n shr p^.right^.n;
      __and : p^.n := p^.left^.n and p^.right^.n;
      __xor : p^.n := p^.left^.n xor p^.right^.n;
      __or  : p^.n := p^.left^.n or p^.right^.n;
      else  begin
              case p^.op of
              __equal      : p^.n := ord(p^.left^.n = p^.right^.n);
              __not_eq     : p^.n := ord(p^.left^.n <> p^.right^.n);
              __greater    : p^.n := ord(p^.left^.n > p^.right^.n);
              __less       : p^.n := ord(p^.left^.n < p^.right^.n);
              __greater_eq : p^.n := ord(p^.left^.n >= p^.right^.n);
              __less_eq    : p^.n := ord(p^.left^.n <= p^.right^.n);
              end;
              p^.return:=booleantyp;
            end;
      end;
      if p^.return=NIL then p^.return:=s32bit_typ;
      p^.op:=__ordconst;
      DestroyTree(p^.left);
      DestroyTree(p^.right);
      CalculateConstants:=TRUE;
    end
    else
     if (p^.left^.op=__realconst) and (p^.right^.op=__realconst) then
    begin
      p^.return:=NIL;
      case p^.op of
      __add :  p^.d := p^.left^.d + p^.right^.d;
      __sub :  p^.d := p^.left^.d - p^.right^.d;
      __mul :  p^.d := p^.left^.d * p^.right^.d;
      __slash: p^.d := p^.left^.d / p^.right^.d;
      else  begin
              case p^.op of
              __equal      : p^.n := ord(p^.left^.d = p^.right^.d);
              __not_eq     : p^.n := ord(p^.left^.d <> p^.right^.d);
              __greater    : p^.n := ord(p^.left^.d > p^.right^.d);
              __less       : p^.n := ord(p^.left^.d < p^.right^.d);
              __greater_eq : p^.n := ord(p^.left^.d >= p^.right^.d);
              __less_eq    : p^.n := ord(p^.left^.d <= p^.right^.d);
              end;
              p^.return:=booleantyp;
            end;
      end;
      p^.op:=__realconst;
      DestroyTree(p^.left);
      DestroyTree(p^.right);
      CalculateConstants:=TRUE;
    end;
  if (p^.left^.op=__realconst) and (p^.right^.op=__ordconst) then
    begin
      p^.return:=NIL;
      case p^.op of
      __add :  p^.d := p^.left^.d + p^.right^.n;
      __sub :  p^.d := p^.left^.d - p^.right^.n;
      __mul :  p^.d := p^.left^.d * p^.right^.n;
      __slash: p^.d := p^.left^.d / p^.right^.n;
      else  begin
              case p^.op of
              __equal      : p^.n := ord(p^.left^.d = p^.right^.n);
              __not_eq     : p^.n := ord(p^.left^.d <> p^.right^.n);
              __greater    : p^.n := ord(p^.left^.d > p^.right^.n);
              __less       : p^.n := ord(p^.left^.d < p^.right^.n);
              __greater_eq : p^.n := ord(p^.left^.d >= p^.right^.n);
              __less_eq    : p^.n := ord(p^.left^.d <= p^.right^.n);
              end;
              p^.return:=booleantyp;
            end;
      end;
      p^.op:=__realconst;
      DestroyTree(p^.left);
      DestroyTree(p^.right);
      CalculateConstants:=TRUE;
    end;
  if (p^.left^.op=__ordconst) and (p^.right^.op=__realconst) then
    begin
      p^.return:=NIL;
      case p^.op of
      __add :  p^.d := p^.left^.n + p^.right^.d;
      __sub :  p^.d := p^.left^.n - p^.right^.d;
      __mul :  p^.d := p^.left^.n * p^.right^.d;
      __slash: p^.d := p^.left^.n / p^.right^.d;
      else  begin
              case p^.op of
              __equal      : p^.n := ord(p^.left^.n = p^.right^.d);
              __not_eq     : p^.n := ord(p^.left^.n <> p^.right^.d);
              __greater    : p^.n := ord(p^.left^.n > p^.right^.d);
              __less       : p^.n := ord(p^.left^.n < p^.right^.d);
              __greater_eq : p^.n := ord(p^.left^.n >= p^.right^.d);
              __less_eq    : p^.n := ord(p^.left^.n <= p^.right^.d);
              end;
              p^.return:=booleantyp;
            end;
      end;
      DestroyTree(p^.left);
      DestroyTree(p^.right);
      p^.op:=__realconst;
      CalculateConstants:=TRUE;
    end;
end;

function IsBoolean(t: TreeType):boolean;
begin
  IsBoolean:=FALSE;
  if (t=__equal) and
     (t=__not_eq) and
     (t=__greater) and
     (t=__less) and
     (t=__greater_eq) and
     (t=__less_eq) then
    IsBoolean:=TRUE;
end;

procedure MaybeTypeConversion(var left, right: pTree);

begin
  if not IsBoolean(left^.op) and
     not IsBoolean(right^.op) and

     (left^.op<>__charconst) and
     (left^.op<>__stringconst) and
     (right^.op<>__charconst) and
     (right^.op<>__stringconst) then
    begin
      if (left^.return^.typedef=_SubRangeDef) and
         (right^.return^.typedef=_SubRangeDef) then
        begin
          if GetSize(left^.return)<4 then left:=GenTypeConvNode(left, s32bit_typ, false);
          if GetSize(right^.return)<4 then right:=GenTypeConvNode(right, s32bit_typ, false);
          left^.line:=left^.left^.line;
          right^.line:=right^.left^.line;
          OptimizeTree(left); {optimize the type conversion}
          OptimizeTree(right); {optimize the type conversion}
        end;
{        else}
          if (left^.return^.typedef<>right^.return^.typedef) or
             (GetSize(left^.return)<>GetSize(right^.return)) then
            begin
              if (left^.return^.typedef=_RealDef) or
                 (left^.op=__realconst) {or
                 {(left^.op=__ordconst) {or
                 (right^.op=__realconst)} then
                begin
                  right:=GenTypeConvNode(right, left^.return, false);
                  right^.line:=right^.left^.line;
                  OptimizeTree(right); {Optimize the conversion}
                end
              else
                begin
                  left:=GenTypeConvNode(left, right^.return, false);
                  left^.line:=left^.left^.line;
                  OptimizeTree(left); {Optimize the conversion}
                end
            end;
          end;
end;

procedure DoNothing(var p : pTree);

begin
end;

procedure DoStrConst(var p: pTree);

begin
  p^.loc.l:=memref;
  p^.return:=stringtyp;
end;

procedure DoOrdConst(var p: pTree);

begin
  p^.loc.l:=ordconst;
  p^.loc.value:=p^.n;
  if p^.return=NIL then p^.return:=s32bit_typ;
end;

procedure DoCharConst(var p: pTree);

begin
  p^.loc.l:=ordconst;
  p^.loc.value:=ord(p^.c);
  if p^.return=NIL then
    begin
      new(p^.return); {new type-record}
      p^.return^.typedef:=_SubRangeDef;
      p^.return^.subrangetyp:=uchar;
    end;
end;

procedure DoSetConst(var p: pTree);
var t: pTree;
begin
  p^.loc.l:=memref;
  t:=p^.left;
  while t<>nil do
    begin
      OptimizeTree(t^.left);
      t:=t^.right;
    end;
  {p^.return is assigned in the parser}
end;

procedure DoRealConst(var p: pTree);


begin
  p^.loc.l:=fpu;
  if p^.return=NIL then p^.return:=realtyp;
end;

procedure DoTypeConv(var p: pTree);

begin
{ OptimizeTree(p^.left); {officially, only needed for typecasted variables}

  p^.loc:=p^.left^.loc; {copy location of variable}
  p^.return:=p^.convtype; {type to convert to}
end;



procedure DoVar(var p : pTree);

begin
  if {(p^.sym^.name='_MEM')  or
     (p^.sym^.name='_MEMW') or
     (p^.sym^.name='_MEML') or}
     (p^.sym^.symbol=_variable) and
     (p^.sym^.absreference) then
    p^.loc.l:=directmem
  else
    if (p^.sym^.name='_PORT')  or
       (p^.sym^.name='_PORTW') or
       (p^.sym^.name='_PORTL') then
      p^.loc.l:=port
    else
      p^.loc.l:=memref; {always in memory}
  p^.return:=p^.sym^.typedef;
end;

procedure DoAddress(var p : pTree);

var tmp, sym:pSymbolRecord;

begin
  if p^.left^.op=__call then p^.left^.op:=__var; {load address instead of calling procedure}
  OptimizeTree(p^.left);
  if p^.return=NIL then
    begin
      sym:=GetPointer(GlobalTable, '_POINTER');
      p^.return:=sym^.typedef;
    end;
  if p^.left^.loc.l<>memref then LineError(p^.line, 'Error while loading address of variable');
end;

procedure DoPtr(var p : pTree);

begin
  OptimizeTree(p^.left);
  p^.loc.l:=memref; {always reference to memory location}
  p^.left^.loc.l:=memref;
  p^.return:=p^.left^.return^.pointerto^.typedef;
end;

procedure DoField(var p : pTree);

begin
  OptimizeTree(p^.left);

  p^.loc:=p^.left^.loc;

  if (p^.left^.loc.l<>memref) and
     (p^.left^.loc.l<>directmem) then LineError(p^.line, 'invalid record'); {should be a memory reference}
  p^.return:=p^.sym_field^.typedef;
  {  p^.return:=p^.left^.return;}
end;

procedure DoIndex(var p : pTree);

begin
  OptimizeTree(p^.left);

  p^.loc:=p^.left^.loc;

  OptimizeTree(p^.right);


  if p^.left^.return^.typedef<>_StringDef then
    p^.right:=GenTypeConvNode(p^.right, s32bit_typ, false);

  p^.right^.line:=p^.right^.left^.line;
  OptimizeTree(p^.right);

{  if (p^.left^.return^.typedef=_ArrayDef) or
     (p^.left^.return^.typedef=_StringDef) then
    begin
      OptimizeTree(p^.right);

      {if p^.left^.return^.typedef=_ArrayDef then}
        p^.return:=p^.left^.return^.definition;
{    end;}
end;


procedure DoAdd(var p: ptree);

var dummy:pointer;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if (p^.left^.op=__charconst) and (p^.right^.op=__stringconst) then
      begin
        p^.op    :=__stringconst;
        p^.s     :=getmemstring(p^.left^.c + p^.right^.s^);
        DestroyTree(p^.left);
        DestroyTree(p^.right);
      end
    else
      if (p^.left^.op=__stringconst) and (p^.right^.op=__charconst) then
        begin
          p^.op    :=__stringconst;
          p^.s     :=getmemstring(p^.left^.s^ + p^.right^.c);
          DestroyTree(p^.left);
          DestroyTree(p^.right);
        end
      else
        if (p^.left^.op=__stringconst) and (p^.right^.op=__stringconst) then
          begin
            p^.op    :=__stringconst;
            p^.s     :=getmemstring(p^.left^.s^ + p^.right^.s^);
            DestroyTree(p^.left);
            DestroyTree(p^.right);
          end
        else
  begin
    if (p^.left^.op=__ordconst) then
      begin {swap left and right tree to allow better code generation}
        dummy:=p^.left;
        p^.left:=p^.right;
        p^.right:=dummy;
      end;
    if (p^.left^.op=__var) and (p^.right^.op=__var) then
      begin
        if (p^.left^.sym<>NIL) and (p^.right^.sym<>NIL) then {skip when there are no symbols}
          if (p^.right^.sym=p^.left^.sym) and
             (p^.right^.sym^.typedef^.typedef=_SubRangeDef) and
             (p^.left^.sym^.typedef^.typedef=_SubRangeDef) and
             (p^.right^.sym^.symbol<>_Func) and
             (p^.left^.sym^.symbol<>_Func) then {don't try this with array, 'coz index-variable can be different}
            begin { X + X = 2 * X (better code-generation) }
              p^.op    :=__mul;
              p^.right^.op:=__ordconst;
              p^.right^.n := 2;
            end;
      end;
        if (p^.left^.return^.typedef=_SubRangeDef) and
           (p^.left^.return^.subrangetyp=uchar) and
           (p^.right^.sym^.typedef^.typedef=_StringDef) then
          begin
            p^.return:=p^.right^.return;
            exit;
          end;
        if (p^.right^.return^.typedef=_SubRangeDef) and
           (p^.right^.return^.subrangetyp=uchar) and
           (p^.left^.sym^.typedef^.typedef=_StringDef) then
          begin
            p^.return:=p^.left^.return;
            exit;
          end;
    MaybeTypeConversion(p^.left, p^.right);
    if p^.return=NIL then
    if (p^.left^.op=__ordconst) or
       (p^.left^.op=__realconst) then p^.return:=p^.right^.return
                                 else p^.return:=p^.left^.return;
    end;
  end;
end;

procedure DoComparison(var p: ptree);

var dummy:pointer;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if (p^.left^.return^.typedef=_SubRangeDef) then
      case p^.left^.return^.subrangetyp of
      s8bit, s16bit, s32bit : p^.signed:=TRUE;
      else                    p^.signed:=FALSE;
      end;
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=booleantyp;
      end;

{    if p^.return=NIL then
    if (p^.left^.op=__ordconst) or
       (p^.left^.op=__realconst) then p^.return:=p^.right^.return
                                 else p^.return:=p^.left^.return;
  end;}
  end;
end;

procedure DoAnd(var p: ptree);

var dummy: ptree;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.left^.op=__ordconst then
      begin {swap left and right tree to allow better code generation}
        dummy:=p^.left;
        p^.left:=p^.right;
        p^.right:=dummy;
      end;
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoOr(var p: ptree);

var dummy: ptree;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.left^.op=__ordconst then
      begin {swap left and right tree to allow better code generation}
        dummy:=p^.left;
        p^.left:=p^.right;
        p^.right:=dummy;
      end;
    MaybeTypeConversion(p^.left, p^.right);
    if p^.return=NIL then p^.return:=p^.left^.return;
  end;
end;

procedure DoXor(var p: ptree);

var dummy: ptree;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.left^.op=__ordconst then
      begin {swap left and right tree to allow better code generation}
        dummy:=p^.left;
        p^.left:=p^.right;
        p^.right:=dummy;
      end;
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoShl(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoShr(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoMul(var p: ptree);

var dummy:pointer;

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.left^.op=__ordconst then
      begin {swap left and right tree to allow fast code generation}
        dummy:=p^.left;
        p^.left:=p^.right;
        p^.right:=dummy;
      end;
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoSlash(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
    begin
      if p^.return=NIL then
        begin
          MaybeTypeConversion(p^.left, p^.right);
          p^.return:=realtyp;
          if p^.left^.return^.typedef<>_RealDef then
            p^.left:=GenTypeConvNode(p^.left, realtyp, false);
          p^.left^.line:=p^.left^.left^.line;
          OptimizeTree(p^.left);
          if p^.right^.return^.typedef<>_RealDef then
            p^.right:=GenTypeConvNode(p^.right, realtyp, false);
          p^.right^.line:=p^.right^.left^.line;
          OptimizeTree(p^.right);
        end;
    end;
end;

procedure DoDiv(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return
      end;
  end;
end;

procedure DoMod(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoSub(var p: ptree);
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if CalculateConstants(p)=FALSE then
  begin
    if p^.return=NIL then
      begin
        MaybeTypeConversion(p^.left, p^.right);
        p^.return:=p^.left^.return;
      end;
  end;
end;

procedure DoMinus(var p: ptree);
begin
  OptimizeTree(p^.left);
  p^.return:=p^.left^.return;
  case p^.left^.op of
  __ordconst:  begin
                 p^.n     := 0 - p^.left^.n;
                 p^.op    := __ordconst;
                 DestroyTree(p^.left)
               end;
  __realconst: begin
                 p^.d     := 0 - p^.left^.d;
                 p^.op    := __realconst;
                 DestroyTree(p^.left)
               end;
  end;
end;

procedure DoNot(var p: ptree);

var oldnode: pTree;

begin
  OptimizeTree(p^.left);

  case p^.left^.op of
  __equal      : begin
                   oldnode:=p^.left;
                   p^.op:=__not_eq;
                   p^.return:=p^.left^.return;
                   p^.left:=p^.left^.left;
                   p^.right:=p^.left^.right;
                   dispose(oldnode);
                 end;
  __not_eq     : begin
                   oldnode:=p^.left;
                   p^.op:=__equal;
                   p^.return:=p^.left^.return;
                   p^.left:=p^.left^.left;
                   p^.right:=p^.left^.right;
                   dispose(oldnode);
                 end;
  __greater    : begin
                   oldnode:=p^.left;
                   p^.op:=__less_eq;
                   p^.return:=oldnode^.return;
                   p^.left:=oldnode^.left;
                   p^.right:=oldnode^.right;
                   dispose(oldnode);
                 end;
  __less       : begin
                   oldnode:=p^.left;
                   p^.op:=__greater_eq;
                   p^.return:=oldnode^.return;
                   p^.left:=oldnode^.left;
                   p^.right:=oldnode^.right;
                   dispose(oldnode);
                 end;
  __greater_eq : begin
                   oldnode:=p^.left;
                   p^.op:=__less;
                   p^.return:=oldnode^.return;
                   p^.left:=oldnode^.left;
                   p^.right:=oldnode^.right;
                   dispose(oldnode);
                 end;
  __less_eq    : begin
                   oldnode:=p^.left;
                   p^.op:=__greater;
                   p^.return:=oldnode^.return;
                   p^.left:=oldnode^.left;
                   p^.right:=oldnode^.right;
                   dispose(oldnode);
                 end;
  end;

  if (p^.left^.return<>NIL) and (p^.left^.return^.typedef=_BooleanDef) then
    begin
      if (p^.left^.op=__ordconst) then
        begin
          p^.op    :=__ordconst;
          p^.n := p^.left^.n xor 1;
          DestroyTree(p^.left);
        end;
    end
  else
    if (p^.left^.op=__ordconst)
      then begin
             p^.op    :=__ordconst;
             p^.n     := not p^.left^.n;
             DestroyTree(p^.left);
          end;
  if p^.return=NIL then p^.return:=p^.left^.return;
end;

procedure DoSubrange(var p : ptree);

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
end;

procedure DoParam(var p : ptree; var sym: pSymbolRecord);

begin
  OptimizeTree(p^.left);
  if p^.right<>NIL then
    if sym=NIL then DoParam(p^.right, sym)
               else begin
                      DoParam(p^.right, sym^.next);
                      OptimizeTree(p^.left);
                    end;
{  else}
  if p^.left^.return=NIL then
    if sym<>NIL then p^.left^.return:=sym^.typedef
                else p^.left^.return:=NIL;
end;

procedure DoInternalProcs(var p:pTree);

begin
  case p^.sym^.internal of
  in_write,
  in_writeln : begin
                 if p^.left<>NIL then DoParam(p^.left, p^.sym^.parameter.first); {there are parameters}
               end;
  in_succ,
  in_pred    : begin
                 OptimizeTree(p^.left^.left); {don't optimize __param but the tree within it}
                 p^.return:=p^.left^.left^.return;
                 if p^.left^.left^.op=__ordconst then
                   begin
                     p^.op:=__ordconst;
                     p^.loc.l:=ordconst;
                     if p^.sym^.internal= in_succ then
                       p^.loc.value:=succ(p^.left^.left^.n)
                     else
                       p^.loc.value:=pred(p^.left^.left^.n);
                     p^.n:=p^.loc.value;
                     DestroyTree(p^.left);
                   end;
               end;
  in_ord     : begin
                 OptimizeTree(p^.left^.left);
                 if p^.left^.left^.op=__ordconst then
                   begin
                     p^.op:=__ordconst;
                     p^.loc.l:=ordconst;
                     p^.loc.value:=p^.left^.left^.n;
                     p^.n:=p^.loc.value;
                     DestroyTree(p^.left);
                   end
                 else
                   begin
                     new(p^.return); {new type-record}
                     p:=GenTypeConvNode(p^.left^.left, s32bit_typ, true);
                     p^.line:=p^.left^.line;
                     OptimizeTree(p); {optimize the type conversion}
                   end;
               end;
  in_chr     : begin
                 OptimizeTree(p^.left^.left);
                 if p^.left^.left^.op=__ordconst then
                   begin
                     p^.op:=__charconst;
                     p^.loc.l:=ordconst;
                     p^.loc.value:=p^.left^.left^.n;
                     p^.n:=p^.loc.value;
                     DestroyTree(p^.left);
                   end
                 else
                   begin
                     new(p^.return); {new type-record}
                     p^.return^.typedef:=_SubRangeDef;
                     p^.return^.subrangetyp:=uchar;
                     p:=GenTypeConvNode(p^.left^.left, p^.return, true);
                     p^.line:=p^.left^.line;
                     OptimizeTree(p); {optimize the type conversion}
                   end;
               end;
  in_sizeof  : begin
                 p^.op:=__ordconst;
                 p^.loc.l:=ordconst;
                 p^.loc.value:=GetSize(p^.left^.left^.return);
                 p^.n:=p^.loc.value;
                 DestroyTree(p^.left);
               end;
  end;
end;

procedure DoCall(var p: pTree);

var sym:pSymbolRecord;

begin
  if p^.sym^.internal<>no then DoInternalProcs(p)
  else
  begin
  if p^.left<>NIL then DoParam(p^.left, p^.sym^.parameter.first); {there are parameters}
  if p^.return=NIL then p^.return:=p^.sym^.ReturnType;
  if p^.sym^.symbol=_func then
    begin
      case p^.return^.typedef of
      _SubrangeDef: begin
                      p^.loc.l:=register;
                      p^.loc.reg:=eax;
                    end;
      _RealDef: p^.loc.l:=fpu;
      else      p^.loc.l:=memref;
      end;
    end;
 end;
end;

procedure DoAssignment(var p : ptree);

begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);

  if p^.left^.op=__call then {assignment to call -> function result}
    begin
      p^.left^.op:=__result;
      p^.left^.return:=p^.left^.sym^.ReturnType;
      p^.left^.loc.l:=memref
    end;

   if (p^.right^.op<>__ordconst) and {should maybe be commented}
      (p^.right^.op<>__charconst) and
      (p^.right^.op<>__stringconst) and
      {(p^.right^.op<>__realconst)  and}
      ((p^.left^.return^.subrangetyp<>p^.right^.return^.subrangetyp) or
      (p^.left^.return^.typedef<>p^.right^.return^.typedef) or {and}
      (GetSize(p^.left^.return)<>GetSize(p^.right^.return))) then
     begin
       p^.right:=GenTypeConvNode(p^.right, p^.left^.return, false);
       p^.right^.line:=p^.right^.left^.line;
       OptimizeTree(p^.right); {Optimize the conversion}
     end;
   p^.return:=p^.left^.return;
end;

procedure DoBlock(var p: pTree);

var
   p1 : ptree;

begin
  p1:=p^.left;
  while p1<>NIL do
  begin
    if (p1^.right<>NIL) then {ignore statement}
      begin
        OptimizeTree(p1^.right);
      end;
    p1:=p1^.left;
  end;
end;

procedure DoFor(var p : pTree);
var tmp:pTree;

begin
  OptimizeTree(p^.left);   {from}
  if (p^.left^.return^.typedef<>_SubRangeDef) and
     (p^.left^.return^.typedef<>_EnumeratedDef) then LineError(p^.line, 'Invalid FOR control variable');

  tmp:=CopyTree(p^.left);  {copy assignment}
  DestroyTree(tmp^.right); {delete the right part ( a := 1 )
                                                        ~ -> this one}
  tmp^.right:=p^.right;    {assign to-value to right part of copied assignment}
  if p^.down_to=TRUE then tmp^.op:=__greater else tmp^.op:=__less;

  p^.right:=tmp;           {tmp-tree contains assignment}

  OptimizeTree(p^.right);  {to}

  OptimizeTree(p^.block);  {block to be executed to - from times}

  if EliminateDeadCode then
    begin
      if (p^.block = NIL) or (p^.block^.left=NIL) then
        begin
          p^.op:=__assign;
          tmp:=p^.left;       {holds old __assign-tree}
          p^.left^:=p^.left^.left^;{contains a __assign, left=var, right=value}
          DestroyTree(tmp);   {remove it}
          {dispose(tmp);}
        end;
    end;
  p^.return:=p^.left^.return;
end;

procedure DoRepeat(var p : ptree);

begin
  OptimizeTree(p^.left);
  if (p^.left^.return^.typedef=_BooleanDef) or
     (p^.left^.op=__ordconst) then
    begin
      OptimizeTree(p^.right);
    end
  else LineError(p^.line, 'Boolean expression expected');
end;

procedure DoWhile(var p : ptree);

begin
  OptimizeTree(p^.left);
  if (p^.left^.return^.typedef=_BooleanDef) or
     (p^.left^.op=__ordconst) then
    begin
      OptimizeTree(p^.right);
    end
  else LineError(p^.line, 'Boolean expression expected');
end;

procedure DoIf(var p : ptree);

var t:ptree;
begin
  OptimizeTree(p^.left);
  if (p^.left^.return^.typedef=_BooleanDef) or
     (p^.left^.op=__ordconst) then
    begin
      OptimizeTree(p^.right);
    end
  else
    LineError(p^.line, 'Boolean expression expected');
  {case p^.left^.op of
  __less, __less_eq,
  __greater, __greater_eq,
  __equal, __not_eq,
  __or, __and, __not      : OptimizeTree(p^.right);
  else                      LineError(p^.line, 'Boolean expression expected');
  end;
  {OptimizeTree(p^.right);}
  if p^.elsetree<>NIL then OptimizeTree(p^.elsetree);
end;

procedure DoCase(var p : ptree);

var t:ptree;
begin
  OptimizeTree(p^.left);
  OptimizeTree(p^.right);
  if p^.elsetree<>NIL then OptimizeTree(p^.elsetree);
end;

procedure DoCaseBlock(var p : ptree);

var t:ptree;
begin
  if p^.left^.op=__caseblock then OptimizeTree(p^.left); {value to evaluate}
  if p^.right<>NIL then OptimizeTree(p^.right);
  if p^.elsetree<>NIL then OptimizeTree(p^.elsetree); {block to execute when true}
end;

procedure OptimizeTree(var p: pTree);


begin
if p<>NIL then
  case p^.op of
  __block      : DoBlock(p);
  __index      : DoIndex(p);
  __field      : DoField(p);
  __assign     : DoAssignment(p);
  __float,
  __subrange   : DoSubRange(p);
  __type       : DoTypeConv(p);
  __ordconst   : DoOrdConst(p);
  __charconst  : DoCharConst(p);
  __stringconst: DoStrConst(p);
  __realconst  : DoRealConst(p);
  __var        : DoVar(p);
  __for        : DoFor(p);
  __if         : DoIf(p);
  __case       : DoCase(p);
  __caseblock  : DoCaseBlock(p);
  __while      : DoWhile(p);
  __repeat     : DoRepeat(p);
  __address    : DoAddress(p);
  __ptr        : DoPtr(p);
  __setconst   : DoSetConst(p);
  __call       : DoCall(p);
  __shl        : DoShl(p);
  __shr        : DoShr(p);
  __and        : DoAnd(p);
  __or         : DoOr(p);
  __xor        : DoXor(p);
  __not        : DoNot(p);
  __add        : DoAdd(p);
  __sub        : DoSub(p);
  __mod        : DoMod(p);
  __slash      : DoSlash(p);
  __div        : DoDiv(p);
  __mul        : DoMul(p);
  __minus      : DoMinus(p);
  __equal,
  __not_eq,
  __greater,
  __less,
  __greater_eq,
  __less_eq    : DoComparison(p);
  __with       : {DoNothing(p)};
  __result     : {DoNothing(p)};
  __break      : {DoNothing(p)};
  __continue   : {DoNothing(p)};
  __exit       : {DoNothing(p)};
  __asm        : {DoNothing(p)};
  __nil        : {DoNothing(p)};
  __empty      : {DoNothing(p)};
  __goto       : {DoNothing(p)};
  __label      : {DoNothing(p)};
  else           begin
                   linecount:=p^.line; {update variable to allow error with (almost) correct linenr.}
                   Abort('Optimizer: '+Numb(Ord(p^.op)));
                 end;
  end;
end;

procedure InitOptimizer;

var s: pSymbolRecord;

begin
  s:=GetPointer(GlobalTable, '_DOUBLE');
  realtyp:=s^.typedef;
  s:=GetPointer(GlobalTable, '_DWORD');
  s32bit_typ:=s^.typedef;
  s:=GetPointer(GlobalTable, '_BOOLEAN');
  booleantyp:=s^.typedef;

  new(stringtyp);
  stringtyp^.typedef:=_StringDef;
end;

begin
end.
