Unit TsrComli;
Interface
Uses DOS;

Const
   Com1 = 0;
   Com2 = 1;
   Com3 = 2;
   Com4 = 3;
   Disablectscheck: Boolean = False; {false If RTS Handshake Is Needed}
   Evenparity:  Boolean = False;   (* Strip Parity? *)
   CtrlKseen:  Boolean = False;   (* Set When ^K Received *)

Procedure INTRinitcom(Chan: Integer);
Procedure INTRsetbaudrate(Speed: Word);

Procedure INTRlowerdtr;
Procedure INTRraisedtr;

Procedure INTRtransmitdata(S:    String);
Procedure INTRflushcom;
Function  INTRreceiveready: Boolean;
Function  INTRreceivedata:  Char;

Procedure INTRuninitcom;


(************** Private *************)

Procedure INTRselectport(Chan: Integer);
Procedure INTRservicetransmit;
Procedure INTRpolltransmit;
Procedure INTRservicereceive;
Procedure INTRcheckinterrupts;

Procedure Controlk;
Procedure Verifytxquespace;

Procedure Cancelxoff;
Procedure Disableint;  Inline($FA);
Procedure Enableint;   Inline($FB);
Procedure Iodelay;     Inline($EB/$00);     {jmp $+2}


Implementation

Const
   Queuesize       = 3000;   {fixed Size Of All Queues}
   Queuehighwater = 2700;   {maximum Queue.Count Before Blocking}
   Queuelowwater  = 2400;   {unblock Queue At This Point}

Type
   Queuerec = Record
      Nextin:  Integer;
      Nextout: Integer;
      Count:    Integer;
      Data:     Array[1..Queuesize] Of Char;
   End;

Const
   Carrierlost = #$E3;         (* Code Returned With Carrier Is Lost *)

   Comchan:     Integer = -1;  (* Currently Selected Com Channel; 0..2 *)
                                (* -1 Indicates Local/No Com Port *)

   Portbase:    Integer = -1;  (* Base Port Number For 8250 Chip *)
                                (* Value = -1 Until Init Is Finished *)

   Portirq:     Integer = -1;  (* Port Irq Number *)

   Oldvector:   Pointer = Nil; (* Pointer To Original Com Interrupt Handler *)
   
   XOFFchar:    Char = ^S;     (* XOFF Character Code *)

Var
   Portintr:    Integer;       (* Interrupt Number For 8250 Chip *)
   Intrmask:    Integer;       (* Interrupt Controller Initialization Code *)

   PrevLCR:     Integer;       (* Previous LCR Contents *)
   PrevIER:     Integer;       (* Previous IER Contents *)
   PrevMCR:     Integer;       (* Previous MCR Contents *)
   PrevICTL:    Integer;       (* Previous ICTL Contents *)

   Xmitactive:  Boolean;       (* Is The Transmitter Active Now?
                                   (Is A THRE Interrupt Expected?) *)

   XOFFactive:  Boolean;       (* Has XOFF Suspended Transmit? *)

   Rxque:        Queuerec;     (* Receive Data Queue *)
   Txque:        Queuerec;     (* Transmit Data Queue *)

   Reg:          Registers;     (* Register Package *)


(*
 * Uart Register Definitions
 *
 *)

Const
   ICTL = $21;                  (* System Interrupt Controller I/O Port *)

   RBR = 0;  (* Receive Buffer Register *)
   THR = 0;  (* Transmit Holding Register *)

   DLM = 1;  (* Divisor Latch MSB *)
   IER = 1;  (* Interrupt Enable Register *)
      IERDAV     = $01;       (* Data Available Interrupt *)
      IERTHRE    = $02;       (* THR Empty Interrupt *)
      IERLSRC    = $04;       (* Line Status Change Interrupt *)
      IERMSR     = $08;       (* Modem Status Interrupt *)


   IIR = 2;  (* Interrupt Identification Register *)
      IIRPENDING = $01;       (* Low When Interrupt Pending *)

      IIRMASK    = $06;       (* Mask For Interrupt Identification *)
        IIRMSR     = $00;       (* Modem Status Change Interrupt *)
        IIRTHRE    = $02;       (* Transmit Holding Reg Empty Interrupt *)
        IIRDAV     = $04;       (* Data Available Interrupt *)
        IIRLSR     = $06;       (* Line Status Change Interrupt *)


   LCR = 3;  (* Line Control Register *)
      LCR5BITS   = $00;       (* 5 Data Bits *)
      LCR7BITS   = $02;       (* 7 Data Bits *)
      LCR8BITS   = $03;       (* 8 Data Bits *)

      LCR1STOP   = $00;       (* 1 Stop Bit *)
      LCR2STOP   = $04;       (* 2 Stop Bits *)

      LCRNPARITY = $00;       (* No Parity *)
      LCREPARITY = $38;       (* Even Parity *)

      LCRNOBREAK = $00;       (* Break Disabled *)
      LCRBREAK   = $40;       (* Break Enabled *)

     {LCRNORMAL  = $00;}      (* Normal *)
      LCRABDL    = $80;       (* Address Baud Divisor Latch *)


   MCR = 4;  (* Modem Control Register *)
      MCRDTR     = $01;       (* Active DTR *)
      MCRRTS     = $02;       (* Active RTS *)
      MCROUT1    = $04;       (* Enable OUT1 *)
      MCROUT2    = $08;       (* Enable OUT2 -- COM INTERRUPT ENABLE *)
      MCRLOOP    = $10;       (* Loopback Mode *)


   LSR = 5;  (* Line Status Register *)
     LSRDAV      = $01;       (* Data Available *)
     LSROERR     = $02;       (* Overrun Error *)
     LSRPERR     = $04;       (* Parity Error *)
     LSRFERR     = $08;       (* Framing Error *)
     LSRBREAK    = $10;       (* Break Received *)
     LSRTHRE     = $20;       (* THR Empty *)
     LSRTSRE     = $40;       (* Transmit Shift Register Empty *)

     LOERRcount:       Integer = 0;    {overrun Error Count}
     LPERRcount:       Integer = 0;    {parity Error Count}
     LFERRcount:       Integer = 0;    {framing Error Count}
     LBREAKcount:      Integer = 0;    {break Received Count}


   MSR = 6;  (* Modem Status Register *)
     MSRDCTS     = $01;       (* Delta CTS *)
     MSRDDSR     = $02;       (* Delta DSR *)
     MSRDRING    = $04;       (* Delta Ring *)
     MSRDRLSD    = $08;       (* Delta Receive Line Signal Detect *)
     MSRCTS      = $10;       (* Clear To Send *)
     MSRDSR      = $20;       (* Data Set Ready *)
     MSRRING     = $40;       (* Ring Detect *)
     MSRRLSD     = $80;       (* Receive Line Signal Detect *)

   {0=Com1, 1=Com2, 2=Com3}
   COMBASETABLE: ARRAY[0..2] OF WORD = ($3F8,$2F8,$3E8);
   COMIRQTABLE:  ARRAY[0..2] OF BYTE = (4, 3, 4);

   IRQMASKTABLE: ARRAY[0..7] OF BYTE = ($01,$02,$04,$08,$10,$20,$40,$80);
   IRQVECTTABLE: ARRAY[0..7] OF BYTE = ($08,$09,$0A,$0B,$0C,$0D,$0E,$0F);


(* ------------------------------------------------------------ *)
Procedure Debugprint(Why,S: String);
Var
   I: Integer;
Const
   Pwhy: String = 'None';
Begin
   If GetEnv('DEBUG') = '' then Exit;

   If Pwhy <> Why then
   Begin
      Writeln;
      Write(Why,': ');
      Pwhy := Why;
   End;

   For I := 1 To Length(S) Do
      Case S[i] Of
      #0..#31:
         Write('^',Chr(Ord(S[i])+Ord('@')));
      Else
         Write(S[i]);
      End;
End;

(* ------------------------------------------------------------ *)
Procedure Giveuptime;
   (* Queue Wait Loop *)
Begin
End;


(* ------------------------------------------------------------ *)
Procedure Controlk;
   (* Process Cancel-Output Command *)
Begin
   Txque.Nextin := 1;
   Txque.Nextout := 1;          (* Throw Away Pending Output *)
   Txque.Count := 0;             
   CtrlKseen := True;
End;


(* ------------------------------------------------------------ *)
Procedure INTRserviceMSR;
  (* Modem Status Change Interrupt *)
Var
   C: Byte;
Begin
   C := Port[ Portbase+MSR ];
   Iodelay;
End;


(* ------------------------------------------------------------ *)
Procedure INTRserviceLSR;
   (* Line Status Change Interrupt *)
Var
   C: Byte;
Begin
   C := Port[ Portbase+LSR ];
   Iodelay;
End;


(* ------------------------------------------------------------ *)
Procedure INTRservicetransmit;
   (* Low-Level Interrupt Service For Transmit, Call Only When Transmit
      Holding Register Is Empty *)
Var
   C:       Char;
Const
   Recur:  Boolean = False;

Begin

(* Prevent Recursion Fb/Bg *)
   If Recur then Exit;
   Recur := True;

(* Drop Out If Transmitter Is Busy *)
   If (Port[ Portbase+LSR ] And LSRTHRE) = 0 then
   Begin
      Iodelay;
      Recur := False;
      Exit;
   End;

   Iodelay;

   (* Stop Transmitting When Queue Is Empty, Or XOFF Is Active
      Or It Is Not CLEAR-To-Send To Modem *)

   Xmitactive := (Txque.Count <> 0) And (Not Xoffactive) And
                  (DisableCTScheck Or ((Port[portbase+MSR] And MSRCTS)>0));

   Iodelay;

   (* Start Next Byte Transmitting *)
   If Xmitactive then
   Begin
      C := Txque.Data[txque.Nextout];
      If Txque.Nextout < Sizeof(Txque.Data) then
         Inc(Txque.Nextout)
      Else
         Txque.Nextout := 1;
      Dec(Txque.Count);

      Port[ Portbase+THR ] := Ord(C); Iodelay;
   End;

   Recur := False;
End;


(* ------------------------------------------------------------ *)
Procedure INTRservicereceive;
   (* Low-Level Interrupt Service For Receive Data,
      Call Only When Receive Data Is Ready *)
Var
   C: Char;
   O: Byte;

Begin
   O := Port[ Portbase+LSR ];
   Iodelay;
   If (O And LSROERR) <> 0 then Inc(LOERRcount);
   If (O And LSRPERR) <> 0 then Inc(LPERRcount);
   If (O And LSRFERR) <> 0 then Inc(LFERRcount);
   If (O And LSRBREAK)<> 0 then Inc(LBREAKcount);
   If (O And LSRDAV) = 0 then Exit;
   C := Chr( Port[ Portbase+RBR ] ); Iodelay;
   If XOFFactive then           (* XOFF Cancelled By Any Character *)
      Cancelxoff
   Else

   If C = XOFFchar then         (* Process XOFF/XON Flow Control *)
      XOFFactive := True
   Else

   If (C = ^K) then              (* Process Cancel-Output Command *)
      Controlk
   Else

   If C = Carrierlost then      (* Ignore This Special Character! *)
   Begin
      {do Nothing}
   End
   Else

   If Rxque.Count < Sizeof(Rxque.Data) then
   Begin
      Inc(Rxque.Count);
      Rxque.Data[rxque.Nextin] := C;
      If Rxque.Nextin < Sizeof(Rxque.Data) then
         Inc(Rxque.Nextin)
      Else
         Rxque.Nextin := 1;
   End;
End;


(* ------------------------------------------------------------ *)
Procedure INTRpolltransmit;
   (* Recover From CTS Or XOF Handshake When Needed *)
Begin
   {no Action If Nothing To Transmit}
   If (Txque.Count = 0) Or (Comchan < 0){local} then
      Exit;

   {check For XON If Output Suspended By XOFF}
   INTRservicereceive;
   INTRservicetransmit;
End;


(* ------------------------------------------------------------ *)
Procedure Cancelxoff;
Begin
   XOFFactive := False;
   INTRpolltransmit;
End;


(* ------------------------------------------------------------ *)
Procedure INTRcheckinterrupts;
   (* Check For And Process Any Pending 8250 Interrupts.
      Can Be Called From TPAS *)
Var
   Status:  Integer;

Begin

(* Get The Interrupt Identification Register *)
   Status := Port[ Portbase+IIR ]; Iodelay;

(* Repeatedly Service Interrupts Until No More Services Possible *)
   While (Status And IIRPENDING) = 0 Do
   Begin
      Disableint;

      Case (Status And IIRMASK) Of
         IIRMSR:   (* Modem Status Change Interrupt *)
            INTRserviceMSR;

         IIRTHRE:  (* Transmit Holding Register Empty Interrupt *)
            INTRservicetransmit;

         IIRDAV:   (* Data Available Interrupt *)
            INTRservicereceive;

         IIRLSR:   (* Line Status Change Interrupt *)
            INTRserviceMSR;
      End;

      Enableint;

  (* Get The Interrupt Identification Register Again *)
      Status := Port[ Portbase+IIR ];
      Iodelay;
   End;

End;


(* ------------------------------------------------------------ *)
Procedure INTRinterrupthandler(Flags,CS,IP,AX,BX,CX,DX,SI,DI,DS,ES,BP: Word);
Interrupt;
   (* Low-Level Interrupt Service Routine.  This Procedure Processes
      All Receive-Ready And Transmit-Ready Interrupts From The 8250 Chip.
      DO NOT Call This Proc From TPAS *)

Begin

(* Service Interrupts Until No More Services Possible *)
   INTRcheckinterrupts;

(* Acknowledge The Interrupt And Return To Foreground Operation *)
   Port[ $20 ] := $20;   {non-Specific EOI} Iodelay;

End;


(* ------------------------------------------------------------ *)
Function INTRreceiveready: Boolean;
   (* See If Any Receive Data Is Ready On The Active Com Port *)
Begin
   INTRpolltransmit;
   INTRreceiveready := Rxque.Count > 0;
End;


(* ------------------------------------------------------------ *)
Procedure INTRflushcom;
   (* Wait For All Pending Transmit Data To Be Sent *)
Begin
   Enableint;
   While Txque.Count > 0 Do
   Begin
      INTRpolltransmit;
      Giveuptime;             (* Give Up Extra Time *)
   End;
End;


(* ------------------------------------------------------------ *)
Procedure Verifytxquespace;
   (* Wait Until There Is Enough Space In The Queue For This Message *)
   (* Or Until Flow Control Is Released *)
Begin
   While Txque.Count > Queuelowwater Do
   Begin
      INTRpolltransmit;
      Giveuptime;             (* Give Up Extra Time *)
   End;
End;


(* ------------------------------------------------------------ *)
Procedure INTRlowerdtr;
   (* Lower DTR To Inhibit Modem Answering *)
Var
   O: Byte;
Begin
   If (Comchan < 0) then Exit;

   O := Port [ Portbase+MCR ];                 Iodelay;
   Port[ Portbase+MCR ] := O And Not MCRDTR;  Iodelay;
End;


(* ------------------------------------------------------------ *)
Procedure INTRraisedtr;
   (* Raise DTR To Allow Modem Answering - Not Supported By BIOS *)
Var
   O: Byte;
Begin
   If (Comchan < 0) then Exit;

   O := Port [ Portbase+MCR ];                       Iodelay;
   Port[ Portbase+MCR ] := O Or (MCRDTR+MCRRTS);   Iodelay;
End;


(* ------------------------------------------------------------ *)
Procedure INTRselectport(Chan: Integer);
   (* Lookup The Port Address For The Specified Com Channel *)
Begin
   Comchan := Chan;
   Xmitactive := False;
   XOFFactive := False;

   If (Chan >= 0) And (Chan <= 2) then
   Begin
      Portbase := COMBASETABLE[chan];
      Portirq := COMIRQTABLE[chan];
      Portintr := IRQVECTTABLE[portirq];
      Intrmask := IRQMASKTABLE[portirq];
   End;

(**
Writeln('[chan=',Chan,' Port Base=',Portbase,' Intr=',Portintr,' Mask=',Intrmask,']');
**)

(* Initialize The Receive And Transmit Queues *)
   Rxque.Nextin := 1;
   Rxque.Nextout := 1;
   Rxque.Count := 0;

   Txque.Nextin := 1;
   Txque.Nextout := 1;
   Txque.Count := 0;

   INTRraisedtr;
End;


(* ------------------------------------------------------------ *)
Procedure INTRinitcom(Chan: Integer);
   (* Initialize Communication Handlers For Operation With The Specified
      Com Port Number.  Must Be Called Before Any Other Services Here *)
Var
   O: Byte;
Begin

(* Initialize Port Numbers, Receive And Transmit Queues *)
   INTRselectport(Chan);

   If Chan < 0 then Exit;

(* Save The Old Interrupt Handler'S Vector *)
   GetIntVec(Portintr, Oldvector);
{writeln('Got Old');}

(* Install A Vector To The New Handler *)
   SetIntVec(Portintr,@INTRinterrupthandler);
{writeln('New Set');}

(* Save Original 8250 Registers *)
   Disableint;
   PrevLCR := Port[ Portbase+LCR ];              Iodelay;
   PrevMCR := Port[ Portbase+MCR ];              Iodelay;
   PrevIER := Port[ Portbase+IER ];              Iodelay;
   PrevICTL  := Port[ ICTL ];                     Iodelay;

(* Clear Divisor Latch If Needed *)
   Port[ Portbase+LCR ] := PrevLCR And Not LCRABDL;
   Iodelay;

(* Initialize The 8250 For Interrupts *)
   O := Port[ Portbase+MCR ];                     Iodelay;
   Port[ Portbase+MCR ] := O Or MCROUT2;         Iodelay;
   Port[ Portbase+IER ] := IERDAV+IERTHRE;      Iodelay;

(* Enable The Interrupt Through The Interrupt Controller *)
   O := Port[ ICTL ];                              Iodelay;
   Port[ ICTL ] := O And (Not Intrmask);          Iodelay;
   Enableint;

(* Initialize The Receive Queues In Case Of An Initial Garbage Byte *)
   Disableint;
   Rxque.Nextin := 1;
   Rxque.Nextout := 1;
   Rxque.Count := 0;
   Enableint;

{writeln('Init Done');}

End;


(* ------------------------------------------------------------ *)
Procedure INTRuninitcom;
   (* Remove Interrupt Handlers For The Com Port
      Must Be Called Before Exit To System *)
Var
   O: Byte;
Begin
   If (Portbase = -1) Or (Oldvector = Nil) then
      Exit;

(* Wait For The Pending Data To Flush From The Queue *)
   INTRflushcom;

(* Attach The Old Handler To The Interrupt Vector *)
   Disableint;

   SetIntVec(Portintr, Oldvector);

   Port[ Portbase+LCR ] := PrevLCR;     Iodelay;
   Port[ Portbase+MCR ] := PrevMCR;     Iodelay;
   Port[ Portbase+IER ] := PrevIER;     Iodelay;
   O := Port[ ICTL ];                     Iodelay;
   Port[ ICTL ] := (O And Not Intrmask) Or (PrevICTL And Intrmask);
   Iodelay;

   Enableint;

(***
Writeln('Prev: LCR=',Itoh(PrevLCR),
             ' MCR=',Itoh(PrevMCR),
             ' IER=',Itoh(PrevIER),
             ' ICTL=',Itoh(PrevICTL));
****)
(***
Writeln(' Now: LCR=',Itoh(Port[ Portbase+LCR ]),
             ' MCR=',Itoh(Port[ Portbase+MCR ]),
             ' IER=',Itoh(Port[ Portbase+IER ]),
             ' ICTL=',Itoh(Port[ ICTL ]));
****)
(***
Writeln('Intrmask=',Itoh(Intrmask),
             ' Vector=',Itoh(Seg(Oldvector)),':',Itoh(Ofs(Oldvector)));
***)

   Oldvector := Nil;
End;


(* ------------------------------------------------------------ *)
Procedure INTRsetbaudrate(Speed: Word);
Var
   Divisor: Word;
   O: Byte;
Begin
   If Comchan < 0 then Exit;
   INTRflushcom;

   Divisor := 115200 Div Speed;
   Disableint;

(* Enable Address Divisor Latch *)
   O := Port[portbase+LCR];              Iodelay;
   Port [portbase+LCR] := O Or LCRABDL; Iodelay;

(* Set The Divisor *)
   Portw[portbase+THR] := Divisor;       Iodelay;

(* Set 8 Bits, 1 Stop, No Parity, No Break, Disable Divisor Latch *)
   PrevLCR := LCR8BITS   Or LCR1STOP   Or
               LCRNPARITY Or LCRNOBREAK;

   Port[ Portbase+LCR ] := PrevLCR;     Iodelay;

   Enableint;

(****
If Setdebug then
Writeln(Dbfd,'Set Baud: LCR=',Itoh(Port[ Portbase+LCR ]),
             ' MCR=',Itoh(Port[ Portbase+MCR ]),
             ' IER=',Itoh(Port[ Portbase+IER ]),
             ' ICTL=',Itoh(Port[ ICTL ]),
             ' Div=',Divisor,
             ' Spd=',Speed);
****)
End;


(* ------------------------------------------------------------ *)
Function INTRreceivedata:  Char;
   (* Wait For And Return 1 Character From The Active Com Port *)
   (* Returns Carrierlost If Carrier Is Not Present *)
Var
   C: Char;

Begin
   If Comchan < 0 then Exit;

   Repeat
      Iodelay;

      If INTRreceiveready then
      Begin
         Disableint;

         {deque From Rxque}
         C := Rxque.Data[rxque.Nextout];
         If Rxque.Nextout < Sizeof(Rxque.Data) then
            Inc(Rxque.Nextout)
         Else
            Rxque.Nextout := 1;
         Dec(Rxque.Count);

         Enableint;

         {strip Parity In 7,E Mode}
         If Evenparity then
            C := Chr( Ord(C) And $7F );

         Debugprint('Recv',C);

         INTRreceivedata := C;
         Exit;
      End;

      {give Up Time While Waiting}
      Giveuptime;

      Iodelay;
   Until Not ((Port[portbase+MSR] And MSRRLSD)<>0);

   {carrier Not Present}
   Cancelxoff;
   INTRreceivedata := Carrierlost;
End;


(* ------------------------------------------------------------ *)
Procedure INTRtransmitdata(S:    String);
   (* Transmits A String Of Characters To The Specified Com Port;
      Does Not Transmit When Carrier Is Not Present *)
Var
   I:    Integer;

Begin
   Debugprint('Xmit',S);

   If Comchan < 0 then Exit;

(* Wait Until There Is Enough Space In The Queue For This Message *)
(* Or Until Flow Control Is Released *)

   If Txque.Count > Queuehighwater then
      Verifytxquespace;


(* Enque The String To Be Transmitted *)
   For I := 1 To Length(S) Do
   Begin
      Disableint;

      Inc(Txque.Count);
      Txque.Data[txque.Nextin] := S[i];
      If Txque.Nextin < Sizeof(Txque.Data) then
         Inc(Txque.Nextin)
      Else
         Txque.Nextin := 1;

      Enableint;
   End;


(* Force An Initial Interrupt To Get Things Rolling (In Case There Are
   No More Pending Transmit-Ready Interrupts *)

   INTRpolltransmit;
End;
End.
