Unit TsrComli;
Interface
uses DOS;

const
   com1 = 0;
   com2 = 1;
   com3 = 2;
   disablectscheck: boolean = false; {false if RTS handshake is needed}
   evenparity:  boolean = false;   (* strip parity? *)
   ctrlKseen:  boolean = false;   (* set when ^K received *)

procedure INTRinitcom(chan: integer);
procedure INTRsetbaudrate(speed: word);

procedure INTRlowerdtr;
procedure INTRraisedtr;

procedure INTRtransmitdata(s:    string);
procedure INTRflushcom;
function  INTRreceiveready: boolean;
function  INTRreceivedata:  char;

procedure INTRuninitcom;


(************** private *************)

procedure INTRselectport(chan: integer);
procedure INTRservicetransmit;
procedure INTRpolltransmit;
procedure INTRservicereceive;
procedure INTRcheckinterrupts;

procedure controlk;
procedure verifytxquespace;

procedure cancelxoff;
procedure disableint;  inline($FA);
procedure enableint;   inline($FB);
procedure iodelay;     inline($EB/$00);     {jmp $+2}


implementation

const
   queuesize       = 3000;   {fixed size of all queues}
   queuehighwater = 2700;   {maximum queue.count before blocking}
   queuelowwater  = 2400;   {unblock queue at this point}

type
   queuerec = record
      nextin:  integer;
      nextout: integer;
      count:    integer;
      data:     array[1..queuesize] of char;
   end;

const
   carrierlost = #$E3;         (* code returned with carrier is lost *)

   comchan:     integer = -1;  (* currently selected com channel; 0..2 *)
                                (* -1 indicates local/no com port *)

   portbase:    integer = -1;  (* base port number for 8250 chip *)
                                (* value = -1 until init is finished *)

   portirq:     integer = -1;  (* port irq number *)

   oldvector:   pointer = nil; (* pointer to original com interrupt handler *)
   
   XOFFchar:    char = ^S;     (* XOFF character code *)

var
   portintr:    integer;       (* interrupt number for 8250 chip *)
   intrmask:    integer;       (* interrupt controller initialization code *)

   prevLCR:     integer;       (* previous LCR contents *)
   prevIER:     integer;       (* previous IER contents *)
   prevMCR:     integer;       (* previous MCR contents *)
   prevICTL:    integer;       (* previous ICTL contents *)

   xmitactive:  boolean;       (* is the transmitter active now?
                                   (is a THRE interrupt expected?) *)

   XOFFactive:  boolean;       (* has XOFF suspended transmit? *)

   rxque:        queuerec;     (* receive data queue *)
   txque:        queuerec;     (* transmit data queue *)

   reg:          registers;     (* register package *)


(*
 * Uart register definitions
 *
 *)

const
   ICTL = $21;                  (* system interrupt controller i/o port *)

   RBR = 0;  (* receive buffer register *)
   THR = 0;  (* transmit holding register *)

   DLM = 1;  (* divisor latch MSB *)
   IER = 1;  (* interrupt enable register *)
      IERDAV     = $01;       (* data available interrupt *)
      IERTHRE    = $02;       (* THR empty interrupt *)
      IERLSRC    = $04;       (* line status change interrupt *)
      IERMSR     = $08;       (* modem status interrupt *)


   IIR = 2;  (* interrupt identification register *)
      IIRPENDING = $01;       (* low when interrupt pending *)

      IIRMASK    = $06;       (* mask for interrupt identification *)
        IIRMSR     = $00;       (* modem status change interrupt *)
        IIRTHRE    = $02;       (* transmit holding reg empty interrupt *)
        IIRDAV     = $04;       (* data available interrupt *)
        IIRLSR     = $06;       (* line status change interrupt *)


   LCR = 3;  (* line control register *)
      LCR5BITS   = $00;       (* 5 data bits *)
      LCR7BITS   = $02;       (* 7 data bits *)
      LCR8BITS   = $03;       (* 8 data bits *)

      LCR1STOP   = $00;       (* 1 stop bit *)
      LCR2STOP   = $04;       (* 2 stop bits *)

      LCRNPARITY = $00;       (* no parity *)
      LCREPARITY = $38;       (* even parity *)

      LCRNOBREAK = $00;       (* break disabled *)
      LCRBREAK   = $40;       (* break enabled *)

     {LCRNORMAL  = $00;}      (* normal *)
      LCRABDL    = $80;       (* address baud divisor latch *)


   MCR = 4;  (* modem control register *)
      MCRDTR     = $01;       (* active DTR *)
      MCRRTS     = $02;       (* active RTS *)
      MCROUT1    = $04;       (* enable OUT1 *)
      MCROUT2    = $08;       (* enable OUT2 -- COM INTERRUPT ENABLE *)
      MCRLOOP    = $10;       (* loopback mode *)


   LSR = 5;  (* line status register *)
     LSRDAV      = $01;       (* data available *)
     LSROERR     = $02;       (* overrun error *)
     LSRPERR     = $04;       (* parity error *)
     LSRFERR     = $08;       (* framing error *)
     LSRBREAK    = $10;       (* break received *)
     LSRTHRE     = $20;       (* THR empty *)
     LSRTSRE     = $40;       (* transmit shift register empty *)

     LOERRcount:       integer = 0;    {overrun error count}
     LPERRcount:       integer = 0;    {parity error count}
     LFERRcount:       integer = 0;    {framing error count}
     LBREAKcount:      integer = 0;    {break received count}


   MSR = 6;  (* modem status register *)
     MSRDCTS     = $01;       (* delta CTS *)
     MSRDDSR     = $02;       (* delta DSR *)
     MSRDRING    = $04;       (* delta ring *)
     MSRDRLSD    = $08;       (* delta receive line signal detect *)
     MSRCTS      = $10;       (* clear to send *)
     MSRDSR      = $20;       (* data set ready *)
     MSRRING     = $40;       (* ring detect *)
     MSRRLSD     = $80;       (* receive line signal detect *)

   {0=com1, 1=com2, 2=com3}
   COMBASETABLE: ARRAY[0..2] OF WORD = ($3F8,$2F8,$3E8);
   COMIRQTABLE:  ARRAY[0..2] OF BYTE = (4, 3, 4);

   IRQMASKTABLE: ARRAY[0..7] OF BYTE = ($01,$02,$04,$08,$10,$20,$40,$80);
   IRQVECTTABLE: ARRAY[0..7] OF BYTE = ($08,$09,$0A,$0B,$0C,$0D,$0E,$0F);


(* ------------------------------------------------------------ *)
procedure debugprint(why,s: string);
var
   i: integer;
const
   pwhy: string = 'none';
begin
   if GetEnv('DEBUG') = '' then exit;

   if pwhy <> why then
   begin
      writeln;
      write(why,': ');
      pwhy := why;
   end;

   for i := 1 to length(s) do
      case s[i] of
      #0..#31:
         write('^',chr(ord(s[i])+ord('@')));
      else
         write(s[i]);
      end;
end;

(* ------------------------------------------------------------ *)
procedure giveuptime;
   (* queue wait loop *)
begin
end;


(* ------------------------------------------------------------ *)
procedure controlk;
   (* process cancel-output command *)
begin
   txque.nextin := 1;
   txque.nextout := 1;          (* throw away pending output *)
   txque.count := 0;             
   ctrlKseen := true;
end;


(* ------------------------------------------------------------ *)
procedure INTRserviceMSR;
  (* modem status change interrupt *)
var
   c: byte;
begin
   c := port[ portbase+MSR ];
   iodelay;
end;


(* ------------------------------------------------------------ *)
procedure INTRserviceLSR;
   (* line status change interrupt *)
var
   c: byte;
begin
   c := port[ portbase+LSR ];
   iodelay;
end;


(* ------------------------------------------------------------ *)
procedure INTRservicetransmit;
   (* low-level interrupt service for transmit, call only when transmit
      holding register is empty *)
var
   c:       char;
const
   recur:  boolean = false;

begin

(* prevent recursion fb/bg *)
   if recur then exit;
   recur := true;

(* drop out if transmitter is busy *)
   if (port[ portbase+LSR ] and LSRTHRE) = 0 then
   begin
      iodelay;
      recur := false;
      exit;
   end;

   iodelay;

   (* stop transmitting when queue is empty, or XOFF is active
      or it is not CLEAR-to-send to modem *)

   xmitactive := (txque.count <> 0) and (not xoffactive) and
                  (disableCTScheck or ((port[portbase+MSR] and MSRCTS)>0));

   iodelay;

   (* start next byte transmitting *)
   if xmitactive then
   begin
      c := txque.data[txque.nextout];
      if txque.nextout < sizeof(txque.data) then
         inc(txque.nextout)
      else
         txque.nextout := 1;
      dec(txque.count);

      port[ portbase+THR ] := ord(c); iodelay;
   end;

   recur := false;
end;


(* ------------------------------------------------------------ *)
procedure INTRservicereceive;
   (* low-level interrupt service for receive data,
      call only when receive data is ready *)
var
   c: char;
   o: byte;

begin
   o := port[ portbase+LSR ];
   iodelay;

(***
   if (o and LSROERR) <> 0 then inc(LOERRcount);
   if (o and LSRPERR) <> 0 then inc(LPERRcount);
   if (o and LSRFERR) <> 0 then inc(LFERRcount);
   if (o and LSRBREAK)<> 0 then inc(LBREAKcount);
***)

   if (o and LSRDAV) = 0 then
      exit;

   c := chr( port[ portbase+RBR ] ); iodelay;

   if XOFFactive then           (* XOFF cancelled by any character *)
      cancelxoff
   else

   if c = XOFFchar then         (* process XOFF/XON flow control *)
      XOFFactive := true
   else

   if (c = ^K) then              (* process cancel-output command *)
      controlk
   else

   if c = carrierlost then      (* ignore this special character! *)
   begin
      {do nothing}
   end
   else

   if rxque.count < sizeof(rxque.data) then
   begin
      inc(rxque.count);
      rxque.data[rxque.nextin] := c;
      if rxque.nextin < sizeof(rxque.data) then
         inc(rxque.nextin)
      else
         rxque.nextin := 1;
   end;
end;


(* ------------------------------------------------------------ *)
procedure INTRpolltransmit;
   (* recover from CTS or XOF handshake when needed *)
begin
   {no action if nothing to transmit}
   if (txque.count = 0) or (comchan < 0){local} then
      exit;

   {check for XON if output suspended by XOFF}
   INTRservicereceive;
   INTRservicetransmit;
end;


(* ------------------------------------------------------------ *)
procedure cancelxoff;
begin
   XOFFactive := false;
   INTRpolltransmit;
end;


(* ------------------------------------------------------------ *)
procedure INTRcheckinterrupts;
   (* check for and process any pending 8250 interrupts.
      can be called from TPAS *)
var
   status:  integer;

begin

(* get the interrupt identification register *)
   status := port[ portbase+IIR ]; iodelay;

(* repeatedly service interrupts until no more services possible *)
   while (status and IIRPENDING) = 0 do
   begin
      disableint;

      case (status and IIRMASK) of
         IIRMSR:   (* modem status change interrupt *)
            INTRserviceMSR;

         IIRTHRE:  (* transmit holding register empty interrupt *)
            INTRservicetransmit;

         IIRDAV:   (* data available interrupt *)
            INTRservicereceive;

         IIRLSR:   (* line status change interrupt *)
            INTRserviceMSR;
      end;

      enableint;

  (* get the interrupt identification register again *)
      status := port[ portbase+IIR ];
      iodelay;
   end;

end;


(* ------------------------------------------------------------ *)
procedure INTRinterrupthandler(Flags,CS,IP,AX,BX,CX,DX,SI,DI,DS,ES,BP: word);
interrupt;
   (* low-level interrupt service routine.  this procedure processes
      all receive-ready and transmit-ready interrupts from the 8250 chip.
      DO NOT call this proc from TPAS *)

begin

(* service interrupts until no more services possible *)
   INTRcheckinterrupts;

(* acknowledge the interrupt and return to foreground operation *)
   port[ $20 ] := $20;   {non-specific EOI} iodelay;

end;


(* ------------------------------------------------------------ *)
function INTRreceiveready: boolean;
   (* see if any receive data is ready on the active com port *)
begin
   INTRpolltransmit;
   INTRreceiveready := rxque.count > 0;
end;


(* ------------------------------------------------------------ *)
procedure INTRflushcom;
   (* wait for all pending transmit data to be sent *)
begin
   enableint;
   while txque.count > 0 do
   begin
      INTRpolltransmit;
      giveuptime;             (* give up extra time *)
   end;
end;


(* ------------------------------------------------------------ *)
procedure verifytxquespace;
   (* wait until there is enough space in the queue for this message *)
   (* or until flow control is released *)
begin
   while txque.count > queuelowwater do
   begin
      INTRpolltransmit;
      giveuptime;             (* give up extra time *)
   end;
end;


(* ------------------------------------------------------------ *)
procedure INTRlowerdtr;
   (* lower DTR to inhibit modem answering *)
var
   o: byte;
begin
   if (comchan < 0) then exit;

   o := port [ portbase+MCR ];                 iodelay;
   port[ portbase+MCR ] := o and not MCRDTR;  iodelay;
end;


(* ------------------------------------------------------------ *)
procedure INTRraisedtr;
   (* raise DTR to allow modem answering - not supported by BIOS *)
var
   o: byte;
begin
   if (comchan < 0) then exit;

   o := port [ portbase+MCR ];                       iodelay;
   port[ portbase+MCR ] := o or (MCRDTR+MCRRTS);   iodelay;
end;


(* ------------------------------------------------------------ *)
procedure INTRselectport(chan: integer);
   (* lookup the port address for the specified com channel *)
begin
   comchan := chan;
   xmitactive := false;
   XOFFactive := false;

   if (chan >= 0) and (chan <= 2) then
   begin
      portbase := COMBASETABLE[chan];
      portirq := COMIRQTABLE[chan];
      portintr := IRQVECTTABLE[portirq];
      intrmask := IRQMASKTABLE[portirq];
   end;

(**
writeln('[chan=',chan,' port base=',portbase,' intr=',portintr,' mask=',intrmask,']');
**)

(* initialize the receive and transmit queues *)
   rxque.nextin := 1;
   rxque.nextout := 1;
   rxque.count := 0;

   txque.nextin := 1;
   txque.nextout := 1;
   txque.count := 0;

   INTRraisedtr;
end;


(* ------------------------------------------------------------ *)
procedure INTRinitcom(chan: integer);
   (* initialize communication handlers for operation with the specified
      com port number.  must be called before any other services here *)
var
   o: byte;
begin

(* initialize port numbers, receive and transmit queues *)
   INTRselectport(chan);

   if chan < 0 then exit;

(* save the old interrupt handler's vector *)
   GetIntVec(portintr, oldvector);
{writeln('got old');}

(* install a vector to the new handler *)
   SetIntVec(portintr,@INTRinterrupthandler);
{writeln('new set');}

(* save original 8250 registers *)
   disableint;
   prevLCR := port[ portbase+LCR ];              iodelay;
   prevMCR := port[ portbase+MCR ];              iodelay;
   prevIER := port[ portbase+IER ];              iodelay;
   prevICTL  := port[ ICTL ];                     iodelay;

(* clear divisor latch if needed *)
   port[ portbase+LCR ] := prevLCR and not LCRABDL;
   iodelay;

(* initialize the 8250 for interrupts *)
   o := port[ portbase+MCR ];                     iodelay;
   port[ portbase+MCR ] := o or MCROUT2;         iodelay;
   port[ portbase+IER ] := IERDAV+IERTHRE;      iodelay;

(* enable the interrupt through the interrupt controller *)
   o := port[ ICTL ];                              iodelay;
   port[ ICTL ] := o and (not intrmask);          iodelay;
   enableint;

(* initialize the receive queues in case of an initial garbage byte *)
   disableint;
   rxque.nextin := 1;
   rxque.nextout := 1;
   rxque.count := 0;
   enableint;

{writeln('init done');}

end;


(* ------------------------------------------------------------ *)
procedure INTRuninitcom;
   (* remove interrupt handlers for the com port
      must be called before exit to system *)
var
   o: byte;
begin
   if (portbase = -1) or (oldvector = nil) then
      exit;

(* wait for the pending data to flush from the queue *)
   INTRflushcom;

(* attach the old handler to the interrupt vector *)
   disableint;

   SetIntVec(portintr, oldvector);

   port[ portbase+LCR ] := prevLCR;     iodelay;
   port[ portbase+MCR ] := prevMCR;     iodelay;
   port[ portbase+IER ] := prevIER;     iodelay;
   o := port[ ICTL ];                     iodelay;
   port[ ICTL ] := (o and not intrmask) or (prevICTL and intrmask);
   iodelay;

   enableint;

(***
writeln('prev: LCR=',itoh(prevLCR),
             ' MCR=',itoh(prevMCR),
             ' IER=',itoh(prevIER),
             ' ICTL=',itoh(prevICTL));
****)
(***
writeln(' now: LCR=',itoh(port[ portbase+LCR ]),
             ' MCR=',itoh(port[ portbase+MCR ]),
             ' IER=',itoh(port[ portbase+IER ]),
             ' ICTL=',itoh(port[ ICTL ]));
****)
(***
writeln('intrmask=',itoh(intrmask),
             ' vector=',itoh(seg(oldvector)),':',itoh(ofs(oldvector)));
***)

   oldvector := nil;
end;


(* ------------------------------------------------------------ *)
procedure INTRsetbaudrate(speed: word);
var
   divisor: word;
   o: byte;
begin
   if comchan < 0 then exit;
   INTRflushcom;

   divisor := 115200 div speed;
   disableint;

(* enable address divisor latch *)
   o := port[portbase+LCR];              iodelay;
   port [portbase+LCR] := o or LCRABDL; iodelay;

(* set the divisor *)
   portw[portbase+THR] := divisor;       iodelay;

(* set 8 bits, 1 stop, no parity, no break, disable divisor latch *)
   prevLCR := LCR8BITS   or LCR1STOP   or
               LCRNPARITY or LCRNOBREAK;

   port[ portbase+LCR ] := prevLCR;     iodelay;

   enableint;

(****
if setdebug then
writeln(dbfd,'set baud: LCR=',itoh(port[ portbase+LCR ]),
             ' MCR=',itoh(port[ portbase+MCR ]),
             ' IER=',itoh(port[ portbase+IER ]),
             ' ICTL=',itoh(port[ ICTL ]),
             ' div=',divisor,
             ' spd=',speed);
****)
end;


(* ------------------------------------------------------------ *)
function INTRreceivedata:  char;
   (* wait for and return 1 character from the active com port *)
   (* returns carrierlost if carrier is not present *)
var
   c: char;

begin
   if comchan < 0 then exit;

   repeat
      iodelay;

      if INTRreceiveready then
      begin
         disableint;

         {deque from rxque}
         c := rxque.data[rxque.nextout];
         if rxque.nextout < sizeof(rxque.data) then
            inc(rxque.nextout)
         else
            rxque.nextout := 1;
         dec(rxque.count);

         enableint;

         {strip parity in 7,E mode}
         if evenparity then
            c := chr( ord(c) and $7f );

         debugprint('recv',c);

         INTRreceivedata := c;
         exit;
      end;

      {give up time while waiting}
      giveuptime;

      iodelay;
   until not ((port[portbase+MSR] and MSRRLSD)<>0);

   {carrier not present}
   cancelxoff;
   INTRreceivedata := carrierlost;
end;


(* ------------------------------------------------------------ *)
procedure INTRtransmitdata(s:    string);
   (* transmits a string of characters to the specified com port;
      does not transmit when carrier is not present *)
var
   i:    integer;

begin
   debugprint('xmit',s);

   if comchan < 0 then exit;

(* wait until there is enough space in the queue for this message *)
(* or until flow control is released *)

   if txque.count > queuehighwater then
      verifytxquespace;


(* enque the string to be transmitted *)
   for i := 1 to length(s) do
   begin
      disableint;

      inc(txque.count);
      txque.data[txque.nextin] := s[i];
      if txque.nextin < sizeof(txque.data) then
         inc(txque.nextin)
      else
         txque.nextin := 1;

      enableint;
   end;


(* force an initial interrupt to get things rolling (in case there are
   no more pending transmit-ready interrupts *)

   INTRpolltransmit;
end;
End.
