{$R-}
{$S-}
{$O-}

Unit SeriellInterface;

Interface
Const
  DSRInput  = $20;
  CTSInput  = $10;
  CDInput   = $80;
  RIInput   = $40;
  DTROutput = $01;
  RTSOutput = $02;
  MaxCom       = 8;      (* Max. sind acht Handler gleichzeitig nutzbar   *)
  NotInstall     = 20000;  (* Der Handler wurde noch nicht installiert      *)
  NoHandler      = 20001;  (* Es ist kein freier Handler mehr vorhanden     *)
  NoChip         = 20002;  (* An der Adresse liegt kein ser. Baustein       *)
  WrongHandler   = 20003;  (* Falsche Handlernummer ( 1 < Com > MaxCom) *)
  WrongBaudRate  = 20100;  (* Ungltige Baudrate                            *)
  WrongStopBit   = 20101;  (* Ungltige Anzahl Stopp-Bits                   *)
  WrongWordLen   = 20102;  (* Ungltige bertragungswort-Lnge              *)
  
Type
  LineZustand  = Boolean;
  ParityType   = (None, Even, Odd, Mark, Space);
  StopBitType  = 1..2;
  WordLenType  = 5..8;
  BaudRateType = 75..115200;
  
  SeriellBuffer = Array [0..$7FFF] Of Char;
  
  SeriellDiscrType = Record
                       PortAdresse,                      (* Basis-Adresse des 8250               *)
                       PortIRQ        : Word;            (* Interrupt-Com der Schnittstelle    *)
                       Transmit       : Boolean;         (* FALSE, wenn Empfangspuffer fast voll *)
                       TransmitMask   : Byte;            (* Maske fr die Statusleitungen        *)
                       BufferSize,                       (* Grsse des Empfangspuffers in Byte   *)
                       BufferFull,                       (* Fll-Grenze fr den Empfangspuffer   *)
                       Top,                              (* erstes Zeichen im Ringpuffers        *)
                       Bottom,                           (* letztes Zeichen im Ringpuffer        *)
                       Anzahl         : Word;            (* Anzahl Zeichen im Ringpuffer         *)
                       Buffer         : ^SeriellBuffer;  (* Pointer auf den Ringpuffer im Heap   *)
                       Install        : Boolean;         (* TRUE, wenn der Handler belegt ist    *)
                       PortInterrupt,                    (* Pointer auf die Interruptroutine     *)
                       OldVector      : POINTER;         (* Ursprnglicher Interrupt-Vektor      *)
                       LineMask,
                       OldIntMask,
                       OldMCR,
                       OldIER         : Byte;
                       CountInt,
                       CountInChar,
                       CountOutChar,
                       CountError,
                       CountOverflow  : Word;
                       NS16550Flag    : Boolean;
                     End;  (* of RECORD *)
  
Var
  SeriellOk    : Boolean;   (* TRUE, wenn kein Fehler erkannt wurde *)
  SeriellError : Word;      (* <> 0, wenn ein Fehler erkannt wurde  *)
  FiFoAktiv    : Boolean;
  SeriellDiscriptor : Array [1..MaxCom] Of SeriellDiscrType;
  
Procedure InstallSeriellHandler (adr, irq, Size : Word ; Var Com : Word);
Procedure DeInstallSeriellHandler (Com : Word);


(* Definition des Handlers <Com> holen.                          *)

Procedure GetHandlerInfo (Com : Word ; Var adr, ir, buflen : Word);


(* Lesen von einer seriellen Schnittstelle.                        *)
(* Die Handlernummer <Com> gibt die Schnittstelle an.            *)

Function  SeriellRead (Com : Word) : Char;


(* Das nchste Zeichen im Buffer holen, aber nicht aus dem Buffer  *)
(* entfernen                                                       *)

Procedure SeriellCheckRead (Com : Word ; Var zeichen : Char ; Var flag : Boolean);
Function SeriellCheck (Com: Word): Boolean;


(* Lesen von einer seriellen Schnittstelle.                        *)
(* Die Handlernummer <Com> gibt die Schnittstelle an.            *)

Procedure SeriellWrite (Com : Word ; zeichen : Char);
Procedure SeriellWriteStr (Com : Word ; S:String);


(* Empfngerpuffer der Schnittstelle <Com> leeren.               *)

Procedure ClearSeriellBuffer (Com : Word);


(* Testen, ob fr die Schnittstelle <Com> ein Zeichen anliegt.   *)

Function  ReceiverReady (Com : Word) : Boolean;


(* Testen, ob die Schnittstelle <Com> ein Zeichen senden kann.   *)

Function  TransmitterReady (Com : Word) : Boolean;


(* Testen, ob CTS-Leitung der Schnittstelle <Com> aktiv ist.     *)

Function  ClearToSend (Com : Word) : Boolean;


(* Testen, ob DSR-Leitung der Schnittstelle <Com> aktiv ist.     *)

Function  DataSetReady (Com : Word) : Boolean;


(* Teste, ob ein Break auf der Leitung erkannt wurde               *)

Function BreakDetected (Com : Word) : Boolean;


(* Testen, ob CD-Leitung der Schnittstelle <Com> aktiv ist.      *)

Function  CarrierDetector (Com : Word) : Boolean;


(* Setzen oder rcksetzen der DTR-Leitung.                         *)

Procedure DataTerminalReady (Com : Word ; zustand : LineZustand);


(* Setzen oder Rcksetzen der RTS-Leitung.                         *)

Procedure RequestToSend (Com : Word ; zustand : LineZustand);


(* Break-Signal ausgeben                                           *)

Procedure SendBreak (Com : Word);


(* Festlegen der Mask fr die Auswertung der Statusleitungen der   *)
(* Schnittstelle.                                                  *)

Procedure SetStatusMask (Com, mask : Word);


(* Festlegen der Mask fr die Behandlung der Statusleitungen der   *)
(* Schnittstelle wenn der Puffer voll ist.                         *)
(* Zum Sperren des Senders werden die angegebenen Ausgnge auf 0   *)
(* gesetzt.                                                        *)

Procedure SetTransmitMask (Com, mask : Word);


(* Testen, ob die Statusleitungen die mit SetStatusMask definiert  *)
(* wurden, gesetzt sind.                                           *)

Function SeriellStatus (Com : Word) : Boolean;


(*******************************************************************)

(* Datenbertragungs-Parameter festlegen.                          *)

Procedure SetParameter (Com   : Word;
                        rate    : BaudRateType;
                        parity  : ParitYType;
                        stopbit : StopBitType;
                        wordlen : WordLenType);


(* Baudrate der Schnittstelle <Com> festlegen.                   *)
(* Fr <baud> sind alle Werte zwischen 75 und 111500 gltig.       *)

Procedure SetBaudrate (Com : Word ; rate : BaudRateType);


(* Aktuelle Baudrate der Schnittstelle <Com> ermitteln           *)

Function  GetBaudrate (Com : Word) : BaudRateType;


(* Parityerzeugung und -Auswertung fr die Schnittstelle <Com>   *)
(* festlegen. Zugelassen sind None,Even oder Odd                   *)

Procedure SetParity (Com : Word ; parity : ParityType);


(* Aktuelle Paritydefinitin der Schnittstelle <Com< ermitteln    *)

Function  GetParity (Com : Word) : ParityType;


(* Anzahl der Stopp-Bit's fr die Schnittstelle <Com> festlegen. *)
(* Zugelassen sind die Werte 1 und 2.                              *)

Procedure SetStopBit (Com : Word ; stopbit : StopBitType);


(* Aktuelle Anzahl Stopp-Bit's fr die Schnittstelle <Com>       *)
(* ermitteln                                                       *)

Function  GetStopBit (Com : Word) : StopBitType;


(* Wort-Lnge fr die Schnittstelle <Com> festlegen.             *)
(* Mgliche Wort-Lngen sind 5,6,7 und 8.                          *)

Procedure SetWordLen (Com : Word ; wordlen : WordLenType);


(* Aktuelle Wort-Lnge der Schnittstelle <Com> ermitteln.        *)

Function  GetWordLen (Com : Word) : WordLenType;


(* Lschen der Schnittstellen-Statistik                            *)

Procedure ClearHandlerStatistic (Com : Word);


(* Zhler fr die Anzahl Interrupts an der Schnittstelle <Com>   *)
(* einfragen.                                                      *)

Function GetIntCounter (Com : Word) : Word;


(* Zhler fr die Anzahl der empfangene Zeichen an der Schnitt-     *)
(* stelle <Com> einfragen.                                        *)

Function GetReceiveCounter (Com : Word) : Word;


(* Zhler fr die Anzahl gesendeten Zeichen an der Schnitt-         *)
(* stelle <Com> einfragen.                                        *)

Function GetSendCounter (Com : Word) : Word;


(* Zhler fr die Anzahl der Empfangsfehler an der Schnitt-         *)
(* stelle <Com> einfragen.                                        *)

Function GetErrorCounter (Com : Word) : Word;


(* Zhler fr die Anzahl der Pufferberlufe an der Schnitt-        *)
(* stelle <Com> einfragen.                                        *)

Function GetOverflowCounter (Com : Word) : Word;


(*************************************************************************)

Implementation

Uses DOS;

Const
  IntrCtrl1      = $20;    (* Basisadresse des ersten Interruptcontroler's  *)
  IntrCtrl2      = $A0;    (* Basisadresse des zweiten Interruptcontroler's *)
  
Var
  i,
  HandlerSize       : Word;      (* Grsses eines Handler-Record's      *)
  altexitproc       : POINTER;   (* Pointer auf die alte Exit-Procedure *)
  Ticker            : LongInt Absolute $40:$6C;
  
  {$L RS232Pas }
  
Procedure SeriellIntrProc1; External;

Procedure SeriellIntrProc2; External;

Procedure SeriellIntrProc3; External;

Procedure SeriellIntrProc4; External;

Procedure SeriellIntrProc5; External;

Procedure SeriellIntrProc6; External;

Procedure SeriellIntrProc7; External;

Procedure SeriellIntrProc8; External;


(*************************************************************************)


Procedure DisableInterrupt; Inline ($FA);

Procedure EnableInterrupt; Inline ($FB);


(*************************************************************************)

Procedure ClearError;

Begin
  SeriellOk := True;
  SeriellError := 0;
End;  (* of ClearError *)


(*************************************************************************)

Procedure SetError (err : Word);

Begin
  SeriellOk := False;
  SeriellError := err;
End;  (* of SetErrror *)


(*************************************************************************)

Procedure InstallSeriellHandler;
Var
  dummy : Byte;
  
  wert  : Word;
Begin
  Com := 1;
  While (SeriellDiscriptor [Com].Install = True) And (Com <= MaxCom) Do Inc (Com);
  If (Com <= MaxCom) Then Begin
    wert := PORT [adr + $06];
    If ( (PORT [adr + $06] And $0F) = 0) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        Transmit := True;
        Top := 0;
        Bottom := 0;
        Anzahl := 0;
        CountInt := 0;
        CountInChar := 0;
        CountOutChar := 0;
        CountError := 0;
        CountOverflow := 0;
        TransmitMask := RTSOutput;
        PortAdresse := adr;
        PortIRQ := irq;
        DisableInterrupt;
        OldIER := PORT [PortAdresse + $01];
        adr := PortAdresse + $04;
        OldMCR := PORT [adr];
        PORT [adr] := OldMCR And $F7;            (* Alle Interrupts mit OUT 2 sperren  *)
        dummy := PORT [PortAdresse + $02];
        If ( (dummy And $C0) > 0) Then
          NS16550Flag := True
        Else Begin
          PORT [PortAdresse + $02] := $01;
          dummy := PORT [PortAdresse + $02];
          NS16550Flag := ( (dummy And $C0) > 0);
        End;  (* of ELSE *)
        
        If NS16550Flag Then Begin
          If FiFoAktiv Then
            PORT [PortAdresse + $02] := $E1
          Else PORT [PortAdresse + $02] := 0;
        End;  (* of IF *)
        
        dummy := PORT [PortAdresse];
        dummy := PORT [PortAdresse + $05];       (* Leitungsstatus-Register lschen    *)
        
        If (PortIRQ <> 0) Then Begin           (* Empfangsintr. nur bei IRQ <> 0 installieren *)
          
          If (Size > $7FFF) Then Size := $7FFF;  (* Buffersize max. $7FFF            *)
          If (MaxAvail < Size) Then            (* wenn zuwenig Platz auf dem Heap, *)
            BufferSize := MaxAvail               (* dann wird der Buffer verkleinert *)
          Else BufferSize := Size;
          
          GetMem (Buffer, BufferSize);          (* Speicher fr den Empfangsbuffer reservieren *)
          
          BufferFull := Word (LongInt (BufferSize) * 90 Div 100);
          If (BufferFull < 10) Then BufferFull := 10;
          
          PORT [PortAdresse + $01] := $01;       (* Interrupt bei Empfang zulassen    *)
          
          adr := PortAdresse + $04;
          wert := PORT [adr];
          PORT [adr] := wert Or TransmitMask Or $08;          (* Die Steuerleitungen setzen        *)
          
          If (PortIRQ < 8) Then Begin                        (* IRQ0 - IRQ7: erster 8259   *)
            GetIntVec ($08 + PortIRQ, OldVector);             (* Interrupt-Vektor retten    *)
            SetIntVec ($08 + PortIRQ, PortInterrupt);         (* und neu setzen             *)
            
            adr := IntrCtrl1 + $01;
            OldIntMask := PORT [adr];
            PORT [adr] := OldIntMask And ($FF XOr 1 ShL PortIRQ);
            OldIntMask := OldIntMask And (1 ShL PortIRQ);
          End  (* of IF THEN *)
          Else Begin                                         (* IRQ8 - IRQ15: zweiter 8259 *)
            GetIntVec ($70 + (PortIRQ - 8), OldVector);       (* Interrupt-Vektor retten    *)
            SetIntVec ($70 + (PortIRQ - 8), PortInterrupt);   (* und neu setzen             *)
            
            adr := IntrCtrl2 + $01;
            OldIntMask := PORT [adr];
            PORT [adr] := OldIntMask And ($FF XOr 1 ShL (PortIRQ - 8) );
            OldIntMask := OldIntMask And (1 ShL (PortIRQ - 8) );
          End;  (* of ELSE *)
        End  (* of IF THEN *)
        Else Begin
          Buffer := Nil;                        (* Ohne Interrupt auch kein Puffer      *)
          OldIntMask := $00;
        End;  (* of ELSE *)
        
        dummy := PORT [PortAdresse];
        dummy := PORT [PortAdresse + $05];       (* Leitungsstatus-Register lschen    *)
        
        EnableInterrupt;
        
        Install := True;                         (* Handler als belegt kennzeichenen     *)
        ClearError;
      End;  (* of WITH *)
    End  (* of IF THEN *)
    Else Begin
      Com := 0;
      SetError (NoChip);
    End;  (* of ELSE *)
  End  (* of IF THEN *)
  Else Begin
    Com := 0;                                  (* Com = 0 wenn kein Handler frei ist *)
    SetError (NoHandler);
  End;  (* of ELSE *)
End;  (* of InstallSeriellHandler *)


(*************************************************************************)

Procedure DeInstallSeriellHandler;

Var
  adr : Word;
  
Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin          (* Nur gltige Handler bearbeiten     *)
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        If (Buffer <> Nil) Then Begin                        (* Wenn ein Empfangspuffer angelegt   *)
          FreeMem (Buffer, BufferSize);                       (* wurde, wird dieser vom Heap        *)
          Buffer := Nil;                                       (* entfernt.                          *)
        End;  (* of IF *)
        
        DisableInterrupt;
        
        PORT [PortAdresse + $01] := OldIER;                       (* alle Interrupts des 8250 sperren   *)
        
        PORT [PortAdresse + $04] := OldMCR;
        
        If (PortIRQ <> 0) Then Begin                         (* Interrupt am 8259 sperren und den  *)
          If (PortIRQ < 8) Then Begin                        (* die Vektor-Adresse restaureien.    *)
            adr := IntrCtrl1 + $01;
            PORT [adr] := PORT [adr] Or OldIntMask;
            SetIntVec ($08 + PortIRQ, OldVector);
          End  (* of IF *)
          Else Begin
            adr := IntrCtrl2 + $01;
            PORT [adr] := PORT [adr] Or OldIntMask;
            SetIntVec ($70 + (PortIRQ - 8), OldVector);
          End;  (* of ELSE *)
        End;  (* of IF *)
        
        EnableInterrupt;
        
        Install := False;                        (* Handler freigeben                  *)
      End  (* of IF *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF *)
  Else SetError (WrongHandler);
End;  (* of DeInstallSeriellHandler *)


(*************************************************************************)

Procedure GetHandlerInfo;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin          (* Nur gltige Handler bearbeiten     *)
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        adr := PortAdresse;
        ir := PortIRQ;
        buflen := BufferSize;
      End  (* of IF *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF *)
  Else SetError (WrongHandler);
End;  (* of GetHandlerInfo *)


(*************************************************************************)

(* Lesen eines Zeichens vom seriellen Com <Com> *)

Function SeriellRead; External;


(*************************************************************************)

(* Lesen eines Zeichens vom seriellen Com <Com> *)
Function SeriellCheck (Com: Word): Boolean;
Begin
  If (Com > 0) And (Com <= MaxCom) Then
  Begin
    If SeriellDiscriptor [Com].Anzahl <> 0 Then SeriellCheck := True Else
      SeriellCheck := False
  End
  Else SetError (WrongHandler);
End;  (* of SeriellCheckRead *)

Procedure SeriellCheckRead;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        If (Anzahl > 0) Then Begin
          zeichen := Buffer^ [Bottom];                (* Zeichen aus dem Puffer holen und     *)
          flag := True;
        End  (* of IF *)
        Else flag := False;
        
        ClearError;
      End  (* of IF THEN *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else SetError (WrongHandler);
End;  (* of SeriellCheckRead *)


(*************************************************************************)

Procedure SeriellWrite; External;

Procedure SeriellWriteStr (Com : Word ; S:String);
Var
  I:Integer;
Begin
  For I:=1 to Length(S) do
    SeriellWrite(Com,S[I]);
End;

(*************************************************************************)

Procedure ClearSeriellBuffer;

  Var
    adr : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          DisableInterrupt;
          
          Anzahl := 0;
          Top := 0;
          Bottom := 0;
          
          If Not (Transmit) Then Begin                       (* Wenn der Puffer fast voll war,       *)
            If (Anzahl < (BufferSize - $10) ) Then Begin      (* teste, ob wieder Platz vorhanden ist *)
              adr := PortAdresse + $04;
              Port [adr] := Port [adr] Or TransmitMask;        (* Wenn ja, Steuerleitungen setzen und  *)
              Transmit := True;                                (* das Flag fr "Puffer voll" lschen.  *)
            End;  (* of IF *)
          End;  (* of IF *)
          
          EnableInterrupt;
          
          ClearError;
        End  (* of IF *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF THEN *)
    Else SetError (WrongHandler);
  End;  (* of ClearSeriellBuffer *)


(*************************************************************************)

Function ReceiverReady; External;


(*************************************************************************)

Function TransmitterReady;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        TransmitterReady := ( (Port [PortAdresse + $05] And $20) > 0);
      End  (* of IF *)
      Else TransmitterReady := False;
    End;  (* of WITH *)
    ClearError;
  End  (* of IF THEN *)
  Else Begin
    TransmitterReady := False;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of TransmitterReady *)


(*************************************************************************)

Function ClearToSend;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        ClearToSend := ( (Port [PortAdresse + $06] And $10) > 0);
      End  (* of IF *)
      Else ClearToSend := False;
    End;  (* of WITH *)
    ClearError;
  End  (* of IF *)
  Else Begin
    ClearToSend := False;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of ClearToSend *)


(*************************************************************************)

Function DataSetReady;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        DataSetReady := ( (Port [PortAdresse + $06] And $20) > 0);
      End  (* of IF *)
      Else DataSetReady := False;
    End;  (* of WITH *)
    ClearError;
  End  (* of IF *)
  Else Begin
    DataSetReady := False;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of DataSetReady *)


(*************************************************************************)

Function CarrierDetector;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        CarrierDetector := ( (Port [PortAdresse + $06] And $80) > 0);
      End  (* of IF *)
      Else CarrierDetector := False;
    End;  (* of WITH *)
    ClearError;
  End  (* of IF *)
  Else Begin
    CarrierDetector := False;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of CarrierDetector *)


(*************************************************************************)

Function BreakDetected;

  Var
    adresse : Word;
    
    Break   : Boolean;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          adresse := PortAdresse + $05;
          Break := ( (Port [adresse] And $08) > 0);
          If Break Then Port [adresse] := Port [adresse] And $F7;
          BreakDetected := Break;
        End  (* of IF *)
        Else BreakDetected := False;
      End;  (* of WITH *)
      ClearError;
    End  (* of IF *)
    Else Begin
      BreakDetected := False;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of BreakDetected *)


(*************************************************************************)

Procedure DataTerminalReady;
Var
  wert,
  adr   : Word;
  
Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        adr := PortAdresse + $04;
        wert := PORT [adr];
        If (zustand = True) Then
          wert := wert Or $01
        Else wert := wert And $FE;
        PORT [adr] := wert;
        ClearError;
      End  (* of IF THEN *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF *)
  Else SetError (WrongHandler);
End;  (* of DataTerminalReady *)


(*************************************************************************)

Procedure RequestToSend;

  Var
    wert,
    adr   : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          adr := PortAdresse + $04;
          wert := PORT [adr];
          If (zustand = True) Then
            wert := wert Or $02
          Else wert := wert And $FD;
          PORT [adr] := wert;
          ClearError;
        End  (* of IF THEN *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF *)
    Else SetError (WrongHandler);
  End;  (* of RequestToSend *)


(*************************************************************************)

Procedure SendBreak;

  Var
    breaktime : LongInt;
    
    teiler,
    adr       : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          adr := PortAdresse + $03;
          DisableInterrupt;
          PORT [adr] := PORT [adr] Or $80;
          teiler := PortW [PortAdresse];
          PORT [adr] := PORT [adr] And $7F;
          EnableInterrupt;
          breaktime := teiler Div 200;
          If (breaktime < 1) Then breaktime := 1;
          breaktime := Ticker + breaktime;
          Port [adr] := Port [adr] Or $40;
          Repeat
          Until (Ticker > breaktime);
          Port [adr] := Port [adr] And $BF;
          ClearError;
        End  (* of IF THEN *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF *)
    Else SetError (WrongHandler);
  End;  (* of SendBreak *)


(*************************************************************************)

Procedure SetStatusMask;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    SeriellDiscriptor [Com].LineMask := (mask Mod $FF);
    ClearError;
  End  (* of IF THEN *)
  Else SetError (WrongHandler);
End;  (* of SetStatusMask *)


(*************************************************************************)

Procedure SetTransmitMask;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    SeriellDiscriptor [Com].TransmitMask := (mask Mod $FF);
    ClearError;
  End  (* of IF THEN *)
  Else SetError (WrongHandler);
End;  (* of SetTransmitMask *)


(*************************************************************************)

Function SeriellStatus;

  Var
    status : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          If ( (Port [PortAdresse + $05] And $20) > 0) Then
            SeriellStatus := ( (Port [PortAdresse + $06] And LineMask) = LineMask)
          Else SeriellStatus := False;
          ClearError;
        End  (* of IF *)
        Else Begin
          SeriellStatus := False;
          SetError (NotInstall);
        End;  (* of ELSE *)
      End;  (* of WITH *)
    End  (* of IF *)
    Else Begin
      SeriellStatus := False;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of SeriellStatus *)


(*************************************************************************)

(* Vor Beendigung des Programmes werden alle noch installierten Handler *)
(* freigegeben.                                                         *)

{$F+}
Procedure SeriellInterfaceExit;
{$F-}

   Var
     adr : Word;
     
   Begin
     For i := 1 To MaxCom Do Begin
       With SeriellDiscriptor [i] Do Begin
         If Install Then Begin
           
           If (Buffer <> Nil) Then Begin                        (* Wenn ein Empfangspuffer angelegt   *)
             FreeMem (Buffer, BufferSize);                       (* wurde, wird dieser vom Heap        *)
             Buffer := Nil;                                       (* entfernt.                          *)
           End;  (* of IF *)
           
           DisableInterrupt;
           
           PORT [PortAdresse + $01] := OldIER;                       (* alle Interrupts des 8250 sperren   *)
           
           PORT [PortAdresse + $04] := OldMCR;
           
           If (PortIRQ <> 0) Then Begin                         (* Interrupt am 8259 sperren und den  *)
             If (PortIRQ < 8) Then Begin                        (* die Vektor-Adresse restaureien.    *)
               adr := IntrCtrl1 + $01;
               PORT [adr] := PORT [adr] Or OldIntMask;
               SetIntVec ($08 + PortIRQ, OldVector);
             End  (* of IF *)
             Else Begin
               adr := IntrCtrl2 + $01;
               PORT [adr] := PORT [adr] Or OldIntMask;
               SetIntVec ($70 + (PortIRQ - 8), OldVector);
             End;  (* of ELSE *)
           End;  (* of IF *)
           
           EnableInterrupt;
           
           Install := False;                        (* Handler freigeben                  *)
         End;  (* of IF *)
       End;  (* of WITH *)
     End;  (* of FOR *)
     
     ExitProc := altexitproc;
   End;  (* of SeriellInterfaceExit *)


(*************************************************************************)

(* Programmieren der seriellen bertragungsparameter. *)

Procedure SetParameter;
Var
  basisadr, wert      : Word;
  
Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        DisableInterrupt;
        basisadr := PortAdresse;
        
        PORT [basisadr + 3] := $80;
        wert := Word (115200 Div rate);
        PORTW [basisadr] := wert;
        
        wert := 0;
        
        Case Parity Of
          Even : wert := wert Or $18;
          Odd : wert := wert Or $08;
          Mark : wert := wert Or $28;
          Space : wert := wert Or $38;
        End;  (* of CASE *)
        
        If (stopbit = 2) Then wert := wert Or $04;
        
        wert := wert + (wordlen - 5);
        
        Port [basisadr + $03] := wert;
        
        wert := Port [basisadr + $05];
        EnableInterrupt;
        ClearError;
      End  (* of IF THEN *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF *)
  Else SetError (WrongHandler);
End;  (* of SetParameter *)


(*************************************************************************)

(* Programmieren der Baudrate <rate> der ser. Schnittstelle an  *)
(* der Basisadresse <basisadr>                                  *)

Procedure SetBaudrate;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          DisableInterrupt;
          basisadr := PortAdresse;
          PORT [basisadr + 3] := PORT [basisadr + 3] Or $80;
          wert := Word (115200 Div rate);
          PORTW [basisadr] := wert;
          PORT [basisadr + 3] := PORT [basisadr + 3] And $7F;
          wert := Port [basisadr + $05];
          ClearError;
          EnableInterrupt;
        End  (* of IF THEN *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF *)
    Else SetError (WrongHandler);
  End;  (* of SetBaudrate *)


(*************************************************************************)

(* Ermitteln der Baudrate der ser. Schnittstelle an *)
(* der Basisdadresse <basisadr>.                    *)

Function GetBaudrate;

  Var
    teiler,
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          DisableInterrupt;
          PORT [basisadr + 3] := PORT [basisadr + 3] Or $80;
          teiler := PORTW [basisadr];
          PORT [basisadr + 3] := PORT [basisadr + 3] And $7F;
          EnableInterrupt;
          If (teiler <> 0) Then
            GetBaudrate := LongInt (115200 Div teiler)
          Else GetBaudrate := 75;
          ClearError;
        End  (* of IF *)
        Else Begin
          GetBaudrate := 75;
          SetError (NotInstall);
        End;  (* of ELSE *)
      End;  (* of WITH *)
    End  (* of IF *)
    Else Begin
      GetBaudrate := 75;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of GetBaudrate *)


(*************************************************************************)

Procedure SetParity;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          DisableInterrupt;
          wert := Port [basisadr + $03];
          
          wert := wert And $C7;
          
          Case Parity Of
            Even : wert := wert Or $18;
            Odd : wert := wert Or $08;
            Mark : wert := wert Or $28;
            Space : wert := wert Or $38;
          End;  (* of CASE *)
          
          Port [basisadr + $03] := wert;
          
          wert := Port [basisadr + $05];
          EnableInterrupt;
        End  (* of IF *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF *)
    Else SetError (WrongHandler);
  End;  (* of SetParity *)


(*************************************************************************)

Function GetParity;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          wert := Port [basisadr + $03] And $38;
          If ( (wert And $08) > 0) Then Begin
            wert := wert ShR 4;
            Case wert Of
              0 : GetParity := Odd;
              1 : GetParity := Even;
              2 : GetParity := Mark;
              3 : GetParity := Space;
            End;  (* of CASE *)
          End  (* of IF THEN *)
          Else GetParity := None;
        End  (* of IF *)
        Else Begin
          GetParity := None;
          SetError (NotInstall);
        End;  (* of ELSE *)
      End;  (* of WITH *)
    End  (* of IF *)
    Else Begin
      GetParity := None;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of GetParity *)


(*************************************************************************)

Procedure SetStopBit;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          DisableInterrupt;
          wert := Port [basisadr + $03];
          If (stopbit = 2) Then
            wert := wert Or $04
          Else wert := wert And $FB;
          Port [basisadr + $03] := wert;
          wert := Port [basisadr + $05];
          EnableInterrupt;
        End  (* of IF THEN *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF THEN *)
    Else SetError (WrongHandler);
  End;  (* of SetStopBit *)

(*************************************************************************)

Function GetStopBit;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          wert := Port [basisadr + $03];
          If ( (wert And $04) > 0) Then
            GetStopBit := 2
          Else GetStopBit := 1;
        End  (* of IF *)
        Else Begin
          GetStopBit := 1;
          SetError (NotInstall);
        End;  (* of ELSE *)
      End;  (* of WITH *)
    End  (* of IF THEN *)
    Else Begin
      GetStopBit := 1;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of GetStopBit *)


(*************************************************************************)

Procedure SetWordLen;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          DisableInterrupt;
          wert := Port [basisadr + $03];
          wert := wert And $FC;
          wert := wert + (wordlen - 5);
          Port [basisadr + $03] := wert;
          wert := Port [basisadr + $05];
          EnableInterrupt;
        End  (* of IF THEN *)
        Else SetError (NotInstall);
      End;  (* of WITH *)
    End  (* of IF *)
    Else SetError (WrongHandler);
  End;  (* of SetWordLen *)

(*************************************************************************)

Function GetWordLen;

  Var
    basisadr,
    wert      : Word;
    
  Begin
    If (Com > 0) And (Com <= MaxCom) Then Begin
      With SeriellDiscriptor [Com] Do Begin
        If Install Then Begin
          basisadr := PortAdresse;
          wert := Port [basisadr + $03];
          GetWordLen := (wert And $03) + 5;
        End  (* of IF THEN *)
        Else Begin
          GetWordLen := 5;
          SetError (NotInstall);
        End;  (* of IF *)
      End;  (* of WITH *)
    End  (* of IF THEN *)
    Else Begin
      GetWordLen := 5;
      SetError (WrongHandler);
    End;  (* of ELSE *)
  End;  (* of GetWordLen *)


(*************************************************************************)

Procedure ClearHandlerStatistic;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        CountInt := 0;
        CountInChar := 0;
        CountOutChar := 0;
        CountError := 0;
        CountOverflow := 0;
        ClearError;
      End  (* of IF THEN *)
      Else SetError (NotInstall);
    End;  (* of WITH *)
  End  (* of IF *)
  Else SetError (WrongHandler);
End;  (* of SetWordLen *)


(*************************************************************************)

Function GetIntCounter;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        GetIntCounter := CountInt;
        ClearError;
      End  (* of IF THEN *)
      Else Begin
        GetIntCounter := 0;
        SetError (NotInstall);
      End;  (* of IF *)
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else Begin
    GetIntCounter := 0;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of GetIntCounter *)


(*************************************************************************)

Function GetReceiveCounter;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        GetReceiveCounter := CountInChar;
        ClearError;
      End  (* of IF THEN *)
      Else Begin
        GetReceiveCounter := 0;
        SetError (NotInstall);
      End;  (* of IF *)
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else Begin
    GetReceiveCounter := 0;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of GetReceiveCounter *)


(*************************************************************************)

Function GetSendCounter;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        GetSendCounter := CountOutChar;
        ClearError;
      End  (* of IF THEN *)
      Else Begin
        GetSendCounter := 0;
        SetError (NotInstall);
      End;  (* of IF *)
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else Begin
    GetSendCounter := 0;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of GetSendCounter *)


(*************************************************************************)

Function GetErrorCounter;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        GetErrorCounter := CountError;
        ClearError;
      End  (* of IF THEN *)
      Else Begin
        GetErrorCounter := 0;
        SetError (NotInstall);
      End;  (* of IF *)
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else Begin
    GetErrorCounter := 0;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of GetErrorCounter *)


(*************************************************************************)

Function GetOverflowCounter;

Begin
  If (Com > 0) And (Com <= MaxCom) Then Begin
    With SeriellDiscriptor [Com] Do Begin
      If Install Then Begin
        GetOverflowCounter := CountOverflow;
        ClearError;
      End  (* of IF THEN *)
      Else Begin
        GetOverflowCounter := 0;
        SetError (NotInstall);
      End;  (* of IF *)
    End;  (* of WITH *)
  End  (* of IF THEN *)
  Else Begin
    GetOverflowCounter := 0;
    SetError (WrongHandler);
  End;  (* of ELSE *)
End;  (* of GetOverflowCounter *)


(*************************************************************************)

Begin
  HandlerSize := SizeOf (SeriellDiscrType);
  
  For i := 1 To MaxCom Do Begin
    With SeriellDiscriptor [i] Do Begin
      Install := False;
      Buffer := Nil;
      OldVector := Nil;
    End;  (* of WITH *)
  End;  (* of FOR *)
  
  SeriellDiscriptor [1].PortInterrupt := @SeriellIntrProc1;
  SeriellDiscriptor [2].PortInterrupt := @SeriellIntrProc2;
  SeriellDiscriptor [3].PortInterrupt := @SeriellIntrProc3;
  SeriellDiscriptor [4].PortInterrupt := @SeriellIntrProc4;
  SeriellDiscriptor [5].PortInterrupt := @SeriellIntrProc5;
  SeriellDiscriptor [6].PortInterrupt := @SeriellIntrProc6;
  SeriellDiscriptor [7].PortInterrupt := @SeriellIntrProc7;
  SeriellDiscriptor [8].PortInterrupt := @SeriellIntrProc8;
  
  altexitproc := ExitProc;
  ExitProc := @SeriellInterfaceExit;
  
  SeriellError := 0;
  SeriellOk := True;
  FiFoAktiv := True;
End.  (* of UNIT SeriellInterface *)