{$R-,V-,S-}
Program DumbTrm;

{--------------------------------------------------------------------------}
{   Program:   DumbTrm                                                     }
{   Author:    Philip R. Burns                                             }
{   Date:      November 11, 1987.                                          }
{              Revised May 23, 1988.                                       }
{   Purpose:   Dumb terminal emulator to demonstrate PibAsync routines.    }
{   Usage:     Compile to .EXE file using Turbo Pascal and execute the     }
{              .EXE file by typing  DUMBTRM  at the DOS prompt.            }
{              You will be prompted for the required communications        }
{              parameters.  If the serial port can be opened successfully, }
{              then a dumb terminal mode is entered.                       }
{              To exit dumb terminal mode, type ^C (ascii 03).             }
{   Needs:     See USES below.                                             }
{   Remarks:   This program doesn't check for input mistakes when the      }
{              communications parameters are being entered.                }
{--------------------------------------------------------------------------}

Uses
  Crt, DOS, GlobType, PibTimer, PibAsync;

Var
  Baud_Rate   : Word              { Baud rate for connection, e.g., 1200  } ;
  Com_Port    : Integer           { Which port, e.g., 1 for COM1:         } ;
  Parity      : Char              { Parity, e.g., E for even parity       } ;
  Data_Bits   : Integer           { How many bits per character, e.g., 8  } ;
  Stop_Bits   : Integer           { How many stop bits -- nearly always 1 } ;
  InBufSize   : Integer           { Size of input buffer                  } ;
  OutBufSize  : Integer           { Size of output buffer                 } ;
  Do_XonXoff  : Char              { 'Y' to do XON/XOFF flow control       } ;
  Do_HardWired: Char              { 'Y' to do XON/XOFF flow control       } ;
  Do_CTS      : Char              { 'Y' to do CTS/RTS flow control        } ;
  Do_DSR      : Char              { 'Y' to do DSR/DTR flow control        } ;
  Com_Addr_Str: String [10]        { Comm port address in hex string       } ;
  Com_Addr    : Integer           { Comm port address                     } ;
  Com_Irq     : Integer           { Comm port IRQ address                 } ;
  Com_Int_No  : Integer           { Comm port interrupt vector number     } ;
  
  {----------------------------------------------------------------------}
  {         Hex_To_Dec --- Convert hex string to decimal number          }
  {----------------------------------------------------------------------}
  
Function Hex_To_Dec ( S : AnyStr; Default : Integer ) : Integer;

{----------------------------------------------------------------------}
{     Function:   Hex_To_Dec                                           }
{     Purpose:    Convert hex string to decimal number                 }
{     Calling Sequence:                                                }
{        Integ := Hex_To_Dec( S: AnyStr; Default: INTEGER ) : INTEGER; }
{           S       --- the hex string                                 }
{           Default --- value to return if S not hex string            }
{           Integ   --- corresponding decimal integer (0 if bad)       }
{----------------------------------------------------------------------}

Var
  I  : Integer;
  Sum: Integer;
  
Begin { Hex_To_Dec }
  Sum        := 0;
  Hex_To_Dec := Default;
  For I := 1 To Length ( S ) Do
    Case S [I] Of
      '0'..'9':  Sum := Sum * 16 + ( Ord (S [I] ) - Ord ('0') );
      'A'..'F':  Sum := Sum * 16 + ( Ord (S [I] ) - Ord ('A') + 10 );
      Else       Exit;
    End;
  Hex_To_Dec := Sum;
End   { Hex_To_Dec } ;

{--------------------------------------------------------------------------}

Procedure Get_Comm_Params;

Var
  YesNo : Char;
  
Begin { Get_Comm_Params }
  { No direct screen writes }
  DirectVideo := True;
  WriteLn;
  Write ('Enter com port (1,2,3,4):                             ');
  ReadLn ( Com_Port );
  If ( Com_Port > 2 ) Then
  Begin
    Write ('Enter com port base address (hex):                    ');
    ReadLn ( Com_Addr_Str );
    Com_Addr := Hex_To_Dec ( Com_Addr_Str , - 1 );
    Write ('Enter com port IRQ level (1 to 15):                   ');
    ReadLn ( Com_Irq );
    Write ('Enter com port interrupt vector number (hex):         ');
    ReadLn ( Com_Addr_Str );
    Com_Int_No := Hex_To_Dec ( Com_Addr_Str , - 1 );
  End;
  Write ('Enter baud rate (1200,2400,9600,19200,38400,57600):   ');
  ReadLn ( Baud_Rate );
  Write ('Enter parity (E=even, N=one, O=odd, M=mark, S=space): ');
  ReadLn ( Parity );
  Parity := UpCase ( Parity );
  Write ('Enter bits per character (7 or 8):                    ');
  ReadLn ( Data_Bits );
  Write ('Enter stop bits (usually 1):                          ');
  ReadLn ( Stop_Bits );
  Write ('Enter size in bytes of async receive buffer:          ');
  ReadLn ( InBufSize );
  Write ('Enter size in bytes of async output buffer:           ');
  ReadLn ( OutBufSize );
  Write ('Use XON/XOFF flow control (Y/N)?                      ');
  ReadLn ( Do_XonXoff );
  Write ('Do CTS/RTS flow control (Y/N)?                        ');
  ReadLn ( Do_CTS );
  Write ('Do DSR/DTR flow control (Y/N)?                        ');
  ReadLn ( Do_DSR );
  Write ('Is connection hard-wired (Y/N)?                       ');
  ReadLn ( Do_HardWired );
End   { Get_Comm_Params } ;

{--------------------------------------------------------------------------}

Function Initialize_Communications : Boolean;

Begin { Initialize_Communications }
  { Set CTS checking }
  Async_Do_CTS         := ( UpCase ( Do_CTS ) = 'Y' );
  { Set DSR checking }
  Async_Do_DSR         := ( UpCase ( Do_DSR ) = 'Y' );
  { Set XON/XOFF to user request }
  Async_Do_XonXoff     := ( UpCase ( Do_XonXoff ) = 'Y' );
  { Set hard-wired as user requests }
  Async_Hard_Wired_On  := ( UpCase ( Do_HardWired ) = 'Y' );
  { Set half-second break duration }
  Async_Break_Length   := 500;
  { Let XON/XOFF break points default. }
  Async_Init ( InBufSize, OutBufSize, 0, 0, 0);
  { If com port 3 or 4, make sure     }
  { port address specified in memory. }
  If ( Com_Port > 2 ) Then
    Async_Setup_Port ( Com_Port, Com_Addr, Com_Irq, Com_Int_No );
  { Try opening the serial port.   }
  If ( Not Async_Open ( Com_Port, Baud_Rate, Parity, Data_Bits, Stop_Bits ) ) Then
  Begin
    WriteLn ('Cannot open serial port.');
    Initialize_Communications := False;
  End
  Else
  Begin
    WriteLn ('Serial port opened, DumbTrm ready.');
    Initialize_Communications := True;
  End;
End   { Initialize_Communications } ;

{--------------------------------------------------------------------------}

Procedure Emulate_Dumb_Terminal;

Var
  Kch   : Char;
  CH    : Char;
  Flags : Byte;
  Do_Async_Status : Boolean;
  Mute_Mode       : Boolean;
  
Const
  Digits : Array [0..15] Of Char = ('0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
  'A', 'B', 'C', 'D', 'E', 'F');
  
Begin { Emulate_Dumb_Terminal }
  { Display <CTRL>key commands    }
  WriteLn ('<CTRL>A toggles line error reporting (now OFF).');
  WriteLn ('<CTRL>B sends a break.');
  WriteLn ('<CTRL>C quits DumbTrm.');
  WriteLn ('<CTRL>E toggles mute mode (now OFF).');
  WriteLn ('<CTRL>N toggles noise rejection mode (now ON).');
  { Line error reporting off      }
  Do_Async_Status := False;
  { Mute mode on to start         }
  Mute_Mode       := False;
  { Begin loop over serial port   }
  { and keyboard.                 }
  Repeat
    { Pick up and display character }
    { from port, if any.            }
    If Async_Receive ( CH ) Then
      Case Ord ( CH ) Of
        0: ;
        7: If ( Not Mute_Mode ) Then Write ( CH )
        Else Write ( '<BEL>' );
        Else Write ( CH );
      End { CASE } ;
    { Read character from keyboard  }
    { and send out port, unless it  }
    { is <CTRL>key command.         }
    If KeyPressed Then
    Begin
      KCh := ReadKey;
      Case KCh Of
        ^A: Begin
          Do_Async_Status := Not Do_Async_Status;
          Case Do_Async_Status Of
            True : 
                   Begin
                     WriteLn ('Line error reporting mode ON');
                   End;
            False: 
                   Begin
                     WriteLn ('Line error reporting mode OFF');
                   End;
          End { CASE } ;
        End;
        ^B: Async_Send_Break;
        ^C: ;
        ^E: 
            Begin
              Mute_Mode := Not Mute_Mode;
              WriteLn;
              Case Mute_Mode Of
                True : 
                       Begin
                         WriteLn ('Mute mode ON');
                       End;
                False: 
                       Begin
                         WriteLn ('Mute mode OFF');
                       End;
              End { CASE } ;
            End;
        ^N: 
            Begin
              Async_Reject_Noise := Not Async_Reject_Noise;
              WriteLn;
              Case Async_Reject_Noise Of
                True : 
                       Begin
                         WriteLn;
                         Write ('New noise character: ');
                         Async_Noise_Char := ReadKey;
                         WriteLn ( Async_Noise_Char );
                         Async_Set_Rejection ( True ,
                         Async_Noise_Char );
                         WriteLn ('Noise rejection ON');
                       End;
                False: 
                       Begin
                         Async_Set_Rejection ( False ,
                         Async_Noise_Char );
                         WriteLn ('Noise rejection OFF');
                       End;
              End { CASE } ;
            End;
        ^Q: 
            Begin
              Async_Send_Now ( CH );
              Async_XOFF_Received := False;
            End;
        Else Async_Send_Now ( Kch );
      End { CASE } ;
    End;
    If Do_Async_Status Then
      If Async_Line_Error ( Flags ) Then
      Begin
        WriteLn;
        WriteLn ( 'Line error = <', Digits [ Flags ShR 4 ],
        Digits [ Flags And $F ], '>' );
      End;
  Until ( Kch = ^C );
End   { Emulate_Dumb_Terminal } ;

{--------------------------------------------------------------------------}

Procedure Finish_Communications;

Begin { Finish_Communications }
  { Close port and drop DTR }
  Async_Close ( False );
  { Release space allocated for buffers }
  Async_Release_Buffers;
End   { Finish_Communications } ;

{--------------------------------------------------------------------------}

Begin { DumbTrm }
  { Request serial port parameters     }
  Get_Comm_Params;
  { Initialize port                    }
  If Initialize_Communications Then
  Begin
    { Emulate dumb terminal until ^C hit }
    Emulate_Dumb_Terminal;
    { Close down port                    }
    Finish_Communications;
  End;
End   { DumbTrm } .