{ Xmodem Checksum/CRC/1K Transfer 1.0

 By Andrew Bartels
 A Product of Digital Innovations

 Placed in the Public Domain, January, 1990, By Andrew Bartels  }

Unit Xmodem;
InterFace
Uses DOS,            { File I/O Routines }
     CRT,            { Video Display Routines / Keyboard }
     IBMCom;         { Use Wayne Conrad's Public Domain COM routines }

Const RX = 1;        { OpCode for Receiving Xmodem }
      SX = 2;        { OpCode for Sending Xmodem   }

      Wt = True;     { Used to determine if OpenFile opens for }
      Rd = False;    { Write (True) or Read (False)            }

Const ACK = #6;      { Acknowledgement     = ASCII Code 6     }
                     { This is sent by receiver to indicate   }
                     { transmitter that the block received    }
                     { had no errors, and that the next block }
                     { can be received.                       }


      CAN = #24;     { Cancel              = ASCII Code 24    }
                     { This signals that the protocol is to   }
                     { be immediately terminated.  Receipt of }
                     { two of these in a row when expecting   }
                     { an SOH, STX, ACK, NAK, or EOT means    }
                     { to abort.                              }

      EOT = #4;      { End Of Transmission = ASCII Code 4    }
                     { Indicates that the tranmitter has     }
                     { successfully transmitted all blocks,  }
                     { and that the protocol is to terminate }

      SOH = #1;      { Start Of Header     = ASCII Code 1     }
                     { This indicates start of a new block    }
                     { of 128 bytes of data.                  }

      STX = #2;      { Start Of Text       = ASCII Code 2     }
                     { This code is used exactly as the SOH   }
                     { by the transmitter to signal block     }
                     { beginning.  However, its use indicates }
                     { the block will have 1K of data         }

      NAK = #21;     { Negative Acknowledgement = ASCII Code  21  }
                     { This is sent to receiver to indicate that  }
                     { the last block failed it's error check, or }
                     { is sent to transmitter to indicate sum    }
                     { error checking is to be used.              }


{ This code has its own built-in error trapping code, designed to report  }
{ errors to the user as accurately as possible.  Each error condition has }
{ a number associated with it, as described below.  Upon receipt of an    }
{ error, the proper error description is printed via the InterpretCode    }
{ procedure, and the code itself becomes the exit code as the program     }
{ terminates.  Error code of zero means no error.                         }

{ Exit Codes To Define Error Conditions Are As Follows:                   }


      NoError                 = 0;  { No error happened - all is well  }
      FileError               = 1;  { Can't open a file                }
      OperatorAbortError      = 2;  { Operator aborted transfer        }
      RemoteCancelError       = 3;  { Remote side aborted transfer     }
      MaxTimeOutError         = 4;  { TimeOut Limit exceeded           }
      BlockOutOfSequenceError = 5;  { Block received out of seq.       }
      EOTTimeOut              = 6;  { Timeout on receiving second EOT. }
      CarrierLostError        = 7;  { Can't xfer if we don't have one  }
{      .
       .
       .
       .                                       }
      InvalidOperation        = 252; { An invalid operation was used      }
      InvalidSwitch           = 253; { An invalid command switch was used }
      InvalidBaudRate         = 254; { An invalid baud rate was specified }
      InvalidCOMPort          = 255; { An Invalid COM Port was specified  }


Type BlockType = Array[1..1024] of Byte;  { Use this type for all blocks }


{ Variables used in opening command line interpretation routines:         }


Var Count         : Integer;      { pointer to next ParamStr item         }
    StrX:String;
    OneK,                         { True if 1K blocks are to be used      }

    CRCMode       : Boolean;      { True if CRC correction is to be used, }
                                  { False if Checksum is to be used       }

    FileName      : String;       { Holds path/filename specified on      }
                                  { command line.                         }

    OpCode        : Byte;         { Determines which transfer operation   }
                                  { was specified.                        }

    ComPort,                      { Holds COM Port specified, or 1 as a   }
                                  { default if none was specified.        }

    Place         : Word;         { Points to correct COM Port I/O Address}

    Hold          : Byte;         { Holds a temp. value when determining  }
                                  { the port's initial baud rate          }

    Baud          : Word;         { Defaults to current COM Port speed, or}
                                  { becomes desired baud when specified   }

    Code,                         { Holds returned error when COM Port is }
                                  { initialized.                          }

    Status        : Word;         { Holds error status when converting the}
                                  { COM Port or Baud rate from string to  }
                                  { a number.                             }

    ExitStatus    : Byte;         { Is assigned an error code indicating  }
                                  { results of operation.  This value is  }
                                  { eventually returned as a process exit }
                                  { code via Halt. See Const defs above.  }


Function ReceiveXmodem(FileName:String):String;
Function ReceiveXmodemCRC(FileName:String):String;

Function SendXmodemOneKCRC(FileName:String):String;
Function SendXmodemCRC(FileName:String):String;
Function SendXmodemOneK(FileName:String):String;
Function SendXmodem(FileName:String):String;

Implementation
{$V-}     {Variables of any length can match in procedure headers}

{$S-,R-}  {Turn stack and range check off now that I'm done deugging}

Function Uppercase (Line : String) : String;  { Make a string all uppercase }
                                              { ...used on filename         }
Var Count : Integer;
Begin
  If Length(Line) > 0
    then
      For Count := 1 to Length(Line) do       { Loop through each character }
        Line[Count] := UpCase(Line[Count]);   { Uppercase it if needed      }
  Uppercase := Line;                          { Return results via function }
End;


{ This procedure updates the CRC value of the block.  The CRC is initially }
{ set to zero, then as data is received or sent, a call for each byte in   }
{ the dat ais made here.  The old CRC value is passed too, and it is       }
{ updated to include the data byte being processed.  The result is that    }
{ once all the data has been sent/received, you have the CRC all ready for }
{ sending or checking against the one from the transmitter.  The only      }
{ thing you need to watch is this:  After the data has been run through    }
{ this routine byte by byte, the CRC you have is bitwise reversed from the }
{ value actually to be sent.  Transmission standards say that the CRC is to}
{ be transmitted high bit first.  But standards also say a byte is sent low}
{ bit first.  Thus, in order to get the desired value to send, use the     }
{ ReverseWord procedure below.  The value you have after ReverseWord       }
{ *** IS NOT *** the CRC, but rather the bit-reversed CRC, all ready to be }
{ transmitted.  The machine on the other end then has to reverse the bits  }
{ back again to get the CRC value.  It's kinda crazy to have to reverse the}
{ bits, but that's the way the IEEE says you do it.              -Andrew   }


Function UpdCRC (Data    : Byte;
                 CRCTemp : Word) : Word;

{ We figure the CRC with an uncommonly small data table here: }

Const Table : Array[0..15] of Word = ($0000,$1081,$2102,$3183,
                                      $4204,$5285,$6306,$7387,
                                      $8408,$9489,$A50A,$B58B,
                                      $C60C,$D68D,$E70E,$F78F);

Var Index : Byte;        { Index pointer to the data table above }
    Power,               { bit pointer used in reversing byte    }
    Power2 : Byte;       { ditto                                 }
    Data2  : Byte;       { second version of byte, bit reversed  }
    Count  : Integer;    { iterates from 1 to 8 for it reversal  }

Begin

 { Before we can figure out the CRC with this byte, we have to reverse it's }
 { bits - due to the fact that data is normally transmitted low-bit first   }

  Data2 := 0;                          { Hold new reversed data here }
  Power := 1;                          { Power of 2 counter going up }
  Power2 := 128;                       { Power of 2 counter going down }
  For Count := 1 to 8 do               { Do 8 bits of the byte         }
    Begin
      If (Data and Power) <> 0         { If lower bit is a 1, then ... }
        then
          Data2 := Data2 or Power2;    { ....set the higher bit        }
      Power2 := Power2 shr 1;          { Shift higher power of 2 down  }
      Power  := Power  shl 1;          { Shift lower power of 2 up     }
    End;
  Data := Data2;                       { Data is assigned it's reverse }

  { Now we can start doing the CRC divide with the following polynomial: }
  {        16     12    5                                                }
  {       X   +  X   + X  + 1   , where X = 2                            }

  { Thus, we're talking about an actual value of $11021.  The divide is  }
  { done with Modulo Two arithmetic (XOR).  We don't actually divide here}
  { because the table already takes care of it for us.  For a very       }
  { clear and easy to understand description of the CRC-CCITT, and other }
  { types of CRC, etc, please see "C Programmer's Guide To NetBIOS," by  }
  { W. David Schwaderer, pages 167-217, published by Howard W. Sams &    }
  { Company.  This code uses the same method of CRC calculation as is    }
  { shown on pages 204 - 205 of the above book.                -AB       }

  Index := ((CRCTemp xor Data) and $000F);  { Figure an index to table }

  { Xor table data into CRC }
  CRCTemp := ((CRCTemp shr 4) and $0FFF) xor (Table[Index]);

  Data := Data shr 4;   {Now, do it all again for lower nibble of Data}

  Index := ((CRCTemp xor Data) and $000F);  { Figure an index to table }

  { Xor table data into CRC }
  CRCTemp := ((CRCTemp shr 4) and $0FFF) xor (Table[Index]);

  UpdCRC := CRCTemp;   { Set function result to new CRC value }
End;


{ This procedure simply reverses the bits of the word sent by parameter. }
{ Used to reverse bits of the CRC before sending.  I'd hope there would  }
{ be a much more efficient way to do such a simple little operation, but }
{ this was all that came to mind at the time I wrote it.  We only use it }
{ once for each 128 or 1024 (in 1K transfers) bytes, so I guess it does  }
{ not need to be extremely efficient.                                    }

Procedure ReverseWord (Var CRC : Word);
Var CRC2   : Word;     { Holds reversed Word     }
    Power,             { bit pointer in reversal }
    Power2 : Word;     { ditto                   }
    Count  : Integer;  { interates for 16 bits   }
Begin
  Power  := 1;                     { Start one pointer at low bit }
  Power2 := 32768;                 { Other pointer at high bit    }
  CRC2   := 0;                     { new word starts out zero.    }

  For Count := 1 to 16 do          { Iterate for all 16 bits      }
    Begin
      If (CRC and Power) <> 0      { If lower bit is 1, then....  }
        then
          CRC2 := CRC2 or Power2;  { ...Set higher bit            }

      Power  := Power  shl 1;      { Shift bit pointers by one    }
      Power2 := Power2 shr 1;
    End;
  CRC := CRC2;                     { Return reversed word to caller }
End;


Procedure Beep;   { Big Deal - Makes A Beep }
Begin
  Sound(500);
  Delay(20);
  NoSound;
End;


{ This procedure is vital to timing of transfers.  We can use it to     }
{ determine if a timeout has occured (i.e., too long of time passes     }
{ between chars received from remote machine).  The SecsToWait param.   }
{ specifies the number of seconds to wait before declaring that a       }
{ timeout has occurred.  This can be from 1 to 60 secs wait.  A value   }
{ over 60 will cause errors in timing.  The TimeDone param is assigned  }
{ True when the time in SecsToWait has passed, while receiving no chars }
{ from the remote end.  If, however, a char comes in before the alloted }
{ SecsToWait passes, then TimeDone becomes False, and Ch returns the    }
{ character received.   This is VERY useful in Xmodem implementation.   }
{ This routine is accurate to within  1 second.                        }


Procedure MonitorReceive (Var Ch         : Char;
                               SecsToWait : Byte;
                           Var TimeDone   : Boolean);

Var Hour,                 { Hold Hours from GetTime             }
    Min,                  { Hold Mins  from GetTime             }
    Sec,                  { Hold Secs  from GetTime             }
    LookFor,              { Hold second value we're waiting for }
    Hund       : Word;    { Hold Hundredths from GetTime        }

    Done       : Boolean; { Used in telling whether we've got a char }
                          { in Repeat/Unit loop.                     }

Begin

  TimeDone := False;      { Automatically assume we've got a character }

  If not Com_RX_Empty     { If one is waiting, receive it & return     }
    then
      Begin
        Ch := Com_RX;
        Exit;
      End;

  GetTime(Hour,Min,Sec,Hund);    { Otherwise, get the start time         }
  LookFor := Sec + SecsToWait;   { Figure destination time to the second }

  While LookFor > 59 do          { Map it to a value from 0 to 59        }
    LookFor := LookFor - 60;

  Done := False;                 { Assume no char received now.          }
  Repeat
    GetTime(Hour,Min,Sec,Hund);  { Get current time now                  }
    TimeDone := Sec = LookFor;   { TimeDone is true if the time is up    }
    If Not Com_RX_Empty          { If we did get a character in, then....}
      then
        Begin
          Ch := Com_RX;          { ....Get it....                        }
          TimeDone := False;     { ....Specify no TimeOut occurred....   }
          Done     := True;      { ....And get us out of the loop.       }
        End;
  Until Done or TimeDone;        { Keep looping until we either run out  }
                                 { of time, or get a character.          }
End;


{ This routine is used to handle opening a file for read/write with error }
{ trapping capabilities.  OpenFile attempts to open the file in FileName  }
{ for Read or Write (depending on Direction), to a record length of       }
{ RecordSize bytes, using File1 for a file handle.  If this operation is  }
{ successful, the function returns True, else False.  A null Filename is  }
{ NOT counted as valid, although Pascal will allow such an operation.     }


Function OpenFile (Var File1        : File;
                       Direction    : Boolean;  {True = Write, False = read}
                       RecordSize   : Integer;
                   Var FileName     : String) : Boolean;

Begin
  If FileName = ''                { If no filename, then we have an error }
    then
      Begin
        OpenFile := False;
      End
    else
      Begin
        Assign(File1,Filename);    { Else, associate Filename with File1 }
        {$I-}
        If Direction                     { Attempt open for Read/Write   }
          then
            ReWrite(File1,RecordSize)
          else
            ReSet(File1,RecordSize);
        {$I+}
        OpenFile := not (IOResult > 0);  { If there was an error, return }
                                         { False, else return True.      }
      End;
End;



{ This procedure will transmit a data block.  The data is passed as a VAR }
{ parameter (to save on stack space), the length of the data (128 or 1024)}
{ comes next.  The procedure also is sent the block number to send the    }
{ data as, as well as whether CRC or Checksum error correction is used.   }


Procedure SendBlock (Var Data : BlockType;
                     DataLen  : Integer;
                     BlockNum : Byte;
                     CRCMode  : Boolean     );

Var Count    : Integer;  { Iteration counter                        }
    CRC      : Word;     { Have WORD ready to handle CRC correction }
    Checksum : Byte;     { Have a BYTE ready incase of checksum     }
Begin
  Case DataLen of              { If data is 128 bytes, then....     }
    128  : Com_TX(SOH);        { ....tell receiver this, else....   }
    1024 : Com_TX(STX);        { ....tell receiver it's 1K long     }
  End;{Case}

  Com_TX(Chr(BlockNum));        { Transmit block number              }
  Com_Tx(Chr(not BlockNum));    { Transmit one's complement of the   }
                                { block number                       }

  If CRCMode                    { If we're doing CRC, then.....      }
    then
      CRC := 0                  { ....initialize CRC variable, else  }
    else
      Checksum := 0;            { initialize the Checksum variable   }

  For Count := 1 to DataLen do  { Loop for all data bytes in block   }
    Begin
      Com_Tx(Chr(Data[Count])); { Transmit the data byte             }
      If CRCMode
        then
          CRC := UpdCRC(Data[Count],CRC)  { If CRC, then update CRC  }
        else
          Checksum := Checksum + Data[Count];  { Else update the Checksum }
    End;
  If CRCMode                     { If CRC, then transmit reversed CRC }
    then
      Begin
        ReverseWord(CRC);
        Com_TX( Chr(Hi(CRC)) );
        Com_TX( Chr(Lo(CRC)) );
      End
    else
      Com_TX(Chr(Checksum));    { else transmit the chechsum }

  Repeat Until Com_TX_Empty;    { Wait until the buffer has been cleared. }
End;



{$I SXRX.INC}   { Include XmodemSend and XmodemReceive procedures here}

Procedure InterpretCode (Code : Byte);
Begin
  Writeln;
  Case Code of
    NoError                 : StrX:=('Receive Complete.');
    FileError               : StrX:=('Error Opening File!');
    OperatorAbortError      : StrX:=('Operator Abort!');
    RemoteCancelError       : StrX:=('Transfer Canceled By Remote!');
    MaxTimeoutError         : StrX:=('Maximum Timeout Count Exceeded!');
    BlockOutOfSequenceError : StrX:=('Block Number Out Of Sequence Error!');
    EOTTimeOut              : StrX:=('Timeout On Second EOT - Protocol Assumed Complete.');
    CarrierLostError        : StrX:=('No Carrier!');
    InvalidOperation        : StrX:=('Missing or Invalid Operation Specified!');
    InvalidSwitch           : StrX:=('Invalid Command Line Switch!');
    InvalidBaudRate         : StrX:=('Invalid Baud Rate!');
    InvalidCOMPort          : StrX:=('Invalid COM Port Number!');
  End;{Case}
  Beep;
  Beep;
  Delay(1000);
  Com_DeInstall;      { No need to keep COM Port running  }
End;

{Function XmodemReceive(CRCMode:Boolean;Var FileName:String):Byte;}
Function ReceiveXmodem(FileName:String):String;
Begin
  InterpretCode(XmodemReceive(False,FileName));
  ReceiveXmodem:=StrX;
End;

Function ReceiveXmodemCRC(FileName:String):String;
Begin
  InterpretCode(XmodemReceive(True,FileName));
  ReceiveXmodemCRC:=StrX;
End;

{Function XmodemSend(CRCMode,OneK:Boolean;Var FileName:String):Byte;}
Function SendXmodemOneKCRC(FileName:String):String;
Begin
  InterpretCode(XmodemSend(True,True,FileName));
  SendXmodemOneKCRC:=StrX;
End;

Function SendXmodemCRC(FileName:String):String;
Begin
  InterpretCode(XmodemSend(True,False,FileName));
  SendXmodemCRC:=StrX;
End;

Function SendXmodemOneK(FileName:String):String;
Begin
  InterpretCode(XmodemSend(False,True,FileName));
  SendXmodemOneK:=StrX;
End;

Function SendXmodem(FileName:String):String;
Begin
  InterpretCode(XmodemSend(False,False,FileName));
  SendXmodem:=StrX;
End;

{Begin
  Case OpCode of
    RX : InterpretCode(XmodemReceive(CRCMode,FileName));
    SX : InterpretCode(XmodemSend(CRCMode,OneK,FileName));
    else InterpretCode(InvalidOperation);
  End;{Case}
End.
