/* File: VGADEMO.C
** Description:
**		VGA sketch demo program.
** Copyright:
**		Copyright 1994, David G. Roberts
*/

#include <assert.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include "gamedefs.h"
#include "mouse.h"
#include "mouseevt.h"
#include "vga.h"

/* constants */
#define TEXT_MODE_NUMBER			(0x03)
#define MOUSE_EVENT_QUEUE_LENGTH	(100)
#define MODE_13H_HORIZ				(320)
#define MODE_13H_VERT				(200)
#define VIDEO_MEM_SEGMENT			(0xA000)

/* types */
typedef struct {
	int Mode;
    int XDim;
    int YDim;
    int BytesPerLine;
    int PixelsPerByte;
    int MaxColors;
    int MouseXDivisor;
    int MouseYDivisor;
} MODE_INFO;

/* module variables */
UINT8 PenColor;
int PenDown;
UINT16 ModeNumber;
MODE_INFO CurrentModeInfo;
static MODE_INFO ModeInfoTable[] = {
	{0xD, 320, 200, 40, 8, 16, 4, 1},
    {0xE, 640, 200, 80, 8, 16, 1, 1},
    {0x10, 640, 350, 80, 8, 16, 1, 1},
    {0x12, 640, 480, 80, 8, 16, 1, 1},
    {0x13, 320, 200, 320, 1, 256, 2, 1}
};

/*
	Function: GetModeInfo
    Description:
    	When supplied with a valid mode, the function returns
        information about it.  The info is returned in the
        MODE_INFO structure pointed to by ModeInfoReturn.
        If the supplied mode number is invalid, the function
        returns FALSE, otherwise it returns TRUE.
*/
int GetModeInfo(UINT16 Mode, MODE_INFO * ModeInfoReturn)
{
    int i;

    for (i = 0; i <= DIM(ModeInfoTable); i++) {
    	if (Mode == ModeInfoTable[i].Mode) {
        	*ModeInfoReturn = ModeInfoTable[i];
            return TRUE;
        }
    }
    return FALSE;
}

/*
	Function: PlotMode13h
    Description:
    	Plots a pixel in mode 013h (assumes that the VGA has already
        been set to that mode).
*/
void PlotMode13h(UINT16 X, UINT16 Y, UINT16 Color)
{
	UINT16 Offset;
	UINT8 far * GraphicsBase;

    /* make sure parameters are in range */
    assert(X <= (MODE_13H_HORIZ - 1));
    assert(Y <= (MODE_13H_VERT - 1));

    /* create a far pointer to the base of Mode 13h memory */
    GraphicsBase = (UINT8 far *) MK_FP(VIDEO_MEM_SEGMENT,0);

    /* compute offset from X,Y coordinates */
    Offset = Y * MODE_13H_HORIZ + X;

    /* set pixel to color */
    *(GraphicsBase + Offset) = Color;
}

/*
	Function: PlotPlanarMode
    Description:
    	Plots a point in a planar mode.
*/
void PlotPlanarMode(MODE_INFO * ModeInfo, UINT16 X, UINT16 Y, UINT16 Color)
{
	UINT16 Offset;
	UINT8 far * GraphicsBase;
    UINT8 PixelNumber;
    UINT8 PixelMask;
    UINT8 Dummy;

    assert(X <= (ModeInfo->XDim - 1));
    assert(Y <= (ModeInfo->YDim - 1));

	/* compute VGA memory location */
    GraphicsBase = (UINT8 far *) MK_FP(VIDEO_MEM_SEGMENT, 0);
    Offset = (Y * ModeInfo->BytesPerLine) + (X / ModeInfo->PixelsPerByte);
	PixelNumber = X % ModeInfo->PixelsPerByte;

    /* read VGA (load latches) */
    Dummy = GraphicsBase[Offset];

    /* set BMR to enable correct pixel location */
    PixelMask = 0x80 >> PixelNumber;
    SetBMR(PixelMask);

    /* set SRR to color */
    SetSRR((UINT8) Color);

    /* set SRER to enable SR function on all planes */
    SetSRER(0xF);

    /* write anything to VGA memory location */
    GraphicsBase[Offset] = Dummy;

    /* clean up */
    SetBMR(0xFF);	/* all pixels */
    SetSRER(0x00);	/* SR function disabled */
}

/*
	Function: VGAPlotPoint
    Description:
    	Plots a point on the screen.
*/
void VGAPlotPoint(MODE_INFO * ModeInfo, UINT16 X, UINT16 Y, UINT16 Color)
{
	if (ModeInfo->Mode == 0x13) {
    	PlotMode13h(X, Y, Color);
    }
    else {
    	PlotPlanarMode(ModeInfo, X, Y, Color);
    }
}

/*
	Function: GetCommandLineMode
    Description:
    	Examines the first parameter supplied on the command line
        and converts it from a string to an integer.  The integer
        is then returned to the caller.  If an error occurs, 0
        is returned instead.
*/
UINT16 GetCommandLineMode(char *argv[])
{
	UINT16 Mode;
    int ScanStatus;

    /* convert argv[1] from ASCII to an integer */
    ScanStatus = sscanf(argv[1], "0x%x", &Mode);

    if (ScanStatus == 1) {
	    return Mode;
    }
    else {
    	return 0;
    }
}

/*
	Function: IncrementPenColor
    Description:
    	Increments the current color of the pen and handles
        wrap-around when the pen has reached the maximum
        number of colors supported by the current mode.
*/
void IncrementPenColor(void)
{
	PenColor++;
    if (PenColor >= CurrentModeInfo.MaxColors) {
    	PenColor = 0;
    }
}

/*
	Function: PlotPointAtCursor
    Description:
    	Takes care of plotting a point at the cursor location
        given the MouseEvent parameter.
*/
void PlotPointAtCursor(MOUSE_EVENT * MouseEvent)
{
	int Xlocation;
    int Ylocation;

    assert(CurrentModeInfo.MouseXDivisor != 0);
    assert(CurrentModeInfo.MouseYDivisor != 0);

    /* convert mouse coordinates from event to screen coordinates */
    Xlocation = MouseEvent->X / CurrentModeInfo.MouseXDivisor;
    Ylocation = MouseEvent->Y / CurrentModeInfo.MouseYDivisor;

    /* plot the point, hiding the mouse cursor temporarily while */
    /* we do the screen update */
    HideMouseCursor();
    VGAPlotPoint(&CurrentModeInfo, Xlocation, Ylocation, PenColor);
    ShowMouseCursor();
}

/*
	Function: DispatchMouseEvent
    Description:
    	Respond to a mouse event by calling the appropriate
        handling function.
*/
void DispatchMouseEvent(MOUSE_EVENT * MouseEvent)
{
	switch (MouseEvent->Event) {
    	case ME_RIGHT_RELEASED:
        	IncrementPenColor();
            break;
        case ME_LEFT_PRESSED:
        	PenDown = TRUE;
            PlotPointAtCursor(MouseEvent);
        	break;
        case ME_LEFT_RELEASED:
        	PenDown = FALSE;
        	break;
        case ME_MOVEMENT:
        	if (PenDown == TRUE) {
            	PlotPointAtCursor(MouseEvent);
            }
        	break;
        default:
        	break;
    }
}

int main(int argc, char *argv[])
{
    int ModeSupported;
    int MouseDriverPresent;
    int DummyKey;
    MOUSE_EVENT MouseEvent;

    /* detect VGA */
    if (!DetectVGA()) {
    	printf("You must have a VGA to run this program.\n");
        return 1;
    }

    /* see if we have a command line parameter.  If not, exit */
    if (argc < 2) {
    	printf("You must supply a graphics mode on the command line.\n");
        return 1;
    }

	/* get vga mode from command line */
    ModeNumber = GetCommandLineMode(argv);

    /* make sure people know what format to use */
    if (ModeNumber == 0) {
    	printf("Enter mode number as \"0xM\"\n");
        return 1;
    }

    /* if mode is one we support, initialize mode */
    /* else print error and exit */
	ModeSupported = GetModeInfo(ModeNumber, &CurrentModeInfo);
    if (ModeSupported == FALSE) {
    	printf("Mode 0x%x is not supported.\n", ModeNumber);
        return 1;
    }
    SetVGAMode(CurrentModeInfo.Mode);

    /* initialize mouse */
    MouseDriverPresent = ResetMouse();
    if (MouseDriverPresent == 0) {
	    SetVGAMode(TEXT_MODE_NUMBER);
        printf("Mouse needed for this program.\n");
    	return 1;
    }

    /* install a handler.  Note that we don't respond to */
    /* right button press events (only right release) */
    InstallMouseEventQueue(MOUSE_EVENT_QUEUE_LENGTH,
    	ME_RIGHT_RELEASED | ME_LEFT_PRESSED | ME_LEFT_RELEASED |
        ME_MOVEMENT);

    /* initialize pen */
    PenDown = FALSE;
    PenColor = 1;	/* note 0 = black */

    ShowMouseCursor();

    /* do polling loop. When key is hit, exit */
    while (!kbhit()) {
    	if (QueryMouseEventWaiting()) {
        	GetMouseEvent(&MouseEvent);
            DispatchMouseEvent(&MouseEvent);
        }
    }

    /* read this so it isn't left hanging around for DOS to find */
    DummyKey = getch();

    /* clean up mouse */
    HideMouseCursor();
    RemoveMouseEventQueue();

    /* enter text mode again */
    SetVGAMode(TEXT_MODE_NUMBER);

    /* terminate program */
    return 0;
}

