/* File: JSTICK.C
** Description:
**   Defines routines for reading the state of the joystick and
**   sensing if a joystick is attached to the game port.
*/

#include <stdio.h>
#include <dos.h>
#include "gamedefs.h"
#include "jstick.h"

/* constants */
#define JOYSTICK_PORT (0x201)

/*
	Function: SenseJoysticks
	Description:
		Tries to determine which joystick axes are available to
		be read.  The routine works by repeatedly reading the
		joystick port location for a very long time.  Those
		axes that time out before we are done reading are assumed
		to be present.  Axes that don't time out by the time that
		we're done are assumed to not be present.  Note that on
		a very fast machine, this routine could be fooled.
*/

void SenseJoysticks(JOYSTICK_STATE * JsState)
{
	register UINT8 portval;
	UINT16 limit;

	/* initialize counters */
	JsState->JsAxisAX = JsState->JsAxisAY = 0;
	JsState->JsAxisBX = JsState->JsAxisBY = 0;
	limit = 0xFFFF;

	/* don't let interrupts happen while we're reading */
	asm cli;

	/* start the timers -- data value of write is unimportant */
	outportb(JOYSTICK_PORT, 0x00);

	/* do until interesting bits all drop to zero */
	while (limit) {
		portval = inportb(JOYSTICK_PORT);
		JsState->JsAxisAX += portval & 1;
		portval >>= 1;
		JsState->JsAxisAY += portval & 1;
		portval >>= 1;
		JsState->JsAxisBX += portval & 1;
		portval >>= 1;
		JsState->JsAxisBY += portval & 1;
		limit--;
	}

    /* enable interrupts */
    asm sti;

	JsState->JsMask = 0;
	if ((JsState->JsAxisAX != 0xFFFF) && (JsState->JsAxisAX != 0))
		JsState->JsMask |= 0x1;
	if ((JsState->JsAxisAY != 0xFFFF) && (JsState->JsAxisAY != 0))
		JsState->JsMask |= 0x2;
	if ((JsState->JsAxisBX != 0xFFFF) && (JsState->JsAxisBX != 0))
		JsState->JsMask |= 0x4;
	if ((JsState->JsAxisBY != 0xFFFF) && (JsState->JsAxisBY != 0))
		JsState->JsMask |= 0x8;

}

/*
	Function: ReadJoysticks
	Description:
		Reads the current values of the joystick positions and
		button states.  Axis values are returned in the JS structure
		variables JsAxisAX/AY/BX/BY and button states in
		the structure variables JsButtonA1/A2/B1/B2.  Note that
		the structure variable JsMask controls which axes of
		which joysticks (either A or B) should be waited for.
		If the B joystick, for instance, is not present, and
		JsMask indicates that JsAxisBY should be waited
		for, the routine will wait an excessively long time.
		The routine SenseJoysticks, above, can be used to try to
		automatically determine which axes of which joysticks
		are present.  Note that for joystick axes not present,
		JsAxisAX/AY/BX/BY are set to garbage values.
*/
void ReadJoysticks(JOYSTICK_STATE * JsState)
{
	register UINT8 portval;
	UINT16 limit;

	/* initialize counters */
	JsState->JsAxisAX = JsState->JsAxisAY = 0;
	JsState->JsAxisBX = JsState->JsAxisBY = 0;
	limit = 0xFFFF;

	/* don't let interrupts happen while we're reading */
	asm cli;

	/* start the timers -- data value of write is unimportant */
	outportb(JOYSTICK_PORT, 0x00);

	/* do until interesting bits all drop to zero */
	while (((portval = inportb(JOYSTICK_PORT)) & JsState->JsMask) && limit) {
		JsState->JsAxisAX += portval & 1;
		portval >>= 1;
		JsState->JsAxisAY += portval & 1;
		portval >>= 1;
		JsState->JsAxisBX += portval & 1;
		portval >>= 1;
		JsState->JsAxisBY += portval & 1;
		limit--;
	}

	/* recognize interrupts again */
	asm sti;

	/* update button variables -- note these are inverted from the port */
	(portval & 0x10) ? (JsState->JsButtonA1 = 0) : (JsState->JsButtonA1 = 1);
	(portval & 0x20) ? (JsState->JsButtonA2 = 0) : (JsState->JsButtonA2 = 1);
	(portval & 0x40) ? (JsState->JsButtonB1 = 0) : (JsState->JsButtonB1 = 1);
	(portval & 0x80) ? (JsState->JsButtonB2 = 0) : (JsState->JsButtonB2 = 1);

}
