/* File: COLLIDE.C
** Description:
**   Routines for testing collisions.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include "gamedefs.h"
#include "collide.h"
#include "bitblt.h"

/*
	Function: CollisionTestRect
    Description:
    	Tests two bounding rectangles to see if they overlap.
		Returns TRUE if so, FALSE otherwise.
*/
BOOL CollisionTestRect(RECT * r1, RECT * r2)
{
	assert(r1->Left <= r1->Right);
    assert(r1->Top <= r1->Bottom);
    assert(r2->Left <= r2->Right);
    assert(r2->Top <= r2->Bottom);

	if (r1->Left > r2->Right || r2->Left > r1->Right ||
    	r1->Top > r2->Bottom || r2->Top > r1->Bottom) {
        return FALSE;
    }
    else {
    	return TRUE;
    }
}

/*
	Function: ComputeBoundingRect
    Description:
    	Calculates the bounding rectangle for a sprite given its
        position, origin offset, width, and height.
*/
void ComputeBoundingRect
	(
    int x,
    int y,
    int OriginX,
    int OriginY,
    int Width,
    int Height,
    RECT * Rect
    )
{
	int Top;
    int Left;
    int Bottom;
    int Right;

    assert(Rect != NULL);

    Top				= y - OriginY;
    Left			= x - OriginX;
    Bottom			= Top + Height - 1;
    Right			= Left + Width - 1;

    Rect->Top		= Top;
    Rect->Left		= Left;
    Rect->Bottom	= Bottom;
    Rect->Right		= Right;
}

/*
	Function: CreateCollisionMap
    Description:
    	Given a far pointer to a LINEAR_BITMAP, the function returns
        a far pointer to a COLLISION_MAP derived from the LINEAR_BITMAP.
        All "solid" pixels from the LINEAR_BITMAP have a corresponding
        bit set to 1 in the COLLISION_MAP while "transparent" pixels
        have a corresponding bit set to 0.
*/
COLLISION_MAP far * CreateCollisionMap(LINEAR_BITMAP far * LinearBM)
{
    UINT16 CollisionMapWidth;
    COLLISION_MAP far * CollisionMap;
    UINT8 far * CollisionMapData;
    UINT8 far * BitmapData;
    UINT16 WidthCounter;
    UINT16 HeightCounter;
    UINT8 WorkingData;
    int BitCounter;

    assert(LinearBM != NULL);
    assert(LinearBM->Width != 0);
    assert(LinearBM->Height != 0);

    /* compute size of collision map, allocate, and initialize */
    CollisionMapWidth = ((LinearBM->Width - 1) / 8) + 1;
    CollisionMap = (COLLISION_MAP far *) farmalloc(sizeof(COLLISION_MAP) +
    	(CollisionMapWidth * LinearBM->Height) - 1);
    CollisionMap->Width		= CollisionMapWidth;
    CollisionMap->Height	= LinearBM->Height;

    /* initialize data pointers */
    CollisionMapData	= (UINT8 far *) &(CollisionMap->Data);
    BitmapData			= (UINT8 far *) &(LinearBM->Data);

    /* interate over all pixels in bitmap and fill in collision map */
    for (HeightCounter = 0; HeightCounter < LinearBM->Height;
		HeightCounter++) {

        WorkingData = 0;
        BitCounter = 7;
        for (WidthCounter = 0; WidthCounter < LinearBM->Width;
        	WidthCounter++) {

            if (*BitmapData++ != 0) {
            	WorkingData |= 1 << BitCounter;
            }
            BitCounter--;
            if (BitCounter < 0) {
            	*CollisionMapData++ = WorkingData;
                WorkingData = 0;
                BitCounter = 7;
            }
        }

        /* do final save of working data at end of line unless */
        /* it's just been done */
		if (BitCounter != 7) {
        	*CollisionMapData++ = WorkingData;
        }
    }

    return CollisionMap;
}

/*
	Function: CollisionTestBitmap
    Description:
    	Tests two objects using COLLISION_MAPs.  The upper left corner
        of each object is specified with (x1, y1) and (x2, y2).
*/
BOOL CollisionTestBitmap
	(
	COLLISION_MAP far * Object1,
	COLLISION_MAP far * Object2,
    int x1,
    int y1,
    int x2,
    int y2
    )
{
	UINT8 far * Data1;
    UINT8 far * Data2;
    COLLISION_MAP far * SwapTemp;
    int DeltaX;
    int DeltaY;
    int Shift;
    int Skip;
    UINT16 WidthCounter1;
    UINT16 WidthCounter2;
    UINT16 HeightCounter1;
    UINT16 HeightCounter2;
    UINT8 Object1Data;
    UINT8 ShiftRegister;
    UINT8 OldObject2Data;
    UINT8 NewObject2Data;
    UINT8 FinalObject2Data;

    assert(Object1 != NULL);
    assert(Object2 != NULL);

    DeltaX = x2 - x1;
    DeltaY = y2 - y1;

    /* swap objects to make the algorithm work */
    if (DeltaX < 0) {
    	SwapTemp	= Object1;
        Object1		= Object2;
        Object2		= SwapTemp;
        DeltaX		= -DeltaX;
        DeltaY		= -DeltaY;
    }

    Data1			= (UINT8 far *) &(Object1->Data);
    Data2			= (UINT8 far *) &(Object2->Data);

    HeightCounter1	= 0;
    HeightCounter2	= 0;

    /* skip rows off the object with the least Y-value */
    if (DeltaY > 0) {
    	Data1 += Object1->Width * DeltaY;
        HeightCounter1 += DeltaY;
    }
    else if (DeltaY < 0) {
    	Data2 += Object2->Width * -DeltaY;
        HeightCounter2 -= DeltaY;
    }

    Shift	= DeltaX % 8;	/* amount to shift object 2 data to right */
    Skip 	= DeltaX / 8;	/* number of bytes to skip at beginning of */
    						/*   object 1 data line */

    while (HeightCounter1 < Object1->Height &&
		HeightCounter2 < Object2->Height) {

        /* potentially skip a few bytes 'cause obj 1 is to left of obj 2 */
	    WidthCounter1	= Skip;
        Data1			+= Skip;

    	WidthCounter2	= 0;
        OldObject2Data	= 0;

        while (WidthCounter1 < Object1->Width &&
        	WidthCounter2 < Object2->Width) {

            /* get data */
            Object1Data		= *Data1++;
            NewObject2Data	= *Data2++;
            /* shift object 2 data to correct delta X differential */
            ShiftRegister	= ((UINT16) OldObject2Data << 8) |
				(UINT16) NewObject2Data;
            ShiftRegister 	>>= Shift;
            FinalObject2Data = ShiftRegister & 0xFF;

            /* return if we have a collision */
            if (Object1Data & FinalObject2Data) {
            	return TRUE;
            }

            OldObject2Data = NewObject2Data;
            WidthCounter1++;
            WidthCounter2++;
        }

		/* correct pointers at end of line */
        Data1 += Object1->Width - WidthCounter1;
        Data2 += Object2->Width - WidthCounter2;

        HeightCounter1++;
        HeightCounter2++;
    }

	/* we got through all that with no collision */
    return FALSE;
}
