/* File: COLLDEMO.C
** Description:
**   Demonstration of collision routines.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include "gamedefs.h"
#include "animate.h"
#include "bitblt.h"
#include "collide.h"
#include "jstick.h"
#include "palette.h"
#include "pcx.h"
#include "retrace.h"
#include "setmodex.h"
#include "vga.h"

/* types */
struct SPRITE_tag {
    int		x;
    int		y;
    int		vx;
    int		vy;
    int		OldX[2];
    int		OldY[2];
    BOOL	Erase[2];
    BOOL	Draw;
    PLANAR_BITMAP far * Bitmap;
    PLANAR_BITMAP far * EraseBitmap;
    COLLISION_MAP far * CollisionMap;
    RECT	Bounds;
};

typedef struct SPRITE_tag SPRITE;

/* constants */
#define NUM_BALLS		15

#define PAGE0_OFFSET	0
#define PAGE1_OFFSET	0x4B00

#define SOUND_FREQUENCY	440

#define ERASE_COLOR		16

#define BLINK_RATE		15

/* globals */
SPRITE Ship;
SPRITE Ball[NUM_BALLS];
int HiddenPage;
UINT16 Offset[2];
UINT16 BallWidth;
UINT16 BallHeight;
UINT16 ShipWidth;
UINT16 ShipHeight;
UINT16 JsXmin;
UINT16 JsYmin;
UINT16 JsXmid;
UINT16 JsYmid;
UINT16 JsXmax;
UINT16 JsYmax;
JOYSTICK_STATE JsState;
UINT16 XRightThird;
UINT16 XLeftThird;
UINT16 YBottomThird;
UINT16 YTopThird;

/*
	Function: CalibrateJsMidpoint
    Description:
    	Calibrates a joystick's range using the midpoint method.
        Read's the value of the joystick at the time it is
        called and computes a min and max range from this.
        Note that this assumes that the joystick is centered
        when the routine is called.
*/
void CalibrateJsMidpoint
	(
    JOYSTICK_STATE * JsState,
    UINT16 * Xmin,
    UINT16 * Ymin,
    UINT16 * Xmid,
    UINT16 * Ymid,
    UINT16 * Xmax,
    UINT16 * Ymax
    )
{
	assert(JsState != NULL);
    assert(Xmin != NULL);
    assert(Ymin != NULL);
    assert(Xmid != NULL);
    assert(Ymid != NULL);
    assert(Xmax != NULL);
    assert(Ymax != NULL);

	/* read midpoint value */
    ReadJoysticks(JsState);
    *Xmid = JsState->JsAxisAX;
    *Ymid = JsState->JsAxisAY;

    /* assume min coordinate is (0,0) and max is twice midpoint */
    *Xmin = *Ymin = 0;

    *Xmax = 2 * *Xmid;
    *Ymax = 2 * *Ymid;
}

/*
	Function: UpdateBallPositions
    Description:
    	Bounces the balls around the screen.
*/
void UpdateBallPositions(void)
{
	int i;

    for (i = 0; i < NUM_BALLS; i++) {
    	if (Ball[i].Erase[HiddenPage]) {
        	Ball[i].x += Ball[i].vx;
        	Ball[i].y += Ball[i].vy;
        	if ((Ball[i].x < 0) || ((Ball[i].x + BallWidth) >=
				MODEX_WIDTH)) {
            	Ball[i].vx = -Ball[i].vx;
            	Ball[i].x += 2 * Ball[i].vx;
        	}
        	if ((Ball[i].y < 0) || ((Ball[i].y + BallHeight) >=
				MODEX_HEIGHT)) {
            	Ball[i].vy = -Ball[i].vy;
            	Ball[i].y += 2 * Ball[i].vy;
        	}
        }
        ComputeBoundingRect(Ball[i].x, Ball[i].y, Ball[i].Bitmap->OriginX,
        	Ball[i].Bitmap->OriginY, BallWidth, BallHeight,
			&(Ball[i].Bounds));
    }
}

/*
	Function: UpdateShipPosition
    Description:
    	Moves the spaceship around the screen.  The velocity of
		the ship is dependent on the position of the joystick.
*/
void UpdateShipPosition(void)
{
	if (Ship.Erase[HiddenPage]) {
    	ReadJoysticks(&JsState);
        if (JsState.JsAxisAX > XRightThird) {
        	Ship.vx = 2;
        }
        else if (JsState.JsAxisAX > XLeftThird) {
        	Ship.vx = 0;
        }
        else {
        	Ship.vx = -2;
        }
        if (JsState.JsAxisAY > YBottomThird) {
        	Ship.vy = 2;
        }
        else if (JsState.JsAxisAY > YTopThird) {
        	Ship.vy = 0;
        }
        else {
        	Ship.vy = -2;
        }
        Ship.x += Ship.vx;
        Ship.y += Ship.vy;
        if ((Ship.x < 0) || ((Ship.x + ShipWidth) >=
			MODEX_WIDTH)) {
            Ship.x -= Ship.vx;
        }
        if ((Ship.y < 0) || ((Ship.y + ShipHeight) >=
			MODEX_HEIGHT)) {
            Ship.y -= Ship.vy;
        }
    }
    ComputeBoundingRect(Ship.x, Ship.y, Ship.Bitmap->OriginX,
    	Ship.Bitmap->OriginY, ShipWidth, ShipHeight,
		&(Ship.Bounds));
}

/*
	Function: CheckCollisions
    Description:
    	Determine if there has been a collision between a ball and
        the ship.  If so, turn on the speaker to sound the alarm!
*/
void CheckCollisions(void)
{
	int i;

    for (i = 0; i < NUM_BALLS; i++) {
    	/* the following statement uses C's short-circuit boolean */
        /* expression capability.  The CollisionTestBitmap */
        /* function call is executed if and only if the */
        /* CollisionTestRect function returns TRUE.  If */
        /* CollisionTestRect returns FALSE, the CollisionTestBitmap */
        /* call is short circuited and not executed. */
    	if (CollisionTestRect(&(Ship.Bounds), &(Ball[i].Bounds)) &&
			CollisionTestBitmap(Ship.CollisionMap, Ball[i].CollisionMap,
			Ship.Bounds.Left, Ship.Bounds.Top, Ball[i].Bounds.Left,
			Ball[i].Bounds.Top)) {

            sound(SOUND_FREQUENCY);
        	break; /* we know it collided, so don't check further */
        }
    }
    /* if no collisions, turn off sound */
    if (i == NUM_BALLS) {
    	nosound();
    }
}

/*
	Function: Erase
    Description:
    	Erase all the sprites from the screen.
*/
void Erase(void)
{
	int i;

	for (i = 0; i < NUM_BALLS; i++) {
    	if (Ball[i].Erase[HiddenPage]) {
    		BltPlanar(Ball[i].EraseBitmap, Ball[i].OldX[HiddenPage],
				Ball[i].OldY[HiddenPage], Offset[HiddenPage]);
        }
    }
    if (Ship.Erase[HiddenPage]) {
    	BltPlanar(Ship.EraseBitmap, Ship.OldX[HiddenPage],
        	Ship.OldY[HiddenPage], Offset[HiddenPage]);
    }

}

/*
	Function: Draw
    Description:
    	Draw all the sprites on the screen at their new coordinates.
*/
void Draw(void)
{
	int i;

	for (i = 0; i < NUM_BALLS; i++) {
    	if (Ball[i].Draw) {
    		BltPlanar(Ball[i].Bitmap, Ball[i].x, Ball[i].y,
				Offset[HiddenPage]);
	        Ball[i].OldX[HiddenPage]	= Ball[i].x;
    	    Ball[i].OldY[HiddenPage]	= Ball[i].y;
        	Ball[i].Erase[HiddenPage]	= TRUE;
        }
    }
    if (Ship.Draw) {
    	BltPlanar(Ship.Bitmap, Ship.x, Ship.y, Offset[HiddenPage]);
        Ship.OldX[HiddenPage]	= Ship.x;
        Ship.OldY[HiddenPage]	= Ship.y;
        Ship.Erase[HiddenPage]	= TRUE;
    }
}

/*
	Function: CreateEraseBitmap
    Description:
    	Takes a bitmap as input and fills it with black.
*/
void CreateEraseBitmap(LINEAR_BITMAP far * Input)
{
	UINT8 far * Data;
    UINT16 Length;

    Length = Input->Width * Input->Height;
    Data = (UINT8 far *) &(Input->Data);

    while (Length > 0) {
    	*Data++ = ERASE_COLOR;
        Length--;
    }
}

int main()
{
    LINEAR_BITMAP far * TempLinear;
    PLANAR_BITMAP far * BallBitmap;
    PLANAR_BITMAP far * BallEraseBitmap;
    COLLISION_MAP far * BallCollisionMap;
    RGB_TUPLE BlinkOff = {42, 0, 0};
    RGB_TUPLE BlinkOn = {63, 21, 21};
    int i;
    int BlinkCounter;
    BOOL On;

    /* detect VGA */
    if (!DetectVGA()) {
    	printf("You must have a VGA to run this program.\n");
        return 1;
    }

    /* detect and calibrate joystick */
    SenseJoysticks(&JsState);
    JsState.JsMask &= 0x3; /* enable only A axes */
    if (JsState.JsMask == 0) {
    	printf("You must have a joystick to run this program.\n");
        return 1;
    }
    CalibrateJsMidpoint(&JsState, &JsXmin, &JsYmin, &JsXmid, &JsYmid,
    	&JsXmax, &JsYmax);
    XRightThird		= (2 * JsXmax) / 3;
    XLeftThird		= JsXmax / 3;
    YBottomThird	= (2 * JsYmax) / 3;
    YTopThird		= JsYmax / 3;

    /* init page offsets */
    Offset[0] = PAGE0_OFFSET;
    Offset[1] = PAGE1_OFFSET;
    HiddenPage = 1;

    /* initialize ball sprites */
    TempLinear = LoadPCX("BALL.PCX", NULL);
    if (TempLinear == NULL) {
    	printf("ERROR: Can't load ball bitmap\n");
        return 1;
    }
    BallWidth	= TempLinear->Width;
    BallHeight	= TempLinear->Height;
    BallBitmap	= LinearToPlanar(TempLinear);
    if (BallBitmap == NULL) {
    	printf("ERROR: Can't load ball bitmap\n");
        return 1;
    }
    BallCollisionMap = CreateCollisionMap(TempLinear);
    if (BallCollisionMap == NULL) {
    	printf("ERROR: Can't load ball bitmap\n");
        return 1;
    }
    CreateEraseBitmap(TempLinear);
    BallEraseBitmap = LinearToPlanar(TempLinear);
    farfree(TempLinear);
    if (BallEraseBitmap == NULL) {
    	printf("ERROR: Can't load ball bitmap\n");
        return 1;
    }
    for (i = 0; i < NUM_BALLS; i++) {
    	Ball[i].x		= random(MODEX_WIDTH - BallWidth);
        Ball[i].y		= random(MODEX_HEIGHT - BallHeight);
        Ball[i].vx 		= random(2) + 1;
        Ball[i].vy		= random(2) + 1;
        if (random(2) == 1) {
        	Ball[i].vx = - Ball[i].vx;
        }
        if (random(2) == 1) {
        	Ball[i].vy = - Ball[i].vy;
        }
        Ball[i].Bitmap		= BallBitmap;
        Ball[i].EraseBitmap	= BallEraseBitmap;
        Ball[i].CollisionMap= BallCollisionMap;
        Ball[i].Draw		= TRUE;
        Ball[i].Erase[0]	= FALSE;
		Ball[i].Erase[1]	= FALSE;
    }

    /* load ship sprite */
    TempLinear	= LoadPCX("SHIP1.PCX", NULL);
    if (TempLinear == NULL) {
    	printf("ERROR: Can't load ship bitmap\n");
        return 1;
    }
    ShipWidth	= TempLinear->Width;
    ShipHeight	= TempLinear->Height;
    Ship.Bitmap			= LinearToPlanar(TempLinear);
    if (Ship.Bitmap == NULL) {
    	printf("ERROR: Can't load ship bitmap\n");
        return 1;
    }
    Ship.CollisionMap = CreateCollisionMap(TempLinear);
    if (Ship.CollisionMap == NULL) {
    	printf("ERROR: Can't load ship bitmap\n");
        return 1;
    }
    CreateEraseBitmap(TempLinear);
    Ship.EraseBitmap	= LinearToPlanar(TempLinear);
    if (Ship.EraseBitmap == NULL) {
    	printf("ERROR: Can't load ship bitmap\n");
        return 1;
    }
    farfree(TempLinear);

    Ship.x			= MODE13H_WIDTH / 2;
    Ship.y			= MODE13H_HEIGHT /2;
    Ship.Draw		= TRUE;
    Ship.Erase[0]	= FALSE;
    Ship.Erase[1]	= FALSE;

    SetModeX();

    BlinkCounter = 0;
    On = TRUE;

	while (!kbhit()) {
    	/* update positions */
        UpdateBallPositions();
        UpdateShipPosition();

        /* check collisions */
        CheckCollisions();

        /* erase old stuff */
        Erase();

        /* draw new stuff */
        Draw();

        /* flip page */
        PageFlip(Offset[HiddenPage]);
        HiddenPage ^= 1; /* flip HiddenPage to other state */

        /* blink the space ship light */
        BlinkCounter++;
        if (BlinkCounter == BLINK_RATE) {
        	if (On) {
        			SetVGAPaletteEntry(4, &BlinkOn);
        	}
        	else {
        		SetVGAPaletteEntry(4, &BlinkOff);
        	}
        	On = !On;
        	BlinkCounter = 0;
        }
    }

    /* shut off sound just in case we were collding with a ball */
    /* when the key was hit */
    nosound();

    /* clean up keys, return to text mode, free memory, and exit */
    getch();

    SetVGAMode(0x3);

    farfree(Ship.Bitmap);
    farfree(Ship.EraseBitmap);
    farfree(Ship.CollisionMap);
    farfree(BallBitmap);
    farfree(BallEraseBitmap);
    farfree(BallCollisionMap);

    return 0;
}