/* File: CLIPSPR
** Description:
**   Clips multiple individual sprites from larger PCX format files.
** Copyright:
**   Copyright 1994, David G. Roberts
** Syntax:
**   clipspr image[.pcx] [output-format [border-color]]
**   Where output-format equals:
**     pcx
**     planar
**     linear
**   Where border-color equals:
**     an integer from 0 to 255
*/

#include <alloc.h>
#include <assert.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gamedefs.h"
#include "filename.h"
#include "bitblt.h"
#include "palette.h"
#include "pcx.h"
#include "vga.h"

#define ERROR 1
#define NO_ERROR 0

#define BORDER_COLOR_DEFAULT	(255)

#define OUTPUT_PCX		0

#define OUTPUT_PLANAR	1
#define OUTPUT_LINEAR	2

#define DEFAULT_IMAGE_EXTENSION	".PCX"
#define SAVE_FILE_EXTENSION		".CSD"

#define USAGE_STRING \
	"Usage: clipspr filename[.ext] [output_format [border_color]]\n"\
    "  Where output_format is:\n"\
    "    'pcx', 'planar', or 'linear'\n"\
    "  Where border_color is:\n"\
    "    an integer from 0 to 255\n"

struct RLN {
	struct RLN	* Next;
    char		* Name;
    RECT		Rect;
    int			Flag;
};

typedef struct RLN RECT_LIST_NODE;

char *OutputFormatExtension[] = { ".pcx", ".pln", ".lin" };

/*
	Function: PrintTitle
    Description:
    	Prints program title, author, copyright info, etc.
*/
void PrintTitle(void)
{
	printf("Clip Sprite\n");
	printf("Version 1.0\n");
    printf("By Dave Roberts\n");
    printf("Copyright 1994, David G. Roberts, All rights reserved.\n");
    printf("------------------------------------------------------\n\n");
}

/*
	Function: ValidateParameters
    Description:
    	Validates and parses the parameters passed on the command
        line.  Returns a filename, a border color, and an output
        format to the calling routine via pointers to appropriate
		holding variables.  Returns NOERROR if all the parameters
        check out, ERROR otherwise.
*/
int ValidateParameters
	(
    int argc,
    char *argv[],
    char **ImageFile,
    int *BorderColor,
    int *OutputFormat
    )
{
	int Result;
    char * Extension;
    char * ExtendedFilename;
    char FormatString[10];

    /* validate number of parameters */
	if (argc < 2 || argc > 4) {
    	fprintf(stderr, "ERROR: Incorrect number of parameters.\n");
		fprintf(stderr, USAGE_STRING);
        return ERROR;
    }

    /* validate image filename */
    Extension = GetExtension(argv[1]);
    if (Extension == NULL) {
    	fprintf(stderr, "ERROR: Out of memory parsing filename param.\n");
        exit(1);
    }

    if (Extension[0] == '\0') {
    	/* if null extension, then allocate room for extension chars */
        ExtendedFilename = (char *) malloc(strlen(argv[1]) +
			strlen(".PCX") + 1);
    }
    else {
    	ExtendedFilename = (char *) malloc(strlen(argv[1]) + 1);
    }
	if (Extension == NULL) {
    	fprintf(stderr, "ERROR: Out of memory parsing filename param.\n");
        exit(1);
    }
    ExtendedFilename[0] = '\0';
    strcat(ExtendedFilename, argv[1]);
    if (Extension[0] == '\0') {
    	/* if null extension, then add the default extension (".PCX") */
    	strcat(ExtendedFilename, DEFAULT_IMAGE_EXTENSION);
    }

    *ImageFile = ExtendedFilename;
    free(Extension);

    /* validate output format */
    if (argc < 3) {
    	*OutputFormat = OUTPUT_PCX;		/* default to PCX */
    }
    else {
	    Result = sscanf(argv[2], "%9s", FormatString);
        if (Result != 1) {
        	fprintf(stderr, "ERROR: Problem interpreting format string.\n");
            fprintf(stderr, USAGE_STRING);
            return ERROR;
        }
        strlwr(FormatString); /* convert string to lower case */
        if (strcmp(FormatString, "pcx") == 0) {
        	*OutputFormat = OUTPUT_PCX;
        }
        else if (strcmp(FormatString, "planar") == 0 ||
			strcmp(FormatString, "pln") == 0) {
            *OutputFormat = OUTPUT_PLANAR;
        }
        else if (strcmp(FormatString, "linear") == 0 ||
			strcmp(FormatString, "lin") == 0) {
            *OutputFormat = OUTPUT_LINEAR;
        }
        else {
        	fprintf(stderr, "ERROR: Invalid format '%s'.\n", FormatString);
            fprintf(stderr, USAGE_STRING);
            return ERROR;
        }
	}


    /* validate border color */
    if (argc < 4) {
    	/* default to 255 */
    	*BorderColor = BORDER_COLOR_DEFAULT;
    }
    else {
    	Result = sscanf(argv[3], "%d", BorderColor);
    	if (Result != 1 || *BorderColor < 0 || *BorderColor > 255) {
    		fprintf(stderr, "ERROR: Border color must be numeric value "
				"from 0 to 255.\n");
        	fprintf(stderr, USAGE_STRING);
        	return ERROR;
    	}
    }
    return NO_ERROR;
}

/*
	Function: SavePlanar
    Description:
    	Save a bitmap out to disk in planar format.  Note that the
        input bitmap is a LINEAR_BITMAP and will be converted
        by the routine.  The format on disk is the binary
        representation in memory.
*/
int SavePlanar(char * Filename, LINEAR_BITMAP far * LinearBM)
{
	FILE * File;
    PLANAR_BITMAP far * PlanarBM;
    unsigned DataLength;
    int Result;
    unsigned i;
    UINT8 far * Data;

    PlanarBM = LinearToPlanar(LinearBM);
    if (PlanarBM == NULL) {
    	return 1; /* error */
    }

    /* open file */
    File = fopen(Filename, "wb");
    if (File == NULL) {
    	return 1; /* error */
    }

    /* write data */
    /*   note, we have to do this byte-by-byte because fwrite */
    /*   doesn't take a far pointer in the small memory model */
    DataLength = PlanarBM->Width * 4 * PlanarBM->Height +
		sizeof(PLANAR_BITMAP) - 1;
    Data = (UINT8 far *) PlanarBM;
    for (i = 0; i < DataLength; i++) {
    	Result = fputc(*Data++, File);
    	if (Result == EOF) {
    		/* write error */
    		fclose(File);
        	return 1;
    	}
    }

    /* close and exit */
    fclose(File);
    return 0;
}

/*
	Function: SaveLinear
    Description:
    	Saves a LINEAR_BITMAP to disk.  The format of the saved
        file is exactly the same as the binary image.
*/
int SaveLinear(char * Filename, LINEAR_BITMAP far * LinearBM)
{
	FILE * File;
    unsigned DataLength;
    int Result;
    unsigned i;
    UINT8 far * Data;

    /* open file */
    File = fopen(Filename, "wb");
    if (File == NULL) {
    	return 1; /* error */
    }

    /* write data */
    /*   note, we have to do this byte-by-byte because fwrite */
    /*   doesn't take a far pointer in the small memory model */
    DataLength = LinearBM->Width * LinearBM->Height +
		sizeof(LINEAR_BITMAP) - 1;
    Data = (UINT8 far *) LinearBM;
    for (i = 0; i < DataLength; i++) {
    	Result = fputc(*Data++, File);
    	if (Result == EOF) {
    		/* write error */
    		fclose(File);
        	return 1;
    	}
    }

    /* close and exit */
    fclose(File);
    return 0;
}

/*
	Function: GetPixel
    Description:
    	Gets a pixel value from a bitmap at the specified coordinates.
*/
int GetPixel(LINEAR_BITMAP far * SourceBM, int x, int y)
{
	UINT8 far * Data;
	unsigned Offset;

    assert(x <= SourceBM->Width);
    assert(y <= SourceBM->Height);

    Data = (UINT8 far *) &(SourceBM->Data);
    Offset = y * SourceBM->Width + x;

    return Data[Offset];
}

/*
	Function: SetPixel
    Description:
    	Sets a pixel value of a bitmap at the specified coordinates.
*/
void SetPixel(LINEAR_BITMAP far * SourceBM, int x, int y, int Color)
{
	UINT8 far * Data;
	unsigned Offset;

    assert(x <= SourceBM->Width);
    assert(y <= SourceBM->Height);

    Data = (UINT8 far *) &(SourceBM->Data);
    Offset = y * SourceBM->Width + x;

    Data[Offset] = Color;
}

/*
	Function: FindSprite
    Description:
    	Searches the source image bitmap until it finds a color
        matching the border color and then locates the corners
        of the bounding rectangle.
*/
RECT_LIST_NODE * FindSprite(LINEAR_BITMAP far * Source, int Border)
{
	unsigned Index;
    UINT8 far * Data;
    unsigned DataLen;
    int Top;
    int Left;
    int Bottom;
    int Bottom1;
    int Bottom2;
    int Right;
    int Right1;
    int Right2;
    RECT_LIST_NODE * Node;

    Data = (UINT8 far *) &(Source->Data);
    DataLen = Source->Width * Source->Height;

    /* try to find a pixel with the border color */
    Index = 0;
    while (Index < DataLen && Data[Index] != Border) {
    	Index++;
    }

    if (Index == DataLen) {
    	/* we didn't find a pixel of the bounding box color */
        return NULL;
    }

    /* convert Index to x,y location of pixel */
	Top = Index / Source->Width;
    Left = Index % Source->Width;
    Bottom = Bottom1 = Bottom2 = Top;
    Right = Right1 = Right2 = Left;
    /* blacken initially found pixel */
    SetPixel(Source, Left, Top, ~Border);

    /* the first pixel found is always the upper right (if the */
    /* border is really a rectangle), so now look for the lower */
    /* right corner */

    /* move to right from upper left */
    while (GetPixel(Source, Right1 + 1, Top) == Border) {
    	Right1++;
        SetPixel(Source, Right1, Top, ~Border);
    }

    /* move down from upper left */
    while (GetPixel(Source, Left, Bottom1 + 1) == Border) {
    	Bottom1++;
        SetPixel(Source, Left, Bottom1, ~Border);
    }

    /* move to right from lower left */
    while (GetPixel(Source, Right2 + 1, Bottom1) == Border) {
    	Right2++;
        SetPixel(Source, Right2, Bottom1, ~Border);
    }

    /* move down from upper right */
    while (GetPixel(Source, Right1, Bottom2 + 1) == Border) {
    	Bottom2++;
        SetPixel(Source, Right1, Bottom2, ~Border);
    }

    if (Right1 != Right2) {
    	fprintf(stderr, "WARNING: Found malformed sprite.\n");
    }

    if (Bottom1 != (Bottom2 + 1)) {
    	fprintf(stderr, "WARNING: Found malformed sprite.\n");
    }

    Right = MAX(Right1, Right2);
    Bottom = MAX(Bottom1, Bottom2);

    Node = (RECT_LIST_NODE *) malloc(sizeof(RECT_LIST_NODE));
    if (Node == NULL) {
    	fprintf(stderr, "ERROR: Out of memory while searching for a sprite.\n");
        exit(1);
    }

    Node->Rect.Top		= Top;
    Node->Rect.Left		= Left;
    Node->Rect.Bottom	= Bottom;
    Node->Rect.Right	= Right;

    return Node;
}

/*
	Function: FindAllSprites
    Description:
    	Finds all the sprites in the source bitmap and puts
        their bounding boxes into a linked list.
*/
RECT_LIST_NODE * FindAllSprites(LINEAR_BITMAP far * Source, int Border)
{
	RECT_LIST_NODE * RectList;
    RECT_LIST_NODE * NewNode;

    RectList = NULL;

    NewNode = FindSprite(Source, Border);
    while (NewNode != NULL) {
    	NewNode->Name		= NULL;
        NewNode->Flag		= FALSE;
    	NewNode->Next		= RectList;
        RectList			= NewNode;

        NewNode = FindSprite(Source, Border);
    }

    return RectList;
}

/*
	Function: LoadOldRects
    Description:
    	Takes the image filename as input, derives the name of
        the saved rectangle filename, and reads in the
        saved rectangles, if possible.  If the saved rectangle
        file doesn't exist, the routine returns NULL to the
        caller.  Otherwise, it returns a linked list of
        rectangles.
*/
RECT_LIST_NODE * LoadOldRects(char * ImageFile)
{
	char * Pathname;
    char * Root;
	char * SaveFilename;
	FILE * SaveFile;
    RECT_LIST_NODE * List;
    RECT_LIST_NODE * Node;
    int Result;
    int Top;
    int Left;
    int Bottom;
    int Right;
    char Name[MAX_ROOT_LEN + 1];

    /* derive save file name from ImageFile name */
    Pathname = GetPath(ImageFile, TRUE);
    Root = GetRoot(ImageFile);
    SaveFilename = (char *) malloc(strlen(Pathname) + strlen(Root) +
    	strlen(SAVE_FILE_EXTENSION) + 1);
    SaveFilename[0] = '\0';
    strcat(SaveFilename, Pathname);
    strcat(SaveFilename, Root);
    strcat(SaveFilename, SAVE_FILE_EXTENSION);
    free(Pathname);
    free(Root);
	printf("Save filename: %s\n", SaveFilename);

    /* open SaveFilename */
    SaveFile = fopen(SaveFilename, "rt");
    free(SaveFilename);

    /* if error, assume it doesn't exist and say we didn't see */
    /* any rectangles */
    if (SaveFile == NULL) {
    	return NULL;
    }

    /* otherwise, read lines in previous saved rectangle data */
    List = NULL;
    while (!feof(SaveFile)) {
		Result = fscanf(SaveFile,"%8s %d %d %d %d\n", Name, &Left, &Top,
			&Right, &Bottom);
        if (Result != 5 || Left > Right || Top > Bottom) {
        	fprintf(stderr, "ERROR: Savefile %s corrupted.\n", SaveFilename);
            exit(1);
        }
		Node = (RECT_LIST_NODE *) malloc(sizeof(RECT_LIST_NODE));
        Node->Rect.Top		= Top;
        Node->Rect.Left		= Left;
        Node->Rect.Bottom	= Bottom;
        Node->Rect.Right	= Right;
        Node->Next			= List;
        Node->Flag			= FALSE;
        Node->Name			= (char *) malloc(strlen(Name) + 1);
        Node->Name[0]		= '\0';
        strcat(Node->Name, Name);
        List				= Node;
    }

    /* close file and return list to caller */
    fclose(SaveFile);
	return List;
}

/*
	Function: PrintRects
    Description:
    	Print all sprites found.
*/
void PrintRects(RECT_LIST_NODE * Rects)
{
	printf("Found sprites at:\n");
	while (Rects != NULL) {
    	printf("  (%d,%d) (%d,%d)\n",
        	Rects->Rect.Left,
			Rects->Rect.Top,
            Rects->Rect.Right,
            Rects->Rect.Bottom);
        Rects = Rects->Next;
    }
}

/*
	Function: ExtractSprite
    Description:
    	Creates a new sprite composed of the pixels enclosed by the
        rectangle passed to the function.  A linear bitmap of the
        enclosed are is passed back to the caller, or NULL if
        some sort of error occurs
*/
LINEAR_BITMAP far * ExtractSprite
	(
    LINEAR_BITMAP far * Source,
    int Left,
    int Top,
    int Right,
    int Bottom
    )
{
	int Width;
    int Height;
    int WidthCounter;
    int HeightCounter;
    LINEAR_BITMAP far * NewBitmap;
    UINT8 far * Data;

    assert(Source != NULL);
    assert(Left < Right);
    assert(Top < Bottom);

    Width = Right - Left - 1;
    Height = Bottom - Top - 1;

    NewBitmap = (LINEAR_BITMAP far *) farmalloc(sizeof(LINEAR_BITMAP) +
    	Width * Height - 1); /* -1 for the data byte in the struct itself */

    NewBitmap->Width	= Width;
    NewBitmap->Height	= Height;
    NewBitmap->OriginX	= 0;
    NewBitmap->OriginY	= 0;

    Data = (UINT8 far *) &(NewBitmap->Data);
    for (HeightCounter = 0; HeightCounter < Height; HeightCounter++) {
    	for (WidthCounter = 0; WidthCounter < Width; WidthCounter++) {
        	*Data++ = GetPixel(Source, Left + WidthCounter + 1,
            	Top + HeightCounter + 1);
        }
    }

    return NewBitmap;
}

/*
	Function: SaveOldSprites
    Description:
    	Checks the sprite rectangles found in the image against the
        stored rectangle locations from last time and if matches
        are found, uses the stored names to save the sprites into
        individual files.
*/
void SaveOldSprites
	(
    int OutputFormat,
	RECT_LIST_NODE * SpriteRects,
	RECT_LIST_NODE * OldSpriteRects,
    LINEAR_BITMAP far * Image,
	UINT8 Palette[256][3]
	)
{
	RECT_LIST_NODE * NewRect;
    RECT_LIST_NODE * OldRect;
    LINEAR_BITMAP far * SpriteBitmap;
    char * Filename;
    int NumOld;
    int SaveResult;

    NumOld = 0;

    NewRect = SpriteRects;
    while (NewRect != NULL) {
    	OldRect = OldSpriteRects;
        while (OldRect != NULL) {
        	if (NewRect->Rect.Top == OldRect->Rect.Top &&
            	NewRect->Rect.Left == OldRect->Rect.Left) {
                break;
            }
        	OldRect = OldRect->Next;
        }
		if (OldRect != NULL) {
        	/* we found a match */
            NumOld++;
            printf("Found sprite: %s at (%d,%d) (%d,%d).\n", OldRect->Name,
				NewRect->Rect.Left, NewRect->Rect.Top,
				NewRect->Rect.Right, NewRect->Rect.Bottom);

            /* print warning if things are a little amiss */
            if (NewRect->Rect.Top != OldRect->Rect.Top ||
            	NewRect->Rect.Bottom != OldRect->Rect.Bottom) {
                printf("WARNING: New and old bitmaps differ in size.\n");
            }

            /* mark both as being dealt with */
            NewRect->Flag = TRUE;
            OldRect->Flag = TRUE;

            /* extract the bitmap from the image bitmap */
            SpriteBitmap = ExtractSprite(Image, NewRect->Rect.Left,
				NewRect->Rect.Top, NewRect->Rect.Right,
				NewRect->Rect.Bottom);

            /* save the extracted bitmap */
            Filename = (char *) malloc(strlen(OldRect->Name) +
				3 + 1 + 1); /* 3 for extension, 1 for '.', 1 for '\0' */
            Filename[0] = '\0';
            strcat(Filename, OldRect->Name);
            strcat(Filename, OutputFormatExtension[OutputFormat]);
            printf("Saving as: %s\n", Filename);
            switch (OutputFormat) {
            	case OUTPUT_PCX:
					SaveResult = SavePCX(Filename, SpriteBitmap, Palette);
                    break;
                case OUTPUT_PLANAR:
                	SaveResult = SavePlanar(Filename, SpriteBitmap);
                    break;
                case OUTPUT_LINEAR:
                	SaveResult = SaveLinear(Filename, SpriteBitmap);
                    break;
                default:
                	fprintf(stderr, "ERROR: Illegal format detected in "
                    	"'SaveOldSprites'.\n");
                    exit(1);
            }
            free(Filename);
            if (SaveResult != 0) {
            	fprintf(stderr, "ERROR: Problem while saving file '%s'.\n",
                	Filename);
                exit(1);
            }

            /* move name over to new sprite and off old sprite */
            NewRect->Name = OldRect->Name;
            OldRect->Name = NULL;

            /* free up the temp bitmap */
            farfree(SpriteBitmap);
        }
    	NewRect = NewRect->Next;
    }
	printf("Found %d old sprites.\n", NumOld);
}

/*
	Function: SaveNewSprites
    Description:
    	Saves all sprites that have been found but not handled by
        SaveOldSprites to disk in the required format.  Allows the
        user to preview the sprites if necessary to figure out what
        they are.
*/
void SaveNewSprites
	(
    int OutputFormat,
	RECT_LIST_NODE * SpriteRects,
	LINEAR_BITMAP far * ImageBitmap,
	UINT8 Palette[256][3]
	)
{
    RECT_LIST_NODE * NewSprite;
    char InLine[80];
    char RootName[MAX_ROOT_LEN + 1];
    char * Filename;
    LINEAR_BITMAP far * NewBitmap;
    int Skip;
    int Done;
    int SaveResult;

    NewSprite = SpriteRects;

    while (NewSprite != NULL) {
    	if (NewSprite->Flag != TRUE) {
            /* handle a new sprite */
            printf("New sprite at (%d,%d) (%d,%d).\n",
				NewSprite->Rect.Left, NewSprite->Rect.Top,
				NewSprite->Rect.Right, NewSprite->Rect.Bottom);

            NewBitmap = ExtractSprite(ImageBitmap, NewSprite->Rect.Left,
				NewSprite->Rect.Top, NewSprite->Rect.Right,
				NewSprite->Rect.Bottom);

            Done = FALSE;
            Skip = FALSE;
            do {
            	printf("Name sprite ('?' to view, '>' to skip) "
					"(no extension): ");

                fgets(InLine, 80, stdin);
                sscanf(InLine, "%8s", RootName);

                if (RootName[0] == '?') {
                	SetMode13h();
                    SetVGAPaletteBlock(Palette, 0, 256);
                    BltLinear(NewBitmap, 0, 0, MK_FP(VIDEO_MEM_SEGMENT,0));
                    getch();
                    SetVGAMode(0x3);
                }
                else if (RootName[0] == '>') {
                	Skip = TRUE;
                    Done = TRUE;
                }
                else if (strlen(RootName) > 0) {
					Done = TRUE;
                }
            } while (!Done);

            if (Skip) {
            	printf("Skipping...\n");
            }
            else {
            	/* save the extracted bitmap */
            	Filename = (char *) malloc(strlen(RootName) +
					3 + 1 + 1); /* 3 for extension, 1 for ".", 1 for '\0' */
            	Filename[0] = '\0';
            	strcat(Filename, RootName);
            	strcat(Filename, OutputFormatExtension[OutputFormat]);
            	printf("Saving as: %s\n", Filename);
	            switch (OutputFormat) {
    	        	case OUTPUT_PCX:
						SaveResult = SavePCX(Filename, NewBitmap, Palette);
            	        break;
                	case OUTPUT_PLANAR:
                		SaveResult = SavePlanar(Filename, NewBitmap);
	                    break;
    	            case OUTPUT_LINEAR:
        	        	SaveResult = SaveLinear(Filename, NewBitmap);
            	        break;
                	default:
                		fprintf(stderr, "ERROR: Illegal format detected in "
                    		"'SaveNewSprites'.\n");
	                    exit(1);
    	        }
            	free(Filename);
	            if (SaveResult != 0) {
    	        	fprintf(stderr, "ERROR: Problem while saving file '%s'.\n",
        	        	Filename);
            	    exit(1);
	            }

            	/* save name away for later */
            	NewSprite->Name = (char *) malloc(strlen(RootName) + 1);
            	NewSprite->Name[0] = '\0';
            	strcat(NewSprite->Name, RootName);
            	NewSprite->Flag = TRUE;
            }

            farfree(NewBitmap);
        }
        NewSprite = NewSprite->Next;
    }
}

/*
	Function: WriteSaveFile
    Description:
    	Writes the save file out to disk.  The save file contains the
        locations and coordinates of all the sprites found and saved
        in this particular run and will be used again to find and
        save the updated sprites in the next run.
*/
void WriteSaveFile(RECT_LIST_NODE * Rects, char * ImageFilename)
{
	char * Pathname;
    char * Root;
	char * SaveFilename;
	FILE * SaveFile;
    RECT_LIST_NODE * List;

    /* derive save file name from ImageFile name */
    Pathname = GetPath(ImageFilename, TRUE);
    Root = GetRoot(ImageFilename);
    SaveFilename = (char *) malloc(strlen(Pathname) + strlen(Root) +
    	strlen(SAVE_FILE_EXTENSION) + 1);
    SaveFilename[0] = '\0';
    strcat(SaveFilename, Pathname);
    strcat(SaveFilename, Root);
    strcat(SaveFilename, SAVE_FILE_EXTENSION);
    free(Pathname);
    free(Root);
	printf("Writing save file: %s\n", SaveFilename);

    /* open SaveFilename */
    SaveFile = fopen(SaveFilename, "wt");
    free(SaveFilename);

    /* if error, exit */
    if (SaveFile == NULL) {
    	return;
    }

    /* write out rectangle data */
    List = Rects;
    while (List != NULL) {
    	if (List->Flag) {
    		fprintf(SaveFile, "%s %d %d %d %d\n", List->Name,
				List->Rect.Left, List->Rect.Top,
				List->Rect.Right, List->Rect.Bottom);
        }
        List = List->Next;
    }

    /* close file and return to caller */
    fclose(SaveFile);
}

int main(int argc, char *argv[])
{
	LINEAR_BITMAP far * ImageBitmap;
    char * ImageFile;
    int BorderColor;
    int OutputFormat;
    RECT_LIST_NODE * SpriteRects;
    RECT_LIST_NODE * OldSpriteRects;
    UINT8 Palette[256][3];

    /* print title stuff */
    PrintTitle();

	/* Check format of all parameters */
    if (ValidateParameters(argc, argv, &ImageFile, &BorderColor,
		&OutputFormat) == ERROR) {
    	return 1;
    }

	/* Load image file as a large linear bitmap */
	ImageBitmap = LoadPCX(ImageFile, Palette);
    if (ImageBitmap == NULL) {
    	fprintf(stderr, "ERROR: Can't open %s\n", ImageFile);
        return 1;
    }

	/* scan though large bitmap and find all bitmaps based on */
	/* boundary color value */
    SpriteRects = FindAllSprites(ImageBitmap, BorderColor);
    if (SpriteRects == NULL) {
    	printf("No sprites found.\n");
        return 1;
    }
    else {
		PrintRects(SpriteRects);
    }

	/* see if "image.sav" exists */
	OldSpriteRects = LoadOldRects(ImageFile);

	/* for each sprite found in the image, see if we've dealt with */
	/* it previously and if so, save it to disk */
    SaveOldSprites(OutputFormat, SpriteRects, OldSpriteRects,
		ImageBitmap, Palette);

	/* for each sprite that was not dealt with previously, */
    SaveNewSprites(OutputFormat, SpriteRects, ImageBitmap, Palette);
		/* show the sprite to the user */
		/* see if there is an old sprite location close to the position of this one, suggest the old sprite name */
			/* as this current name */
		/* get the new name from the user */
		/* save sprite under new name in the current format */
		/* generate a sprite position history record */
	/* rewrite the "image.sav" file with all the sprite position history records */
    WriteSaveFile(SpriteRects, ImageFile);

	/* exit */
    farfree(ImageBitmap);
    free(ImageFile);

    return 0;
}